<?php
/**
 * Plugin Name: MxChat Multi-Bot Manager
 * Plugin URI: https://mxchat.ai/multi-bot
 * Description: Create and manage multiple specialized chatbots with dedicated knowledge databases for different purposes (support, sales, technical, etc.).
 * Version: 1.0.2
 * Author: MxChat
 * Author URI: https://mxchat.ai
 * Text Domain: mxchat-multi-bot
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * License: Commercial
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MXCHAT_MULTI_BOT_VERSION', '1.0.2');
define('MXCHAT_MULTI_BOT_PLUGIN_FILE', __FILE__);
define('MXCHAT_MULTI_BOT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_MULTI_BOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_MULTI_BOT_PLUGIN_BASENAME', plugin_basename(__FILE__));

function is_mxchat_active_multi_bot() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_multi_bot()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Multi-Bot Manager</strong> requires the MxChat plugin to be installed and active.', 'mxchat-multi-bot'); ?></p>
        </div>
        <?php
    });
    return;
}

// Include update checker
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerMultiBot::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-multi-bot'
]);

/**
 * Main MxChat Multi-Bot Manager class
 */
class MxChat_Multi_Bot_Manager {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Options
     */
    private $options;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->options = get_option('mxchat_options', array());
        
        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        
        // Only load features if pro is active
        if ($this->is_license_active()) {
            $this->init_components();
        }
        
        // Always add license management hooks
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Check if MxChat Pro is activated
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }
    
    /**
     * Admin notice for pro requirement
     */
    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Multi-Bot Manager requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-multi-bot'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }
    
    /**
     * Initialize components (only if license is active)
     */
    private function init_components() {
        // Load dependencies
        $this->load_dependencies();
        
        // Initialize hooks
        $this->init_hooks();
    }
    
    /**
     * Initialize the plugin
     */
    public function init() {
        // Load text domain
        load_plugin_textdomain('mxchat-multi-bot', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        require_once MXCHAT_MULTI_BOT_PLUGIN_DIR . 'includes/class-database.php';
        require_once MXCHAT_MULTI_BOT_PLUGIN_DIR . 'includes/class-multi-bot-manager.php';
        require_once MXCHAT_MULTI_BOT_PLUGIN_DIR . 'includes/class-admin-interface.php';
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Initialize database
        MxChat_Multi_Bot_Database::get_instance();
        
        // Initialize core manager
        MxChat_Multi_Bot_Core_Manager::get_instance();
        
        // Initialize admin interface
        if (is_admin()) {
            MxChat_Multi_Bot_Admin::get_instance();
        }
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Add database integrity check for admin users
        add_action('admin_init', array($this, 'check_database_integrity'));
        
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'mxchat-max',
            __('Multi-Bot Manager', 'mxchat-multi-bot'),
            __('Multi-Bot Manager', 'mxchat-multi-bot'),
            'manage_options',
            'mxchat-multi-bot',
            array($this, 'admin_page_callback')
        );
    }
    
    /**
     * Admin page callback
     */
    public function admin_page_callback() {
        // Get the admin interface instance and display the page
        $admin = MxChat_Multi_Bot_Admin::get_instance();
        $admin->bot_list_page();
    }
    
    /**
     * Check database integrity
     */
    public function check_database_integrity() {
        // Only run for admin users
        if (!current_user_can('manage_options')) {
            return;
        }
        
        global $wpdb;
        $multi_bots_table = $wpdb->prefix . 'mxchat_multi_bots';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$multi_bots_table}'");
        
        // If table doesn't exist, show admin notice
        if (!$table_exists) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php _e('<strong>MxChat Multi-Bot Manager</strong> database issue detected. Please deactivate and reactivate the plugin to resolve this issue.', 'mxchat-multi-bot'); ?></p>
                </div>
                <?php
            });
        }
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables
        require_once MXCHAT_MULTI_BOT_PLUGIN_DIR . 'includes/class-database.php';
        MxChat_Multi_Bot_Database::create_tables();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up transients and caches
        delete_transient('mxchat_multi_bot_license_check');
    }
    
    /**
     * Get license status
     */
    public function is_licensed() {
        return $this->is_license_active();
    }
    
    /**
     * Get plugin version
     */
    public function get_version() {
        return MXCHAT_MULTI_BOT_VERSION;
    }
    
    /**
     * Prevent cloning
     */
    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-multi-bot'), '1.0.2');
    }
    
    /**
     * Prevent unserializing
     */
    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-multi-bot'), '1.0.2');
    }
}

/**
 * Returns the main instance of MxChat_Multi_Bot_Manager
 */
function MxChat_Multi_Bot_Manager() {
    return MxChat_Multi_Bot_Manager::get_instance();
}

// Initialize the plugin
add_action('plugins_loaded', 'MxChat_Multi_Bot_Manager');