/**
 * MxChat Perplexity Integration Admin JavaScript
 */
(function($) {
    'use strict';

    // DOM Ready
    $(function() {
        // Initialize variables
        const apiKeyField = $('#mxchat_perplexity_api_key');
        const responseMode = $('#mxchat_perplexity_response_mode');
        const modelSelect = $('#mxchat_perplexity_model');
        const testApiButton = $('#mxchat_perplexity_test_api');
        const statusContainer = $('.mxchat-perplexity-status-container');
        
        // Add password visibility toggle
        if (apiKeyField.length) {
            setupPasswordToggle();
        }
        
        // Test API connection
        if (testApiButton.length) {
            testApiButton.on('click', function(e) {
                e.preventDefault();
                testApiConnection();
            });
        }
        
        // Show/hide specific options based on response mode
        if (responseMode.length) {
            responseMode.on('change', function() {
                toggleResponseModeOptions($(this).val());
            });
            
            // Initialize visibility on page load
            toggleResponseModeOptions(responseMode.val());
        }

        /**
         * Setup password visibility toggle for API key field
         */
        function setupPasswordToggle() {
            // Find the existing toggle button
            const toggleButton = $('.mxchat-toggle-visibility');
            const apiKeyField = $('#mxchat_perplexity_api_key');
            
            // Handle toggle button click
            toggleButton.on('click', function() {
                const inputType = apiKeyField.attr('type');
                
                if (inputType === 'password') {
                    apiKeyField.attr('type', 'text');
                    toggleButton.find('.dashicons')
                        .removeClass('dashicons-visibility')
                        .addClass('dashicons-hidden');
                } else {
                    apiKeyField.attr('type', 'password');
                    toggleButton.find('.dashicons')
                        .removeClass('dashicons-hidden')
                        .addClass('dashicons-visibility');
                }
            });
        }
        
        /**
         * Toggle options based on response mode
         */
        function toggleResponseModeOptions(mode) {
            const structuredOptions = $('.mxchat-perplexity-structured-options');
            
            if (mode === 'structured') {
                structuredOptions.fadeIn(300);
            } else {
                structuredOptions.fadeOut(300);
            }
        }
        
        /**
         * Test API connection
         */
        function testApiConnection() {
            const apiKey = apiKeyField.val();
            
            if (!apiKey) {
                showStatus('error', 'Please enter an API key before testing the connection.');
                return;
            }
            
            // Add loading indicator to button
            testApiButton.prop('disabled', true);
            const originalText = testApiButton.text();
            testApiButton.html('<span class="mxchat-perplexity-loading"></span> ' + 'Testing...');
            
            // Make AJAX request to test the connection
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_perplexity_test_api',
                    api_key: apiKey,
                    model: modelSelect.val(),
                    nonce: mxchatPerplexity.nonce
                },
                success: function(response) {
                    if (response.success) {
                        showStatus('success', 'API connection successful! The Perplexity API is working correctly.');
                    } else {
                        showStatus('error', 'API connection failed: ' + (response.data || 'Unknown error'));
                    }
                },
                error: function() {
                    showStatus('error', 'API connection test failed. Please check your connection and try again.');
                },
                complete: function() {
                    // Reset button state
                    testApiButton.prop('disabled', false);
                    testApiButton.text(originalText);
                }
            });
        }
        
        /**
         * Show status message
         * 
         * @param {string} type - 'success', 'error', or 'warning'
         * @param {string} message - The message to display
         */
        function showStatus(type, message) {
            const iconClass = type === 'success' ? 'dashicons-yes-alt' : 
                            (type === 'warning' ? 'dashicons-warning' : 'dashicons-no-alt');
            
            const statusHtml = `
                <div class="mxchat-perplexity-status ${type}">
                    <span class="dashicons ${iconClass} mxchat-perplexity-status-icon"></span>
                    <span>${message}</span>
                </div>
            `;
            
            // Clear any existing status and add the new one
            statusContainer.html(statusHtml);
            
            // Auto-dismiss success messages after 5 seconds
            if (type === 'success') {
                setTimeout(function() {
                    statusContainer.find('.mxchat-perplexity-status').fadeOut(500, function() {
                        $(this).remove();
                    });
                }, 5000);
            }
        }
        
        /**
         * Initialize tooltip help icons
         */
        function initTooltips() {
            $('.mxchat-perplexity-help-tip').on('click', function(e) {
                e.preventDefault();
            }).each(function() {
                $(this).tipTip({
                    'attribute': 'data-tip',
                    'fadeIn': 50,
                    'fadeOut': 50,
                    'delay': 200,
                    'keepAlive': true,
                    'maxWidth': '300px'
                });
            });
        }
        
        // Initialize tooltips if the TipTip plugin is available
        if (typeof $.fn.tipTip !== 'undefined') {
            initTooltips();
        }
        
        /**
         * Save settings via AJAX (optional functionality)
         */
        $('#mxchat-perplexity-ajax-save').on('click', function(e) {
            e.preventDefault();
            
            const form = $(this).closest('form');
            const formData = form.serialize();
            
            // Add loading state
            $(this).prop('disabled', true);
            const originalText = $(this).text();
            $(this).html('<span class="mxchat-perplexity-loading"></span> Saving...');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: formData + '&action=mxchat_perplexity_save_settings&nonce=' + mxchatPerplexity.nonce,
                success: function(response) {
                    if (response.success) {
                        showStatus('success', 'Settings saved successfully!');
                    } else {
                        showStatus('error', 'Failed to save settings: ' + (response.data || 'Unknown error'));
                    }
                },
                error: function() {
                    showStatus('error', 'Failed to save settings due to a server error. Please try again.');
                },
                complete: function() {
                    // Reset button state
                    $('#mxchat-perplexity-ajax-save').prop('disabled', false);
                    $('#mxchat-perplexity-ajax-save').text(originalText);
                }
            });
        });
    });
})(jQuery);