// mxchat-perplexity.js
(function($) {
    'use strict';
    
    // Debug flag for additional logging
    const DEBUG = true;
    
    // Log function with prefix for easier debugging
    function debugLog(message) {
        if (DEBUG) {
            console.log('🔍 PERPLEXITY: ' + message);
        }
    }
    
    // Add a console message on script load
    debugLog('MxChat Perplexity script loaded!');
    
    // Flag to track if web search mode is enabled
    let webSearchEnabled = false;
    
    // Initialize when document is ready
    $(document).ready(function() {
        debugLog('Document ready, initializing Perplexity integration');
        
        // Create the status indicator container after the button
        createStatusIndicator();
        
        // Try to find the button immediately
        if ($('#perplexity-search-btn').length > 0) {
            debugLog('Perplexity button found immediately!');
            initPerplexityButton();
        } else {
            // Set up a MutationObserver to watch for the button
            debugLog('Button not found, setting up observer');
            setupButtonObserver();
            
            // Also try with timeouts as a fallback
            setTimeout(checkForButton, 1000);
        }
        
        // IMPORTANT: Hook into the core plugin's AJAX calls
        setupCorePluginIntegration();
    });
    
    // Create the status indicator container
    function createStatusIndicator() {
        // Check if the perplexity button exists
        const perplexityBtn = $('#perplexity-search-btn');
        if (perplexityBtn.length === 0) {
            debugLog('Perplexity button not found, delaying status indicator creation');
            setTimeout(createStatusIndicator, 1000);
            return;
        }
        
        // Create status indicator HTML if it doesn't exist
        if ($('#active-perplexity-container').length === 0) {
            debugLog('Creating status indicator container');
            
            const statusHtml = `
                <div id="active-perplexity-container" class="active-pdf-container" style="display: none;">
                    <span id="active-perplexity-status" class="active-pdf-name">Perplexity Search Active</span>
                </div>
            `;
            
            // Insert after the perplexity button
            perplexityBtn.after(statusHtml);
        }
    }
    
    // Set cookie helper function
    function setCookie(name, value, days) {
        let expires = "";
        if (days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toUTCString();
        }
        document.cookie = name + "=" + (value || "") + expires + "; path=/";
        debugLog('Cookie set: ' + name + '=' + value);
    }
    
    // Get cookie helper function
    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for (let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }
    
    // Check for the button with increasing delays
    function checkForButton(attempt = 1) {
        if ($('#perplexity-search-btn').length > 0) {
            debugLog('Perplexity button found on attempt ' + attempt);
            initPerplexityButton();
            return;
        }
        
        if (attempt < 5) {
            debugLog('Button not found, retry attempt ' + (attempt + 1));
            setTimeout(() => checkForButton(attempt + 1), 1000 * attempt);
        } else {
            debugLog('Failed to find button after multiple attempts');
        }
    }
    
    // Set up observer to watch for the button being added
    function setupButtonObserver() {
        const observer = new MutationObserver(function(mutations) {
            if ($('#perplexity-search-btn').length > 0) {
                debugLog('Perplexity button found via observer!');
                initPerplexityButton();
                observer.disconnect();
            }
        });
        
        observer.observe(document.body, { childList: true, subtree: true });
    }
    
    // Initialize the button functionality
    function initPerplexityButton() {
        const perplexityButton = $('#perplexity-search-btn');
        
        if (perplexityButton.length === 0) {
            debugLog('Button disappeared, something went wrong');
            return;
        }
        
        debugLog('Initializing button with ID: ' + perplexityButton.attr('id'));
        
        // Create the status indicator if it doesn't exist
        createStatusIndicator();
        
        // Load saved state from localStorage and cookies
        const localStorageState = localStorage.getItem('mxchat_perplexity_enabled') === 'true';
        const cookieState = getCookie('mxchat_perplexity_enabled') === 'true';
        
        // Use either source of truth
        webSearchEnabled = localStorageState || cookieState;
        
        debugLog('Loaded saved state: ' + (webSearchEnabled ? 'enabled' : 'disabled'));
        
        // Update button appearance
        updateButtonState();
        
        // Handle button click
        perplexityButton.off('click').on('click', function(e) {
            debugLog('Button clicked!');
            e.preventDefault();
            e.stopPropagation();
            
            // Toggle state
            setWebSearchEnabled(!webSearchEnabled);
            
            return false;
        });
        
        debugLog('Button initialization complete');
    }
    
    // Helper function to set the web search state
    function setWebSearchEnabled(enabled) {
        webSearchEnabled = enabled;
        
        // Save state to localStorage
        localStorage.setItem('mxchat_perplexity_enabled', webSearchEnabled);
        
        // Save to cookie as well (for PHP to detect)
        setCookie('mxchat_perplexity_enabled', webSearchEnabled ? 'true' : 'false', 7);
        
        // Update button appearance
        updateButtonState();
    }
    
    // Update the button's visual state
function updateButtonState() {
    const button = $('#perplexity-search-btn');
    const statusContainer = $('#active-perplexity-container');
    
    if (webSearchEnabled) {
        // Update button
        button.addClass('active');
        button.attr('title', 'Perplexity Search Mode Active - Click to Disable');
        
        // Show status indicator with flex display and add active class
        if (statusContainer.length) {
            statusContainer.css({
                'display': 'flex',
                'align-items': 'center',
                'gap': '8px'
            }).addClass('active');
        }
        
        debugLog('Button set to active state');
    } else {
        // Update button
        button.removeClass('active');
        button.attr('title', 'Web Search Mode - Click to Enable');
        
        // Hide status indicator and remove active class
        if (statusContainer.length) {
            statusContainer.css('display', 'none').removeClass('active');
        }
        
        debugLog('Button set to inactive state');
    }
}
    
    // This is the critical function that hooks into the core plugin's AJAX calls
    function setupCorePluginIntegration() {
        debugLog('Setting up core plugin integration');
        
        // Method 1: Hook into jQuery's ajaxSend event to intercept ALL AJAX calls
        $(document).ajaxSend(function(event, jqXHR, settings) {
            // Check if settings.data exists
            if (!settings.data) {
                return;
            }
            
            // FIX: Handle both string and object data formats
            let isMxChatRequest = false;
            
            if (typeof settings.data === 'string') {
                // If data is a string, check if it contains the action parameter
                isMxChatRequest = settings.data.indexOf('action=mxchat_handle_chat_request') !== -1;
            } else if (typeof settings.data === 'object') {
                // If data is an object, check for action property
                isMxChatRequest = settings.data.action === 'mxchat_handle_chat_request';
            }
            
            // Only proceed if this is a mxchat request
            if (isMxChatRequest) {
                debugLog('Intercepted core plugin AJAX call');
                
                // If web search is enabled, add the flag to the request
                if (webSearchEnabled) {
                    debugLog('Adding perplexity_web_search=true to request');
                    
                    // Handle different data formats
                    if (typeof settings.data === 'string') {
                        // Append the parameter to the existing data string
                        if (settings.data.indexOf('perplexity_web_search=') === -1) {
                            settings.data += '&perplexity_web_search=true';
                        }
                    } else if (typeof settings.data === 'object') {
                        // Add property to data object
                        settings.data.perplexity_web_search = 'true';
                    }
                }
            }
        });
        
        // Method 2: Override the send button click handler (more direct approach)
        const originalSendButtonHandler = typeof window.mxchatSendMessage === 'function' ? 
            window.mxchatSendMessage : null;
            
        if (originalSendButtonHandler) {
            debugLog('Found original send button handler, overriding');
            
            window.mxchatSendMessage = function(messageText) {
                // If web search is enabled, add our flag to window.mxchatAjaxData
                if (webSearchEnabled && window.mxchatAjaxData) {
                    debugLog('Adding perplexity_web_search=true to mxchatAjaxData');
                    window.mxchatAjaxData.perplexity_web_search = 'true';
                }
                
                // Call the original handler
                return originalSendButtonHandler(messageText);
            };
        } else {
            // Try to find and override the click handler on the send button
            $('#send-button').off('click.perplexity').on('click.perplexity', function(e) {
                // Let the event continue, but add our flag to any form data
                if (webSearchEnabled) {
                    debugLog('Adding hidden input for perplexity_web_search');
                    
                    // Add a hidden input to the form if it doesn't exist
                    if ($('input[name="perplexity_web_search"]').length === 0) {
                        $('body').append('<input type="hidden" name="perplexity_web_search" value="true">');
                    } else {
                        $('input[name="perplexity_web_search"]').val('true');
                    }
                }
            });
        }
        
        // Method 3: Also try to hook directly into any chat-related AJAX functions
        if (typeof $.fn.mxchatSubmitMessage === 'function') {
            debugLog('Found mxchatSubmitMessage function, overriding');
            
            const originalSubmitMessage = $.fn.mxchatSubmitMessage;
            $.fn.mxchatSubmitMessage = function(options) {
                if (webSearchEnabled && options && options.data) {
                    debugLog('Adding perplexity_web_search=true to mxchatSubmitMessage data');
                    options.data.perplexity_web_search = 'true';
                }
                return originalSubmitMessage.apply(this, arguments);
            };
        }
    }
    
    // Add a user message to the chat
    function addUserMessage(message) {
        const chatContainer = $('#chat-box');
        if (chatContainer.length === 0) {
            debugLog('Chat container not found');
            return;
        }
        
        const userMessage = $('<div class="user-message"></div>').text(message);
        chatContainer.append(userMessage);
        chatContainer.scrollTop(chatContainer[0].scrollHeight);
    }
    
    // Add a system message
    function addSystemMessage(message) {
        const chatContainer = $('#chat-box');
        if (chatContainer.length === 0) {
            debugLog('Chat container not found');
            return;
        }
        
        const systemMessage = $('<div class="system-message"></div>').text(message);
        chatContainer.append(systemMessage);
        chatContainer.scrollTop(chatContainer[0].scrollHeight);
        debugLog('Added system message: ' + message);
    }
    
})(jQuery);