<?php
/**
 * Admin class for the Perplexity integration
 */
class MxChat_Perplexity_Admin {

    private $loader;
    private $option_group = 'mxchat_perplexity_options';

public function __construct($loader) {
        $this->loader = $loader;
        
        // Add settings page
        $this->loader->add_action('admin_init', $this, 'register_settings');
        // Add admin assets (CSS and JS)
        $this->loader->add_action('admin_enqueue_scripts', $this, 'enqueue_admin_assets');
    }

    /**
     * Enqueue admin assets (CSS and JS)
     */
    public function enqueue_admin_assets($hook) {
        // Log the hook for verification
        //error_log('MxChat Perplexity Admin Hook: ' . $hook);

        // Check for the correct hook
        if ($hook !== 'mxchat_page_mxchat-perplexity') {
            //error_log('MxChat Perplexity: Not enqueuing assets, wrong hook: ' . $hook);
            return;
        }

        // Enqueue CSS
        $css_url = MXCHAT_PERPLEXITY_PLUGIN_URL . 'assets/css/mxchat-perplexity-admin.css';
        //error_log('MxChat Perplexity: Attempting to enqueue CSS at: ' . $css_url);
        wp_enqueue_style(
            'mxchat-perplexity-admin',
            $css_url,
            array(),
            MXCHAT_PERPLEXITY_VERSION,
            'all'
        );
        if (wp_style_is('mxchat-perplexity-admin', 'enqueued')) {
            //error_log('MxChat Perplexity: CSS successfully enqueued');
        } else {
            //error_log('MxChat Perplexity: Failed to enqueue CSS');
        }

        // Enqueue JS
        $js_url = MXCHAT_PERPLEXITY_PLUGIN_URL . 'assets/js/mxchat-perplexity-admin.js';
        //error_log('MxChat Perplexity: Attempting to enqueue JS at: ' . $js_url);
        wp_enqueue_script(
            'mxchat-perplexity-admin',
            $js_url,
            array('jquery'), // Add dependencies if needed (jQuery is common)
            MXCHAT_PERPLEXITY_VERSION,
            true // Load in footer
        );
        if (wp_script_is('mxchat-perplexity-admin', 'enqueued')) {
            //error_log('MxChat Perplexity: JS successfully enqueued');
        } else {
            //error_log('MxChat Perplexity: Failed to enqueue JS');
        }
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting(
            $this->option_group,
            'mxchat_perplexity_api_key',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '',
            )
        );

        register_setting(
            $this->option_group,
            'mxchat_perplexity_response_mode',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'conversational',
            )
        );

        register_setting(
            $this->option_group,
            'mxchat_perplexity_model',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'sonar',
            )
        );
        
register_setting(
    $this->option_group,
    'mxchat_perplexity_show_logo',
    array(
        'type' => 'boolean',
        'sanitize_callback' => function($value) {
            // Convert various input values to a standardized format
            if ($value === '1' || $value === 'on' || $value === 'true' || $value === true) {
                return '1'; // Store as '1' for consistency
            }
            return ''; // Empty string for false/unchecked
        },
        'default' => true,
    )
);


        add_settings_section(
            'mxchat_perplexity_api_section',
            __('Perplexity API Settings', 'mxchat-perplexity'),
            array($this, 'api_settings_section_callback'),
            'mxchat-perplexity'
        );

        add_settings_field(
            'mxchat_perplexity_api_key',
            __('API Key', 'mxchat-perplexity'),
            array($this, 'api_key_field_callback'),
            'mxchat-perplexity',
            'mxchat_perplexity_api_section'
        );

        add_settings_field(
            'mxchat_perplexity_model',
            __('Model', 'mxchat-perplexity'),
            array($this, 'model_field_callback'),
            'mxchat-perplexity',
            'mxchat_perplexity_api_section'
        );

        add_settings_field(
            'mxchat_perplexity_response_mode',
            __('Response Mode', 'mxchat-perplexity'),
            array($this, 'response_mode_field_callback'),
            'mxchat-perplexity',
            'mxchat_perplexity_api_section'
        );
        
        add_settings_field(
            'mxchat_perplexity_show_logo',
            __('Perplexity Logo', 'mxchat-perplexity'),
            array($this, 'show_logo_field_callback'),
            'mxchat-perplexity',
            'mxchat_perplexity_api_section'
        );

        add_settings_section(
            'mxchat_perplexity_usage_section',
            __('Usage Instructions', 'mxchat-perplexity'),
            array($this, 'usage_section_callback'),
            'mxchat-perplexity'
        );
    }

    /**
     * API settings section callback
     */
    public function api_settings_section_callback() {
        echo '<p>' . __('Configure your Perplexity API settings to enable web search capabilities in your chatbot.', 'mxchat-perplexity') . '</p>';
    }

    /**
     * Usage section callback
     */
    public function usage_section_callback() {
        echo '<p>' . __('Learn how to set up and use the Perplexity integration with your MxChat chatbot.', 'mxchat-perplexity') . '</p>';
    }

    /**
     * API key field callback
     */
    public function api_key_field_callback() {
        $api_key = get_option('mxchat_perplexity_api_key', '');
        ?>
        <input type="password" 
               id="mxchat_perplexity_api_key" 
               name="mxchat_perplexity_api_key" 
               value="<?php echo esc_attr($api_key); ?>" 
               class="regular-text" />
        <p class="description">
            <?php echo sprintf(
                __('Enter your Perplexity API key. Need an API key? <a href="%s" target="_blank">Get one here</a>.', 'mxchat-perplexity'),
                'https://docs.perplexity.ai/guides/getting-started'
            ); ?>
        </p>
        <?php
    }

/**
 * Model field callback
 */
public function model_field_callback() {
    $model = get_option('mxchat_perplexity_model', 'sonar');
    ?>
    <select id="mxchat_perplexity_model" name="mxchat_perplexity_model">
        <option value="sonar-pro" <?php selected($model, 'sonar-pro'); ?>>
            <?php _e('Sonar Pro (200k context)', 'mxchat-perplexity'); ?>
        </option>
        <option value="sonar-reasoning" <?php selected($model, 'sonar-reasoning'); ?>>
            <?php _e('Sonar Reasoning (128k context)', 'mxchat-perplexity'); ?>
        </option>
        <option value="sonar-reasoning-pro" <?php selected($model, 'sonar-reasoning-pro'); ?>>
            <?php _e('Sonar Reasoning Pro (128k context)', 'mxchat-perplexity'); ?>
        </option>
        <option value="sonar-deep-research" <?php selected($model, 'sonar-deep-research'); ?>>
            <?php _e('Sonar Deep Research (128k context)', 'mxchat-perplexity'); ?>
        </option>
    </select>
    <p class="description">
        <?php _e('Select which Perplexity model to use for searches. Different models have different capabilities and costs.', 'mxchat-perplexity'); ?>
    </p>
    <?php
}
    /**
     * Response mode field callback
     */
    public function response_mode_field_callback() {
        $mode = get_option('mxchat_perplexity_response_mode', 'conversational');
        ?>
        <select id="mxchat_perplexity_response_mode" name="mxchat_perplexity_response_mode">
            <option value="conversational" <?php selected($mode, 'conversational'); ?>>
                <?php _e('Conversational', 'mxchat-perplexity'); ?>
            </option>
            <option value="structured" <?php selected($mode, 'structured'); ?>>
                <?php _e('Structured (with sources)', 'mxchat-perplexity'); ?>
            </option>
        </select>
        <p class="description">
            <?php _e('Select how the search results should be presented to the user. Conversational mode provides a more natural response, while structured mode includes source citations.', 'mxchat-perplexity'); ?>
        </p>
        <?php
    }
    
    /**
     * Toggle field callback for showing/hiding Perplexity logo
     */
    public function show_logo_field_callback() {
        $show_logo = get_option('mxchat_perplexity_show_logo', true);
        ?>
        <div class="mxchat-toggle-container">
            <label class="mxchat-toggle-switch">
                <input type="checkbox"
                       id="mxchat_perplexity_show_logo"
                       name="mxchat_perplexity_show_logo"
                       value="1"
                       <?php checked($show_logo); ?>>
                <span class="mxchat-toggle-slider"></span>
            </label>
            <span class="mxchat-toggle-label">
                <?php esc_html_e('Show Perplexity Logo', 'mxchat-perplexity'); ?>
            </span>
        </div>
        <p class="description">
            <?php esc_html_e('Enable to show the Perplexity logo in the chatbot interface. Disable to hide it.', 'mxchat-perplexity'); ?>
        </p>
        <?php
    }

    /**
     * Display the admin page
     */
    public function display_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap mxchat-wrapper">
            <!-- Hero Section -->
            <div class="mxchat-hero">
                <h1 class="mxchat-main-title">
                    <span class="mxchat-gradient-text">Perplexity</span> Integration
                </h1>
                <p class="mxchat-hero-subtitle">
                    <?php esc_html_e('Enhance your AI chatbot with real-time web search capabilities using Perplexity. Keep your responses current and factual with the latest information.', 'mxchat-perplexity'); ?>
                </p>
            </div>

            <div class="mxchat-content">
                <!-- API Settings Card -->
                <div class="mxchat-card">
                    <h2><?php esc_html_e('Perplexity API Settings', 'mxchat-perplexity'); ?></h2>
                    <p><?php esc_html_e('Configure your Perplexity API settings to enable web search capabilities in your chatbot.', 'mxchat-perplexity'); ?></p>
                    
                    <form action="options.php" method="post">
                        <?php settings_fields($this->option_group); ?>
                        
                        <div class="mxchat-form-group">
                            <label for="mxchat_perplexity_api_key"><?php esc_html_e('API Key', 'mxchat-perplexity'); ?></label>
                            <div class="mxchat-api-key-field">
                                <input type="password" 
                                    id="mxchat_perplexity_api_key" 
                                    name="mxchat_perplexity_api_key" 
                                    value="<?php echo esc_attr(get_option('mxchat_perplexity_api_key', '')); ?>" 
                                    class="regular-text" />
                                <button type="button" class="mxchat-toggle-visibility">
                                    <span class="dashicons dashicons-visibility"></span>
                                </button>
                            </div>
                            <p class="description">
                                <?php echo sprintf(
                                    __('Enter your Perplexity API key. Need an API key? <a href="%s" target="_blank">Get one here</a>.', 'mxchat-perplexity'),
                                    'https://docs.perplexity.ai/guides/getting-started'
                                ); ?>
                            </p>
                        </div>

                        <div class="mxchat-form-group">
                            <label for="mxchat_perplexity_model"><?php esc_html_e('Model', 'mxchat-perplexity'); ?></label>
                            <select id="mxchat_perplexity_model" name="mxchat_perplexity_model">

                                <option value="sonar-pro" <?php selected(get_option('mxchat_perplexity_model', 'sonar'), 'sonar-pro'); ?>>
                                    <?php esc_html_e('Sonar Pro (200k context)', 'mxchat-perplexity'); ?>
                                </option>
                                <option value="sonar-reasoning" <?php selected(get_option('mxchat_perplexity_model', 'sonar'), 'sonar-reasoning'); ?>>
                                    <?php esc_html_e('Sonar Reasoning (128k context)', 'mxchat-perplexity'); ?>
                                </option>
                                <option value="sonar-reasoning-pro" <?php selected(get_option('mxchat_perplexity_model', 'sonar'), 'sonar-reasoning-pro'); ?>>
                                    <?php esc_html_e('Sonar Reasoning Pro (128k context)', 'mxchat-perplexity'); ?>
                                </option>
                                <option value="sonar-deep-research" <?php selected(get_option('mxchat_perplexity_model', 'sonar'), 'sonar-deep-research'); ?>>
                                    <?php esc_html_e('Sonar Deep Research (128k context)', 'mxchat-perplexity'); ?>
                                </option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('Select which Perplexity model to use for searches. Different models have different capabilities and costs.', 'mxchat-perplexity'); ?>
                            </p>
                        </div>

                        <div class="mxchat-form-group">
                            <label for="mxchat_perplexity_response_mode"><?php esc_html_e('Response Mode', 'mxchat-perplexity'); ?></label>
                            <select id="mxchat_perplexity_response_mode" name="mxchat_perplexity_response_mode">
                                <option value="conversational" <?php selected(get_option('mxchat_perplexity_response_mode', 'conversational'), 'conversational'); ?>>
                                    <?php esc_html_e('Conversational', 'mxchat-perplexity'); ?>
                                </option>
                                <option value="structured" <?php selected(get_option('mxchat_perplexity_response_mode', 'conversational'), 'structured'); ?>>
                                    <?php esc_html_e('Structured (with sources)', 'mxchat-perplexity'); ?>
                                </option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('Select how search results should be presented. Conversational provides a more natural response, while structured includes citations.', 'mxchat-perplexity'); ?>
                            </p>
                        </div>
                        
                        <div class="mxchat-form-group">
                            <label for="mxchat_perplexity_show_logo"><?php esc_html_e('Perplexity Logo', 'mxchat-perplexity'); ?></label>
                            <div class="mxchat-toggle-container">
                                <label class="mxchat-toggle-switch">
                                    <input type="checkbox"
                                           id="mxchat_perplexity_show_logo"
                                           name="mxchat_perplexity_show_logo"
                                           value="1"
                                           <?php checked(get_option('mxchat_perplexity_show_logo', true)); ?>>
                                    <span class="mxchat-toggle-slider"></span>
                                </label>
                                <span class="mxchat-toggle-label">
                                    <?php esc_html_e('Show Perplexity Logo', 'mxchat-perplexity'); ?>
                                </span>
                            </div>
                            <p class="description">
                                <?php esc_html_e('Enable to show the Perplexity logo in the chatbot toolbar. Disable to hide it.', 'mxchat-perplexity'); ?>
                            </p>
                        </div>
                        
                        <div class="mxchat-form-actions">
                            <?php submit_button(__('Save Settings', 'mxchat-perplexity'), 'mxchat-button-primary'); ?>
                        </div>
                    </form>
                </div>

                <!-- Usage Instructions Card -->
                <div class="mxchat-card">
                    <h2><?php esc_html_e('How to Use Perplexity Integration', 'mxchat-perplexity'); ?></h2>
                    
                    <div class="mxchat-method-card">
                        <h4><?php esc_html_e('Setup Instructions', 'mxchat-perplexity'); ?></h4>
                        <ol>
                            <li><?php esc_html_e('Enter your Perplexity API key above', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Go to the Intents tab in the main MxChat settings', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Enable the "Perplexity Research" intent', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Train your chatbot with examples like "search for..." or "research..."', 'mxchat-perplexity'); ?></li>
                        </ol>
                    </div>
                    
                    <div class="mxchat-method-card">
                        <h4><?php esc_html_e('Using the Web Search Toolbar Button', 'mxchat-perplexity'); ?></h4>
                        <p><?php esc_html_e('The Perplexity integration adds a web search button to your chatbot toolbar:', 'mxchat-perplexity'); ?></p>
                        <ol>
                            <li><?php esc_html_e('Enable the toolbar in MxChat → Integrations → Toolbar Settings', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('The Perplexity web search icon will appear in your chatbot toolbar', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Click the icon to toggle web search mode on/off', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('When active, ALL user queries will automatically search the web using Perplexity', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Click again to return to normal chatbot operation', 'mxchat-perplexity'); ?></li>
                        </ol>
                        <div class="mxchat-note">
                            <strong><?php esc_html_e('Pro Tip:', 'mxchat-perplexity'); ?></strong> 
                            <?php esc_html_e('The web search toggle is perfect for when users need up-to-date information or research on specific topics without having to explicitly ask to search the web.', 'mxchat-perplexity'); ?>
                        </div>
                    </div>

                    <div class="mxchat-method-card">
                        <h4><?php esc_html_e('How It Works', 'mxchat-perplexity'); ?></h4>
                        <p><?php esc_html_e('When a user asks your chatbot to search for information (or when web search mode is active):', 'mxchat-perplexity'); ?></p>
                        <ol>
                            <li><?php esc_html_e('The chatbot sends the query to Perplexity API', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Perplexity searches the web for up-to-date information', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('The search results are processed and returned', 'mxchat-perplexity'); ?></li>
                            <li><?php esc_html_e('Results are presented in your selected response mode (conversational or structured)', 'mxchat-perplexity'); ?></li>
                        </ol>
                        <div class="mxchat-note">
                            <strong><?php esc_html_e('Note:', 'mxchat-perplexity'); ?></strong> 
                            <?php esc_html_e('Unlike traditional search, the Perplexity integration provides conversational responses based on up-to-date web information, making your chatbot more helpful and knowledgeable.', 'mxchat-perplexity'); ?>
                        </div>
                    </div>
                    
                    <div class="mxchat-method-card">
                        <h4><?php esc_html_e('Choosing a Response Mode', 'mxchat-perplexity'); ?></h4>
                        <p><?php esc_html_e('You can choose how Perplexity search results are presented to your users:', 'mxchat-perplexity'); ?></p>
                        <ul>
                            <li><strong><?php esc_html_e('Conversational Mode:', 'mxchat-perplexity'); ?></strong> <?php esc_html_e('Delivers information in a natural, conversational tone as if the chatbot already knew the answer.', 'mxchat-perplexity'); ?></li>
                            <li><strong><?php esc_html_e('Structured Mode:', 'mxchat-perplexity'); ?></strong> <?php esc_html_e('Presents information with clear formatting and source citations for maximum transparency.', 'mxchat-perplexity'); ?></li>
                        </ul>
                    </div>
                </div>

            </div>
        </div>
        <?php
    }
}