<?php
/**
 * Class for handling Perplexity API integration
 */
class MxChat_Perplexity_Handler {

    private $loader;
    private $api_key;
    private $response_mode;

    public function __construct($loader) {
        $this->loader = $loader;
        $this->api_key = get_option('mxchat_perplexity_api_key', '');
        $this->response_mode = get_option('mxchat_perplexity_response_mode', 'conversational');
    }
    
  /**
 * Register AJAX handler for web search
 */
public function register_ajax_handlers() {
    add_action('wp_ajax_mxchat_perplexity_search', array($this, 'ajax_perplexity_search'));
    add_action('wp_ajax_nopriv_mxchat_perplexity_search', array($this, 'ajax_perplexity_search'));
}
  
    
    
/**
 * Enqueue frontend scripts for the Perplexity integration
 */
public function enqueue_frontend_scripts() {
    // Debug that we're attempting to load scripts
    //error_log('🔍 PERPLEXITY DEBUG: Attempting to enqueue frontend scripts');
    
    // Don't rely on the filter - just check if we're on a frontend page
    if (is_admin()) {
        //error_log('🔍 PERPLEXITY DEBUG: Not loading scripts on admin page');
        return;
    }
    
    // Verify file exists
    $js_file = MXCHAT_PERPLEXITY_PLUGIN_DIR . 'assets/js/mxchat-perplexity.js';
    $js_url = MXCHAT_PERPLEXITY_PLUGIN_URL . 'assets/js/mxchat-perplexity.js';
    
    if (!file_exists($js_file)) {
        //error_log('🔍 PERPLEXITY DEBUG: JavaScript file not found at: ' . $js_file);
        return;
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: JS file found, enqueueing from: ' . $js_url);
    
    // Enqueue the JavaScript with direct script loading
    wp_enqueue_script(
        'mxchat-perplexity-frontend',
        $js_url,
        array('jquery'),
        filemtime($js_file), // Use file time as version for cache busting
        true
    );
    
    // Pass data to JavaScript
    wp_localize_script('mxchat-perplexity-frontend', 'mxchatPerplexity', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('mxchat_perplexity_nonce'),
        'debug' => true
    ));
    
    // Confirm script was enqueued
    if (wp_script_is('mxchat-perplexity-frontend', 'enqueued')) {
        //error_log('🔍 PERPLEXITY DEBUG: Script successfully enqueued');
    } else {
        //error_log('🔍 PERPLEXITY DEBUG: Failed to enqueue script');
    }
    

}

/**
 * Filter messages before they're processed to check for web search mode
 * This function runs via the 'mxchat_pre_process_message' filter
 * 
 * @param string $message The user message
 * @param string $user_id The user ID
 * @param string $session_id The session ID
 * @return mixed The original message or a response array if web search handled it
 */
public function check_web_search_mode($message, $user_id, $session_id) {
    // Debug all request data
    //error_log('🔍 PERPLEXITY DEBUG: Checking for web search mode');
    //error_log('🔍 PERPLEXITY DEBUG: POST params: ' . print_r($_POST, true));
    
    // Check if web search mode is enabled
    $web_search_enabled = false;
    
    if (isset($_POST['perplexity_web_search']) && $_POST['perplexity_web_search'] === 'true') {
        $web_search_enabled = true;
        //error_log('🔍 PERPLEXITY DEBUG: Web search enabled via POST param');
    } 
    else if (isset($_REQUEST['perplexity_web_search']) && $_REQUEST['perplexity_web_search'] === 'true') {
        $web_search_enabled = true;
        //error_log('🔍 PERPLEXITY DEBUG: Web search enabled via REQUEST param');
    }
    
    // Additional check: Check localStorage via JavaScript cookie
    if (!$web_search_enabled && isset($_COOKIE['mxchat_perplexity_enabled']) && $_COOKIE['mxchat_perplexity_enabled'] === 'true') {
        $web_search_enabled = true;
        //error_log('🔍 PERPLEXITY DEBUG: Web search enabled via cookie');
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: Web search mode: ' . ($web_search_enabled ? 'ENABLED' : 'DISABLED'));
    
    if ($web_search_enabled) {
        //error_log('🔍 PERPLEXITY DEBUG: Processing message with Perplexity: ' . $message);
        
        // Process with Perplexity search
        $result = $this->handle_perplexity_search(false, $message, $user_id, $session_id);
        
        if ($result !== false) {
            //error_log('🔍 PERPLEXITY DEBUG: Perplexity search successful, returning result');
            
            // Return the result for the core plugin to use
            return $result;
        }
        
        //error_log('🔍 PERPLEXITY DEBUG: Perplexity search failed, falling back to standard processing');
    }
    
    // If we get here, either web search is disabled or it failed - continue with normal processing
    return $message;
}


/**
 * Handle the perplexity search intent
 * 
 * @param mixed $default Default return value (false)
 * @param string $message User's message
 * @param string $user_id User ID
 * @return array|bool Response compatible with MxChat's main handler
 */
public function handle_perplexity_search($default, $message, $user_id, $session_id = null) {
    // Debug all parameters
    //error_log('🔍 PERPLEXITY DEBUG: Incoming message: ' . $message);
    
    // Get the response mode from settings
    $response_mode = get_option('mxchat_perplexity_response_mode', 'conversational');
    
    // Clean up the query to remove duplications
    $search_query = $this->clean_search_query($message);
    //error_log('🔍 PERPLEXITY DEBUG: Cleaned search query: ' . $search_query);
    
    // Check if API key is set
    if (empty($this->api_key)) {
        //error_log('❌ PERPLEXITY DEBUG: Missing API key');
        return [
            'text' => __('The Perplexity API key is not configured. Please set it up in the plugin settings.', 'mxchat-perplexity'),
            'html' => ''
        ];
    }
    
    // Handle empty query
    if (empty($search_query) || $search_query === '.') {
        //error_log('❌ PERPLEXITY DEBUG: Invalid or empty search query');
        return [
            'text' => __('I need more information to search for. Could you please provide a more specific question?', 'mxchat-perplexity'),
            'html' => ''
        ];
    }
    
    // Verify we have a valid search query before proceeding
    if (strlen($search_query) < 3) {
        //error_log('❌ PERPLEXITY DEBUG: Search query too short: ' . $search_query);
        return [
            'text' => __('Your question seems too short. Could you provide more details for a better search?', 'mxchat-perplexity'),
            'html' => ''
        ];
    }
    
    // Handle empty query
    if (empty($search_query)) {
        //error_log('❌ PERPLEXITY DEBUG: Invalid or empty search query');
        return [
            'text' => __('I need more information to search for. Could you please provide a more specific question?', 'mxchat-perplexity'),
            'html' => ''
        ];
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: Using cleaned search query: ' . $search_query);
    
    // Modify the system prompt based on the response mode
    $system_prompt = ($response_mode === 'conversational') 
        ? 'You are a helpful research assistant. Search the web for accurate and up-to-date information to answer the user\'s query. Provide your response in a natural, conversational way without bullet points or numbered lists. Do not prefix with phrases like "Based on my search" or "According to the information". Just respond naturally as if you knew the information. Include relevant facts but present them in a flowing narrative rather than structured format. Do not include any citation numbers or anything of that sort. IMPORTANT: Do not use markdown headings (# or ##) for organization - use bold text (**text**) for emphasis instead. Limit your use of formatting and provide a more cohesive narrative.'
        : 'You are a helpful research assistant. Search the web for accurate and up-to-date information to answer the user\'s query. Include relevant facts, data, and cite sources. IMPORTANT: Do not use markdown headings (# or ##) for organization - use bold text (**text**) for emphasis instead. Limit your use of section headers and provide information in a more cohesive format.';
    
    // Get the model name
    $model = get_option('mxchat_perplexity_model', 'sonar');
    
    // Perform the search with the appropriate system prompt
    $search_results = $this->perform_perplexity_search($search_query, $system_prompt);

    if (is_wp_error($search_results)) {
        //error_log('❌ PERPLEXITY DEBUG: Search error: ' . $search_results->get_error_message());
        return [
            'text' => $search_results->get_error_message(),
            'html' => ''
        ];
    }

    // Log what we got back from Perplexity
    if (isset($search_results['choices']) && !empty($search_results['choices'])) {
        $content = $search_results['choices'][0]['message']['content'] ?? 'No content';
        //error_log('✅ PERPLEXITY DEBUG: Got response content: ' . substr($content, 0, 200) . '...');
    } else {
        //error_log('⚠️ PERPLEXITY DEBUG: Unexpected response structure from Perplexity');
        return [
            'text' => __('I couldn\'t find relevant information about your query. Would you like to try a different search?', 'mxchat-perplexity'),
            'html' => ''
        ];
    }

    // Get the content from Perplexity
    $perplexity_content = $search_results['choices'][0]['message']['content'] ?? '';
    
    // Remove Chain of Thought for deep research and reasoning models
    if ($model === 'sonar-deep-research' || strpos($model, 'reasoning') !== false) {
        $perplexity_content = $this->remove_chain_of_thought($perplexity_content);
        //error_log('🔍 PERPLEXITY DEBUG: Removed Chain of Thought from ' . $model . ' response');
    }
    
    $citations = isset($search_results['citations']) ? $search_results['citations'] : [];
    
    // Choose response format based on selected mode
    if ($response_mode === 'conversational') {
        // For conversational mode, return the response directly without any prefix
        //error_log('✅ PERPLEXITY DEBUG: Using conversational mode - returning natural response');
        
       $perplexity_content = preg_replace('/^(#{1,6})\s+(.+)$/m', '**$2**', $perplexity_content);
        
        return [
            'text' => $perplexity_content,
            'html' => '',
            'skip_markdown_processing' => true // Add a flag to signal your main handler
        ];
    } else {
        // For structured mode, format the content directly with sources
        //error_log('✅ PERPLEXITY DEBUG: Using structured mode - formatting with sources');
        
        // Convert markdown to HTML
        $html_content = $this->convert_markdown_to_html($perplexity_content);
        
        // Format citations
        $citations_html = '';
        if (!empty($citations)) {
            $citations_html = '<div class="perplexity-citations"><h4 style="color: inherit;">Sources:</h4><ul>';
            foreach ($citations as $index => $citation) {
                $citations_html .= '<li>[' . ($index + 1) . '] ';
                if (filter_var($citation, FILTER_VALIDATE_URL)) {
                    $citations_html .= '<a href="' . esc_url($citation) . '" target="_blank">' . esc_html($citation) . '</a>';
                } else {
                    $citations_html .= esc_html($citation);
                }
                $citations_html .= '</li>';
            }
            $citations_html .= '</ul></div>';
        }
        
        // Build the intro text
        $intro_text = "I searched for \"" . esc_html($search_query) . "\" and found this information:";
        
        // Build HTML version with proper formatting - use $html_content directly
        $html_response = '<div class="perplexity-result">';
        $html_response .= '<div class="perplexity-content">' . $html_content . '</div>';
        $html_response .= $citations_html;
        $html_response .= '</div>';
        
        // Return formatted content
        return [
            'text' => $intro_text,
            'html' => $html_response
        ];
    }
}


/**
 * Clean response from reasoning models to remove Chain of Thought sections
 * 
 * @param string $content The response content from Perplexity
 * @return string Cleaned content without CoT sections
 */
private function remove_chain_of_thought($content) {
    // Remove content between <think> tags
    $content = preg_replace('/<think>.*?<\/think>/s', '', $content);
    
    // Also check for incomplete think tags (if the closing tag is missing)
    if (strpos($content, '<think>') !== false) {
        $parts = explode('<think>', $content, 2);
        $content = $parts[0]; // Keep only the content before the <think> tag
    }
    
    // Clean up any leftover artifacts
    $content = preg_replace('/\n{3,}/', "\n\n", $content); // Remove excessive newlines
    $content = preg_replace('/^[\s\n]+|[\s\n]+$/', '', $content); // Trim whitespace and newlines
    
    return $content;
}


/**
 * Format search results for AI to use in generating a response
 * 
 * @param array $results Search results from Perplexity
 * @param string $query Original user query
 * @return array Context for AI processing
 */
private function format_search_results_for_ai($results, $query) {
    //error_log('🔍 PERPLEXITY DEBUG: Formatting search results for AI');

    // Extract the content
    $content = '';
    if (isset($results['choices']) && !empty($results['choices'])) {
        $content = $results['choices'][0]['message']['content'] ?? '';
    }
    
    // Extract citations
    $citations = [];
    if (isset($results['citations']) && is_array($results['citations'])) {
        $citations = $results['citations'];
    }
    
    // Build context for the AI generator in your core plugin
    // This format will be used as context in your generate_response call
    $context = "I found the following information about your search for \"{$query}\":\n\n";
    $context .= $content;
    
    if (!empty($citations)) {
        $context .= "\n\nSources:\n";
        foreach ($citations as $index => $citation) {
            $context .= "- " . $citation . "\n";
        }
    }
    
    $context .= "\n\nPlease use this information to provide a helpful, conversational response to the user's query.";
    
    //error_log('✅ PERPLEXITY DEBUG: Created AI context of ' . strlen($context) . ' characters, passing to main AI');
    
    // Return a special format that works with your core plugin's expected structure
    // This text will be passed to your main AI model to generate a conversational response
    return [
        'text' => '__MXCHAT_CONTEXT_MARKER__' . $context,
        'html' => ''
    ];
}

/**
 * Clean up a search query to remove duplicate text
 * 
 * @param string $query The original query
 * @return string The cleaned query
 */
private function clean_search_query($query) {
    //error_log('🔍 PERPLEXITY DEBUG: Original query before cleaning: ' . $query);
    
    // Safety check - return original if empty or too short
    if (empty($query) || strlen(trim($query)) < 2) {
        //error_log('⚠️ PERPLEXITY DEBUG: Query too short, returning original');
        return $query;
    }
    
    // Store original for comparison
    $original_query = $query;
    
    // Remove any obvious escape slashes from the query (like the \' in your example)
    $cleaned_query = stripslashes($query);
    
    // Basic sanitization - remove any HTML tags
    $cleaned_query = strip_tags($cleaned_query);
    
    // Trim whitespace
    $cleaned_query = trim($cleaned_query);
    
    // SAFETY CHECK: If cleaning resulted in an extremely short query, use original
    if (strlen($cleaned_query) < 3) {
        //error_log('⚠️ PERPLEXITY DEBUG: Cleaning made query too short, reverting to original');
        return $original_query;
    }
    
    // If we've made significant changes, log it
    if ($cleaned_query !== $original_query) {
        //error_log('🔍 PERPLEXITY DEBUG: Original: ' . $original_query);
        //error_log('🔍 PERPLEXITY DEBUG: Cleaned: ' . $cleaned_query);
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: Final query after cleaning: ' . $cleaned_query);
    return $cleaned_query;
}

/**
 * Convert markdown text to HTML
 * 
 * @param string $markdown Markdown formatted text
 * @return string HTML formatted text
 */
private function convert_markdown_to_html($markdown) {
    // PRE-PROCESSING: Convert heading markers to bold text BEFORE regular conversion
    // This ensures ## Headings become **Headings** before any HTML conversion
    $markdown = preg_replace('/^#{1,6}\s+(.*?)$/m', '**$1**', $markdown);
    
    // Process in specific order to avoid conflicts
    
    // 1. Process code blocks first (if any)
    $markdown = preg_replace('/```(.*?)```/s', '<pre><code>$1</code></pre>', $markdown);
    
    // 2. Process bold (**text**)
    $markdown = preg_replace('/\*\*(.*?)\*\*/s', '<strong>$1</strong>', $markdown);
    
    // 3. Process numbered lists (maintain the numbers)
    $markdown = preg_replace_callback('/^(\d+)\.\s+(.*)$/m', function($matches) {
        return "<div class=\"numbered-item\"><span class=\"number\">{$matches[1]}.</span> <span class=\"content\">{$matches[2]}</span></div>";
    }, $markdown);
    
    // 4. Process bullet lists
    $markdown = preg_replace('/^- (.*?)$/m', '<li>$1</li>', $markdown);
    
    // 5. Wrap consecutive list items in ul tags
    $markdown = preg_replace('/((?:<li>.*?<\/li>\s*)+)/', '<ul>$1</ul>', $markdown);
    
    // 6. Process italics (after bold)
    $markdown = preg_replace('/\*(.*?)\*/s', '<em>$1</em>', $markdown);
    
    // 7. Process links - [text](url)
    $markdown = preg_replace('/\[(.*?)\]\((.*?)\)/', '<a href="$2" target="_blank">$1</a>', $markdown);
    
    // 8. Process paragraphs (double newlines)
    $markdown = preg_replace('/\n\n+/', '</p><p>', $markdown);
    
    // 9. Process line breaks (single newlines)
    $markdown = str_replace("\n", "<br>", $markdown);
    
    // Wrap in a paragraph if not already
    if (strpos($markdown, '<p>') === false) {
        $markdown = '<p>' . $markdown . '</p>';
    }
    
    // POST-PROCESSING: Remove any heading tags that might still be in the content
    $markdown = preg_replace('/<h[1-6][^>]*>(.*?)<\/h[1-6]>/i', '<p><strong>$1</strong></p>', $markdown);
    
    return $markdown;
}


/**
 * Format Perplexity content for display
 * 
 * @param string $content Markdown content from Perplexity
 * @param array $citations Citations array if available
 * @param string $search_query The original search query
 * @return array Formatted response with text and HTML
 */
private function format_perplexity_response($content, $citations, $search_query) {
    // Convert markdown headings to proper HTML without double breaks
    $content = preg_replace('/##\s(.*?)(\n|$)/', '<h3>$1</h3>', $content);
    
    // Convert markdown bold to HTML bold
    $content = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $content);
    
    // Convert markdown lists to proper HTML lists
    $content = preg_replace_callback('/(?:^|\n)- (.*?)(?:\n|$)/', function($matches) {
        return '<li>' . $matches[1] . '</li>';
    }, $content);
    
    // Wrap list items in ul tags
    $content = preg_replace('/(<li>.*?<\/li>)+/', '<ul>$0</ul>', $content);
    
    // Fix any doubled ul tags
    $content = str_replace('</ul><ul>', '', $content);
    
    // Format citations section
    $citations_html = '';
    if (!empty($citations)) {
        $citations_html = '<div class="perplexity-citations"><h4 style="color: inherit;">Sources:</h4><ul>';
        foreach ($citations as $index => $citation) {
            // Format the URL as a proper link
            $citation_html = '<li>[' . ($index + 1) . '] ';
            if (filter_var($citation, FILTER_VALIDATE_URL)) {
                $citation_html .= '<a href="' . esc_url($citation) . '" target="_blank">' . esc_html($citation) . '</a>';
            } else {
                $citation_html .= esc_html($citation);
            }
            $citations_html .= $citation_html . '</li>';
        }
        $citations_html .= '</ul></div>';
    }
    
    // Build text version (with minimal formatting)
    $text_response = "I searched for \"" . esc_html($search_query) . "\" and found the following information:\n\n";
    $text_response .= strip_tags(str_replace(['<h3>', '</h3>', '<li>'], ["\n\n", "\n", "\n- "], $content));
    
    // Add citations to text version
    if (!empty($citations)) {
        $text_response .= "\n\nSources:\n";
        foreach ($citations as $index => $citation) {
            $text_response .= "[" . ($index + 1) . "] " . $citation . "\n";
        }
    }
    
    // Build HTML version (with proper formatting)
    $html_response = '<div class="perplexity-result">';
    $html_response .= '<p>I searched for "<strong>' . esc_html($search_query) . '</strong>" and found this information:</p>';
    $html_response .= '<div class="perplexity-content">' . $content . '</div>';
    $html_response .= $citations_html;
    $html_response .= '</div>';
    
    return [
        'text' => $text_response,
        'html' => $html_response
    ];
}

/**
 * Extract the actual search query from user message
 * 
 * @param string $query Original user query
 * @return string Refined search query
 */
private function extract_search_query($query) {
    // Log the original query
    //error_log('🔍 PERPLEXITY DEBUG: Original query: ' . $query);
    
    // Remove common search prefixes but preserve the actual query
    $prefixes = array(
        'search for ',
        'search ',
        'look up ',
        'find information about ',
        'find info on ',
        'find information on ',
        'tell me about ',
        'research ',
        'can you find ',
        'please find ',
        'what is ',
        'what are ',
        'tell me ',
    );

    $cleaned_query = $query;
    foreach ($prefixes as $prefix) {
        if (stripos($query, $prefix) === 0) {
            $cleaned_query = trim(substr($query, strlen($prefix)));
            break;
        }
    }
    
    // Safety check - don't allow empty or single-character queries
    if (empty($cleaned_query) || strlen($cleaned_query) <= 1) {
        //error_log('⚠️ PERPLEXITY DEBUG: Extracted query was too short, using original: ' . $query);
        return $query;
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: Extracted query: ' . $cleaned_query);
    return $cleaned_query;
}


/**
 * Perform the search using Perplexity API
 * 
 * @param string $query The search query
 * @param string $system_prompt The system prompt to use
 * @return array|WP_Error Search results or error
 */
private function perform_perplexity_search($query, $system_prompt = null) {
    $api_url = 'https://api.perplexity.ai/chat/completions';
    
    // Get the model from settings or use a default valid model
    $model = get_option('mxchat_perplexity_model', 'sonar');
    
    // Use default system prompt if none provided
    if ($system_prompt === null) {
        $system_prompt = 'You are a helpful research assistant. Search the web for accurate and up-to-date information to answer the user\'s query. Include relevant facts, data, and cite sources.';
    }
    
    // Set timeout based on model
    $timeout = 30; // Default timeout
    if ($model === 'sonar-deep-research') {
        $timeout = 300; // 5 minutes for deep research model
    } else if (strpos($model, 'reasoning') !== false) {
        $timeout = 120; // 2 minutes for reasoning models
    } else if ($model === 'sonar-pro') {
        $timeout = 60; // 1 minute for pro model
    }
    
    // For debugging
    //error_log('🔍 PERPLEXITY DEBUG: Using model: ' . $model);
    //error_log('🔍 PERPLEXITY DEBUG: Using system prompt: ' . substr($system_prompt, 0, 100) . '...');
    //error_log('🔍 PERPLEXITY DEBUG: Using timeout: ' . $timeout . ' seconds');
    
    // Uncomment this line for easier debugging when needed
    //error_log('🔍 PERPLEXITY DEBUG: Request to ' . $model . ' with timeout of ' . $timeout . ' seconds');
    
    $request_args = array(
        'method' => 'POST',
        'timeout' => $timeout, // Use dynamic timeout based on model
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->api_key,
        ),
        'body' => json_encode(array(
            'model' => $model,
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => $system_prompt
                ),
                array(
                    'role' => 'user',
                    'content' => $query
                )
            ),
            'options' => array(
                'search' => true // Ensure web search is enabled
            )
        )),
    );
    // Log the request for debugging
    //error_log('🔍 PERPLEXITY DEBUG: Sending API request to: ' . $api_url);
    
    $response = wp_remote_request($api_url, $request_args);
    if (is_wp_error($response)) {
        //error_log('❌ PERPLEXITY DEBUG: WP Error: ' . $response->get_error_message());
        return new WP_Error(
            'perplexity_timeout',
            sprintf(__('The request timed out after %d seconds. Deep research queries may take longer to process. Please try again or use a different model.', 'mxchat-perplexity'), $timeout),
            array('status' => 408)
        );
    }
    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        $error_body = wp_remote_retrieve_body($response);
        //error_log('❌ PERPLEXITY DEBUG: API Error (' . $response_code . '): ' . $error_body);
        return new WP_Error(
            'perplexity_api_error',
            sprintf(__('Perplexity API error: %s', 'mxchat-perplexity'), $error_body),
            array('status' => $response_code)
        );
    }
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    if (empty($data) || !is_array($data)) {
        //error_log('❌ PERPLEXITY DEBUG: Invalid response format');
        return new WP_Error('perplexity_api_error', __('Invalid response from Perplexity API', 'mxchat-perplexity'));
    }
    
    // Log the raw response content for deep research model to examine CoT format
    if ($model === 'sonar-deep-research' || strpos($model, 'reasoning') !== false) {
        if (isset($data['choices']) && !empty($data['choices'])) {
            $content = $data['choices'][0]['message']['content'] ?? '';
            //error_log('🔍 PERPLEXITY DEBUG: Raw ' . $model . ' response content:');
            //error_log(substr($content, 0, 1000) . '...'); // Log first 1000 chars to see format
        }
    }
    
    //error_log('✅ PERPLEXITY DEBUG: API request successful');
    return $data;
}


    /**
     * Log debug information if debugging is enabled
     * 
     * @param mixed $data Data to log
     * @param string $label Optional label for the log entry
     */
    private function log_debug($data, $label = '') {
        if (defined('WP_DEBUG') && WP_DEBUG === true) {
            $log_dir = WP_CONTENT_DIR . '/perplexity-logs';
            
            if (!file_exists($log_dir)) {
                mkdir($log_dir, 0755, true);
            }
            
            $log_file = $log_dir . '/perplexity-debug.log';
            $timestamp = current_time('mysql');
            
            $log_entry = "[{$timestamp}] " . ($label ? "{$label}: " : "");
            $log_entry .= (is_array($data) || is_object($data)) ? print_r($data, true) : $data;
            $log_entry .= "\n\n" . str_repeat('-', 50) . "\n\n";
            
            file_put_contents($log_file, $log_entry, FILE_APPEND);
        }
    }
    
    
/**
 * AJAX handler for Perplexity search
 */
public function ajax_perplexity_search() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_perplexity_nonce')) {
        wp_send_json_error('Invalid security token');
        return;
    }
    
    // Get message and session
    $message = isset($_POST['message']) ? sanitize_text_field($_POST['message']) : '';
    $session_id = isset($_POST['session_id']) ? sanitize_text_field($_POST['session_id']) : '';
    
    if (empty($message)) {
        wp_send_json_error('Empty message');
        return;
    }
    
    //error_log('🔍 PERPLEXITY DEBUG: AJAX search for query: ' . $message);
    
    // Process with Perplexity
    $result = $this->handle_perplexity_search(false, $message, '1', $session_id);
    
    if ($result === false) {
        wp_send_json_error('Search failed');
        return;
    }
    
    wp_send_json_success($result);
}
    
    
}