<?php
/**
 * Plugin Name: MxChat Perplexity Integration
 * Plugin URI: https://mxchat.ai/
 * Description: Perplexity integration add-on for MxChat (Pro only)
 * Version: 1.0.5
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-perplexity
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

define('MXCHAT_PERPLEXITY_VERSION', '1.0.5');
define('MXCHAT_PERPLEXITY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_PERPLEXITY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_PERPLEXITY_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Autoloader for classes
spl_autoload_register(function ($class) {
    $prefix = 'MxChat_Perplexity_';
    $base_dir = MXCHAT_PERPLEXITY_PLUGIN_DIR . 'includes/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-mxchat-perplexity-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    if (file_exists($file)) {
        require $file;
    }
});

// Check if MxChat core plugin is active
function is_mxchat_perplexity_requirements_met() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        $mxchat_active = isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php']);
    }
    
    // Check site-specific plugins if not found in network
    if (!isset($mxchat_active)) {
        $active_plugins = apply_filters('active_plugins', get_option('active_plugins'));
        $mxchat_active = in_array('mxchat-basic/mxchat-basic.php', $active_plugins);
    }
    
    return $mxchat_active;
}

// Update checker for plugin updates (placeholder - implement as needed)
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerPerplexity::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-perplexity'
]);

// Main plugin class
final class MxChat_Perplexity {
    private static $instance = null;
    public $loader;
    public $admin;
    public $perplexity_handler;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->check_dependencies();
        $this->init_components();
        register_deactivation_hook(__FILE__, array($this, 'handle_deactivation'));
    }

    private function check_dependencies() {
        if (!is_mxchat_perplexity_requirements_met()) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php _e('<strong>MxChat Perplexity Integration</strong> requires MxChat plugin to be installed and active.', 'mxchat-perplexity'); ?></p>
                </div>
                <?php
            });
            return;
        }
    }

    private function init_components() {
        require_once MXCHAT_PERPLEXITY_PLUGIN_DIR . 'includes/class-mxchat-perplexity-loader.php';
        $this->loader = new MxChat_Perplexity_Loader();
    
        if ($this->is_license_active()) {
            // Initialize admin first
            require_once MXCHAT_PERPLEXITY_PLUGIN_DIR . 'includes/class-mxchat-perplexity-admin.php';
            $this->admin = new MxChat_Perplexity_Admin($this->loader);
            
            // Initialize Perplexity handler (only once!)
            require_once MXCHAT_PERPLEXITY_PLUGIN_DIR . 'includes/class-mxchat-perplexity-handler.php';
            $this->perplexity_handler = new MxChat_Perplexity_Handler($this->loader);
    
            
            // Add Perplexity callbacks to the main plugin
            add_filter('mxchat_available_callbacks', array($this, 'add_perplexity_callbacks'));
            
            // Add the filter for handling the Perplexity search intent
            add_filter('mxchat_handle_perplexity_search', array($this->perplexity_handler, 'handle_perplexity_search'), 10, 4);
            
            // Hook the web search mode checker
            $this->loader->add_filter('mxchat_pre_process_message', $this->perplexity_handler, 'check_web_search_mode', 10, 3);
            
            // Add frontend scripts
            add_action('wp_enqueue_scripts', array($this->perplexity_handler, 'enqueue_frontend_scripts'), 999);
            
            // Add admin menu
            add_action('admin_menu', array($this, 'add_admin_menu'), 100);
            
            // Add this in your init_components method
            $this->loader->add_action('init', $this->perplexity_handler, 'register_ajax_handlers');
            
            // ADD THESE LINES - Logo visibility filter and CSS
            add_filter('mxchat_perplexity_should_show_logo', array($this, 'should_show_perplexity_logo'));
            add_action('wp_head', array($this, 'add_perplexity_logo_css'));
            
            // Run the loader
            $this->loader->run();
        } else {
            add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        }
    }
    
/**
 * Determine if the Perplexity logo should be shown
 * 
 * @return bool Whether to show the Perplexity logo
 */
public function should_show_perplexity_logo() {
    // Get the setting, default to true (show logo)
    $show_logo = get_option('mxchat_perplexity_show_logo', true);
    
    // Convert to boolean - handles various formats WordPress might store
    if (is_string($show_logo)) {
        // Convert "1", "true", "on", "yes" to true
        return in_array(strtolower($show_logo), ['1', 'true', 'on', 'yes'], true);
    } elseif (is_numeric($show_logo)) {
        // Convert 1 to true, 0 to false
        return (bool)$show_logo;
    }
    
    // If it's already a boolean or other type, cast to boolean
    return (bool)$show_logo;
}
    
/**
 * Add CSS for the Perplexity logo visibility
 */
public function add_perplexity_logo_css() {
    // Always add the base styles
    echo '<style>
        /* Base styles for the button */
        #perplexity-search-btn {
            display: block !important;
            align-items: center;
            justify-content: center;
        }
        
        /* Hide the button when disabled */
        .mxchat-perplexity-disabled #perplexity-search-btn {
            display: none !important;
        }
    </style>';
    
    // Add a class to the body to control visibility
    if (!$this->should_show_perplexity_logo()) {
        add_filter('body_class', function($classes) {
            $classes[] = 'mxchat-perplexity-disabled';
            return $classes;
        });
    }
}

    public function add_perplexity_callbacks($callbacks) {
        $perplexity_callbacks = array(
            'mxchat_handle_perplexity_search' => array(
                'label' => __('Perplexity Research', 'mxchat-perplexity'),
                'pro_only' => true,
                'group' => __('Search Features', 'mxchat-perplexity'),
            ),
        );

        return array_merge($callbacks, $perplexity_callbacks);
    }

    public function add_admin_menu() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php',  // Network settings menu
                __('Perplexity Integration', 'mxchat-perplexity'),
                __('Perplexity', 'mxchat-perplexity'),
                'manage_network_options',
                'mxchat-perplexity',
                array($this->admin, 'display_admin_page')
            );
        } else {
            add_submenu_page(
                'mxchat-max',
                __('Perplexity Integration', 'mxchat-perplexity'),
                __('Perplexity', 'mxchat-perplexity'),
                'manage_options',
                'mxchat-perplexity',
                array($this->admin, 'display_admin_page')
            );
        }
    }

    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    public function handle_deactivation() {
        // Any cleanup needed upon deactivation
    }

    public function show_pro_requirement_notice() {
        $class = 'notice notice-warning';
        $message = sprintf(
            __('MxChat Perplexity Integration requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-perplexity'),
            admin_url('admin.php?page=mxchat-activation')
        );
        printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
    }

    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-perplexity'), '1.0.5');
    }

    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-perplexity'), '1.0.5');
    }
}

function MxChat_Perplexity() {
    return MxChat_Perplexity::instance();
}

add_action('plugins_loaded', 'MxChat_Perplexity', 20);

register_activation_hook(__FILE__, function() {
    require_once MXCHAT_PERPLEXITY_PLUGIN_DIR . 'includes/class-mxchat-perplexity-activator.php';
    MxChat_Perplexity_Activator::activate();
});