// assets/js/admin.js - Updated with AI integration
jQuery(document).ready(function($) {
    'use strict';
    
    // Simple debounce function
    function debounce(func, wait) {
        let timeout;
        return function(...args) {
            clearTimeout(timeout);
            timeout = setTimeout(() => func.apply(this, args), wait);
        };
    }

    // Function to update preview elements
    function updatePreviewColor(selector, property, value) {
        const elements = document.querySelectorAll(selector);
        elements.forEach(element => {
            element.style[property] = value;
        });
    }

    // Initialize color pickers with live preview
    $('.my-color-field').wpColorPicker({
        change: debounce(function(event, ui) {
            var $input = $(event.target);
            var name = $input.attr('name');
            var value = ui.color.toString();
            var nonce = $input.data('nonce');
            
            // Update preview based on the color field
            updatePreviewColorsForField(name, value);
            
            // Create feedback container
            const feedbackContainer = $('<div class="feedback-container"></div>');
            const spinner = $('<div class="saving-spinner"></div>');
            const successIcon = $('<div class="success-icon">✔</div>');

            // Position feedback container next to the color picker
            $input.closest('.wp-picker-container').after(feedbackContainer);
            feedbackContainer.append(spinner);
            
            // Determine which AJAX action to use
            let ajaxAction = 'mxchat_save_theme_setting';
            let ajaxData = {
                action: ajaxAction,
                name: name,
                value: value,
                _ajax_nonce: nonce
            };
            
            // Use special AJAX action for input container colors
            if (name === 'input_container_bg_color') {
                ajaxAction = 'mxchat_save_input_container_bg';
                ajaxData = {
                    action: ajaxAction,
                    color: value,
                    _ajax_nonce: nonce
                };
            } else if (name === 'input_container_border_color') {
                ajaxAction = 'mxchat_save_input_container_border';
                ajaxData = {
                    action: ajaxAction,
                    color: value,
                    _ajax_nonce: nonce
                };
            }
            
            // Save via AJAX
            $.ajax({
                url: mxchatTheme.ajaxurl,
                type: 'POST',
                data: ajaxData,
                success: function(response) {
                    if (response.success) {
                        spinner.fadeOut(200, function() {
                            feedbackContainer.append(successIcon);
                            successIcon.fadeIn(200).delay(1000).fadeOut(200, function() {
                                feedbackContainer.remove();
                            });
                        });
                    } else {
                        alert('Error saving: ' + (response.data?.message || 'Unknown error'));
                        feedbackContainer.remove();
                    }
                },
                error: function() {
                    alert('An error occurred while saving.');
                    feedbackContainer.remove();
                }
            });
        }, 500) // 500ms debounce delay
    });
    
    // Centralized function to update preview colors
    function updatePreviewColorsForField(name, value) {
        switch(name) {
            case 'close_button_color':
                updatePreviewColor('.chatbot-title', 'color', value);
                updatePreviewColor('.exit-chat', 'color', value);
                updatePreviewColor('.exit-chat svg path:last-child', 'fill', value);
                break;
            case 'chatbot_bg_color':
                updatePreviewColor('#mxchat-chatbot', 'backgroundColor', value);
                break;
            case 'user_message_bg_color':
                updatePreviewColor('.user-message', 'backgroundColor', value);
                break;
            case 'user_message_font_color':
                updatePreviewColor('.user-message', 'color', value);
                break;
            case 'bot_message_bg_color':
                updatePreviewColor('.bot-message', 'backgroundColor', value);
                break;
            case 'bot_message_font_color':
                updatePreviewColor('.bot-message', 'color', value);
                break;
            case 'top_bar_bg_color':
                updatePreviewColor('.chatbot-top-bar', 'backgroundColor', value);
                break;
            case 'send_button_font_color':
                updatePreviewColor('#send-button svg', 'fill', value);
                break;
            case 'chat_input_font_color':
                updatePreviewColor('#chat-input', 'color', value);
                break;
            case 'input_container_bg_color':
                updatePreviewColor('#input-container', 'backgroundColor', value);
                break;
            case 'input_container_border_color':
                updatePreviewColor('#input-container', 'borderColor', value);
                break;
            case 'chatbot_background_color':
                updatePreviewColor('#floating-chatbot', 'backgroundColor', value);
                break;
            case 'icon_color':
                updatePreviewColor('.exit-chat svg', 'fill', value);
                break;
            case 'live_agent_message_bg_color':
                updatePreviewColor('.agent-message', 'backgroundColor', value);
                break;
            case 'live_agent_message_font_color':
                updatePreviewColor('.agent-message', 'color', value);
                break;
            case 'mode_indicator_bg_color':
                updatePreviewColor('.chat-mode-indicator', 'backgroundColor', value);
                break;
            case 'mode_indicator_font_color':
                updatePreviewColor('.chat-mode-indicator', 'color', value);
                break;
            case 'toolbar_icon_color':
                updatePreviewColor('#pdf-upload-btn svg', 'fill', value);
                updatePreviewColor('#word-upload-btn svg', 'fill', value);
                updatePreviewColor('.toolbar-btn svg', 'fill', value);
                break;
            case 'quick_questions_toggle_color':
                updatePreviewColor('.questions-collapse-btn svg', 'stroke', value);
                updatePreviewColor('.questions-toggle-btn svg', 'stroke', value);
                break;
        }
    }
    
    // Icon URL input handler
    $('.icon-url-field').on('change', function() {
        var $input = $(this);
        var value = $input.val();
        var name = $input.attr('name').replace('mxchat_options[', '').replace(']', '');
        var nonce = $input.data('nonce');
        var $preview = $input.siblings('.icon-preview');

        // Update field preview
        if (value) {
            $preview.show().find('img').attr('src', value);
        } else {
            $preview.hide();
        }
        
        // Save via AJAX
        $.ajax({
            url: mxchatTheme.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_save_theme_setting',
                name: name,
                value: value,
                _ajax_nonce: nonce
            },
            success: function(response) {
                if (response.success) {
                    // Update preview
                    if (name === 'title_icon') {
                        $('.chatbot-title-icon').attr('src', value);
                    }
                    // Feedback indicator
                    const feedbackContainer = $('<div class="feedback-container"></div>');
                    const successIcon = $('<div class="success-icon">✔</div>');
                    $input.after(feedbackContainer);
                    feedbackContainer.append(successIcon);
                    successIcon.fadeIn(200).delay(1000).fadeOut(200, function() {
                        feedbackContainer.remove();
                    });
                } else {
                    alert('Error saving: ' + (response.data?.message || 'Unknown error'));
                }
            },
            error: function() {
                alert('An error occurred while saving.');
            }
        });
    });
    
    // Initialize icons
    function initializeIcons() {
        const titleIconInput = $('#title_icon');
        if (titleIconInput.length && titleIconInput.val()) {
            $('.chatbot-title-icon').attr('src', titleIconInput.val());
        }
    }
    
    // Initialize preview with current colors
    function initializePreview() {
        $('.my-color-field').each(function() {
            const name = $(this).attr('name');
            const value = $(this).val();
            if (value) {
                updatePreviewColorsForField(name, value);
            }
        });
    }

    // Apply CSS theme to preview
    function applyThemeToPreview(css) {
        // Remove existing AI theme CSS
        $('#mxchat-ai-preview-css').remove();
        
        if (css) {
            // Add new AI theme CSS
            $('<style id="mxchat-ai-preview-css">' + css + '</style>').appendTo('head');
        }
    }

    // Reset preview to manual settings
    function resetPreviewToManual() {
        $('#mxchat-ai-preview-css').remove();
        initializePreview();
    }

    // Initialize everything when page loads
    initializeIcons();
    initializePreview();
    
    // Make functions globally available for AI generator
    window.MxChatThemeAdmin = {
        updatePreviewColor: updatePreviewColor,
        updatePreviewColorsForField: updatePreviewColorsForField,
        applyThemeToPreview: applyThemeToPreview,
        resetPreviewToManual: resetPreviewToManual,
        initializePreview: initializePreview
    };
    
    // Handle AI theme preview integration
    $(document).on('click', '#preview-ai-theme', function() {
        if (window.MxChatAiThemeGenerator && window.MxChatAiThemeGenerator.generatedCSS) {
            applyThemeToPreview(window.MxChatAiThemeGenerator.generatedCSS);
        }
    });
    
    // Handle resetting to manual theme
    $(document).on('click', '#reset-to-manual', function() {
        resetPreviewToManual();
    });
    
    // Auto-apply AI themes on generation
    $(document).on('mxchat:theme-generated', function(e, css) {
        applyThemeToPreview(css);
    });
    
    // Handle active AI theme removal
    $(document).on('mxchat:ai-theme-removed', function() {
        resetPreviewToManual();
    });
    
    // ========================================
    // Position Settings Auto-Save
    // ========================================
    
    // Debounce for position settings
    let positionSaveTimeouts = {};
    
    function savePositionSetting(fieldId, value, $statusElement) {
        const nonce = $('#' + fieldId).data('nonce');
        
        // Show saving indicator
        $statusElement.removeClass('success error').addClass('saving').text('Saving...');
        
        $.ajax({
            url: mxchatTheme.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_save_position_setting',
                nonce: nonce,
                setting_name: fieldId,
                value: value
            },
            success: function(response) {
                if (response.success) {
                    $statusElement.removeClass('saving').addClass('success').text('Saved ✔');
                    
                    // Update the live preview position
                    updatePositionPreview(fieldId, value);
                    
                    // Clear success message after 2 seconds
                    setTimeout(function() {
                        $statusElement.removeClass('success').text('');
                    }, 2000);
                } else {
                    $statusElement.removeClass('saving').addClass('error').text('Error');
                    setTimeout(function() {
                        $statusElement.removeClass('error').text('');
                    }, 3000);
                }
            },
            error: function() {
                $statusElement.removeClass('saving').addClass('error').text('Error');
                setTimeout(function() {
                    $statusElement.removeClass('error').text('');
                }, 3000);
            }
        });
    }
    
    // Update preview with new position
    function updatePositionPreview(fieldId, value) {
        const $style = $('#mxchat-position-css');
        
        if ($style.length === 0) {
            return; // No style tag to update
        }
        
        let css = $style.html();
        
        // Update CSS based on field
        switch(fieldId) {
            case 'widget_position':
                css = css.replace(/(#floating-chatbot-button\s*\{[^}]*bottom:\s*)\d+(px[^}]*\})/i, '$1' + value + '$2');
                break;
            case 'prechat_position':
                css = css.replace(/(#pre-chat-message\s*\{[^}]*bottom:\s*)\d+(px[^}]*\})/i, '$1' + value + '$2');
                break;
            case 'widget_position_mobile':
                // Match the mobile media query section for widget
                css = css.replace(/(@media[^{]*\{[^}]*#floating-chatbot-button\s*\{[^}]*bottom:\s*)\d+(px[^}]*\})/i, '$1' + value + '$2');
                break;
            case 'prechat_position_mobile':
                // Match the mobile media query section for pre-chat
                css = css.replace(/(@media[^{]*\{[^}]*#pre-chat-message\s*\{[^}]*bottom:\s*)\d+(px[^}]*\})/i, '$1' + value + '$2');
                break;
        }
        
        $style.html(css);
    }
    
    // Handle range slider input
    $('.position-range-field').on('input', function() {
        const $range = $(this);
        const fieldId = $range.attr('id');
        const value = parseInt($range.val());
        
        // Update the corresponding number input
        $('#' + fieldId + '_number').val(value);
        
        // Clear any existing timeout for this field
        if (positionSaveTimeouts[fieldId]) {
            clearTimeout(positionSaveTimeouts[fieldId]);
        }
        
        // Set new timeout to save after 500ms of no changes
        const $statusElement = $('.position-save-status[data-field="' + fieldId + '"]');
        positionSaveTimeouts[fieldId] = setTimeout(function() {
            savePositionSetting(fieldId, value, $statusElement);
        }, 500);
    });
    
    // Handle number input
    $('.position-number-field').on('input', function() {
        const $number = $(this);
        const rangeId = $number.data('range-id');
        const value = parseInt($number.val());
        
        // Update the corresponding range input
        $('#' + rangeId).val(value);
        
        // Clear any existing timeout for this field
        if (positionSaveTimeouts[rangeId]) {
            clearTimeout(positionSaveTimeouts[rangeId]);
        }
        
        // Set new timeout to save after 500ms of no changes
        const $statusElement = $('.position-save-status[data-field="' + rangeId + '"]');
        positionSaveTimeouts[rangeId] = setTimeout(function() {
            savePositionSetting(rangeId, value, $statusElement);
        }, 500);
    });
    

});