jQuery(document).ready(function($) {
    'use strict';
    
    const MxChatAiThemeGenerator = {
        
        selectedModelId: '',
        selectedModelName: '',
        generatedCSS: '',
        
        init: function() {
            this.bindEvents();
            this.initModelSelector();
            this.checkFormValidity();
        },
        
        bindEvents: function() {
            // Generate theme button
            $('#generate-ai-theme').on('click', this.generateTheme.bind(this));
            
            // Preview theme button
            $('#preview-ai-theme').on('click', this.previewTheme.bind(this));
            
            // Apply theme button (now with auto-save)
            $('#apply-ai-theme').on('click', this.applyAndSaveTheme.bind(this));
            
            // Regenerate theme button
            $('#regenerate-theme').on('click', this.regenerateTheme.bind(this));
            
            // Copy CSS code
            $('#copy-css-code').on('click', this.copyCSSCode.bind(this));
            
            // View generated CSS toggle
            $('#view-generated-css').on('click', this.toggleGeneratedCSS.bind(this));
            
            // View active CSS
            $('#view-active-css').on('click', this.toggleActiveCSS.bind(this));
            
            // Remove active theme
            $('#remove-active-theme').on('click', this.removeActiveTheme.bind(this));
            
            // Saved theme actions
            $(document).on('click', '.mxchat-theme-preview-btn', this.previewSavedTheme.bind(this));
            $(document).on('click', '.mxchat-theme-apply-btn', this.applySavedTheme.bind(this));
            $(document).on('click', '.mxchat-theme-delete-btn', this.deleteSavedTheme.bind(this));
            $(document).on('click', '.mxchat-theme-edit-btn', this.editSavedTheme.bind(this));
            
            // CSS Editor actions
            $(document).on('click', '#edit-active-css', this.editActiveTheme.bind(this));
            $(document).on('click', '#save-css-editor', this.saveCSSEditorChanges.bind(this));
            $(document).on('click', '#cancel-css-editor', this.cancelCSSEditor.bind(this));

            // Copy buttons
            $(document).on('click', '.mxchat-copy-button', this.copyToClipboard.bind(this));
            
            // Form validation
            $('#ai-theme-description').on('input', this.checkFormValidity.bind(this));
            
            // Close modal on outside click
            $('.mxchat-css-editor-modal').on('click', function(e) {
                if (e.target === this) {
                    $(this).hide();
                }
            });
            
            // Modal close buttons
            $(document).on('click', '.mxchat-css-editor-modal-close', function() {
                $(this).closest('.mxchat-css-editor-modal').hide();
            });
        },
        
        initModelSelector: function() {
            // The model selector is now handled by MxChatThemeModelSelector
            // Just bind to the selection event
            const self = this;
            
            $(document).on('click', '.mxchat-theme-model-selector-card', function(e) {
                const modelId = $(this).data('value');
                const modelName = $(this).find('.mxchat-theme-model-selector-title').text();
                
                self.setSelectedModel(modelId, modelName);
            });
        },
        
        setSelectedModel: function(modelId, modelName) {
            this.selectedModelId = modelId;
            this.selectedModelName = modelName;
            
            $('#selected-model-name').text(modelName);
            $('#selected-model-id').val(modelId);
            
            this.checkFormValidity();
        },
        
        checkFormValidity: function() {
            const description = $('#ai-theme-description').val().trim();
            const hasModel = $('#selected-model-id').val() !== '' || this.selectedModelId !== '';
            
            const isValid = description.length > 10 && hasModel;
            $('#generate-ai-theme').prop('disabled', !isValid);
            
            return isValid;
        },
        
        generateTheme: function() {
            const description = $('#ai-theme-description').val().trim();
            const modelId = this.selectedModelId || $('#selected-model-id').val();
            
            if (!description || !modelId) {
                this.showNotification('Please provide a theme description and select a model.', 'error');
                return;
            }
            
            this.setLoadingState(true);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_generate_ai_theme',
                    description: description,
                    model_id: modelId,
                    _ajax_nonce: window.mxchatAiThemeNonce
                },
                success: (response) => {
                    this.setLoadingState(false);
                    
                    if (response.success) {
                        this.generatedCSS = response.data.css;
                        this.showGeneratedTheme(response.data.css);
                        this.showNotification(response.data.message, 'success');
                        
                        // Auto-preview the generated theme
                        this.applyTempCSS(response.data.css);
                    } else {
                        this.showNotification(response.data.message || 'Failed to generate theme', 'error');
                    }
                },
                error: (xhr, status, error) => {
                    this.setLoadingState(false);
                    this.showNotification('An error occurred while generating the theme: ' + error, 'error');
                }
            });
        },
        
        regenerateTheme: function() {
            this.generateTheme();
        },
                
        showGeneratedTheme: function(css) {
            $('#generated-css-code').text(css);
            $('#ai-results-section').show();
            
            // Update button with SVG icon
            const checkSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="20,6 9,17 4,12"/></svg>';
            $('#apply-ai-theme').html(checkSvg + ' Apply & Save Theme');
            
            // Hide the CSS preview by default
            $('#generated-css-preview').hide();
            
            // Scroll to results
            $('html, body').animate({
                scrollTop: $('#ai-results-section').offset().top - 50
            }, 500);
        },
        
        toggleGeneratedCSS: function() {
            $('#generated-css-preview').toggle();
        },
        
        previewTheme: function() {
            if (!this.generatedCSS) {
                this.showNotification('No theme to preview', 'error');
                return;
            }
            
            this.applyTempCSS(this.generatedCSS);
            this.showNotification('Theme preview applied! Check the chatbot preview.', 'info');
            
            // Auto-scroll to the chatbot preview
            $('html, body').animate({
                scrollTop: $('#floating-chatbot').offset().top - 50
            }, 500);
        },
        
        previewSavedTheme: function(e) {
            const css = $(e.currentTarget).data('css');
            this.applyTempCSS(css);
            this.showNotification('Saved theme preview applied!', 'info');
            
            // Auto-scroll to the chatbot preview
            $('html, body').animate({
                scrollTop: $('#floating-chatbot').offset().top - 50
            }, 500);
        },
        
        // Edit saved theme
        editSavedTheme: function(e) {
            const $button = $(e.currentTarget);
            const css = $button.data('css');
            const themeName = $button.closest('.mxchat-saved-theme-card').find('h4').text();
            
            this.openCSSEditor(css, themeName, 'saved');
        },
        
        // Edit active theme
        editActiveTheme: function() {
            const css = $('#active-css-code').text();
            const themeName = $('.mxchat-applied-theme-name').text();
            
            this.openCSSEditor(css, themeName, 'active');
        },
        
        // Open CSS Editor Modal
openCSSEditor: function(css, themeName, source) {
    // Set editor content
    $('#css-editor-textarea').val(css);
    $('#css-editor-theme-name').text(themeName);
    $('#css-editor-source').val(source);
    $('#css-editor-original-name').val(themeName);
    
    // Show modal
    $('#mxchat-css-editor-modal').show();
    
},

saveCSSEditorChanges: function() {
    const css = $('#css-editor-textarea').val().trim();
    const source = $('#css-editor-source').val();
    const originalName = $('#css-editor-original-name').val();
    
    console.log('Saving CSS changes:', { css: css.length, source, originalName }); // Debug log
    
    if (!css) {
        const errorMsg = 'CSS cannot be empty';
        this.showNotification(errorMsg, 'error');
        this.highlightCSSError(errorMsg);
        return;
    }
    
    // Basic client-side validation
    const braceCount = (css.match(/{/g) || []).length - (css.match(/}/g) || []).length;
    if (braceCount !== 0) {
        const errorMsg = 'CSS Syntax Error: Missing ' + Math.abs(braceCount) + ' ' + (braceCount > 0 ? 'closing' : 'opening') + ' brace(s)';
        this.showNotification(errorMsg, 'error');
        this.highlightCSSError(errorMsg);
        return;
    }
    
    const $button = $('#save-css-editor');
    const originalHtml = $button.html();
    const spinnerSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="mxchat-spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';
    $button.html(spinnerSvg + ' Saving...').prop('disabled', true);
    
    $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_update_theme_css',
            css: css,
            theme_name: originalName,
            source: source,
            _ajax_nonce: window.mxchatAiThemeNonce
        },
        success: (response) => {
            console.log('AJAX Success Response:', response); // Debug log
            $button.html(originalHtml).prop('disabled', false);
            
            if (response.success) {
                this.showNotification('Theme updated successfully!', 'success');
                $('#mxchat-css-editor-modal').hide();
                
                // Update UI based on source
                if (source === 'active') {
                    this.updateActiveThemeUI(response.data.css || css);
                } else if (source === 'saved') {
                    this.updateSavedThemeUI(originalName, response.data.css || css);
                }
                
                // Apply the updated CSS
                this.clearAllAIThemeCSS();
                $('<style id="mxchat-ai-theme-css">' + (response.data.css || css) + '</style>').appendTo('head');
                
            } else {
                // Enhanced error message display
                let errorMessage = 'Failed to update theme';
                
                if (response.data && response.data.message) {
                    errorMessage = response.data.message;
                } else if (response.message) {
                    errorMessage = response.message;
                }
                
                console.log('Server Error:', errorMessage); // Debug log
                this.showNotification(errorMessage, 'error');
                
                // If it's a validation error, highlight the issue
                if (errorMessage.toLowerCase().includes('validation') || 
                    errorMessage.toLowerCase().includes('syntax') || 
                    errorMessage.toLowerCase().includes('brace') ||
                    errorMessage.toLowerCase().includes('quote') ||
                    errorMessage.toLowerCase().includes('css')) {
                    this.highlightCSSError(errorMessage);
                }
            }
        },
        error: (xhr, status, error) => {
            console.log('AJAX Error:', { xhr, status, error }); // Debug log
            $button.html(originalHtml).prop('disabled', false);
            
            let errorMessage = 'Network error: Unable to save changes. Please try again.';
            
            // Try to get more specific error info
            if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                errorMessage = xhr.responseJSON.data.message;
            } else if (xhr.responseText) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.data && response.data.message) {
                        errorMessage = response.data.message;
                    }
                } catch (e) {
                    // Keep default error message
                }
            }
            
            this.showNotification(errorMessage, 'error');
            this.highlightCSSError(errorMessage);
        }
    });
},

// Enhanced error detection with line numbers
highlightCSSError: function(errorMessage) {
    const $textarea = $('#css-editor-textarea');
    const css = $textarea.val();
    
    // Add error highlighting to textarea
    $textarea.addClass('mxchat-css-error');
    
    // Only try to find specific error location if we have an error message
    if (errorMessage) {
        // Try to find the specific error location
        const errorInfo = this.findCSSErrorLocation(css, errorMessage);
        
        if (errorInfo) {
            // Show error details near the textarea
            this.showInlineError(errorInfo);
            
            // Try to position cursor near the error
            if (errorInfo.position) {
                $textarea.focus();
                $textarea[0].setSelectionRange(errorInfo.position, errorInfo.position);
            }
        }
    } else {
        // Just focus on textarea if no specific error message
        $textarea.focus();
    }
    
    // Remove error highlighting after user starts typing
    $textarea.one('input', function() {
        $(this).removeClass('mxchat-css-error');
        $('.mxchat-css-inline-error').remove();
    });
},

// Find specific error location in CSS
findCSSErrorLocation: function(css, errorMessage) {
    // Add null check for errorMessage
    if (!errorMessage) {
        return null;
    }
    
    const lines = css.split('\n');
    let errorInfo = null;
    
    // Check for brace errors
    if (errorMessage.includes('brace')) {
        let braceCount = 0;
        let lastOpenBrace = -1;
        
        for (let i = 0; i < css.length; i++) {
            if (css[i] === '{') {
                braceCount++;
                lastOpenBrace = i;
            } else if (css[i] === '}') {
                braceCount--;
                if (braceCount < 0) {
                    // Extra closing brace found
                    const lineInfo = this.getLineFromPosition(css, i);
                    return {
                        line: lineInfo.line,
                        column: lineInfo.column,
                        position: i,
                        message: 'Unexpected closing brace',
                        type: 'brace'
                    };
                }
            }
        }
        
        // Missing closing brace
        if (braceCount > 0 && lastOpenBrace >= 0) {
            const lineInfo = this.getLineFromPosition(css, lastOpenBrace);
            return {
                line: lineInfo.line,
                column: lineInfo.column,
                position: lastOpenBrace,
                message: 'Missing closing brace for this opening brace',
                type: 'brace'
            };
        }
    }
    
    // Check for quote errors
    if (errorMessage.includes('quote')) {
        const quotePattern = /(['"])/g;
        let match;
        let inString = false;
        let stringChar = null;
        
        while ((match = quotePattern.exec(css)) !== null) {
            if (!inString) {
                inString = true;
                stringChar = match[1];
            } else if (match[1] === stringChar) {
                inString = false;
                stringChar = null;
            }
        }
        
        if (inString) {
            const lineInfo = this.getLineFromPosition(css, css.lastIndexOf(stringChar));
            return {
                line: lineInfo.line,
                column: lineInfo.column,
                position: css.lastIndexOf(stringChar),
                message: 'Unclosed string literal',
                type: 'quote'
            };
        }
    }
    
    return null;
},

// Get line and column from character position
getLineFromPosition: function(text, position) {
    const lines = text.substring(0, position).split('\n');
    return {
        line: lines.length,
        column: lines[lines.length - 1].length + 1
    };
},

// Show inline error message
showInlineError: function(errorInfo) {
    // Remove existing error messages
    $('.mxchat-css-inline-error').remove();
    
    const errorHtml = `
        <div class="mxchat-css-inline-error">
            <div class="mxchat-error-icon">⚠️</div>
            <div class="mxchat-error-details">
                <strong>Line ${errorInfo.line}, Column ${errorInfo.column}:</strong>
                <span>${errorInfo.message}</span>
            </div>
        </div>
    `;
    
    // Add after the textarea
    $('#css-editor-textarea').after(errorHtml);
},
        
        // Cancel CSS Editor
        cancelCSSEditor: function() {
            if (confirm('Are you sure you want to discard your changes?')) {
                $('#mxchat-css-editor-modal').hide();
                // Remove any temporary preview
                this.clearAllAIThemeCSS();
                // Reapply the original active theme if exists
                const activeCSS = $('#active-css-code').text();
                if (activeCSS) {
                    $('<style id="mxchat-ai-theme-css">' + activeCSS + '</style>').appendTo('head');
                }
            }
        },
        
        // Update active theme UI after editing
        updateActiveThemeUI: function(css) {
            $('#active-css-code').text(css);
            $('.mxchat-applied-theme-name').append(' <small>(Edited)</small>');
        },
        
        // Update saved theme UI after editing
        updateSavedThemeUI: function(themeName, css) {
            // Find the theme card and update its CSS data
            $('.mxchat-saved-theme-card').each(function() {
                const cardThemeName = $(this).find('h4').text();
                if (cardThemeName === themeName) {
                    $(this).find('.mxchat-theme-preview-btn').data('css', css);
                    $(this).find('.mxchat-theme-apply-btn').data('css', css);
                    $(this).find('.mxchat-theme-edit-btn').data('css', css);
                    
                    // Add edited indicator
                    const $title = $(this).find('h4');
                    if (!$title.find('small').length) {
                        $title.append(' <small>(Edited)</small>');
                    }
                }
            });
        },
                
        // Apply and save in one action (no page refresh)
        applyAndSaveTheme: function() {
            if (!this.generatedCSS) {
                this.showNotification('No theme to apply', 'error');
                return;
            }
            
            const description = $('#ai-theme-description').val().trim();
            
            // Show loading state with SVG spinner
            const $button = $('#apply-ai-theme');
            const originalText = $button.html();
            const spinnerSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="mxchat-spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';
            $button.html(spinnerSvg + ' Applying...').prop('disabled', true);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_apply_and_save_theme',
                    css: this.generatedCSS,
                    description: description,
                    _ajax_nonce: window.mxchatAiThemeNonce
                },
                success: (response) => {
                    $button.html(originalText).prop('disabled', false);
                    
                    if (response.success) {
                        this.showNotification(response.data.message, 'success');
                        
                        // Remove temporary CSS and replace with permanent CSS
                        $('#mxchat-temp-ai-css').remove();
                        $('#mxchat-ai-theme-css').remove(); // Remove any existing permanent CSS
                        
                        // Add the CSS as permanent (same ID as PHP output)
                        $('<style id="mxchat-ai-theme-css">' + this.generatedCSS + '</style>').appendTo('head');
                        
                        // Update the UI to show the applied theme
                        this.updateAppliedThemeSection(response.data.theme_name, this.generatedCSS);
                        
                        // Hide the generated theme section since it's now applied
                        $('#ai-results-section').fadeOut();
                        
                        // Clear the form for next use
                        $('#ai-theme-description').val('');
                        this.generatedCSS = '';
                        this.checkFormValidity();
                        
                    } else {
                        this.showNotification(response.data.message || 'Failed to apply theme', 'error');
                    }
                },
                error: () => {
                    $button.html(originalText).prop('disabled', false);
                    this.showNotification('An error occurred while applying the theme.', 'error');
                }
            });
        },

        applySavedTheme: function(e) {
            const $button = $(e.currentTarget);
            const css = $button.data('css');
            const themeName = $button.closest('.mxchat-saved-theme-card').find('h4').text();
            
            // Show loading state with SVG spinner
            const originalHtml = $button.html();
            const spinnerSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="mxchat-spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';
            $button.html(spinnerSvg).prop('disabled', true);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_apply_generated_theme',
                    css: css,
                    theme_name: themeName,
                    _ajax_nonce: window.mxchatAiThemeNonce
                },
                success: (response) => {
                    $button.html(originalHtml).prop('disabled', false);
                    
                    if (response.success) {
                        this.showNotification('Saved theme applied successfully!', 'success');
                        
                        // Remove temporary CSS and replace with permanent CSS
                        $('#mxchat-temp-ai-css').remove();
                        $('#mxchat-ai-theme-css').remove(); // Remove any existing permanent CSS
                        
                        // Add the CSS as permanent (same ID as PHP output)
                        $('<style id="mxchat-ai-theme-css">' + css + '</style>').appendTo('head');
                        
                        this.updateAppliedThemeSection(themeName, css);
                    } else {
                        this.showNotification(response.data.message || 'Failed to apply theme', 'error');
                    }
                },
                error: () => {
                    $button.html(originalHtml).prop('disabled', false);
                    this.showNotification('An error occurred while applying the theme.', 'error');
                }
            });
        },
        
        clearAllAIThemeCSS: function() {
            // Remove both temporary and permanent AI theme CSS
            $('#mxchat-temp-ai-css').remove();
            $('#mxchat-ai-theme-css').remove();
        },
        
        deleteSavedTheme: function(e) {
            const $button = $(e.currentTarget);
            const themeName = $button.data('theme');
            
            if (!confirm(`Are you sure you want to delete the theme "${themeName}"?`)) {
                return;
            }
            
            // Show loading state with SVG spinner
            const originalHtml = $button.html();
            const spinnerSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="mxchat-spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';
            $button.html(spinnerSvg).prop('disabled', true);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_delete_saved_theme',
                    theme_name: themeName,
                    _ajax_nonce: window.mxchatAiThemeNonce
                },
                success: (response) => {
                    if (response.success) {
                        // Remove the theme card from UI
                        $button.closest('.mxchat-saved-theme-card').fadeOut(300, function() {
                            $(this).remove();
                            
                            // Check if no themes left
                            if ($('.mxchat-saved-theme-card').length === 0) {
                                $('.mxchat-saved-themes-section').fadeOut();
                            }
                        });
                        
                        this.showNotification(response.data.message, 'success');
                    } else {
                        $button.html(originalHtml).prop('disabled', false);
                        this.showNotification(response.data.message || 'Failed to delete theme', 'error');
                    }
                },
                error: () => {
                    $button.html(originalHtml).prop('disabled', false);
                    this.showNotification('An error occurred while deleting the theme.', 'error');
                }
            });
        },
        
        // Update the applied theme section without page refresh
        updateAppliedThemeSection: function(themeName, css) {
            // If css is not provided, use the current generatedCSS or get from DOM
            const activeCSS = css || this.generatedCSS || $('#mxchat-ai-theme-css').text();
            
            let $section = $('.mxchat-active-ai-theme-section');
            
            if ($section.length === 0) {
                // Create the section with SVG icons
                const codeSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="16,18 22,12 16,6"/><polyline points="8,6 2,12 8,18"/></svg>';
                const editSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/></svg>';
                const removeSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>';
                const copySvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="9" y="9" width="13" height="13" rx="2" ry="2"/><path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"/></svg>';
                
                const sectionHtml = `
                    <div class="mxchat-active-ai-theme-section">
                        <h3>Currently Applied AI Theme</h3>
                        <div class="mxchat-applied-theme-info">
                            <p><strong>Theme Name:</strong> <span class="mxchat-applied-theme-name">${themeName}</span></p>
                            <p><small>Applied: Just now</small></p>
                        </div>
                        <div class="mxchat-active-theme-actions">
                            <button id="view-active-css" class="mxchat-button-secondary">${codeSvg} View CSS</button>
                            <button id="edit-active-css" class="mxchat-button-secondary">${editSvg} Edit CSS</button>
                            <button id="remove-active-theme" class="mxchat-button-outline">${removeSvg} Remove Theme</button>
                        </div>
                        <div id="active-css-display" class="mxchat-css-code-container" style="display: none;">
                            <button class="mxchat-copy-button" data-copy-target="#active-css-code">
                                ${copySvg} Copy
                            </button>
                            <pre id="active-css-code" class="mxchat-css-code">${activeCSS}</pre>
                        </div>
                    </div>
                `;
                $('.mxchat-ai-generator-container').append(sectionHtml);
            } else {
                // Update existing section
                $section.find('.mxchat-applied-theme-name').text(themeName);
                $section.find('small').text('Applied: Just now');
                $section.find('#active-css-code').text(activeCSS);
                $section.show();
            }
        },
        
        toggleActiveCSS: function() {
            $('#active-css-display').toggle();
        },
        
        removeActiveTheme: function() {
            if (!confirm('Are you sure you want to remove the active AI theme?')) {
                return;
            }
            
            const $button = $('#remove-active-theme');
            const originalHtml = $button.html();
            const spinnerSvg = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="mxchat-spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';
            $button.html(spinnerSvg + ' Removing...').prop('disabled', true);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'mxchat_remove_active_theme',
                    _ajax_nonce: window.mxchatAiThemeNonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotification('Active theme removed successfully!', 'success');
                        $('.mxchat-active-ai-theme-section').fadeOut();
                        
                        // Remove both temp and permanent AI theme CSS
                        $('#mxchat-temp-ai-css').remove();
                        $('#mxchat-ai-theme-css').remove();
                    } else {
                        $button.html(originalHtml).prop('disabled', false);
                        this.showNotification('Failed to remove active theme', 'error');
                    }
                },
                error: () => {
                    $button.html(originalHtml).prop('disabled', false);
                    this.showNotification('An error occurred while removing the theme.', 'error');
                }
            });
        },
        
        applyTempCSS: function(css) {
            // Remove existing temp CSS
            $('#mxchat-temp-ai-css').remove();
            
            // Remove existing permanent AI theme CSS to prevent conflicts
            $('#mxchat-ai-theme-css').remove();
            
            // Apply new temp CSS with higher specificity to override any remaining styles
            $('<style id="mxchat-temp-ai-css">' + css + '</style>').appendTo('head');
        },
        
        copyCSSCode: function() {
            if (this.generatedCSS) {
                this.copyToClipboard(null, this.generatedCSS);
            }
        },
        
        copyToClipboard: function(e, text) {
            let textToCopy = text;
            
            if (!textToCopy && e) {
                const target = $(e.currentTarget).data('copy-target');
                textToCopy = target ? $(target).text() : '';
            }
            
            if (!textToCopy) return;
            
            navigator.clipboard.writeText(textToCopy).then(() => {
                this.showNotification('CSS copied to clipboard!', 'success');
            }).catch(() => {
                // Fallback for older browsers
                const textArea = document.createElement('textarea');
                textArea.value = textToCopy;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                this.showNotification('CSS copied to clipboard!', 'success');
            });
        },
        
        setLoadingState: function(loading) {
            const $button = $('#generate-ai-theme');
            const $text = $button.find('.button-text');
            const $loading = $button.find('.button-loading');
            
            if (loading) {
                $text.hide();
                $loading.show();
                $button.prop('disabled', true);
            } else {
                $text.show();
                $loading.hide();
                this.checkFormValidity();
            }
        },
        
showNotification: function(message, type) {
    // Remove existing notifications
    $('.mxchat-notification').remove();
    
    // Create notification element
    const notificationClass = 'mxchat-notification mxchat-notification-' + type;
    const notification = $('<div class="' + notificationClass + '">' + message + '</div>');
    
    // Add to body
    $('body').append(notification);
    
    // Show with animation
    notification.css('opacity', '0').animate({opacity: 1}, 300);
    
    // Auto remove after 7 seconds for errors, 5 seconds for others
    const timeout = type === 'error' ? 7000 : 5000;
    setTimeout(() => {
        notification.fadeOut(300, () => notification.remove());
    }, timeout);
    
    // Add click to dismiss
    notification.on('click', () => {
        notification.fadeOut(300, () => notification.remove());
    });
},
    };
    
    // Initialize the AI Theme Generator
    MxChatAiThemeGenerator.init();
    
    // Make it globally accessible
    window.MxChatAiThemeGenerator = MxChatAiThemeGenerator;
});