jQuery(document).ready(function($) {
    'use strict';
    
    const MxChatThemeModelSelector = {
        
        models: {},
        selectedModelId: '',
        selectedModelName: '',
        
        init: function() {
            this.initModels();
            this.bindEvents();
            this.createModelModal();
        },
        
        initModels: function() {
            // Define models - same structure as your core plugin
            this.models = {
                gemini: [
                    { value: 'gemini-2.0-flash', label: 'Gemini 2.0 Flash', description: 'Next-Gen features, speed & multimodal generation' },
                    { value: 'gemini-2.0-flash-lite', label: 'Gemini 2.0 Flash-Lite', description: 'Cost-efficient with low latency' },
                    { value: 'gemini-1.5-pro', label: 'Gemini 1.5 Pro', description: 'Complex reasoning tasks requiring more intelligence' },
                    { value: 'gemini-1.5-flash', label: 'Gemini 1.5 Flash', description: 'Fast and versatile performance' },
                ],
                openai: [
                    { value: 'gpt-5', label: 'GPT-5', description: 'Flagship for coding, reasoning, and agentic tasks across domains' },
                    { value: 'gpt-5-mini', label: 'GPT-5 Mini', description: 'Faster, more cost-efficient for well-defined tasks and precise prompts' },
                    { value: 'gpt-5-nano', label: 'GPT-5 Nano', description: 'Fastest and cheapest; ideal for summarization and classification' },
                    { value: 'gpt-4.1-2025-04-14', label: 'GPT-4.1', description: 'Flagship model for complex tasks' },
                    { value: 'gpt-4o', label: 'GPT-4o', description: 'Recommended for most use cases' },
                    { value: 'gpt-4o-mini', label: 'GPT-4o Mini', description: 'Fast and lightweight' },
                    { value: 'gpt-4-turbo', label: 'GPT-4 Turbo', description: 'High-performance model' },
                    { value: 'gpt-4', label: 'GPT-4', description: 'High intelligence model' },
                    { value: 'gpt-3.5-turbo', label: 'GPT-3.5 Turbo', description: 'Affordable and fast' },
                ],
                        claude: [
                        { value: 'claude-sonnet-4-5-20250929', label: 'Claude Sonnet 4.5', description: 'Best for complex agents and coding' },
                        { value: 'claude-opus-4-1-20250805', label: 'Claude Opus 4.1', description: 'Exceptional for specialized complex tasks' },
                        { value: 'claude-haiku-4-5-20251001', label: 'Claude Haiku 4.5', description: 'Fastest and most intelligent Haiku' },
                        { value: 'claude-opus-4-20250514', label: 'Claude 4 Opus', description: 'Most capable Claude model' },
                        { value: 'claude-sonnet-4-20250514', label: 'Claude 4 Sonnet', description: 'High performance' },
                        { value: 'claude-3-7-sonnet-20250219', label: 'Claude 3.7 Sonnet', description: 'High intelligence' },
                        { value: 'claude-3-opus-20240229', label: 'Claude 3 Opus', description: 'Highly complex tasks' },
                        { value: 'claude-3-sonnet-20240229', label: 'Claude 3 Sonnet', description: 'Balanced performance' },
                        { value: 'claude-3-haiku-20240307', label: 'Claude 3 Haiku', description: 'Fastest Claude model' },
                    ],
                xai: [
                    { value: 'grok-4-0709', label: 'Grok 4', description: 'Latest flagship model - unparalleled performance in natural language, math and reasoning' }, // Added Grok-4
                    { value: 'grok-3-beta', label: 'Grok-3', description: 'Powerful model with 131K context' },
                    { value: 'grok-3-fast-beta', label: 'Grok-3 Fast', description: 'High performance with faster responses' },
                    { value: 'grok-3-mini-beta', label: 'Grok-3 Mini', description: 'Affordable model with good performance' },
                    { value: 'grok-3-mini-fast-beta', label: 'Grok-3 Mini Fast', description: 'Quick and cost-effective' },
                    { value: 'grok-2', label: 'Grok 2', description: 'Latest X.AI model' },
                ],
                deepseek: [
                    { value: 'deepseek-chat', label: 'DeepSeek-V3', description: 'Advanced AI assistant' },
                ],
            };
        },
        
bindEvents: function() {
    const self = this;
    
    // Handle model selector button click with better debugging
    $(document).on('click', '#mxchat-ai-model-selector .mxchat-model-selector-btn', function(e) {
        //console.log('Model selector button clicked'); // Debug line
        e.preventDefault();
        self.openModelModal();
    });
    
    // Alternative binding in case the above doesn't work
    $(document).on('click', '.mxchat-model-selector-btn', function(e) {
        //console.log('Alternative model selector button clicked'); // Debug line
        e.preventDefault();
        if ($(this).closest('#mxchat-ai-model-selector').length) {
            self.openModelModal();
        }
    });
},
        
        createModelModal: function() {
            const self = this;
            
            // Create modal HTML with MxChat styling
            const modalHtml = `
                <div id="mxchat-theme-model-selector-modal" class="mxchat-theme-model-selector-modal">
                    <div class="mxchat-theme-model-selector-modal-content">
                        <div class="mxchat-theme-model-selector-modal-header">
                            <h3>Select AI Model</h3>
                            <span class="mxchat-theme-model-selector-modal-close">
                                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <line x1="18" y1="6" x2="6" y2="18"/>
                                    <line x1="6" y1="6" x2="18" y2="18"/>
                                </svg>
                            </span>
                        </div>
                        <div class="mxchat-theme-model-selector-modal-body">
                            <div class="mxchat-theme-model-search-container">
                                <input type="text" id="mxchat-theme-model-search-input" class="mxchat-theme-model-search-input" placeholder="Search models...">
                            </div>
                            <div class="mxchat-theme-model-selector-categories">
                                <button class="mxchat-theme-model-category-btn active" data-category="all">All</button>
                                <button class="mxchat-theme-model-category-btn" data-category="claude">Claude</button>
                                <button class="mxchat-theme-model-category-btn" data-category="openai">OpenAI</button>
                                <button class="mxchat-theme-model-category-btn" data-category="gemini">Google Gemini</button>
                                <button class="mxchat-theme-model-category-btn" data-category="xai">X.AI</button>
                                <button class="mxchat-theme-model-category-btn" data-category="deepseek">DeepSeek</button>
                            </div>
                            <div class="mxchat-theme-model-selector-grid" id="mxchat-theme-models-grid"></div>
                        </div>
                        <div class="mxchat-theme-model-selector-modal-footer">
                            <button id="mxchat-theme-cancel-model-selection" class="button mxchat-theme-model-cancel-btn">Cancel</button>
                        </div>
                    </div>
                </div>
            `;
            
            // Add modal to page
            $('body').append(modalHtml);
            
            // Bind modal events
            this.bindModalEvents();
        },
        
        bindModalEvents: function() {
            const self = this;
            
            // Close modal
            $(document).on('click', '.mxchat-theme-model-selector-modal-close, #mxchat-theme-cancel-model-selection', function() {
                $('#mxchat-theme-model-selector-modal').css('display', 'none'); // Use css() instead of hide()
            });
            
            // Close on outside click
            $(document).on('click', '#mxchat-theme-model-selector-modal', function(e) {
                if (e.target === this) {
                    $(this).css('display', 'none'); // Use css() instead of hide()
                }
            });
            
            // Category filtering
            $(document).on('click', '.mxchat-theme-model-category-btn', function() {
                $('.mxchat-theme-model-category-btn').removeClass('active');
                $(this).addClass('active');
                const category = $(this).data('category');
                const searchTerm = $('#mxchat-theme-model-search-input').val();
                self.populateModelsGrid(searchTerm, category);
            });
            
            // Search functionality
            $(document).on('input', '#mxchat-theme-model-search-input', function() {
                const searchTerm = $(this).val();
                const activeCategory = $('.mxchat-theme-model-category-btn.active').data('category');
                self.populateModelsGrid(searchTerm, activeCategory);
            });
            
            // Model selection
            $(document).on('click', '.mxchat-theme-model-selector-card', function() {
                const modelId = $(this).data('value');
                const modelName = $(this).find('.mxchat-theme-model-selector-title').text();
                
                self.selectModel(modelId, modelName);
                $('#mxchat-theme-model-selector-modal').hide();
            });
        },
        
openModelModal: function() {
    //console.log('Opening model modal'); // Debug line
    $('#mxchat-theme-model-selector-modal').css('display', 'flex'); // Use flex instead of show()
    this.populateModelsGrid('', 'all');
    //console.log('Modal should be visible now'); // Debug line
},
        populateModelsGrid: function(filter = '', category = 'all') {
            const $grid = $('#mxchat-theme-models-grid');
            $grid.empty();
            
            let allModels = [];
            
            // Collect models based on category
            Object.keys(this.models).forEach(key => {
                if (category === 'all' || category === key) {
                    allModels = allModels.concat(this.models[key].map(model => {
                        return {...model, category: key};
                    }));
                }
            });
            
            // Filter by search term
            if (filter) {
                const lowerFilter = filter.toLowerCase();
                allModels = allModels.filter(model => 
                    model.label.toLowerCase().includes(lowerFilter) || 
                    (model.description && model.description.toLowerCase().includes(lowerFilter)) ||
                    (model.category && model.category.toLowerCase().includes(lowerFilter))
                );
            }
            
            // Show message if no models
            if (allModels.length === 0) {
                $grid.html('<div class="mxchat-theme-no-models-message">No models match your search criteria</div>');
                return;
            }
            
            // Create model cards
            allModels.forEach(model => {
                const isSelected = this.selectedModelId === model.value;
                const providerLabel = this.getProviderLabel(model.category);
                
                const $modelCard = $(`
                    <div class="mxchat-theme-model-selector-card ${isSelected ? 'mxchat-theme-model-selected' : ''}" data-value="${model.value}">
                        <div class="mxchat-theme-model-selector-icon">${this.getModelIcon(model.value)}</div>
                        <div class="mxchat-theme-model-selector-info">
                            <h4 class="mxchat-theme-model-selector-title">${model.label}</h4>
                            <p class="mxchat-theme-model-selector-description">
                                <span class="mxchat-theme-model-provider">${providerLabel}</span>
                                ${model.description ? ' - ' + model.description : ''}
                            </p>
                        </div>
                        ${isSelected ? '<div class="mxchat-theme-model-selector-checkmark">✓</div>' : ''}
                    </div>
                `);
                $grid.append($modelCard);
            });
        },
        
        getProviderLabel: function(category) {
            const providers = {
                'gemini': 'Google Gemini',
                'openai': 'OpenAI',
                'claude': 'Claude',
                'xai': 'X.AI',
                'deepseek': 'DeepSeek'
            };
            
            return providers[category] || category;
        },
        
        getModelIcon: function(modelValue) {
            if (modelValue.startsWith('gemini-')) {
                return '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 48 48" class="mxchat-theme-model-icon-gemini"><defs><path id="a" d="M44.5 20H24v8.5h11.8C34.7 33.9 30.1 37 24 37c-7.2 0-13-5.8-13-13s5.8-13 13-13c3.1 0 5.9 1.1 8.1 2.9l6.4-6.4C34.6 4.1 29.6 2 24 2 11.8 2 2 11.8 2 24s9.8 22 22 22c11 0 21-8 21-22 0-1.3-.2-2.7-.5-4z"></path></defs><clipPath id="b"><use xlink:href="#a" overflow="visible"></use></clipPath><path clip-path="url(#b)" fill="#FBBC05" d="M0 37V11l17 13z"></path><path clip-path="url(#b)" fill="#EA4335" d="M0 11l17 13 7-6.1L48 14V0H0z"></path><path clip-path="url(#b)" fill="#34A853" d="M0 37l30-23 7.9 1L48 0v48H0z"></path><path clip-path="url(#b)" fill="#4285F4" d="M48 48L17 24l-4-3 35-10z"></path></svg>';
            }
            if (modelValue.startsWith('gpt-')) {
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 320" class="mxchat-theme-model-icon-openai"><path fill="currentColor" d="M297 131a80.6 80.6 0 0 0-93.7-104.2 80.6 80.6 0 0 0-137 29A80.6 80.6 0 0 0 23 189a80.6 80.6 0 0 0 93.7 104.2 80.6 80.6 0 0 0 137-29A80.7 80.7 0 0 0 297.1 131zM176.9 299c-14 .1-27.6-4.8-38.4-13.8l1.9-1 63.7-36.9c3.3-1.8 5.3-5.3 5.2-9v-89.9l27 15.6c.3.1.4.4.5.7v74.4a60 60 0 0 1-60 60zM47.9 244a59.7 59.7 0 0 1-7.1-40.1l1.9 1.1 63.7 36.8c3.2 1.9 7.2 1.9 10.5 0l77.8-45V228c0 .3-.2.6-.4.8L129.9 266a60 60 0 0 1-82-22zM31.2 105c7-12.2 18-21.5 31.2-26.3v75.8c0 3.7 2 7.2 5.2 9l77.8 45-27 15.5a1 1 0 0 1-.9 0L53.1 187a60 60 0 0 1-22-82zm221.2 51.5-77.8-45 27-15.5a1 1 0 0 1 .9 0l64.4 37.1a60 60 0 0 1-9.3 108.2v-75.8c0-3.7-2-7.2-5.2-9zm26.8-40.4-1.9-1.1-63.7-36.8a10.4 10.4 0 0 0-10.5 0L125.4 123V92c0-.3 0-.6.3-.8L190.1 54a60 60 0 0 1 89.1 62.1zm-168.5 55.4-27-15.5a1 1 0 0 1-.4-.7V80.9a60 60 0 0 1 98.3-46.1l-1.9 1L116 72.8a10.3 10.3 0 0 0-5.2 9v89.8zm14.6-31.5 34.7-20 34.6 20v40L160 200l-34.7-20z"></path></svg>';
            }
            if (modelValue.startsWith('claude-')) {
                return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 176" fill="none" class="mxchat-theme-model-icon-claude"><path fill="currentColor" d="m147.487 0l70.081 175.78H256L185.919 0zM66.183 106.221l23.98-61.774l23.98 61.774zM70.07 0L0 175.78h39.18l14.33-36.914h73.308l14.328 36.914h39.179L110.255 0z"></path></svg>';
            }
            if (modelValue.startsWith('grok-')) {
                return '<svg fill="currentColor" fill-rule="evenodd" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" class="mxchat-theme-model-icon-xai"><path d="M6.469 8.776L16.512 23h-4.464L2.005 8.776H6.47zm-.004 7.9l2.233 3.164L6.467 23H2l4.465-6.324zM22 2.582V23h-3.659V7.764L22 2.582zM22 1l-9.952 14.095-2.233-3.163L17.533 1H22z"></path></svg>';
            }
            if (modelValue.startsWith('deepseek-')) {
                return '<svg height="1em" viewBox="0 0 24 24" width="1em" xmlns="http://www.w3.org/2000/svg" class="mxchat-theme-model-icon-deepseek"><path d="M23.748 4.482c-.254-.124-.364.113-.512.234-.051.039-.094.09-.137.136-.372.397-.806.657-1.373.626-.829-.046-1.537.214-2.163.848-.133-.782-.575-1.248-1.247-1.548-.352-.156-.708-.311-.955-.65-.172-.241-.219-.51-.305-.774-.055-.16-.11-.323-.293-.35-.2-.031-.278.136-.356.276-.313.572-.434 1.202-.422 1.84.027 1.436.633 2.58 1.838 3.393.137.093.172.187.129.323-.082.28-.18.552-.266.833-.055.179-.137.217-.329.14a5.526 5.526 0 01-1.736-1.18c-.857-.828-1.631-1.742-2.597-2.458a11.365 11.365 0 00-.689-.471c-.985-.957.13-1.743.388-1.836.27-.098.093-.432-.779-.428-.872.004-1.67.295-2.687.684a3.055 3.055 0 01-.465.137 9.597 9.597 0 00-2.883-.102c-1.885.21-3.39 1.102-4.497 2.623C.082 8.606-.231 10.684.152 12.85c.403 2.284 1.569 4.175 3.36 5.653 1.858 1.533 3.997 2.284 6.438 2.14 1.482-.085 3.133-.284 4.994-1.86.47.234.962.327 1.78.397.63.059 1.236-.03 1.705-.128.735-.156.684-.837.419-.961-2.155-1.004-1.682-.595-2.113-.926 1.096-1.296 2.746-2.642 3.392-7.003.05-.347.007-.565 0-.845-.004-.17.035-.237.23-.256a4.173 4.173 0 001.545-.475c1.396-.763 1.96-2.015 2.093-3.517.02-.23-.004-.467-.247-.588zM11.581 18c-2.089-1.642-3.102-2.183-3.52-2.16-.392.024-.321.471-.235.763.09.288.207.486.371.739.114.167.192.416-.113.603-.673.416-1.842-.14-1.897-.167-1.361-.802-2.5-1.86-3.301-3.307-.774-1.393-1.224-2.887-1.298-4.482-.02-.386.093-.522.477-.592a4.696 4.696 0 011.529-.039c2.132.312 3.946 1.265 5.468 2.774.868.86 1.525 1.887 2.202 2.891.72 1.066 1.494 2.082 2.48 2.914.348.292.625.514.891.677-.802.09-2.14.11-3.054-.614zm1-6.44a.306.306 0 01.415-.287.302.302 0 01.2.288.306.306 0 01-.31.307.303.303 0 01-.304-.308zm3.11 1.596c-.2.081-.399.151-.59.16a1.245 1.245 0 01-.798-.254c-.274-.23-.47-.358-.552-.758a1.73 1.73 0 01.016-.588c.07-.327-.008-.537-.239-.727-.187-.156-.426-.199-.688-.199a.559.559 0 01-.254-.078c-.11-.054-.2-.19-.114-.358.028-.054.16-.186.192-.21.356-.202.767-.136 1.146.016.352.144.618.408 1.001.782.391.451.462.576.685.914.176.265.336.537.445.848.067.195-.019.354-.25.452z" fill="currentColor"></path></svg>';
            }
            return '<span class="dashicons dashicons-admin-generic mxchat-theme-model-icon-generic"></span>';
        },
        
        selectModel: function(modelId, modelName) {
            this.selectedModelId = modelId;
            this.selectedModelName = modelName;
            
            // Update the button text
            $('#selected-model-name').text(modelName);
            $('#selected-model-id').val(modelId);
            
            // Update AI generator if available
            if (window.MxChatAiThemeGenerator) {
                window.MxChatAiThemeGenerator.selectedModelId = modelId;
                window.MxChatAiThemeGenerator.selectedModelName = modelName;
                if (typeof window.MxChatAiThemeGenerator.checkFormValidity === 'function') {
                    window.MxChatAiThemeGenerator.checkFormValidity();
                }
            }
            
            // Close modal with css() instead of hide()
            $('#mxchat-theme-model-selector-modal').css('display', 'none');
            
            // Visual feedback in the grid
            $('.mxchat-theme-model-selector-card').removeClass('mxchat-theme-model-selected');
            $('.mxchat-theme-model-selector-checkmark').remove();
            const $selectedCard = $('.mxchat-theme-model-selector-card[data-value="' + modelId + '"]');
            $selectedCard.addClass('mxchat-theme-model-selected');
            $selectedCard.append('<div class="mxchat-theme-model-selector-checkmark">✓</div>');
        },
    };
    
    // Initialize the model selector
    MxChatThemeModelSelector.init();
    
    // Make it globally available
    window.MxChatThemeModelSelector = MxChatThemeModelSelector;
});