<?php
/**
 * AI Theme Generator Class
 * Integrates with MxChat core plugin's model selection and API infrastructure
 */

class MxChat_AI_Theme_Generator {

    private $core_options;
    private $theme_options;
    private $api_keys;
    
public function __construct() {
    $this->core_options = get_option('mxchat_options', array());
    $this->theme_options = get_option('mxchat_theme_options', array());
    $this->init_api_keys();
    
    add_action('wp_ajax_mxchat_generate_ai_theme', array($this, 'generate_ai_theme_callback'));
    add_action('wp_ajax_mxchat_apply_generated_theme', array($this, 'apply_generated_theme_callback'));
    add_action('wp_ajax_mxchat_apply_and_save_theme', array($this, 'apply_and_save_theme_callback'));
    add_action('wp_ajax_mxchat_delete_saved_theme', array($this, 'delete_saved_theme_callback'));
    add_action('wp_ajax_mxchat_remove_active_theme', array($this, 'remove_active_theme_callback'));
    add_action('wp_ajax_mxchat_update_theme_css', array($this, 'update_theme_css_callback')); // ADD THIS LINE
}
    
    /**
     * Initialize API keys from core plugin
     */
    private function init_api_keys() {
        // Get the main plugin options - same way as admin chat addon
        $main_plugin_options = get_option('mxchat_options', array());
        
        $this->api_keys = array(
            'openai' => isset($main_plugin_options['api_key']) ? $main_plugin_options['api_key'] : '',
            'claude' => isset($main_plugin_options['claude_api_key']) ? $main_plugin_options['claude_api_key'] : '',
            'gemini' => isset($main_plugin_options['gemini_api_key']) ? $main_plugin_options['gemini_api_key'] : '',
            'xai' => isset($main_plugin_options['xai_api_key']) ? $main_plugin_options['xai_api_key'] : '',
            'deepseek' => isset($main_plugin_options['deepseek_api_key']) ? $main_plugin_options['deepseek_api_key'] : '',
        );
    }
    
    /**
     * Get available models from core plugin
     */
    private function get_available_models() {
        // Use the same method as your core plugin to get models
        $models = array();
        
        // Check if the core plugin function exists
        if (function_exists('mxchat_get_available_models')) {
            $models = mxchat_get_available_models();
        } else {
            // Fallback - get models from options or define basic set
            $models = $this->get_fallback_models();
        }
        
        return $models;
    }
    
    /**
     * Fallback models if core function not available
     */
    private function get_fallback_models() {
        return array(
            'claude' => array(
                array('value' => 'claude-4-sonnet-20250514', 'label' => 'Claude 4 Sonnet', 'description' => 'Advanced Intelligence'),
                array('value' => 'claude-3-5-sonnet-20241022', 'label' => 'Claude 3.5 Sonnet', 'description' => 'Intelligent'),
            ),
            'openai' => array(
                array('value' => 'gpt-4o', 'label' => 'GPT-4o', 'description' => 'Recommended for most use cases'),
                array('value' => 'gpt-4o-mini', 'label' => 'GPT-4o Mini', 'description' => 'Fast and lightweight'),
            ),
            'deepseek' => array(
                array('value' => 'deepseek-chat', 'label' => 'DeepSeek-V3', 'description' => 'Advanced AI assistant'),
            ),
            'gemini' => array(
                array('value' => 'gemini-1.5-pro', 'label' => 'Gemini 1.5 Pro', 'description' => 'Google\'s most capable model'),
                array('value' => 'gemini-1.5-flash', 'label' => 'Gemini 1.5 Flash', 'description' => 'Fast and efficient'),
            ),
            'xai' => array(
                array('value' => 'grok-beta', 'label' => 'Grok Beta', 'description' => 'xAI\'s reasoning model'),
            )
        );
    }
    
    /**
     * Generate AI theme based on user description
     */
    public function generate_ai_theme_callback() {
        check_ajax_referer('mxchat_ai_theme_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-theme')));
        }
        
        $description = sanitize_textarea_field($_POST['description'] ?? '');
        $model_id = sanitize_text_field($_POST['model_id'] ?? '');
        
        if (empty($description)) {
            wp_send_json_error(array('message' => __('Theme description is required', 'mxchat-theme')));
        }
        
        if (empty($model_id)) {
            wp_send_json_error(array('message' => __('Please select a model', 'mxchat-theme')));
        }
        
        try {
            $css_code = $this->call_ai_api($description, $model_id);
            
            if ($css_code) {
                // Use CSS Builder to validate and sanitize
                require_once dirname(__FILE__) . '/class-theme-css-builder.php';
                $css_builder = new MxChat_Theme_CSS_Builder();
                
                $sanitized_css = $css_builder->validate_and_sanitize_css($css_code);
                
                if (is_wp_error($sanitized_css)) {
                    wp_send_json_error(array('message' => 'Generated CSS validation failed: ' . $sanitized_css->get_error_message()));
                    return;
                }
                
                wp_send_json_success(array(
                    'css' => $sanitized_css,
                    'message' => __('Theme generated successfully!', 'mxchat-theme')
                ));
            } else {
                wp_send_json_error(array('message' => __('Failed to generate theme', 'mxchat-theme')));
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * Apply generated theme without page refresh
     */
    public function apply_generated_theme_callback() {
        check_ajax_referer('mxchat_ai_theme_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-theme')));
        }
        
        $css = wp_unslash($_POST['css'] ?? '');
        $theme_name = sanitize_text_field($_POST['theme_name'] ?? '');
        
        if (empty($css)) {
            wp_send_json_error(array('message' => __('No CSS provided', 'mxchat-theme')));
        }
        
        // Save the CSS as active theme
        $this->theme_options['active_ai_theme_css'] = $css;
        $this->theme_options['active_ai_theme_name'] = $theme_name;
        $this->theme_options['active_ai_theme_applied_at'] = current_time('mysql');
        
        update_option('mxchat_theme_options', $this->theme_options);
        
        wp_send_json_success(array(
            'message' => __('Theme applied successfully!', 'mxchat-theme'),
            'theme_name' => $theme_name,
            'no_refresh' => true
        ));
    }
    
    /**
     * Apply and save theme in one action
     */
    public function apply_and_save_theme_callback() {
        check_ajax_referer('mxchat_ai_theme_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-theme')));
        }
        
        $css = wp_unslash($_POST['css'] ?? '');
        $description = sanitize_textarea_field($_POST['description'] ?? '');
        $custom_name = sanitize_text_field($_POST['custom_name'] ?? '');
        
        if (empty($css)) {
            wp_send_json_error(array('message' => __('No CSS provided', 'mxchat-theme')));
        }
        
        // Generate theme name if not provided
        $theme_name = $custom_name ?: $this->generate_theme_name($description);
        
        // Save to saved themes first
        if (!isset($this->theme_options['saved_ai_themes'])) {
            $this->theme_options['saved_ai_themes'] = array();
        }
        
        $this->theme_options['saved_ai_themes'][$theme_name] = array(
            'css' => $css,
            'created' => current_time('mysql'),
            'description' => $description
        );
        
        // Apply as active theme
        $this->theme_options['active_ai_theme_css'] = $css;
        $this->theme_options['active_ai_theme_name'] = $theme_name;
        $this->theme_options['active_ai_theme_applied_at'] = current_time('mysql');
        
        update_option('mxchat_theme_options', $this->theme_options);
        
        wp_send_json_success(array(
            'message' => sprintf(__('Theme "%s" saved and applied successfully!', 'mxchat-theme'), $theme_name),
            'theme_name' => $theme_name,
            'no_refresh' => true
        ));
    }
    
    
/**
 * Update theme CSS - NEW METHOD for editing functionality
 */
public function update_theme_css_callback() {
    check_ajax_referer('mxchat_ai_theme_nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized access', 'mxchat-theme')));
    }
    
    $css = wp_unslash($_POST['css'] ?? '');
    $theme_name = sanitize_text_field($_POST['theme_name'] ?? '');
    $source = sanitize_text_field($_POST['source'] ?? ''); // 'active' or 'saved'
    
    if (empty($css)) {
        wp_send_json_error(array('message' => __('CSS cannot be empty', 'mxchat-theme')));
    }
    
    if (empty($theme_name)) {
        wp_send_json_error(array('message' => __('Theme name is required', 'mxchat-theme')));
    }
    
    // Enhanced CSS validation with detailed error messages
    $validation_result = $this->validate_css_detailed($css);
    if (is_wp_error($validation_result)) {
        wp_send_json_error(array('message' => $validation_result->get_error_message()));
        return;
    }
    
    // Update based on source
    if ($source === 'active') {
        // Update active theme
        $this->theme_options['active_ai_theme_css'] = $css;
        $this->theme_options['active_ai_theme_name'] = $theme_name;
        $this->theme_options['active_ai_theme_applied_at'] = current_time('mysql');
        
        // Also update in saved themes if it exists there
        if (isset($this->theme_options['saved_ai_themes'][$theme_name])) {
            $this->theme_options['saved_ai_themes'][$theme_name]['css'] = $css;
            $this->theme_options['saved_ai_themes'][$theme_name]['updated'] = current_time('mysql');
        }
        
    } else if ($source === 'saved') {
        // Update saved theme
        if (isset($this->theme_options['saved_ai_themes'][$theme_name])) {
            $this->theme_options['saved_ai_themes'][$theme_name]['css'] = $css;
            $this->theme_options['saved_ai_themes'][$theme_name]['updated'] = current_time('mysql');
            
            // If this is also the active theme, update that too
            $active_theme_name = $this->theme_options['active_ai_theme_name'] ?? '';
            if ($active_theme_name === $theme_name) {
                $this->theme_options['active_ai_theme_css'] = $css;
                $this->theme_options['active_ai_theme_applied_at'] = current_time('mysql');
            }
        } else {
            wp_send_json_error(array('message' => __('Saved theme not found', 'mxchat-theme')));
            return;
        }
    } else {
        wp_send_json_error(array('message' => __('Invalid source specified', 'mxchat-theme')));
        return;
    }
    
    update_option('mxchat_theme_options', $this->theme_options);
    
    wp_send_json_success(array(
        'message' => sprintf(__('Theme "%s" updated successfully!', 'mxchat-theme'), $theme_name),
        'theme_name' => $theme_name,
        'css' => $css,
        'no_refresh' => true
    ));
}

/**
 * Enhanced CSS validation with detailed error messages - NEW
 */
private function validate_css_detailed($css) {
    // Basic CSS validation
    if (empty(trim($css))) {
        return new WP_Error('empty_css', 'CSS cannot be empty');
    }
    
    // Check for basic CSS syntax - brace matching
    $open_braces = substr_count($css, '{');
    $close_braces = substr_count($css, '}');
    $brace_diff = $open_braces - $close_braces;
    
    if ($brace_diff > 0) {
        return new WP_Error('syntax_error', sprintf('CSS Syntax Error: Missing %d closing brace(s) "}"', $brace_diff));
    } elseif ($brace_diff < 0) {
        return new WP_Error('syntax_error', sprintf('CSS Syntax Error: Missing %d opening brace(s) "{"', abs($brace_diff)));
    }
    
    // Check for potentially dangerous content
    $dangerous_patterns = array(
        '/javascript:/i' => 'JavaScript URLs are not allowed in CSS',
        '/expression\s*\(/i' => 'CSS expressions are not allowed for security reasons',
        '/import\s*[\'"]/i' => 'CSS @import statements are not allowed',
        '/@import\s+url\s*\(/i' => 'CSS @import URLs are not allowed',
        '/behavior\s*:/i' => 'CSS behavior property is not allowed',
        '/moz-binding/i' => 'Mozilla binding is not allowed',
        '/data:(?!image)/i' => 'Data URLs (except images) are not allowed',
    );
    
    foreach ($dangerous_patterns as $pattern => $message) {
        if (preg_match($pattern, $css)) {
            return new WP_Error('dangerous_css', $message);
        }
    }
    
    // Check for unclosed strings
    $single_quotes = substr_count($css, "'") - substr_count($css, "\\'");
    $double_quotes = substr_count($css, '"') - substr_count($css, '\\"');
    
    if ($single_quotes % 2 !== 0) {
        return new WP_Error('syntax_error', 'CSS Syntax Error: Unclosed single quote');
    }
    
    if ($double_quotes % 2 !== 0) {
        return new WP_Error('syntax_error', 'CSS Syntax Error: Unclosed double quote');
    }
    
    return true;
}

/**
 * Delete saved theme - UPDATED to handle active theme removal
 */
public function delete_saved_theme_callback() {
    check_ajax_referer('mxchat_ai_theme_nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-theme')));
    }
    
    $theme_name = sanitize_text_field($_POST['theme_name'] ?? '');
    
    if (empty($theme_name)) {
        wp_send_json_error(array('message' => __('Theme name is required', 'mxchat-theme')));
    }
    
    // Remove from saved themes
    if (isset($this->theme_options['saved_ai_themes'][$theme_name])) {
        unset($this->theme_options['saved_ai_themes'][$theme_name]);
        
        // NEW: If this was the active theme, remove it from active as well
        $active_theme_name = $this->theme_options['active_ai_theme_name'] ?? '';
        if ($active_theme_name === $theme_name) {
            unset($this->theme_options['active_ai_theme_css']);
            unset($this->theme_options['active_ai_theme_name']);
            unset($this->theme_options['active_ai_theme_applied_at']);
        }
        
        update_option('mxchat_theme_options', $this->theme_options);
        
        wp_send_json_success(array(
            'message' => sprintf(__('Theme "%s" deleted successfully!', 'mxchat-theme'), $theme_name),
            'removed_active' => ($active_theme_name === $theme_name), // NEW
            'no_refresh' => true
        ));
    } else {
        wp_send_json_error(array('message' => __('Theme not found', 'mxchat-theme')));
    }
}
    
    /**
     * Remove active AI theme
     */
    public function remove_active_theme_callback() {
        check_ajax_referer('mxchat_ai_theme_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'mxchat-theme')));
        }
        
        // Clear the active theme
        unset($this->theme_options['active_ai_theme_css']);
        unset($this->theme_options['active_ai_theme_name']);
        unset($this->theme_options['active_ai_theme_applied_at']);
        
        update_option('mxchat_theme_options', $this->theme_options);
        
        wp_send_json_success(array(
            'message' => __('Active theme removed successfully!', 'mxchat-theme'),
            'no_refresh' => true
        ));
    }
    
    /**
     * Generate a theme name based on description
     */
    private function generate_theme_name($description) {
        // Extract first few words and create a name
        $words = explode(' ', $description);
        $name_words = array_slice($words, 0, 3);
        $base_name = ucwords(implode(' ', $name_words));
        
        // Add timestamp to make it unique
        $timestamp = date('M j, H:i');
        
        return $base_name . ' - ' . $timestamp;
    }
    
    /**
     * Call AI API using core plugin's infrastructure
     */
    private function call_ai_api($description, $model_id) {
        // Get the provider from model ID
        $provider = $this->get_provider_from_model($model_id);
        
        // Build the prompt
        $prompt = $this->build_theme_prompt($description);
        
        // Use core plugin's API calling mechanism
        if (function_exists('mxchat_call_api')) {
            $response = mxchat_call_api($provider, $model_id, $prompt);
            return $this->extract_css_from_response($response);
        }
        
        // Fallback to manual API call
        return $this->manual_api_call($provider, $model_id, $prompt);
    }
    
    /**
     * Build the prompt for AI using the CSS Builder
     */
    private function build_theme_prompt($description) {
        // Use the CSS Builder to create a proper prompt
        require_once dirname(__FILE__) . '/class-theme-css-builder.php';
        $css_builder = new MxChat_Theme_CSS_Builder();
        
        return $css_builder->build_ai_prompt($description);
    }
    
    /**
     * Get provider from model ID
     */
    private function get_provider_from_model($model_id) {
        if (strpos($model_id, 'claude-') === 0) return 'claude';
        if (strpos($model_id, 'gpt-') === 0) return 'openai';
        if (strpos($model_id, 'deepseek-') === 0) return 'deepseek';
        if (strpos($model_id, 'grok-') === 0) return 'xai';
        if (strpos($model_id, 'gemini-') === 0) return 'gemini';
        
        return 'claude'; // Default fallback
    }
    
    /**
     * Extract CSS from AI response
     */
    private function extract_css_from_response($response) {
        // Remove markdown code blocks if present
        $css = preg_replace('/```css\s*/', '', $response);
        $css = preg_replace('/```\s*$/', '', $css);
        
        // Remove any non-CSS content before the first CSS rule
        $css = preg_replace('/^[^{]*?(?=\s*[.#]|\s*@)/', '', $css);
        
        return trim($css);
    }
    
    /**
     * Manual API call fallback
     */
    private function manual_api_call($provider, $model_id, $prompt) {
        switch ($provider) {
            case 'claude':
                return $this->call_claude_api($model_id, $prompt);
            case 'openai':
                return $this->call_openai_api($model_id, $prompt);
            case 'deepseek':
                return $this->call_deepseek_api($model_id, $prompt);
            case 'gemini':
                return $this->call_gemini_api($model_id, $prompt);
            case 'xai':
                return $this->call_xai_api($model_id, $prompt);
            default:
                throw new Exception('Unsupported provider: ' . $provider);
        }
    }
    
    /**
     * Claude API call - using same pattern as admin chat addon
     */
    private function call_claude_api($model_id, $prompt) {
        $api_key = $this->api_keys['claude'];
        
        if (empty($api_key)) {
            throw new Exception('Claude API key not configured');
        }
        
        // Create messages array with system message
        $messages = array(
            array(
                'role' => 'user',
                'content' => $prompt
            )
        );
        
        $system_message = 'You are a CSS expert specializing in chatbot interface design. Generate clean, valid CSS code with !important declarations.';
        
        $body = json_encode(array(
            'model' => $model_id,
            'max_tokens' => 4096,
            'temperature' => 0.7,
            'messages' => $messages,
            'system' => $system_message
        ));
        
        $args = array(
            'body' => $body,
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01'
            ),
            'timeout' => 120,
            'blocking' => true,
            'sslverify' => true,
        );
        
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', $args);
        
        if (is_wp_error($response)) {
            throw new Exception('Claude API call failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($response_code !== 200) {
            $error_message = isset($response_body['error']['message']) 
                ? $response_body['error']['message'] 
                : 'API returned status code ' . $response_code;
            throw new Exception($error_message);
        }
        
        if (isset($response_body['content'][0]['text'])) {
            return $this->extract_css_from_response($response_body['content'][0]['text']);
        }
        
        throw new Exception('Invalid Claude API response format');
    }
    
    /**
     * OpenAI API call - using same pattern as admin chat addon
     */
    private function call_openai_api($model_id, $prompt) {
        $api_key = $this->api_keys['openai'];
        
        if (empty($api_key)) {
            throw new Exception('OpenAI API key not configured');
        }
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'body' => json_encode(array(
                'model' => $model_id,
                'messages' => array(
                    array(
                        'role' => 'user',
                        'content' => $prompt
                    )
                ),
                'temperature' => 1
            )),
            'timeout' => 120
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('OpenAI API call failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($response_code !== 200) {
            $error_message = isset($data['error']['message']) 
                ? $data['error']['message'] 
                : 'API returned status code ' . $response_code;
            throw new Exception($error_message);
        }
        
        if (isset($data['choices'][0]['message']['content'])) {
            return $this->extract_css_from_response($data['choices'][0]['message']['content']);
        }
        
        throw new Exception('Invalid OpenAI API response format');
    }
    
    /**
     * DeepSeek API call - using same pattern as admin chat addon
     */
    private function call_deepseek_api($model_id, $prompt) {
        $api_key = $this->api_keys['deepseek'];
        
        if (empty($api_key)) {
            throw new Exception('DeepSeek API key not configured');
        }
        
        $response = wp_remote_post('https://api.deepseek.com/v1/chat/completions', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'body' => json_encode(array(
                'model' => $model_id,
                'messages' => array(
                    array(
                        'role' => 'user',
                        'content' => $prompt
                    )
                ),
                'max_tokens' => 2000
            )),
            'timeout' => 120
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('DeepSeek API call failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($response_code !== 200) {
            $error_message = isset($data['error']['message']) 
                ? $data['error']['message'] 
                : 'API returned status code ' . $response_code;
            throw new Exception($error_message);
        }
        
        if (isset($data['choices'][0]['message']['content'])) {
            return $this->extract_css_from_response($data['choices'][0]['message']['content']);
        }
        
        throw new Exception('Invalid DeepSeek API response format');
    }
    
    /**
     * Gemini API call - using Google AI Studio API
     */
    private function call_gemini_api($model_id, $prompt) {
        $api_key = $this->api_keys['gemini'];
        
        if (empty($api_key)) {
            throw new Exception('Gemini API key not configured');
        }
        
        $system_instruction = 'You are a CSS expert specializing in chatbot interface design. Generate clean, valid CSS code with !important declarations.';
        
        $body = json_encode(array(
            'contents' => array(
                array(
                    'parts' => array(
                        array('text' => $prompt)
                    )
                )
            ),
            'systemInstruction' => array(
                'parts' => array(
                    array('text' => $system_instruction)
                )
            ),
            'generationConfig' => array(
                'temperature' => 0.7,
                'maxOutputTokens' => 2048,
                'topP' => 0.8,
                'topK' => 10
            )
        ));
        
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model_id}:generateContent?key={$api_key}";
        
        $args = array(
            'body' => $body,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'timeout' => 120,
            'blocking' => true,
            'sslverify' => true,
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('Gemini API call failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($response_code !== 200) {
            $error_message = isset($response_body['error']['message']) 
                ? $response_body['error']['message'] 
                : 'API returned status code ' . $response_code;
            throw new Exception($error_message);
        }
        
        if (isset($response_body['candidates'][0]['content']['parts'][0]['text'])) {
            return $this->extract_css_from_response($response_body['candidates'][0]['content']['parts'][0]['text']);
        }
        
        throw new Exception('Invalid Gemini API response format');
    }
    
    /**
     * xAI API call - using same pattern as OpenAI (compatible API)
     */
    private function call_xai_api($model_id, $prompt) {
        $api_key = $this->api_keys['xai'];
        
        if (empty($api_key)) {
            throw new Exception('xAI API key not configured');
        }
        
        $response = wp_remote_post('https://api.x.ai/v1/chat/completions', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'body' => json_encode(array(
                'model' => $model_id,
                'messages' => array(
                    array(
                        'role' => 'system',
                        'content' => 'You are a CSS expert specializing in chatbot interface design. Generate clean, valid CSS code with !important declarations.'
                    ),
                    array(
                        'role' => 'user',
                        'content' => $prompt
                    )
                ),
                'max_tokens' => 2000,
                'temperature' => 0.7
            )),
            'timeout' => 120
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('xAI API call failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($response_code !== 200) {
            $error_message = isset($data['error']['message']) 
                ? $data['error']['message'] 
                : 'API returned status code ' . $response_code;
            throw new Exception($error_message);
        }
        
        if (isset($data['choices'][0]['message']['content'])) {
            return $this->extract_css_from_response($data['choices'][0]['message']['content']);
        }
        
        throw new Exception('Invalid xAI API response format');
    }
    
    /**
     * Get the current active AI theme CSS
     */
    public function get_active_ai_theme_css() {
        return $this->theme_options['active_ai_theme_css'] ?? '';
    }
    
    /**
     * Get the current active AI theme name
     */
    public function get_active_ai_theme_name() {
        return $this->theme_options['active_ai_theme_name'] ?? '';
    }
    
    /**
     * Get saved AI themes
     */
    public function get_saved_ai_themes() {
        return $this->theme_options['saved_ai_themes'] ?? array();
    }
}