<?php
/**
 * CSS Theme Builder Helper Class
 * File: mxchat-theme/includes/class-theme-css-builder.php
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class MxChat_Theme_CSS_Builder
 * 
 * Handles CSS generation, validation, and manipulation for AI-generated themes
 */
class MxChat_Theme_CSS_Builder {
    
    /**
     * Valid CSS selectors for the chatbot
     */
private $valid_selectors = array(
    // Main containers
    '#mxchat-chatbot',
    '#floating-chatbot',
    '#mxchat-chatbot-wrapper',
    '#chat-container',
    '#chat-box',
    '#input-container',
    
    // Top bar elements
    '.chatbot-top-bar',
    '.chatbot-title',
    '.chatbot-title-container',
    '.chatbot-title-group',
    '.chatbot-title-icon',
    '.exit-chat',
    '.exit-chat svg',
    '.exit-chat svg path',
    '.chat-mode-indicator',
    
    // Message elements
    '.bot-message',
    '.user-message', 
    '.agent-message',
    '.bot-message div[dir="auto"]',
    '.user-message div[dir="auto"]',
    '.agent-message div[dir="auto"]',
    '.bot-message a',
    '.user-message a',
    '.agent-message a',
    '.bot-message p a',
    '.user-message p a',
    '.agent-message p a',
    
    // Thinking dots elements - NEW
    '.bot-message.temporary-message',
    '.thinking-dots-container',
    '.thinking-dots',
    '.thinking-dots .dot',
    
    // Input elements
    '#chat-input',
    '#chat-input::placeholder',
    '#chat-input::-webkit-input-placeholder',
    '#chat-input::-moz-placeholder',
    '#send-button',
    '#send-button svg',
    '#mxchat-chatbot #chat-container #input-container',
    
    // Toolbar elements
    '.chat-toolbar',
    '.toolbar-btn',
    '.toolbar-btn svg',
    '.woo-cart-count',
    '.remove-pdf-btn svg',
    
    // Active file containers
    '.active-pdf-container',
    '.active-pdf-name',
    '.remove-pdf-btn',
    
    // Popular questions
    '#mxchat-popular-questions',
    '.mxchat-popular-questions-title',
    '.mxchat-popular-questions-container',
    '.mxchat-popular-question',
    '.questions-collapse-btn svg',
    '.questions-toggle-btn svg',
    
    // Footer
    '.chatbot-footer',
    '.privacy-notice',
    '.privacy-notice a',
    
    // Floating button
    '#floating-chatbot-button',
    
    // Mobile specific
    '#floating-chatbot.visible',
    
    // Pre-chat message popup
    '#pre-chat-message',
    '.close-pre-chat-message',
    '#pre-chat-message .close-pre-chat-message',
    
    // WooCommerce product card
    '.mxchat-add-to-cart-button',
    '.mxchat-product-link',
    
    // Email blocker
    '.email-blocker h2',
    '.email-blocker p',
    '.email-blocker input[type="email"]',
    '.email-blocker button',
    '.email-blocker-header'
);
    
    /**
     * Dangerous CSS properties that should be filtered out
     */
    private $dangerous_properties = array(
        'javascript',
        'expression',
        'behavior',
        '@import',
        'url(',
        'eval(',
        'script',
        'vbscript',
        'onload',
        'onerror',
        'onclick'
    );
    
    /**
     * CSS properties that are safe to use
     */
    private $allowed_properties = array(
        'color',
        'background-color',
        'background',
        'border',
        'border-color',
        'border-radius',
        'border-width',
        'border-style',
        'padding',
        'margin',
        'font-size',
        'font-weight',
        'font-family',
        'line-height',
        'text-align',
        'text-decoration',
        'box-shadow',
        'opacity',
        'transform',
        'transition',
        'width',
        'height',
        'max-width',
        'max-height',
        'min-width',
        'min-height',
        'display',
        'position',
        'top',
        'right',
        'bottom',
        'left',
        'z-index',
        'overflow',
        'float',
        'clear',
        'visibility',
        'cursor',
        'outline',
        'fill',
        'stroke'
    );
    
    /**
     * Build CSS prompt for AI with comprehensive selector information
     * 
     * @param string $description User's theme description
     * @return string Formatted prompt for AI
     */
public function build_ai_prompt($description) {
$selector_reference = $this->get_selector_reference();

$prompt = "You are a CSS expert specializing in chatbot interface design. Generate CSS code based on this description: \"{$description}\"

CRITICAL REQUIREMENTS:
1. Use !important on ALL CSS properties (but only once per property)
2. Only use the selectors provided below - do not create new ones
3. For SVG icons, use 'fill' and 'stroke' properties instead of 'color'
4. Focus on colors, backgrounds, borders, and visual styling only
5. Ensure good contrast and accessibility (minimum 4.5:1 contrast ratio)
6. Return ONLY valid CSS code, no explanations or markdown formatting
7. Do not use any external URLs or imports
8. Keep all values simple and avoid complex calculations
9. Use valid CSS color values (hex, rgb, rgba, or named colors)
10. ALWAYS style thinking dots to match bot message colors
11. ALWAYS include quick questions toggle button colors in your themes
12. ALWAYS add mobile border fix: #floating-chatbot.visible on mobile should have border: none

IMPORTANT STYLING GUIDELINES:
- DO NOT add borders to: #send-button, #chat-input, .exit-chat, #mxchat-chatbot-wrapper
- DO NOT add backgrounds to: .toolbar-btn, .chatbot-footer, .chat-toolbar, .exit-chat, #chat-box, #chat-input
- DO NOT add border-radius to: #mxchat-chatbot, #input-container, .bot-message, .user-message (keep default radius)
- For borders around input area, use: #input-container (not individual elements, and NO border-radius)
- For backgrounds in input area, use: #mxchat-chatbot #chat-container #input-container (NOT #chat-input)
- For borders around entire chatbot, use: #floating-chatbot.visible (not wrapper)
- For toolbar icons, only change 'fill' color, no backgrounds or borders
- Keep footer and toolbar areas clean with minimal styling
- #chat-box inherits background from #mxchat-chatbot, do not override
- For bot messages, use both .bot-message AND .bot-message div[dir=\"auto\"] to target inline styled intro message
- For user messages, use both .user-message AND .user-message div[dir=\"auto\"] for consistency
- For #mxchat-chatbot, ALWAYS use background-color property, NEVER use background property
- THINKING DOTS: Always make .thinking-dots .dot background-color match the text color of bot messages (opposite of bot message background)
- QUICK QUESTIONS TOGGLE: Always style .questions-collapse-btn svg and .questions-toggle-btn svg with stroke property
- MOBILE FIX: Always include @media (max-width: 768px) { #floating-chatbot.visible { border: none !important; } }

AVAILABLE SELECTORS AND THEIR PURPOSES:
{$selector_reference}

EXAMPLE OUTPUT FORMAT:
#mxchat-chatbot {
    background-color: #your-main-bg-color !important;
}

.bot-message {
    background-color: #your-bot-bg !important;
    color: #your-bot-text !important;
}

.bot-message div[dir=\"auto\"] {
    color: #your-bot-text !important;
}

.thinking-dots .dot {
    background-color: #your-bot-text !important;
}

.user-message {
    background-color: #your-user-bg !important;
    color: #your-user-text !important;
}

.bot-message a {
    color: #your-bot-link-color !important;
    text-decoration: underline !important;
}

.user-message a {
    color: #your-user-link-color !important;
    text-decoration: underline !important;
}

.bot-message p a {
    color: #your-bot-link-color !important;
    text-decoration: underline !important;
}

.toolbar-btn svg {
    fill: #your-icon-color !important;
    stroke: #your-icon-color !important;
    color: #your-icon-color !important;
}

.questions-collapse-btn svg,
.questions-toggle-btn svg {
    stroke: #your-toggle-color !important;
}

#input-container {
    border: 2px solid #your-border-color !important;
}

@media (max-width: 768px) {
    #floating-chatbot.visible {
        border: none !important;
    }
}

Generate CSS that matches this description: \"{$description}\"

CSS OUTPUT:";

return $prompt;
}

    /**
     * Get detailed selector reference for AI prompt
     * 
     * @return string Formatted selector reference
     */
private function get_selector_reference() {
    return "
/* MAIN CONTAINERS */
#mxchat-chatbot - Main chatbot interface background and styling (DO NOT add border-radius, ALWAYS use background-color not background)
#floating-chatbot - Overall floating container styling
#floating-chatbot.visible - Use this for borders around the entire chat box (NOT #mxchat-chatbot-wrapper)
#mxchat-chatbot-wrapper - Wrapper container (DO NOT add borders here)
#chat-container - Container for the conversation area
#chat-box - Scrollable conversation area (DO NOT add backgrounds - inherits from #mxchat-chatbot)
#input-container - Input field container styling (use for borders but NO border-radius)

/* TOP BAR ELEMENTS */
.chatbot-top-bar - Header bar background and styling
.chatbot-title - Title text color and font styling
.chatbot-title-icon - Icon next to the title
.exit-chat - Close/minimize button styling (DO NOT add backgrounds or borders)
.exit-chat svg - Close button icon color (use 'fill' property only)
.chat-mode-indicator - Mode indicator background and text color

/* MESSAGE ELEMENTS */
.bot-message - Bot message background color, text color, and styling (DO NOT add border-radius)
.user-message - User message background color, text color, and styling (DO NOT add border-radius)
.agent-message - Live agent message background color and text color
.bot-message div[dir=\"auto\"] - Target text inside bot messages (including intro message with inline styles)
.user-message div[dir=\"auto\"] - Target text inside user messages for consistency
.agent-message div[dir=\"auto\"] - Target text inside agent messages for consistency
.bot-message a - Hyperlinks within bot messages
.user-message a - Hyperlinks within user messages
.agent-message a - Hyperlinks within agent messages
.bot-message p a - Hyperlinks within paragraphs in bot messages
.user-message p a - Hyperlinks within paragraphs in user messages
.agent-message p a - Hyperlinks within paragraphs in agent messages

/* THINKING DOTS ELEMENTS */
.bot-message.temporary-message - Temporary thinking message container (inherits bot message styling)
.thinking-dots-container - Container for the thinking dots animation
.thinking-dots - Wrapper for individual dots
.thinking-dots .dot - Individual thinking dots (should match bot message background/text colors)

/* INPUT ELEMENTS */
#chat-input - Input field text color (DO NOT add borders or backgrounds here)
#chat-input::placeholder - Input field placeholder text color
#chat-input::-webkit-input-placeholder - Input field placeholder text color (Webkit browsers)
#chat-input::-moz-placeholder - Input field placeholder text color (Firefox)
#send-button - Send button container styling (DO NOT add borders here)
#send-button svg - Send button icon color (use 'fill' property)
#mxchat-chatbot #chat-container #input-container - Input area background styling (use this for backgrounds, not #chat-input)

/* TOOLBAR ELEMENTS */
.chat-toolbar - Toolbar container (DO NOT add backgrounds)
.toolbar-btn - Toolbar button styling (DO NOT add backgrounds or borders)
.toolbar-btn svg - ALL toolbar icon colors (use 'fill', 'stroke', and 'color' properties with same value)
.woo-cart-count - Cart count badge text color
.remove-pdf-btn svg - Remove file button icon color (fill, stroke, and color)

/* POPULAR QUESTIONS */
.mxchat-popular-questions-title - Questions section title styling
.mxchat-popular-question - Individual question button styling
.questions-collapse-btn svg - Quick questions collapse button icon (use 'stroke' property only)
.questions-toggle-btn svg - Quick questions expand button icon (use 'stroke' property only)

/* FOOTER */
.chatbot-footer - Footer area styling (DO NOT add backgrounds)
.privacy-notice - Privacy notice text color
.privacy-notice a - Privacy policy link color

/* FLOATING BUTTON */
#floating-chatbot-button - Main floating button background and styling

/* PRE-CHAT MESSAGE */
#pre-chat-message - Pre-chat popup message background and text color
.close-pre-chat-message - Close button for pre-chat message
#pre-chat-message .close-pre-chat-message - Pre-chat close button background and text color

/* WOOCOMMERCE PRODUCT CARD */
.mxchat-add-to-cart-button - Add to cart button styling
.mxchat-product-link - Product link text color

/* EMAIL BLOCKER */
.email-blocker h2 - Email blocker title color
.email-blocker button - Email submit button styling
.email-blocker-header - Email blocker header text color";
}

    /**
     * Validate and sanitize CSS code
     * 
     * @param string $css Raw CSS code
     * @return string|WP_Error Sanitized CSS or error
     */
    public function validate_and_sanitize_css($css) {
        if (empty($css)) {
            return new WP_Error('empty_css', 'CSS code is empty');
        }
        
        // Remove any potential security threats
        $css = $this->remove_dangerous_css($css);
        
        // Validate CSS syntax
        $validation_result = $this->validate_css_syntax($css);
        if (is_wp_error($validation_result)) {
            return $validation_result;
        }
        
        // Filter selectors to only allowed ones
        $css = $this->filter_allowed_selectors($css);
        
        // Ensure all properties have !important
        $css = $this->ensure_important_declarations($css);
        
        // Final cleanup
        $css = $this->cleanup_css($css);
        
        return $css;
    }
    
    /**
     * Remove dangerous CSS patterns
     * 
     * @param string $css CSS code
     * @return string Cleaned CSS
     */
    private function remove_dangerous_css($css) {
        // Remove potentially dangerous patterns
        $dangerous_patterns = array(
            '/javascript\s*:/i',
            '/expression\s*\(/i',
            '/behavior\s*:/i',
            '/@import/i',
            '/url\s*\([^)]*javascript:/i',
            '/url\s*\([^)]*data:/i',
            '/url\s*\([^)]*vbscript:/i',
            '/on\w+\s*=/i',
            '/eval\s*\(/i',
            '/script/i',
            '/vbscript/i'
        );
        
        foreach ($dangerous_patterns as $pattern) {
            $css = preg_replace($pattern, '', $css);
        }
        
        return $css;
    }
    
    /**
     * Basic CSS syntax validation
     * 
     * @param string $css CSS code
     * @return true|WP_Error True if valid, error if not
     */
    private function validate_css_syntax($css) {
        // Check for balanced braces
        $open_braces = substr_count($css, '{');
        $close_braces = substr_count($css, '}');
        
        if ($open_braces !== $close_braces) {
            return new WP_Error('invalid_css_syntax', 'Unbalanced CSS braces');
        }
        
        // Check for basic CSS structure
        if (!preg_match('/[^{}]+\{[^{}]*\}/', $css)) {
            return new WP_Error('invalid_css_structure', 'Invalid CSS structure');
        }
        
        return true;
    }
    
    /**
     * Filter CSS to only include allowed selectors
     * 
     * @param string $css CSS code
     * @return string Filtered CSS
     */
    private function filter_allowed_selectors($css) {
        $filtered_css = '';
        
        // Split CSS into rules
        preg_match_all('/([^{}]+)\{([^{}]*)\}/', $css, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $selector = trim($match[1]);
            $properties = trim($match[2]);
            
            // Check if selector is allowed
            if ($this->is_selector_allowed($selector)) {
                // Filter properties
                $filtered_properties = $this->filter_properties($properties);
                if (!empty($filtered_properties)) {
                    $filtered_css .= $selector . " {\n" . $filtered_properties . "\n}\n\n";
                }
            }
        }
        
        return $filtered_css;
    }
    
    /**
     * Check if a CSS selector is allowed
     * 
     * @param string $selector CSS selector
     * @return bool True if allowed
     */
    private function is_selector_allowed($selector) {
        // Clean the selector
        $selector = trim($selector);
        
        // Check against our list of valid selectors
        foreach ($this->valid_selectors as $valid_selector) {
            if (strpos($selector, $valid_selector) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Filter CSS properties to only safe ones
     * 
     * @param string $properties CSS properties string
     * @return string Filtered properties
     */
    private function filter_properties($properties) {
        $filtered_properties = '';
        $property_lines = explode(';', $properties);
        
        foreach ($property_lines as $property_line) {
            $property_line = trim($property_line);
            if (empty($property_line)) {
                continue;
            }
            
            // Split property and value
            $parts = explode(':', $property_line, 2);
            if (count($parts) !== 2) {
                continue;
            }
            
            $property = trim($parts[0]);
            $value = trim($parts[1]);
            
            // Remove !important for now, we'll add it back later
            $value = str_replace('!important', '', $value);
            $value = trim($value);
            
            // Check if property is allowed
            if ($this->is_property_allowed($property)) {
                // Validate the value
                $validated_value = $this->validate_property_value($property, $value);
                if ($validated_value !== false) {
                    $filtered_properties .= "    {$property}: {$validated_value};\n";
                }
            }
        }
        
        return $filtered_properties;
    }
    
    /**
     * Check if a CSS property is allowed
     * 
     * @param string $property CSS property name
     * @return bool True if allowed
     */
    private function is_property_allowed($property) {
        return in_array(strtolower($property), $this->allowed_properties);
    }
    
    /**
     * Validate a CSS property value
     * 
     * @param string $property Property name
     * @param string $value Property value
     * @return string|false Validated value or false if invalid
     */
    private function validate_property_value($property, $value) {
        // Remove any dangerous content from value
        foreach ($this->dangerous_properties as $dangerous) {
            if (stripos($value, $dangerous) !== false) {
                return false;
            }
        }
        
        // Fix common AI mistakes
        $value = $this->fix_common_css_mistakes($value);
        
        // Basic validation based on property type
        switch (strtolower($property)) {
            case 'color':
            case 'background-color':
            case 'border-color':
            case 'fill': // Add fill for SVG elements
                return $this->validate_color_value($value);
                
            case 'font-size':
            case 'width':
            case 'height':
            case 'padding':
            case 'margin':
            case 'border-width':
                return $this->validate_size_value($value);
                
            case 'opacity':
                return $this->validate_opacity_value($value);
                
            default:
                // For other properties, just ensure no dangerous content
                return esc_attr($value);
        }
    }
    
    /**
     * Fix common CSS mistakes made by AI
     * 
     * @param string $value CSS value
     * @return string Fixed value
     */
    private function fix_common_css_mistakes($value) {
        // Remove duplicate !important declarations
        $value = preg_replace('/(!important\s*)+/', '', $value);
        
        // Fix invalid filter values
        if (strpos($value, 'filter:') !== false) {
            return '';
        }
        
        // Fix text-shadow issues (often causes problems)
        if (strpos($value, 'text-shadow') !== false) {
            return '';
        }
        
        return trim($value);
    }
    
    /**
     * Validate color values
     * 
     * @param string $value Color value
     * @return string|false Valid color or false
     */
    private function validate_color_value($value) {
        // Remove whitespace
        $value = trim($value);
        
        // Check for hex colors
        if (preg_match('/^#([a-fA-F0-9]{3}|[a-fA-F0-9]{6})$/', $value)) {
            return $value;
        }
        
        // Check for rgb/rgba
        if (preg_match('/^rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*(?:,\s*([\d.]+))?\s*\)$/', $value)) {
            return $value;
        }
        
        // Check for named colors (basic list)
        $named_colors = array('red', 'blue', 'green', 'white', 'black', 'yellow', 'purple', 'orange', 'pink', 'gray', 'grey', 'transparent');
        if (in_array(strtolower($value), $named_colors)) {
            return $value;
        }
        
        return false;
    }
    
    /**
     * Validate size values
     * 
     * @param string $value Size value
     * @return string|false Valid size or false
     */
    private function validate_size_value($value) {
        // Allow numbers with px, em, rem, %, vh, vw units
        if (preg_match('/^[\d.]+\s*(px|em|rem|%|vh|vw|pt|pc|in|cm|mm|ex|ch)$/', trim($value))) {
            return trim($value);
        }
        
        // Allow just numbers (will default to px)
        if (is_numeric(trim($value))) {
            return trim($value) . 'px';
        }
        
        // Allow 'auto', 'inherit', 'initial'
        $valid_keywords = array('auto', 'inherit', 'initial', 'none');
        if (in_array(strtolower(trim($value)), $valid_keywords)) {
            return trim($value);
        }
        
        return false;
    }
    
    /**
     * Validate opacity values
     * 
     * @param string $value Opacity value
     * @return string|false Valid opacity or false
     */
    private function validate_opacity_value($value) {
        $value = trim($value);
        
        if (is_numeric($value)) {
            $numeric_value = floatval($value);
            if ($numeric_value >= 0 && $numeric_value <= 1) {
                return $value;
            }
        }
        
        return false;
    }
    
    /**
     * Ensure all CSS properties have !important declarations
     * 
     * @param string $css CSS code
     * @return string CSS with !important added
     */
    private function ensure_important_declarations($css) {
        // First, remove any existing !important declarations to avoid duplicates
        $css = preg_replace('/\s*!important\s*/', '', $css);
        
        // Then add !important to all property declarations
        $css = preg_replace_callback('/([^;}]+:\s*[^;}]+)(\s*[;}])/', function($matches) {
            return trim($matches[1]) . ' !important' . $matches[2];
        }, $css);
        
        return $css;
    }
    
    /**
     * Clean up CSS formatting
     * 
     * @param string $css CSS code
     * @return string Cleaned CSS
     */
    private function cleanup_css($css) {
        // Remove extra whitespace
        $css = preg_replace('/\s+/', ' ', $css);
        
        // Fix spacing around braces
        $css = preg_replace('/\s*{\s*/', " {\n", $css);
        $css = preg_replace('/\s*}\s*/', "\n}\n\n", $css);
        
        // Fix semicolons
        $css = preg_replace('/;\s*/', ";\n", $css);
        
        // Remove trailing semicolons before closing braces
        $css = preg_replace('/;\s*\n\s*}/', "\n}", $css);
        
        return trim($css);
    }
    
    /**
     * Extract CSS from AI response that might contain markdown or other formatting
     * 
     * @param string $response AI response
     * @return string Extracted CSS
     */
    public function extract_css_from_response($response) {
        // Remove markdown code blocks if present
        $css = preg_replace('/```css\s*/', '', $response);
        $css = preg_replace('/```\s*$/', '', $css);
        
        // Remove any text before the first CSS rule
        $css = preg_replace('/^[^{]*?(?=\s*[.#]|\s*@)/', '', $css);
        
        // Remove any text after the last CSS rule
        $css = preg_replace('/}\s*[^}]*$/', '}', $css);
        
        return trim($css);
    }
    
    /**
     * Generate default theme CSS based on a color scheme
     * 
     * @param string $primary_color Primary color
     * @param string $secondary_color Secondary color
     * @param string $text_color Text color
     * @return string Generated CSS
     */
    public function generate_default_theme($primary_color = '#7873f5', $secondary_color = '#fa73e6', $text_color = '#333333') {
        $css = "
    /* AI Generated Theme */
    .chatbot-top-bar {
        background: linear-gradient(135deg, {$primary_color}, {$secondary_color}) !important;
        color: white !important;
    }
    
    .chatbot-title {
        color: white !important;
    }
    
    .exit-chat svg {
        fill: white !important;
    }
    
    .bot-message {
        background-color: {$primary_color} !important;
        color: white !important;
    }
    
    .bot-message div[dir=\"auto\"] {
        color: white !important;
    }
    
    .thinking-dots .dot {
        background-color: white !important;
    }
    
    .user-message {
        background-color: #f0f0f0 !important;
        color: {$text_color} !important;
    }
    
    .user-message div[dir=\"auto\"] {
        color: {$text_color} !important;
    }
    
    #chat-input {
        color: {$text_color} !important;
        border-color: {$primary_color} !important;
    }
    
    #send-button svg {
        fill: {$primary_color} !important;
    }
    
    .toolbar-btn svg {
        fill: {$primary_color} !important;
    }
    
    .mxchat-popular-question {
        border-color: {$primary_color} !important;
        color: {$text_color} !important;
    }
    
    .mxchat-popular-question:hover {
        background-color: {$primary_color} !important;
        color: white !important;
    }
    
    #floating-chatbot-button {
        background: linear-gradient(135deg, {$primary_color}, {$secondary_color}) !important;
    }";
    
        return $this->validate_and_sanitize_css($css);
    }

    /**
     * Convert theme description to color suggestions
     * 
     * @param string $description Theme description
     * @return array Color suggestions
     */
    public function extract_colors_from_description($description) {
        $colors = array();
        
        // Common color mappings
        $color_mappings = array(
            'blue' => '#2196f3',
            'red' => '#f44336',
            'green' => '#4caf50',
            'purple' => '#9c27b0',
            'orange' => '#ff9800',
            'pink' => '#e91e63',
            'dark' => '#212121',
            'light' => '#ffffff',
            'black' => '#000000',
            'white' => '#ffffff',
            'gray' => '#9e9e9e',
            'grey' => '#9e9e9e'
        );
        
        $description_lower = strtolower($description);
        
        foreach ($color_mappings as $color_name => $hex_value) {
            if (strpos($description_lower, $color_name) !== false) {
                $colors[] = $hex_value;
            }
        }
        
        return $colors;
    }
}

// Initialize the CSS builder
if (!function_exists('mxchat_get_css_builder')) {
    function mxchat_get_css_builder() {
        return new MxChat_Theme_CSS_Builder();
    }
}