<?php
/**
 * Plugin Name: MxChat Theme Customizer
 * Plugin URI: https://mxchat.ai/
 * Description: Theme customization add-on for MxChat (Pro only)
 * Version: 1.1.3
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-theme
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

function is_mxchat_active_themes() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_themes()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Theme Customizer</strong> requires the MxChat plugin to be installed and active.', 'mxchat-theme'); ?></p>
        </div>
        <?php
    });
    return;
}

require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerThemes::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-theme'
]);

// Load AI Theme Generator
require_once dirname(__FILE__) . '/includes/class-ai-theme-generator.php';

class MxChat_Theme_Addon {
    private $options;
    private $ai_generator;
    
    public function __construct() {
        $this->options = get_option('mxchat_options', array());
        
        // Check for plugin upgrades
        $this->check_version_upgrade();
        
        // Initialize AI Theme Generator
        $this->ai_generator = new MxChat_AI_Theme_Generator();
        
        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        
        // Only load features if pro is active
        if ($this->is_license_active()) {
            add_action('admin_menu', array($this, 'add_theme_menu'), 20);
            add_action('admin_init', array($this, 'init_settings'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
            add_action('wp_ajax_mxchat_save_theme_setting', array($this, 'save_theme_setting_callback'));
            add_action('wp_ajax_mxchat_save_position_setting', array($this, 'save_position_setting_callback'));
            add_action('wp_ajax_mxchat_save_input_container_bg', array($this, 'save_input_container_bg_callback'));
            add_action('wp_ajax_mxchat_save_input_container_border', array($this, 'save_input_container_border_callback'));
            
            // Add AI theme CSS to frontend
            add_action('wp_head', array($this, 'output_ai_theme_css'));
            add_action('admin_head', array($this, 'output_ai_theme_css'));
            
            // Add position CSS to frontend
            add_action('wp_head', array($this, 'output_position_css'));
            add_action('admin_head', array($this, 'output_position_css'));
            
            // Add input container CSS to frontend
            add_action('wp_head', array($this, 'output_input_container_css'));
            add_action('admin_head', array($this, 'output_input_container_css'));
        }
    }

    /**
     * Check for plugin version upgrades and run upgrade routines
     */
    private function check_version_upgrade() {
        $current_version = '1.1.3';
        $saved_version = get_option('mxchat_theme_version', '0');
        
        // If version changed, run upgrade routines
        if (version_compare($saved_version, $current_version, '<')) {
            // Upgrade to 1.1.3 - Reset position settings to new defaults
            if (version_compare($saved_version, '1.1.3', '<')) {
                $this->upgrade_to_1_1_3();
            }
            
            // Update the saved version
            update_option('mxchat_theme_version', $current_version);
        }
    }
    
    /**
     * Upgrade routine for version 1.1.3
     * Resets widget position settings to corrected default values
     */
    private function upgrade_to_1_1_3() {
        $position_settings = get_option('mxchat_position_settings', array());
        
        // Update with new default values
        $position_settings['widget_position_desktop'] = '30';
        $position_settings['widget_position_mobile'] = '15';
        $position_settings['prechat_position_desktop'] = '85';
        $position_settings['prechat_position_mobile'] = '70';
        
        update_option('mxchat_position_settings', $position_settings);
    }

    /**
     * Check if MX Chat Pro is activated
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    /**
     * Admin notice for pro requirement
     */
    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Theme Customizer requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-theme'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    public function add_theme_menu() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php', // Network settings menu
                esc_html__('Theme Settings', 'mxchat-theme'),
                esc_html__('Theme Settings', 'mxchat-theme'),
                'manage_network_options',
                'mxchat-theme-settings',
                array($this, 'render_theme_page')
            );
        } else {
            add_submenu_page(
                'mxchat-max',
                esc_html__('Theme Settings', 'mxchat-theme'),
                esc_html__('Theme Settings', 'mxchat-theme'),
                'manage_options',
                'mxchat-theme-settings',
                array($this, 'render_theme_page')
            );
        }
    }

    public function init_settings() {
        add_settings_section(
            'mxchat_theme_section',
            esc_html__('Theme Settings', 'mxchat'),
            array($this, 'render_section_info'),
            'mxchat-theme'
        );

        foreach ($this->get_color_fields() as $field_id => $title) {
            add_settings_field(
                $field_id,
                $title,
                array($this, 'render_color_field'),
                'mxchat-theme',
                'mxchat_theme_section',
                array('field_id' => $field_id)
            );
        }

        add_settings_field(
            'custom_icon',
            esc_html__('Custom Chatbot Icon (PNG)', 'mxchat-theme'),
            array($this, 'render_icon_field'),
            'mxchat-theme',
            'mxchat_theme_section',
            array('field_id' => 'custom_icon')
        );

        add_settings_field(
            'title_icon',
            esc_html__('Title Bar Icon (PNG)', 'mxchat-theme'),
            array($this, 'render_icon_field'),
            'mxchat-theme',
            'mxchat_theme_section',
            array('field_id' => 'title_icon')
        );
        
        // Add position settings section
        add_settings_section(
            'mxchat_position_section',
            esc_html__('Position Settings', 'mxchat-theme'),
            array($this, 'render_position_section_info'),
            'mxchat-theme'
        );
        
        add_settings_field(
            'widget_position',
            esc_html__('Widget Position (Desktop)', 'mxchat-theme'),
            array($this, 'render_position_field'),
            'mxchat-theme',
            'mxchat_position_section',
            array('field_id' => 'widget_position', 'default' => 30, 'description' => 'Distance from bottom (px)')
        );
        
        add_settings_field(
            'prechat_position',
            esc_html__('Pre-Chat Popup Position (Desktop)', 'mxchat-theme'),
            array($this, 'render_position_field'),
            'mxchat-theme',
            'mxchat_position_section',
            array('field_id' => 'prechat_position', 'default' => 85, 'description' => 'Distance from bottom (px)')
        );
        
        add_settings_field(
            'widget_position_mobile',
            esc_html__('Widget Position (Mobile)', 'mxchat-theme'),
            array($this, 'render_position_field'),
            'mxchat-theme',
            'mxchat_position_section',
            array('field_id' => 'widget_position_mobile', 'default' => 15, 'description' => 'Distance from bottom (px) - Mobile screens')
        );
        
        add_settings_field(
            'prechat_position_mobile',
            esc_html__('Pre-Chat Popup Position (Mobile)', 'mxchat-theme'),
            array($this, 'render_position_field'),
            'mxchat-theme',
            'mxchat_position_section',
            array('field_id' => 'prechat_position_mobile', 'default' => 70, 'description' => 'Distance from bottom (px) - Mobile screens')
        );
    }

    public function save_theme_setting_callback() {
        // Check pro status first
        if (!$this->is_license_active()) {
            wp_send_json_error(['message' => esc_html__('Pro license required', 'mxchat-theme')]);
            return;
        }

        check_ajax_referer('mxchat_save_setting_nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Unauthorized', 'mxchat-theme')]);
        }

        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $value = isset($_POST['value']) ? sanitize_text_field(wp_unslash($_POST['value'])) : '';

        if (empty($name)) {
            wp_send_json_error(['message' => esc_html__('Invalid field name', 'mxchat-theme')]);
        }

        $options = get_option('mxchat_options', []);
        
        $current_value = isset($options[$name]) ? $options[$name] : '';
        
        if ($current_value !== $value) {
            if ($name === 'custom_icon' || $name === 'title_icon') {
                $options[$name] = esc_url_raw($value);
            } else {
                $options[$name] = sanitize_hex_color($value);
            }
            
            $updated = update_option('mxchat_options', $options);
            
            if ($updated) {
                wp_send_json_success(['message' => esc_html__('Setting saved', 'mxchat-theme')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Failed to save setting', 'mxchat-theme')]);
            }
        } else {
            wp_send_json_success(['message' => esc_html__('Setting unchanged', 'mxchat-theme')]);
        }
    }

    /**
     * AJAX callback to save position settings (widget and pre-chat popup)
     */
    public function save_position_setting_callback() {
        check_ajax_referer('mxchat_theme_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Permission denied', 'mxchat-theme')]);
            return;
        }
        
        $setting_name = sanitize_text_field($_POST['setting_name']);
        $value = intval($_POST['value']);
        
        // Validate setting name
        $allowed_settings = array('widget_position', 'prechat_position', 'widget_position_mobile', 'prechat_position_mobile');
        if (!in_array($setting_name, $allowed_settings)) {
            wp_send_json_error(['message' => esc_html__('Invalid setting name', 'mxchat-theme')]);
            return;
        }
        
        // Validate value range (reasonable limits for positioning)
        if ($value < 0 || $value > 1000) {
            wp_send_json_error(['message' => esc_html__('Value out of range', 'mxchat-theme')]);
            return;
        }
        
        // Get position settings (separate option for positions)
        $position_settings = get_option('mxchat_position_settings', array());
        
        // Update the specific setting
        if (!isset($position_settings[$setting_name]) || $position_settings[$setting_name] !== $value) {
            $position_settings[$setting_name] = $value;
            
            $updated = update_option('mxchat_position_settings', $position_settings);
            
            if ($updated) {
                wp_send_json_success(['message' => esc_html__('Position saved', 'mxchat-theme')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Failed to save position', 'mxchat-theme')]);
            }
        } else {
            wp_send_json_success(['message' => esc_html__('Position unchanged', 'mxchat-theme')]);
        }
    }

    /**
     * AJAX callback to save input container background color
     */
    public function save_input_container_bg_callback() {
        check_ajax_referer('mxchat_save_setting_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Permission denied', 'mxchat-theme')]);
            return;
        }
        
        $color = sanitize_hex_color($_POST['color']);
        
        if (!$color) {
            wp_send_json_error(['message' => esc_html__('Invalid color format', 'mxchat-theme')]);
            return;
        }
        
        // Get current input container settings
        $input_container_settings = get_option('mxchat_input_container_settings', array());
        
        // Update the background color
        if (!isset($input_container_settings['background_color']) || $input_container_settings['background_color'] !== $color) {
            $input_container_settings['background_color'] = $color;
            
            $updated = update_option('mxchat_input_container_settings', $input_container_settings);
            
            if ($updated) {
                wp_send_json_success(['message' => esc_html__('Input container background color saved', 'mxchat-theme')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Failed to save color', 'mxchat-theme')]);
            }
        } else {
            wp_send_json_success(['message' => esc_html__('Color unchanged', 'mxchat-theme')]);
        }
    }

    /**
     * AJAX callback to save input container border color
     */
    public function save_input_container_border_callback() {
        check_ajax_referer('mxchat_save_setting_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Permission denied', 'mxchat-theme')]);
            return;
        }
        
        $color = sanitize_hex_color($_POST['color']);
        
        if (!$color) {
            wp_send_json_error(['message' => esc_html__('Invalid color format', 'mxchat-theme')]);
            return;
        }
        
        // Get current input container settings
        $input_container_settings = get_option('mxchat_input_container_settings', array());
        
        // Update the border color
        if (!isset($input_container_settings['border_color']) || $input_container_settings['border_color'] !== $color) {
            $input_container_settings['border_color'] = $color;
            
            $updated = update_option('mxchat_input_container_settings', $input_container_settings);
            
            if ($updated) {
                wp_send_json_success(['message' => esc_html__('Input container border color saved', 'mxchat-theme')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Failed to save color', 'mxchat-theme')]);
            }
        } else {
            wp_send_json_success(['message' => esc_html__('Color unchanged', 'mxchat-theme')]);
        }
    }

    private function get_color_fields() {
    return array(
        'close_button_color' => esc_html__('Close Button & Title Color', 'mxchat-theme'),
        'chatbot_bg_color' => esc_html__('Chatbot Background Color', 'mxchat-theme'),
        'user_message_bg_color' => esc_html__('User Message Background Color', 'mxchat-theme'),
        'user_message_font_color' => esc_html__('User Message Font Color', 'mxchat-theme'),
        'bot_message_bg_color' => esc_html__('Bot Message Background Color', 'mxchat-theme'),
        'bot_message_font_color' => esc_html__('Bot Message Font Color', 'mxchat-theme'),
        'top_bar_bg_color' => esc_html__('Top Bar Background Color', 'mxchat-theme'),
        'send_button_font_color' => esc_html__('Send Button Color', 'mxchat-theme'),
        'chat_input_font_color' => esc_html__('Chat Input Font Color', 'mxchat-theme'),
        'input_container_bg_color' => esc_html__('Input Container Background Color', 'mxchat-theme'),
        'input_container_border_color' => esc_html__('Input Container Border Color', 'mxchat-theme'),
        'chatbot_background_color' => esc_html__('Floating Widget Background Color', 'mxchat-theme'),
        'icon_color' => esc_html__('Chatbot Icon Color', 'mxchat-theme'),
        'live_agent_message_bg_color' => esc_html__('Live Agent Background Color', 'mxchat-theme'),
        'live_agent_message_font_color' => esc_html__('Live Agent Font Color', 'mxchat-theme'),
        'mode_indicator_bg_color' => esc_html__('Mode Indicator Background Color', 'mxchat-theme'),
        'mode_indicator_font_color' => esc_html__('Mode Indicator Font Color', 'mxchat-theme'),
        'toolbar_icon_color' => esc_html__('Toolbar Icon Color', 'mxchat-theme'),
        'quick_questions_toggle_color' => esc_html__('Quick Questions Toggle Color', 'mxchat-theme') // NEW LINE
    );
}

    public function render_section_info() {
        echo '<div class="mxchat-section-info">';
        echo '</div>';
    }

    public function render_color_field($args) {
        $field_id = $args['field_id'];
        $default_colors = [
            'close_button_color' => '#4a4a4a',
            'chatbot_bg_color' => '#f9f9f9',
            'user_message_bg_color' => '#ffffff',
            'user_message_font_color' => '#000000',
            'bot_message_bg_color' => '#ffffff',
            'bot_message_font_color' => '#000000',
            'top_bar_bg_color' => '#ffffff',
            'send_button_font_color' => '#ffffff',
            'chat_input_font_color' => '#000000',
            'input_container_bg_color' => '#ffffff',
            'input_container_border_color' => '#cccccc',
            'chatbot_background_color' => '#ffffff',
            'icon_color' => '#000000',
            'live_agent_message_bg_color' => '#ffffff',
            'live_agent_message_font_color' => '#000000',
            'mode_indicator_bg_color' => '#ffffff',
            'mode_indicator_font_color' => '#000000',
            'toolbar_icon_color' => '#000000',
            'quick_questions_toggle_color' => '#212121' // NEW LINE - matches your current stroke color
        ];
        
        $default_color = isset($default_colors[$field_id]) ? $default_colors[$field_id] : '#ffffff';
        
        // Handle input container colors separately as they're stored in a different option
        if ($field_id === 'input_container_bg_color') {
            $input_container_settings = get_option('mxchat_input_container_settings', array());
            $value = isset($input_container_settings['background_color']) ? $input_container_settings['background_color'] : $default_color;
        } elseif ($field_id === 'input_container_border_color') {
            $input_container_settings = get_option('mxchat_input_container_settings', array());
            $value = isset($input_container_settings['border_color']) ? $input_container_settings['border_color'] : $default_color;
        } else {
            $value = isset($this->options[$field_id]) ? $this->options[$field_id] : $default_color;
        }
        
        printf(
            '<input type="text" 
                   id="%s" 
                   name="%s" 
                   value="%s" 
                   class="my-color-field" 
                   data-nonce="%s"
                   data-default-color="%s" />',
            esc_attr($field_id),
            esc_attr($field_id),
            esc_attr($value),
            wp_create_nonce('mxchat_save_setting_nonce'),
            esc_attr($default_color)
        );
    }
    
    public function render_icon_field($args) {
        $field_id = $args['field_id'];
        $value = isset($this->options[$field_id]) ? esc_url($this->options[$field_id]) : '';
        
        echo '<div class="icon-field-wrapper">';
        printf(
            '<input type="url" 
                   id="%s" 
                   name="mxchat_options[%s]" 
                   value="%s" 
                   placeholder="%s" 
                   class="regular-text icon-url-field" 
                   data-nonce="%s" />',
            esc_attr($field_id),
            esc_attr($field_id),
            esc_attr($value),
            esc_attr__('Enter PNG URL', 'mxchat-theme'),
            wp_create_nonce('mxchat_save_setting_nonce')
        );

        echo '<div class="icon-preview" style="margin-top: 10px;' . (empty($value) ? 'display:none;' : '') . '">';
        if (!empty($value)) {
            echo '<img src="' . esc_url($value) . '" alt="' . esc_attr__('Icon Preview', 'mxchat-theme') . '" 
                  style="max-width: 48px; height: auto;" />';
        }
        echo '</div>';
        
        echo '<p class="description">' . 
             esc_html__('Upload your PNG icon and paste the URL here. Recommended size: 48x48 pixels. Refresh page after save to view.', 'mxchat-theme') . 
             '</p>';
        
        echo '</div>'; // Close icon-field-wrapper
    }

    /**
     * Render position settings section info
     */
    public function render_position_section_info() {
        echo '<div class="mxchat-section-info">';
        echo '<p>' . esc_html__('Adjust the vertical position of the chatbot widget and pre-chat popup. Settings auto-save as you adjust.', 'mxchat-theme') . '</p>';
        echo '</div>';
    }
    
    /**
     * Render position input field
     */
    public function render_position_field($args) {
        $field_id = $args['field_id'];
        $default = isset($args['default']) ? $args['default'] : 0;
        $description = isset($args['description']) ? $args['description'] : '';
        
        // Get position settings
        $position_settings = get_option('mxchat_position_settings', array());
        $value = isset($position_settings[$field_id]) ? intval($position_settings[$field_id]) : $default;
        
        echo '<div class="position-field-wrapper">';
        printf(
            '<input type="range" 
                   id="%s" 
                   name="mxchat_position_settings[%s]" 
                   value="%d" 
                   min="0" 
                   max="500" 
                   step="5"
                   class="position-range-field" 
                   data-setting-name="%s"
                   data-nonce="%s" />',
            esc_attr($field_id),
            esc_attr($field_id),
            esc_attr($value),
            esc_attr($field_id),
            wp_create_nonce('mxchat_theme_nonce')
        );
        
        printf(
            '<input type="number" 
                   id="%s_number" 
                   class="position-number-field small-text" 
                   value="%d" 
                   min="0" 
                   max="500" 
                   step="5"
                   data-range-id="%s" />',
            esc_attr($field_id),
            esc_attr($value),
            esc_attr($field_id)
        );
        
        echo '<span class="position-unit">px</span>';
        
        if (!empty($description)) {
            echo '<p class="description">' . esc_html($description) . '</p>';
        }
        
        echo '<span class="position-save-status" data-field="' . esc_attr($field_id) . '"></span>';
        echo '</div>';
    }

    public function enqueue_admin_assets($hook) {
    if ('mxchat_page_mxchat-theme-settings' !== $hook) {
        return;
    }

    $version = '1.1.0'; // Consistent plugin version

    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    
    wp_enqueue_style(
        'mxchat-theme-admin',
        plugin_dir_url(__FILE__) . 'assets/css/admin.css',
        array(),
        $version
    );
    
    wp_enqueue_style(
        'mxchat-chatbot-preview',
        plugin_dir_url(__FILE__) . 'assets/css/chatbot-rendering.css',
        array(),
        $version
    );
    
    wp_enqueue_style(
        'mxchat-ai-generator',
        plugin_dir_url(__FILE__) . 'assets/css/ai-generator.css',
        array('mxchat-theme-admin'),
        $version
    );
    
    wp_enqueue_script(
        'mxchat-theme-admin',
        plugin_dir_url(__FILE__) . 'assets/js/admin.js',
        array('wp-color-picker', 'jquery'),
        $version,
        true
    );
    
    wp_enqueue_script(
        'mxchat-ai-generator',
        plugin_dir_url(__FILE__) . 'assets/js/ai-generator.js',
        array('jquery', 'mxchat-theme-admin'),
        $version,
        true
    );
    
    wp_enqueue_script(
        'mxchat-model-selector-integration',
        plugin_dir_url(__FILE__) . 'assets/js/model-selector-integration.js',
        array('jquery'),
        $version,
        true
    );

    wp_localize_script('mxchat-theme-admin', 'mxchatTheme', array(
        'ajaxurl' => admin_url('admin-ajax.php')
    ));
}

    /**
     * Output AI theme CSS to frontend and admin
     */
    public function output_ai_theme_css() {
        // Only output on frontend OR on the specific theme settings page
        if (is_admin()) {
            // Check if we're on the theme settings page
            $current_screen = get_current_screen();
            if (!$current_screen || $current_screen->id !== 'mxchat_page_mxchat-theme-settings') {
                return; // Don't output CSS on other admin pages
            }
        }
        
        $active_css = $this->ai_generator->get_active_ai_theme_css();
        
        if (!empty($active_css)) {
            // Wrap the CSS to be more specific to the preview container
            $scoped_css = $this->scope_css_to_preview($active_css);
            echo '<style id="mxchat-ai-theme-css">' . wp_strip_all_tags($scoped_css) . '</style>';
        }
    }
    
    /**
     * Scope CSS selectors to only apply within the preview container
     */
    private function scope_css_to_preview($css) {
        // If we're in admin, scope the CSS to the preview container
        if (is_admin()) {
            // Add the preview container scope to all CSS rules
            // This assumes your preview is wrapped in .mxchat-preview-container
            $css = preg_replace('/([^{}]+){/', '.mxchat-preview-container $1 {', $css);
            
            // Handle any edge cases where the scoping might have doubled up
            $css = str_replace('.mxchat-preview-container .mxchat-preview-container', '.mxchat-preview-container', $css);
        }
        
        return $css;
    }

    /**
     * Output position CSS for widget and pre-chat popup
     */
    public function output_position_css() {
        // Only output on frontend OR on the specific theme settings page
        if (is_admin()) {
            // Check if we're on the theme settings page
            $current_screen = get_current_screen();
            if (!$current_screen || $current_screen->id !== 'mxchat_page_mxchat-theme-settings') {
                return; // Don't output CSS on other admin pages
            }
        }
        
        // Get position settings with defaults
        $position_settings = get_option('mxchat_position_settings', array());
        
        // Default values (matching your current CSS)
        $widget_position = isset($position_settings['widget_position']) ? intval($position_settings['widget_position']) : 130;
        $prechat_position = isset($position_settings['prechat_position']) ? intval($position_settings['prechat_position']) : 185;
        $widget_position_mobile = isset($position_settings['widget_position_mobile']) ? intval($position_settings['widget_position_mobile']) : 75;
        $prechat_position_mobile = isset($position_settings['prechat_position_mobile']) ? intval($position_settings['prechat_position_mobile']) : 130;
        
        ?>
        <style id="mxchat-position-css">
            #floating-chatbot-button {
                bottom: <?php echo esc_attr($widget_position); ?>px !important;
            }

            #pre-chat-message {
                bottom: <?php echo esc_attr($prechat_position); ?>px !important;
            }

            @media (max-width: 550px) {
                #floating-chatbot-button {
                    bottom: <?php echo esc_attr($widget_position_mobile); ?>px !important;
                }
               
                #pre-chat-message {
                    bottom: <?php echo esc_attr($prechat_position_mobile); ?>px !important;
                }
            }
        </style>
        <?php
    }
    
    /**
     * Output input container CSS
     */
    public function output_input_container_css() {
        // Only output on frontend OR on the specific theme settings page
        if (is_admin()) {
            // Check if we're on the theme settings page
            $current_screen = get_current_screen();
            if (!$current_screen || $current_screen->id !== 'mxchat_page_mxchat-theme-settings') {
                return; // Don't output CSS on other admin pages
            }
        }
        
        // Get input container settings
        $input_container_settings = get_option('mxchat_input_container_settings', array());
        
        // Get colors
        $background_color = isset($input_container_settings['background_color']) ? $input_container_settings['background_color'] : '';
        $border_color = isset($input_container_settings['border_color']) ? $input_container_settings['border_color'] : '';
        
        // Only output CSS if at least one color is set
        if (!empty($background_color) || !empty($border_color)) {
            ?>
            <style id="mxchat-input-container-css">
                #mxchat-chatbot #chat-container #input-container {
                    <?php if (!empty($background_color)): ?>
                    background-color: <?php echo esc_attr($background_color); ?> !important;
                    <?php endif; ?>
                    <?php if (!empty($border_color)): ?>
                    border-color: <?php echo esc_attr($border_color); ?> !important;
                    <?php endif; ?>
                }
            </style>
            <?php
        }
    }

    /**
     * Render the theme settings page
     */
    public function render_theme_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap mxchat-wrapper">
            <div class="mxchat-hero">
                <h1 class="mxchat-main-title">
                    <span class="mxchat-gradient-text">Theme</span> Customizer
                </h1>
                <p class="mxchat-hero-subtitle">
                    <?php esc_html_e('Customize the appearance of your chatbot interface with custom colors and styles.', 'mxchat-theme'); ?>
                </p>
            </div>
    
            <div class="mxchat-content">
                <!-- AI Theme Generator Section -->
                <?php $this->render_ai_theme_generator_section(); ?>
                
                <div class="mxchat-card">
                    <div class="mxchat-settings-layout">
                        <div class="mxchat-settings-column">
                            <div class="mxchat-settings-form">
                                <?php do_settings_sections('mxchat-theme'); ?>
                            </div>
                        </div>
                        <div class="mxchat-preview-column">
                            <h2><?php esc_html_e('Live Preview', 'mxchat-theme'); ?></h2>
                            <div class="mxchat-preview-container">
                                <!-- Added .visible class and proper structure -->
                                <div id="floating-chatbot" class="preview visible">
                                    <div id="mxchat-chatbot-wrapper">
                                        <div class="chatbot-top-bar" id="exit-chat-button">
                                            <?php
                                            $has_title_icon = isset($this->options['title_icon']) && !empty(trim($this->options['title_icon']));
                                            $title_icon_url = $has_title_icon 
                                                ? esc_url($this->options['title_icon']) 
                                                : 'https://mxchat.ai/wp-content/uploads/2024/12/title-bar-icon.png';
                                            ?>
                                            
                                            <div class="chatbot-title-container">
                                                <div class="chatbot-title-group">
                                                    <?php if ($has_title_icon): ?>
                                                        <img src="<?php echo $title_icon_url; ?>" 
                                                             alt="" 
                                                             class="chatbot-title-icon"
                                                             onerror="this.style.display='none'; this.nextElementSibling.style.marginLeft='0';">
                                                    <?php endif; ?>
                                                    <p class="chatbot-title" style="<?php echo !$has_title_icon ? 'margin-left: 0;' : ''; ?>">MxChat</p>
                                                </div>
                                                <span class="chat-mode-indicator">AI Agent</span>
                                            </div>
                                            <button class="exit-chat" type="button" aria-label="Minimize">
                                                <svg xmlns="http://www.w3.org/2000/svg" height="24" viewBox="0 0 24 24" width="24" id="ic-minimize">
                                                    <path d="M0 0h24v24H0z" fill="none"></path>
                                                    <path d="M11.67 3.87L9.9 2.1 0 12l9.9 9.9 1.77-1.77L3.54 12z"></path>
                                                </svg>
                                                <span>Minimize</span>
                                            </button>
                                        </div>
                                        <div id="mxchat-chatbot">
                                            <div id="chat-container">
                                                <div id="chat-box">
                                                    <!-- Fixed message structure with proper div[dir="auto"] -->
                                                    <div class="bot-message">
                                                        <div dir="auto">This is what a bot message looks like! I'm always here, never sleep, and definitely don't dream of electric sheep. Adjust my colors as you see fit!</div>
                                                    </div>
                                                    <div class="user-message">
                                                        <div dir="auto">User messages appear like this! This is you, typing away, making important decisions… like what color looks best.</div>
                                                    </div>
                                                    <div class="agent-message">
                                                        <div dir="auto">Boom! A real human has entered the chat. Live agents' messages look like this—feel free to tweak them too!</div>
                                                    </div>
                                                    <!-- Added message with hyperlinks for testing -->
                                                    <div class="bot-message">
                                                        <div dir="auto">
                                                            <p>Great choice! Your item has been added to your cart for $33.56. You can view your cart <a href="#" target="_self">here</a> or proceed to checkout <a href="#" target="_self">here</a>. Feel free to continue shopping! 😊</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="quick-questions-preview" style="padding: 10px; border-top: 1px solid #eee;">
                                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px;">
                                                <span style="font-size: 12px; color: #666;">Quick Questions:</span>
                                                <!-- Collapse button (down arrow) -->
                                                <button class="questions-collapse-btn" aria-label="Hide Quick Questions" style="background: none; border: none; cursor: pointer;">
                                                    <svg width="25" height="25" viewBox="0 0 24 24" fill="none" stroke="#212121" stroke-width="2">
                                                        <polyline points="6,9 12,15 18,9"></polyline>
                                                    </svg>
                                                </button>
                                                <!-- Expand button (up arrow) -->
                                                <button class="questions-toggle-btn" aria-label="Show Quick Questions" style="background: none; border: none; cursor: pointer;">
                                                    <svg width="25" height="25" viewBox="0 0 24 24" fill="none" stroke="#212121" stroke-width="2">
                                                        <polyline points="18,15 12,9 6,15"></polyline>
                                                    </svg>
                                                </button>
                                            </div>
                                        </div>
                                            <div id="input-container">
                                                <!-- Added proper structure for input area background -->
                                                <div id="mxchat-chatbot" style="display: none;"></div> <!-- Hidden helper for CSS selector -->
                                                <textarea id="chat-input" dir="auto" placeholder="Enter chat here"></textarea>
                                                <button id="send-button">
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
                                                        <path d="M498.1 5.6c10.1 7 15.4 19.1 13.5 31.2l-64 416c-1.5 9.7-7.4 18.2-16 23s-18.9 5.4-28 1.6L284 427.7l-68.5 74.1c-8.9 9.7-22.9 12.9-35.2 8.1S160 493.2 160 480V396.4c0-4 1.5-7.8 4.2-10.7L331.8 202.8c5.8-6.3 5.6-16-.4-22s-15.7-6.4-22-.7L106 360.8 17.7 316.6C7.1 311.3 .3 300.7 0 288.9s5.9-22.8 16.1-28.7l448-256c10.7-6.1 23.9-5.5 34 1.4z"></path>
                                                    </svg>
                                                </button>
                                            </div>
                                            <div class="chat-toolbar">
                                                <input type="file" id="pdf-upload" accept=".pdf" style="display: none;">
                                                <button id="pdf-upload-btn" class="toolbar-btn" title="Upload PDF">
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" stroke="currentColor">
                                                        <path d="M64 464l48 0 0 48-48 0c-35.3 0-64-28.7-64-64L0 64C0 28.7 28.7 0 64 0L229.5 0c17 0 33.3 6.7 45.3 18.7l90.5 90.5c12 12 18.7 28.3 18.7 45.3L384 304l-48 0 0-144-80 0c-17.7 0-32-14.3-32-32l0-80L64 48c-8.8 0-16 7.2-16 16l0 384c0 8.8 7.2 16 16 16zM176 352l32 0c30.9 0 56 25.1 56 56s-25.1 56-56 56l-16 0 0 32c0 8.8-7.2 16-16 16s-16-7.2-16-16l0-48 0-80c0-8.8 7.2-16 16-16zm32 80c13.3 0 24-10.7 24-24s-10.7-24-24-24l-16 0 0 48 16 0zm96-80l32 0c26.5 0 48 21.5 48 48l0 64c0 26.5-21.5 48-48 48l-32 0c-8.8 0-16-7.2-16-16l0-128c0-8.8 7.2-16 16-16zm32 128c8.8 0 16-7.2 16-16l0-64c0-8.8-7.2-16-16-16l-16 0 0 96 16 0zm80-112c0-8.8 7.2-16 16-16l48 0c8.8 0 16 7.2 16 16s-7.2 16-16 16l-32 0 0 32 32 0c8.8 0 16 7.2 16 16s-7.2 16-16 16l-32 0 0 48c0 8.8-7.2 16-16 16s-16-7.2-16-16l0-64 0-64z"></path>
                                                    </svg>
                                                </button>
                                                <input type="file" id="word-upload" accept=".docx" style="display: none;">
                                                <button id="word-upload-btn" class="toolbar-btn" title="Upload Word Document">
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512" stroke="currentColor">
                                                        <path d="M48 448L48 64c0-8.8 7.2-16 16-16l160 0 0 80c0 17.7 14.3 32 32 32l80 0 0 288c0 8.8-7.2 16-16 16L64 464c-8.8 0-16-7.2-16-16zM64 0C28.7 0 0 28.7 0 64L0 448c0 35.3 28.7 64 64 64l256 0c35.3 0 64-28.7 64-64l0-293.5c0-17-6.7-33.3-18.7-45.3L274.7 18.7C262.7 6.7 246.5 0 229.5 0L64 0zm55 241.1c-3.8-12.7-17.2-19.9-29.9-16.1s-19.9 17.2-16.1 29.9l48 160c3 10.2 12.4 17.1 23 17.1s19.9-7 23-17.1l25-83.4 25 83.4c3 10.2 12.4 17.1 23 17.1s19.9-7 23-17.1l48-160c3.8-12.7-3.4-26.1-16.1-29.9s-26.1 3.4-29.9 16.1l-25 83.4-25-83.4c-3-10.2-12.4-17.1-23-17.1s-19.9 7-23 17.1l-25 83.4-25-83.4z"></path>
                                                    </svg>
                                                </button>
                                                <div id="active-pdf-container" class="active-pdf-container" style="display: none;">
                                                    <span id="active-pdf-name" class="active-pdf-name"></span>
                                                    <button id="remove-pdf-btn" class="remove-pdf-btn" title="Remove PDF">
                                                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="12" height="12" fill="none" stroke="currentColor" stroke-width="2">
                                                            <line x1="18" y1="6" x2="6" y2="18"></line>
                                                            <line x1="6" y1="6" x2="18" y2="18"></line>
                                                        </svg>
                                                    </button>
                                                </div>
                                                <div id="active-word-container" class="active-pdf-container" style="display: none;">
                                                    <span id="active-word-name" class="active-pdf-name"></span>
                                                    <button id="remove-word-btn" class="remove-pdf-btn" title="Remove Word Document">
                                                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="12" height="12" fill="none" stroke="currentColor" stroke-width="2">
                                                            <line x1="18" y1="6" x2="6" y2="18"></line>
                                                            <line x1="6" y1="6" x2="18" y2="18"></line>
                                                        </svg>
                                                    </button>
                                                </div>
                                            </div>
                                            
                                            <div class="chatbot-footer">
                                                <p class="privacy-notice">By chatting, you agree to our <a href="#" target="_blank">privacy policy</a>.</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render AI Theme Generator Section
     */
    private function render_ai_theme_generator_section() {
        include dirname(__FILE__) . '/templates/ai-theme-generator.php';
    }
}

// Initialize the plugin
new MxChat_Theme_Addon();