/**
 * MxChat Trigger - Admin Sidebar Navigation JavaScript
 *
 * @package MxChat_Trigger
 * @since 2.2.0
 */

(function($) {
    'use strict';

    /**
     * Sidebar Navigation Controller
     */
    var SidebarNav = {
        /**
         * Initialize the sidebar navigation
         */
        init: function() {
            this.bindEvents();
            this.initCopyButtons();
            this.initToggle();
            this.restoreLastSection();
        },

        /**
         * Bind navigation events
         */
        bindEvents: function() {
            var self = this;

            // Desktop navigation
            $(document).on('click', '.mxch-nav-link[data-target]', function(e) {
                e.preventDefault();
                var target = $(this).data('target');
                self.switchSection(target);
                self.updateActiveNavLink($(this));
                self.saveLastSection(target);
            });

            // Mobile navigation
            $(document).on('click', '.mxch-mobile-nav-link[data-target]', function(e) {
                e.preventDefault();
                var target = $(this).data('target');
                self.switchSection(target);
                self.updateActiveMobileNavLink($(this));
                self.closeMobileMenu();
                self.saveLastSection(target);
            });

            // Mobile menu toggle
            $(document).on('click', '.mxch-mobile-menu-btn', function(e) {
                e.preventDefault();
                self.openMobileMenu();
            });

            // Mobile menu close
            $(document).on('click', '.mxch-mobile-menu-close, .mxch-mobile-overlay', function(e) {
                e.preventDefault();
                self.closeMobileMenu();
            });

            // Close mobile menu on escape
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    self.closeMobileMenu();
                }
            });
        },

        /**
         * Switch to a specific section
         */
        switchSection: function(sectionId) {
            // Hide all sections
            $('.mxch-section').removeClass('active');

            // Show target section
            $('#' + sectionId).addClass('active');

            // Scroll to top of content
            $('.mxch-content').scrollTop(0);
        },

        /**
         * Update active state on desktop nav links
         */
        updateActiveNavLink: function($clickedLink) {
            $('.mxch-nav-link').removeClass('active');
            $clickedLink.addClass('active');
        },

        /**
         * Update active state on mobile nav links
         */
        updateActiveMobileNavLink: function($clickedLink) {
            $('.mxch-mobile-nav-link').removeClass('active');
            $clickedLink.addClass('active');

            // Also update desktop nav
            var target = $clickedLink.data('target');
            $('.mxch-nav-link').removeClass('active');
            $('.mxch-nav-link[data-target="' + target + '"]').addClass('active');
        },

        /**
         * Open mobile menu
         */
        openMobileMenu: function() {
            $('.mxch-mobile-menu').addClass('active');
            $('.mxch-mobile-overlay').addClass('active');
            $('body').css('overflow', 'hidden');
        },

        /**
         * Close mobile menu
         */
        closeMobileMenu: function() {
            $('.mxch-mobile-menu').removeClass('active');
            $('.mxch-mobile-overlay').removeClass('active');
            $('body').css('overflow', '');
        },

        /**
         * Save last visited section
         */
        saveLastSection: function(sectionId) {
            if (typeof localStorage !== 'undefined') {
                localStorage.setItem('mxch_trigger_last_section', sectionId);
            }
        },

        /**
         * Restore last visited section
         */
        restoreLastSection: function() {
            if (typeof localStorage !== 'undefined') {
                var lastSection = localStorage.getItem('mxch_trigger_last_section');
                if (lastSection && $('#' + lastSection).length) {
                    this.switchSection(lastSection);
                    this.updateActiveNavLink($('.mxch-nav-link[data-target="' + lastSection + '"]'));
                    this.updateActiveMobileNavLink($('.mxch-mobile-nav-link[data-target="' + lastSection + '"]'));
                }
            }
        },

        /**
         * Initialize copy to clipboard buttons
         */
        initCopyButtons: function() {
            // Code block copy buttons
            $(document).on('click', '.mxch-copy-btn', function(e) {
                e.preventDefault();
                var targetId = $(this).data('copy-target');
                var codeText = $('#' + targetId).text();
                var $btn = $(this);

                navigator.clipboard.writeText(codeText).then(function() {
                    var originalHtml = $btn.html();
                    $btn.html('<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg> Copied!');
                    setTimeout(function() {
                        $btn.html(originalHtml);
                    }, 2000);
                }).catch(function() {
                    // Fallback for older browsers
                    var textArea = document.createElement('textarea');
                    textArea.value = codeText;
                    textArea.style.position = 'fixed';
                    textArea.style.opacity = '0';
                    document.body.appendChild(textArea);
                    textArea.select();
                    try {
                        document.execCommand('copy');
                        var originalHtml = $btn.html();
                        $btn.html('<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg> Copied!');
                        setTimeout(function() {
                            $btn.html(originalHtml);
                        }, 2000);
                    } catch (err) {
                        console.error('Copy failed:', err);
                    }
                    document.body.removeChild(textArea);
                });
            });

            // Inline copy buttons (for bot IDs)
            $(document).on('click', '.mxch-copy-inline', function(e) {
                e.preventDefault();
                var textToCopy = $(this).data('copy-text');
                var $btn = $(this);

                navigator.clipboard.writeText(textToCopy).then(function() {
                    var originalHtml = $btn.html();
                    $btn.html('<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg>');
                    setTimeout(function() {
                        $btn.html(originalHtml);
                    }, 1500);
                });
            });
        },

        /**
         * Initialize the enable/disable toggle
         */
        initToggle: function() {
            $('#triggers_enabled').on('change', function() {
                var value = $(this).is(':checked') ? 'on' : 'off';
                var $toggle = $(this).closest('.mxch-toggle');

                // Add loading state
                $toggle.addClass('loading');

                $.ajax({
                    url: ajaxurl,
                    method: 'POST',
                    data: {
                        action: 'mxchat_trigger_save_setting',
                        nonce: window.mxchatTriggerNonce,
                        name: 'enabled',
                        value: value
                    },
                    success: function(response) {
                        $toggle.removeClass('loading');
                        if (response.success) {
                            // Optional: Show success feedback
                        }
                    },
                    error: function() {
                        $toggle.removeClass('loading');
                        // Revert toggle on error
                        $('#triggers_enabled').prop('checked', value !== 'on');
                    }
                });
            });
        }
    };

    /**
     * Initialize when DOM is ready
     */
    $(document).ready(function() {
        // Only initialize on the triggers admin page
        if ($('.mxch-admin-wrapper').length) {
            SidebarNav.init();
        }
    });

})(jQuery);
