/**
 * MxChat Triggers - Frontend JavaScript
 *
 * Enables any HTML element to trigger the MxChat chatbot using data attributes:
 * - data-mxchat-trigger="message" : Opens chat and sends the specified message
 * - data-mxchat-open              : Just opens the chat without sending a message
 * - data-mxchat-bot-id="bot_id"   : Target a specific bot (multi-bot support)
 *
 * @package MxChat Trigger
 * @since 2.1.0
 */

(function($) {
    'use strict';

    // Configuration
    var config = {
        maxWaitAttempts: 50,      // Max times to check if chat is ready (50 * 100ms = 5 seconds)
        checkInterval: 100,       // Milliseconds between ready checks
        messageDelay: 300         // Delay before sending message after opening chat
    };

    /**
     * Get the bot ID from an element or return default
     * @param {jQuery|HTMLElement} element - The trigger element
     * @returns {string} The bot ID
     */
    function getBotId(element) {
        var $element = $(element);
        var botId = $element.attr('data-mxchat-bot-id') || $element.data('mxchat-bot-id');

        // If no bot specified, try to find the first/default bot on page
        if (!botId) {
            // Check if there's only one bot on the page
            var $wrappers = $('.mxchat-chatbot-wrapper');
            if ($wrappers.length === 1) {
                botId = $wrappers.attr('data-bot-id') || 'default';
            } else {
                botId = 'default';
            }
        }

        return botId;
    }

    /**
     * Get selectors for a specific bot
     * Uses class-based selectors with bot_id suffix for multi-bot support
     * @param {string} botId - The bot ID
     * @returns {object} Object with jQuery selectors
     */
    function getSelectorsForBot(botId) {
        // New class-based selector pattern with ID suffix
        return {
            chatbot: '#floating-chatbot-' + botId,
            chatButton: '#floating-chatbot-button-' + botId,
            notificationBadge: '#chat-notification-badge-' + botId,
            preChatMessage: '#pre-chat-message-' + botId,
            chatInput: '#chat-input-' + botId,
            sendButton: '#send-button-' + botId,
            chatBox: '#chat-box-' + botId,
            wrapper: '.mxchat-chatbot-wrapper[data-bot-id="' + botId + '"]'
        };
    }

    /**
     * Check if a specific bot's chat system is ready
     * @param {string} botId - The bot ID to check
     * @returns {boolean}
     */
    function isChatReady(botId) {
        botId = botId || 'default';
        var selectors = getSelectorsForBot(botId);

        // Check for new class-based structure first
        if ($(selectors.chatbot).length > 0) {
            return true;
        }

        // Fallback: check for any chatbot wrapper with this bot ID
        if ($(selectors.wrapper).length > 0) {
            return true;
        }

        // Legacy fallback: check old ID-based selector (for backwards compatibility)
        if (botId === 'default' && $('#floating-chatbot').length > 0) {
            return true;
        }

        return false;
    }

    /**
     * Get the actual chatbot element (handles both new and legacy selectors)
     * @param {string} botId - The bot ID
     * @returns {jQuery} The chatbot element
     */
    function getChatbotElement(botId) {
        botId = botId || 'default';
        var selectors = getSelectorsForBot(botId);

        // Try new selector first
        var $chatbot = $(selectors.chatbot);
        if ($chatbot.length > 0) {
            return $chatbot;
        }

        // Try wrapper-based selector
        var $wrapper = $(selectors.wrapper);
        if ($wrapper.length > 0) {
            var $floating = $wrapper.find('.floating-chatbot');
            if ($floating.length > 0) {
                return $floating;
            }
        }

        // Legacy fallback for default bot
        if (botId === 'default') {
            return $('#floating-chatbot');
        }

        return $();
    }

    /**
     * Get the chat button element
     * @param {string} botId - The bot ID
     * @returns {jQuery} The chat button element
     */
    function getChatButtonElement(botId) {
        botId = botId || 'default';
        var selectors = getSelectorsForBot(botId);

        var $button = $(selectors.chatButton);
        if ($button.length > 0) {
            return $button;
        }

        // Legacy fallback
        if (botId === 'default') {
            return $('#floating-chatbot-button');
        }

        return $();
    }

    /**
     * Open the chat window for a specific bot
     * @param {string} botId - The bot ID to open
     */
    function openChat(botId) {
        botId = botId || 'default';
        var selectors = getSelectorsForBot(botId);

        var $chatbot = getChatbotElement(botId);
        var $chatButton = getChatButtonElement(botId);

        if ($chatbot.length === 0) {
            if (typeof mxchatTriggers !== 'undefined' && mxchatTriggers.debug) {
                console.warn('MxChat Triggers: Could not find chatbot for bot ID:', botId);
            }
            return;
        }

        if ($chatbot.hasClass('hidden')) {
            $chatbot.removeClass('hidden').addClass('visible');
            $chatButton.addClass('hidden');

            // Hide notification badge
            $(selectors.notificationBadge).hide();
            // Legacy fallback
            if (botId === 'default') {
                $('#chat-notification-badge').hide();
            }

            // Disable scroll on mobile
            if (window.innerWidth <= 768) {
                $('body').css('overflow', 'hidden');
            }

            // Hide pre-chat message
            $(selectors.preChatMessage).fadeOut(250);
            // Legacy fallback
            if (botId === 'default') {
                $('#pre-chat-message').fadeOut(250);
            }
        }

        // Dispatch custom event with bot ID
        document.dispatchEvent(new CustomEvent('mxchat:trigger:open', {
            detail: { botId: botId }
        }));
    }

    /**
     * Send a message to a specific bot's chatbot
     * @param {string} message - The message to send
     * @param {string} botId - The bot ID
     */
    function sendMessage(message, botId) {
        if (!message || typeof message !== 'string') {
            return;
        }

        botId = botId || 'default';

        // Check for MxChatInstances (new multi-bot architecture)
        if (typeof window.MxChatInstances !== 'undefined' && window.MxChatInstances.get) {
            var instance = window.MxChatInstances.get(botId);
            if (instance) {
                // Use instance-specific methods if available
                sendMessageViaInstance(message, botId);
                return;
            }
        }

        // Fallback to global functions (legacy/single bot mode)
        sendMessageViaGlobal(message, botId);
    }

    /**
     * Send message using MxChatInstances (multi-bot)
     * @param {string} message - The message to send
     * @param {string} botId - The bot ID
     */
    function sendMessageViaInstance(message, botId) {
        // Use the global appendMessage function with bot ID
        if (typeof window.appendMessage === 'function') {
            window.appendMessage('user', message, botId);
        }

        // Collapse quick questions if they exist
        if (typeof window.hasQuickQuestions === 'function' &&
            typeof window.collapseQuickQuestions === 'function' &&
            window.hasQuickQuestions(botId)) {
            window.collapseQuickQuestions(botId);
        }

        // Send to the chatbot
        if (typeof window.callMxChat === 'function') {
            // Show thinking indicator
            if (typeof window.appendThinkingMessage === 'function') {
                window.appendThinkingMessage(botId);
            }

            // Scroll to bottom
            if (typeof window.scrollToBottom === 'function') {
                window.scrollToBottom(botId);
            }

            // Check if streaming should be used
            if (typeof window.shouldUseStreaming === 'function' && window.shouldUseStreaming(botId)) {
                if (typeof window.callMxChatStream === 'function') {
                    window.callMxChatStream(message, function(response) {
                        if (typeof window.replaceLastMessage === 'function') {
                            window.replaceLastMessage('bot', response, botId);
                        }
                    }, botId);
                } else {
                    window.callMxChat(message, function(response) {
                        if (typeof window.replaceLastMessage === 'function') {
                            window.replaceLastMessage('bot', response, botId);
                        }
                    }, botId);
                }
            } else {
                window.callMxChat(message, function(response) {
                    if (typeof window.replaceLastMessage === 'function') {
                        window.replaceLastMessage('bot', response, botId);
                    }
                }, botId);
            }
        }

        // Dispatch custom event with bot ID
        document.dispatchEvent(new CustomEvent('mxchat:trigger:message', {
            detail: { message: message, botId: botId }
        }));
    }

    /**
     * Send message using global functions (legacy single-bot mode)
     * @param {string} message - The message to send
     * @param {string} botId - The bot ID (used for events)
     */
    function sendMessageViaGlobal(message, botId) {
        // Use the global appendMessage function to show the user's message
        if (typeof window.appendMessage === 'function') {
            window.appendMessage('user', message);
        }

        // Collapse quick questions if they exist
        if (typeof window.hasQuickQuestions === 'function' &&
            typeof window.collapseQuickQuestions === 'function' &&
            window.hasQuickQuestions()) {
            window.collapseQuickQuestions();
        }

        // Send to the chatbot using the best available method
        if (typeof window.callMxChat === 'function') {
            // Show thinking indicator
            if (typeof window.appendThinkingMessage === 'function') {
                window.appendThinkingMessage();
            }

            // Scroll to bottom
            if (typeof window.scrollToBottom === 'function') {
                window.scrollToBottom();
            }

            // Check if streaming should be used
            if (typeof window.shouldUseStreaming === 'function' && window.shouldUseStreaming()) {
                // Use streaming
                if (typeof window.callMxChatStream === 'function') {
                    window.callMxChatStream(message, function(response) {
                        if (typeof window.replaceLastMessage === 'function') {
                            window.replaceLastMessage('bot', response);
                        }
                    });
                } else {
                    // Fallback to non-streaming
                    window.callMxChat(message, function(response) {
                        if (typeof window.replaceLastMessage === 'function') {
                            window.replaceLastMessage('bot', response);
                        }
                    });
                }
            } else {
                // Use non-streaming
                window.callMxChat(message, function(response) {
                    if (typeof window.replaceLastMessage === 'function') {
                        window.replaceLastMessage('bot', response);
                    }
                });
            }
        }

        // Dispatch custom event
        document.dispatchEvent(new CustomEvent('mxchat:trigger:message', {
            detail: { message: message, botId: botId }
        }));
    }

    /**
     * Open chat and optionally send a message
     * @param {string} message - The message to send (optional)
     * @param {string} botId - The bot ID
     */
    function openChatAndSend(message, botId) {
        botId = botId || 'default';

        // Open the chat first
        openChat(botId);

        // If a message is provided, send it after a short delay
        if (message) {
            setTimeout(function() {
                sendMessage(message, botId);
            }, config.messageDelay);
        }
    }

    /**
     * Wait for chat to be ready, then execute callback
     * @param {function} callback - Function to call when ready
     * @param {string} botId - The bot ID to wait for
     */
    function whenReady(callback, botId) {
        botId = botId || 'default';

        if (isChatReady(botId)) {
            callback();
            return;
        }

        var attempts = 0;
        var checkInterval = setInterval(function() {
            attempts++;

            if (isChatReady(botId)) {
                clearInterval(checkInterval);
                callback();
            } else if (attempts >= config.maxWaitAttempts) {
                clearInterval(checkInterval);
                if (typeof mxchatTriggers !== 'undefined' && mxchatTriggers.debug) {
                    console.warn('MxChat Triggers: Chat system not ready for bot "' + botId + '" after ' + (config.maxWaitAttempts * config.checkInterval / 1000) + ' seconds');
                }
            }
        }, config.checkInterval);
    }

    /**
     * Handle trigger click
     */
    function handleTriggerClick(e) {
        e.preventDefault();

        var $element = $(this);
        var message = $element.attr('data-mxchat-trigger');
        var botId = getBotId($element);

        whenReady(function() {
            openChatAndSend(message, botId);
        }, botId);
    }

    /**
     * Handle open-only click
     */
    function handleOpenClick(e) {
        e.preventDefault();

        var $element = $(this);
        var botId = getBotId($element);

        whenReady(function() {
            openChat(botId);
        }, botId);
    }

    /**
     * Get list of available bots on the current page
     * @returns {array} Array of bot IDs
     */
    function getAvailableBots() {
        var bots = [];
        $('.mxchat-chatbot-wrapper').each(function() {
            var botId = $(this).attr('data-bot-id');
            if (botId && bots.indexOf(botId) === -1) {
                bots.push(botId);
            }
        });

        // Also check for legacy single-bot setup
        if (bots.length === 0 && $('#floating-chatbot').length > 0) {
            bots.push('default');
        }

        return bots;
    }

    /**
     * Initialize triggers
     */
    function init() {
        // Bind click events using event delegation for dynamic elements
        $(document).on('click', '[data-mxchat-trigger]', handleTriggerClick);
        $(document).on('click', '[data-mxchat-open]', handleOpenClick);

        // Log initialization in debug mode
        if (typeof mxchatTriggers !== 'undefined' && mxchatTriggers.debug) {
            console.log('MxChat Triggers: Initialized (v2.1.0 with multi-bot support)');

            // Count triggers on page
            var triggerCount = $('[data-mxchat-trigger]').length;
            var openCount = $('[data-mxchat-open]').length;
            var availableBots = getAvailableBots();

            if (triggerCount > 0 || openCount > 0) {
                console.log('MxChat Triggers: Found ' + triggerCount + ' message triggers and ' + openCount + ' open triggers');
            }

            if (availableBots.length > 0) {
                console.log('MxChat Triggers: Available bots on page:', availableBots);
            }
        }
    }

    // Expose public API with multi-bot support

    /**
     * Open chat for a specific bot
     * @param {string} botId - Optional bot ID (defaults to 'default')
     */
    window.mxchatOpen = function(botId) {
        botId = botId || 'default';
        whenReady(function() {
            openChat(botId);
        }, botId);
    };

    /**
     * Open chat and send a message to a specific bot
     * @param {string} message - The message to send
     * @param {string} botId - Optional bot ID (defaults to 'default')
     */
    window.mxchatOpenAndSend = function(message, botId) {
        botId = botId || 'default';
        whenReady(function() {
            openChatAndSend(message, botId);
        }, botId);
    };

    /**
     * Check if a specific bot's chat is ready
     * @param {string} botId - Optional bot ID (defaults to 'default')
     * @returns {boolean}
     */
    window.mxchatIsReady = function(botId) {
        return isChatReady(botId || 'default');
    };

    /**
     * Send a message to a specific bot (chat must be open)
     * @param {string} message - The message to send
     * @param {string} botId - Optional bot ID (defaults to 'default')
     */
    window.mxchatSendMessage = function(message, botId) {
        botId = botId || 'default';
        whenReady(function() {
            sendMessage(message, botId);
        }, botId);
    };

    /**
     * Get list of available bots on the page
     * @returns {array} Array of bot IDs
     */
    window.mxchatGetAvailableBots = function() {
        return getAvailableBots();
    };

    // Initialize when DOM is ready
    $(document).ready(init);

})(jQuery);
