<?php
/**
 * Plugin Name: MxChat Trigger
 * Plugin URI: https://mxchat.ai/
 * Description: Add chat trigger buttons that open MxChat and send predefined messages. Requires MxChat plugin.
 * Version: 2.2.1
 * Author: MxChat
 * Author URI: https://mxchat.ai
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: mxchat-trigger
 * Requires Plugins: mxchat-basic
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MXCHAT_TRIGGER_VERSION', '2.2.1');
define('MXCHAT_TRIGGER_PATH', plugin_dir_path(__FILE__));
define('MXCHAT_TRIGGER_URL', plugin_dir_url(__FILE__));

/**
 * Check if MxChat is installed and active (multisite aware)
 */
function is_mxchat_active_trigger() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

// Check if MxChat is installed and active
if (!is_mxchat_active_trigger()) {
    add_action('admin_notices', function() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Trigger</strong> requires the MxChat plugin to be installed and active.', 'mxchat-trigger'); ?></p>
        </div>
        <?php
    });
    return;
}

// Include update checker
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerTrigger::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-trigger'
]);

/**
 * Main MxChat Trigger Class
 */
class MxChat_Trigger {

    /**
     * Plugin instance
     */
    private static $instance = null;

    /**
     * Plugin options
     */
    private $options;

    /**
     * Get plugin instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->options = get_option('mxchat_trigger_options', array(
            'enabled' => 'on',
        ));

        // Add admin notice for pro requirement
        add_action('admin_notices', array($this, 'show_pro_requirement_notice'));

        // Only load features if pro is active
        if ($this->is_license_active()) {
            $this->init_components();
        }

        // Always add hooks for text domain loading
        add_action('plugins_loaded', array($this, 'load_textdomain'));
    }

    /**
     * Check if MxChat Pro is activated
     */
    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    /**
     * Admin notice for pro requirement
     */
    public function show_pro_requirement_notice() {
        if (!$this->is_license_active()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Trigger requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-trigger'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    /**
     * Load plugin text domain
     */
    public function load_textdomain() {
        load_plugin_textdomain('mxchat-trigger', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /**
     * Initialize components (only if license is active)
     */
    private function init_components() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'), 20);
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_mxchat_trigger_save_setting', array($this, 'ajax_save_setting'));
        add_action('wp_ajax_mxchat_trigger_get_bots', array($this, 'ajax_get_bots'));

        // Frontend hooks
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'), 20);

        // Keep shortcode for backwards compatibility
        add_shortcode('mxchat_trigger', array($this, 'trigger_shortcode'));
    }

    /**
     * Check if Multi-Bot add-on is active
     */
    private function is_multi_bot_active() {
        return class_exists('MxChat_Multi_Bot_Core_Manager');
    }

    /**
     * Get available bots from Multi-Bot plugin
     */
    private function get_available_bots() {
        $bots = array(
            'default' => __('Default Bot', 'mxchat-trigger')
        );

        if ($this->is_multi_bot_active()) {
            try {
                $manager = MxChat_Multi_Bot_Core_Manager::get_instance();
                if (method_exists($manager, 'get_available_bots')) {
                    $multi_bots = $manager->get_available_bots();
                    if (is_array($multi_bots)) {
                        $bots = $multi_bots;
                    }
                }
            } catch (Exception $e) {
                // Silently fail, return default bots array
            }
        }

        return $bots;
    }

    /**
     * Add submenu under MxChat menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'mxchat-max',                                    // Parent slug (MxChat's menu)
            esc_html__('Chat Triggers', 'mxchat-trigger'),   // Page title
            esc_html__('Triggers', 'mxchat-trigger'),        // Menu title
            'manage_options',                                 // Capability
            'mxchat-triggers',                               // Menu slug
            array($this, 'render_admin_page')                // Callback
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('mxchat_trigger_options', 'mxchat_trigger_options', array(
            'sanitize_callback' => array($this, 'sanitize_options'),
        ));
    }

    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = array();
        $sanitized['enabled'] = isset($input['enabled']) && $input['enabled'] === 'on' ? 'on' : 'off';
        return $sanitized;
    }

    /**
     * AJAX handler for saving settings
     */
    public function ajax_save_setting() {
        // Check pro status first
        if (!$this->is_license_active()) {
            wp_send_json_error(array('message' => __('Pro license required', 'mxchat-trigger')));
            return;
        }

        check_ajax_referer('mxchat_trigger_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'));
        }

        $name = sanitize_text_field($_POST['name'] ?? '');
        $value = sanitize_text_field($_POST['value'] ?? '');

        $options = get_option('mxchat_trigger_options', array());
        $options[$name] = $value;
        update_option('mxchat_trigger_options', $options);

        // Update local options
        $this->options = $options;

        wp_send_json_success(array('message' => 'Setting saved'));
    }

    /**
     * AJAX handler for getting available bots
     */
    public function ajax_get_bots() {
        check_ajax_referer('mxchat_trigger_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'));
        }

        $bots = $this->get_available_bots();
        wp_send_json_success(array('bots' => $bots));
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our admin page
        // Hook format for submenu: {parent_slug}_page_{menu_slug}
        if (strpos($hook, 'mxchat-triggers') === false) {
            return;
        }

        // Admin CSS
        wp_enqueue_style(
            'mxchat-trigger-admin',
            MXCHAT_TRIGGER_URL . 'assets/css/admin.css',
            array(),
            MXCHAT_TRIGGER_VERSION
        );

        // Admin JavaScript
        wp_enqueue_script(
            'mxchat-trigger-admin',
            MXCHAT_TRIGGER_URL . 'assets/js/admin.js',
            array('jquery'),
            MXCHAT_TRIGGER_VERSION,
            true
        );
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        $enabled = isset($this->options['enabled']) ? $this->options['enabled'] : 'on';

        if ($enabled !== 'on') {
            return;
        }

        // Enqueue trigger JavaScript
        wp_enqueue_script(
            'mxchat-triggers',
            MXCHAT_TRIGGER_URL . 'assets/js/mxchat-triggers.js',
            array('jquery'),
            MXCHAT_TRIGGER_VERSION,
            true
        );

        // Enqueue optional CSS helper classes
        wp_enqueue_style(
            'mxchat-triggers',
            MXCHAT_TRIGGER_URL . 'assets/css/mxchat-triggers.css',
            array(),
            MXCHAT_TRIGGER_VERSION
        );

        // Pass settings to JS
        wp_localize_script('mxchat-triggers', 'mxchatTriggers', array(
            'debug' => defined('WP_DEBUG') && WP_DEBUG,
            'multiBot' => $this->is_multi_bot_active(),
        ));
    }

    /**
     * Backwards-compatible shortcode with multi-bot support
     */
    public function trigger_shortcode($atts) {
        $atts = shortcode_atts(array(
            'message' => 'Hello!',
            'text' => 'Chat with Us',
            'class' => '',
            'bot_id' => '',  // New: support for specific bot
            'open_only' => 'no',  // New: just open without message
        ), $atts);

        $classes = 'mxchat-btn mxchat-btn-primary';
        if (!empty($atts['class'])) {
            $classes .= ' ' . esc_attr($atts['class']);
        }

        // Build data attributes
        $data_attrs = '';
        if ($atts['open_only'] === 'yes') {
            $data_attrs = 'data-mxchat-open';
        } else {
            $data_attrs = sprintf('data-mxchat-trigger="%s"', esc_attr($atts['message']));
        }

        // Add bot ID if specified
        if (!empty($atts['bot_id'])) {
            $data_attrs .= sprintf(' data-mxchat-bot-id="%s"', esc_attr($atts['bot_id']));
        }

        return sprintf(
            '<button class="%s" %s>%s</button>',
            esc_attr($classes),
            $data_attrs,
            esc_html($atts['text'])
        );
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        // Include the new sidebar-based admin page template
        require_once MXCHAT_TRIGGER_PATH . 'includes/admin-page.php';
        mxchat_trigger_render_admin_page($this);
    }

    /**
     * Get license status (public method)
     */
    public function is_licensed() {
        return $this->is_license_active();
    }

    /**
     * Get plugin version
     */
    public function get_version() {
        return MXCHAT_TRIGGER_VERSION;
    }
}

// Initialize the plugin
MxChat_Trigger::get_instance();
