jQuery(document).ready(function($) {
    'use strict';

    const VeoAdmin = {
        currentOperation: null,
        pollInterval: null,
        maxPollAttempts: 120,
        pollAttempts: 0,
        currentPage: 1,
        isLoading: false,
        searchTimeout: null,
        
        init: function() {
            this.bindEvents();
            this.initToggles();
            this.initCharacterCounters();
            this.initModelDependencies();
        },

bindEvents: function() {
    // Modal controls
    $('#open-settings, #setup-now').on('click', this.openSettings);
    $('.mxchat-veo-modal-close').on('click', this.closeModal);
    $('.mxchat-veo-modal-overlay').on('click', this.handleModalOverlayClick);
    
    // Form controls
    $('#mxchat_veo_api_key').siblings('.mxchat-veo-toggle-visibility').on('click', this.togglePasswordVisibility);
    $('#toggle-advanced').on('click', this.toggleAdvancedOptions);
    $('#generate-video').on('click', this.generateVideo);
    $('#save-video').on('click', this.saveVideo);
    $('#veo_model').on('change', this.onModelChange);
    $('#veo_prompt, #veo_negative_prompt').on('input', this.updateCharacterCount);
    $('.mxchat-veo-history-item').on('click', this.playHistoryVideo);
    $('.mxchat-veo-history-download').on('click', this.downloadVideo);
    $('.mxchat-veo-history-delete').on('click', this.deleteVideo);

    // NEW: Image selection events - FIXED debounce
    $('#select-image').on('click', this.openImageBrowser);
    $('#remove-image').on('click', this.removeSelectedImage);
    $('#image-search').on('input', this.debounceSearch); // Changed this line
    $(document).on('click', '.mxchat-veo-image-item', this.selectImage);
    $(document).on('click', '.mxchat-veo-pagination-btn', this.changePage);

    // Keyboard shortcuts
    $(document).on('keydown', this.handleKeyboardShortcuts);
},

// Add this new method to handle search with simple debouncing
debounceSearch: function() {
    clearTimeout(VeoAdmin.searchTimeout);
    VeoAdmin.searchTimeout = setTimeout(function() {
        VeoAdmin.searchImages();
    }, 300);
},
        openSettings: function(e) {
            e.preventDefault();
            $('#settings-modal').addClass('active').fadeIn(300);
            $('body').addClass('modal-open');
        },

        closeModal: function(e) {
            e.preventDefault();
            $('.mxchat-veo-modal-overlay').removeClass('active').fadeOut(300);
            $('body').removeClass('modal-open');
        },

        handleModalOverlayClick: function(e) {
            if (e.target === this) {
                VeoAdmin.closeModal(e);
            }
        },

        initToggles: function() {
            $('.mxchat-veo-toggle input[type="checkbox"]').on('change', function() {
                const $toggle = $(this).closest('.mxchat-veo-toggle');
                $toggle.toggleClass('active', this.checked);
            });

            $('.mxchat-veo-toggle input[type="checkbox"]:checked').each(function() {
                $(this).closest('.mxchat-veo-toggle').addClass('active');
            });
        },

        initCharacterCounters: function() {
            const $prompt = $('#veo_prompt');
            const $negativePrompt = $('#veo_negative_prompt');
            
            if ($prompt.length) {
                this.updateCharacterCount.call($prompt[0]);
            }
            if ($negativePrompt.length) {
                this.updateCharacterCount.call($negativePrompt[0]);
            }
        },

        initModelDependencies: function() {
            this.populateModelOptions();
        },

        // Model configuration data
        modelConfig: {
            // Google Veo models
            'veo-3.1-generate-preview': {
                provider: 'google',
                resolutions: ['720p', '1080p'],
                durations: [4, 6, 8],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1024,
                supportsNegativePrompt: true,
                time: '11s - 6min'
            },
            'veo-3.1-fast-generate-preview': {
                provider: 'google',
                resolutions: ['720p', '1080p'],
                durations: [4, 6, 8],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1024,
                supportsNegativePrompt: true,
                time: '8s - 3min'
            },
            'veo-3.0-generate-001': {
                provider: 'google',
                resolutions: ['720p', '1080p'],
                durations: [4, 6, 8],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1024,
                supportsNegativePrompt: true,
                time: '11s - 6min'
            },
            'veo-3.0-fast-generate-001': {
                provider: 'google',
                resolutions: ['720p', '1080p'],
                durations: [4, 6, 8],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1024,
                supportsNegativePrompt: true,
                time: '8s - 3min'
            },
            'veo-2.0-generate-001': {
                provider: 'google',
                resolutions: ['720p'],
                durations: [5, 6, 8],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1024,
                supportsNegativePrompt: true,
                time: '15s - 5min'
            },
            // OpenAI Sora models - only supports '4', '8', '12' seconds
            'sora-2': {
                provider: 'openai',
                resolutions: ['720p', '1080p'],
                durations: [4, 8, 12],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '720p',
                defaultDuration: 8,
                maxPromptLength: 1000,
                supportsNegativePrompt: false,
                time: '1min - 5min'
            },
            'sora-2-pro': {
                provider: 'openai',
                resolutions: ['720p', '1080p'],
                durations: [4, 8, 12],
                aspectRatios: ['16:9', '9:16'],
                defaultResolution: '1080p',
                defaultDuration: 8,
                maxPromptLength: 1000,
                supportsNegativePrompt: false,
                time: '2min - 10min'
            }
        },

        populateModelOptions: function() {
            const selectedModel = $('#veo_model').val();
            const config = VeoAdmin.modelConfig[selectedModel];

            if (!config) return;

            // Populate resolution dropdown
            const $resolution = $('#veo_resolution');
            const currentResolution = $resolution.val();
            $resolution.empty();

            config.resolutions.forEach(function(res) {
                let label = res;
                if (res === '480p') label = '480p (SD)';
                else if (res === '720p') label = '720p (HD)';
                else if (res === '1080p') label = '1080p (Full HD)';
                $resolution.append($('<option>', {
                    value: res,
                    text: label
                }));
            });

            // Restore previous selection if valid, otherwise use default
            if (config.resolutions.includes(currentResolution)) {
                $resolution.val(currentResolution);
            } else {
                $resolution.val(config.defaultResolution);
            }

            // Populate duration dropdown
            const $duration = $('#veo_duration');
            const currentDuration = parseInt($duration.val());
            $duration.empty();

            config.durations.forEach(function(dur) {
                $duration.append($('<option>', {
                    value: dur,
                    text: dur + ' seconds'
                }));
            });

            // Restore previous selection if valid, otherwise use default
            if (config.durations.includes(currentDuration)) {
                $duration.val(currentDuration);
            } else {
                $duration.val(config.defaultDuration);
            }

            // Update aspect ratio options based on model
            const $aspectRatio = $('#veo_aspect_ratio');
            const currentAspectRatio = $aspectRatio.val();
            $aspectRatio.find('option').each(function() {
                const value = $(this).val();
                const isSupported = config.aspectRatios.includes(value);
                $(this).prop('disabled', !isSupported);
                if (value === '1:1') {
                    $(this).toggle(config.aspectRatios.includes('1:1'));
                }
            });

            // Reset to 16:9 if current aspect ratio not supported
            if (!config.aspectRatios.includes(currentAspectRatio)) {
                $aspectRatio.val('16:9');
            }

            // Show/hide negative prompt based on provider
            const $negativePromptRow = $('.mxchat-veo-negative-prompt-row');
            if (config.supportsNegativePrompt) {
                $negativePromptRow.show();
            } else {
                $negativePromptRow.hide();
                $('#veo_negative_prompt').val('');
            }

            // Update max prompt length
            const maxLength = config.maxPromptLength || 1024;
            $('.mxchat-veo-char-max').text(maxLength);

            // Update estimated time
            $('.mxchat-veo-status-time').text('Estimated time: ' + config.time);
        },

        togglePasswordVisibility: function(e) {
            e.preventDefault();
            const $input = $(this).siblings('input');
            const $icon = $(this).find('.dashicons');
            
            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        },

        toggleAdvancedOptions: function(e) {
            e.preventDefault();
            const $button = $(this);
            const $options = $('#advanced-options');
            const $icon = $button.find('.dashicons');
            const isVisible = $options.hasClass('active');
            
            if (isVisible) {
                $options.removeClass('active').slideUp(300);
                $icon.removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
            } else {
                $options.addClass('active').slideDown(300);
                $icon.removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
            }
        },

        updateCharacterCount: function() {
            const $textarea = $(this);
            const currentLength = $textarea.val().length;
            const selectedModel = $('#veo_model').val();
            const config = VeoAdmin.modelConfig[selectedModel];
            const maxLength = config ? config.maxPromptLength : 1024;
            const $counter = $textarea.siblings('.mxchat-veo-char-counter');

            if ($counter.length) {
                const $count = $counter.find('.mxchat-veo-char-count');
                $count.text(currentLength);

                $counter.toggleClass('warning', currentLength > maxLength * 0.9);
                $counter.toggleClass('error', currentLength > maxLength);
            }

            $textarea.toggleClass('error', currentLength > maxLength);
        },

        onModelChange: function() {
            VeoAdmin.populateModelOptions();
        },

        updateModelInfo: function(model) {
            // Model info is now handled by populateModelOptions
            // This function is kept for backward compatibility
            const config = VeoAdmin.modelConfig[model];
            if (config) {
                $('.mxchat-veo-status-time').text('Estimated time: ' + config.time);
            }
        },

        // NEW: Image browser functions
        openImageBrowser: function(e) {
            e.preventDefault();
            $('#image-browser-modal').addClass('active').fadeIn(300);
            $('body').addClass('modal-open');
            VeoAdmin.currentPage = 1;
            VeoAdmin.loadImages(1);
        },

        loadImages: function(page = 1, search = '') {
            if (VeoAdmin.isLoading) return;
            
            const $grid = $('#image-grid');
            const $loading = $('#images-loading');
            const $noResults = $('#no-images');
            
            VeoAdmin.isLoading = true;
            $loading.show();
            $grid.empty();
            $noResults.hide();
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_get_media_images',
                    nonce: mxchatVeo.nonce,
                    page: page,
                    search: search
                },
                success: function(response) {
                    VeoAdmin.isLoading = false;
                    $loading.hide();
                    
                    if (response.success && response.data.images.length > 0) {
                        VeoAdmin.renderImages(response.data.images);
                        VeoAdmin.renderPagination(response.data.current_page, response.data.total_pages);
                        VeoAdmin.currentPage = response.data.current_page;
                    } else {
                        $noResults.show();
                    }
                },
                error: function() {
                    VeoAdmin.isLoading = false;
                    $loading.hide();
                    VeoAdmin.showNotice('error', 'Failed to load images');
                }
            });
        },

        renderImages: function(images) {
            const $grid = $('#image-grid');
            
            images.forEach(function(image) {
                const $item = $(`
                    <div class="mxchat-veo-image-item" data-image-id="${image.id}" data-image-url="${image.url}" data-full-url="${image.full_url}">
                        <div class="mxchat-veo-image-thumb">
                            <img src="${image.url}" alt="${image.alt || image.title}" loading="lazy">
                            <div class="mxchat-veo-image-select-overlay">
                                <span class="dashicons dashicons-yes"></span>
                            </div>
                        </div>
                        <div class="mxchat-veo-image-info">
                            <p class="mxchat-veo-image-title">${image.title || 'Untitled'}</p>
                        </div>
                    </div>
                `);
                
                $grid.append($item);
            });
        },

        renderPagination: function(currentPage, totalPages) {
            const $pagination = $('#image-pagination');
            $pagination.empty();
            
            if (totalPages <= 1) return;
            
            // Previous button
            if (currentPage > 1) {
                $pagination.append(`
                    <button class="mxchat-veo-pagination-btn" data-page="${currentPage - 1}">
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                `);
            }
            
            // Page numbers (show current and 2 on each side)
            const start = Math.max(1, currentPage - 2);
            const end = Math.min(totalPages, currentPage + 2);
            
            for (let i = start; i <= end; i++) {
                const isActive = i === currentPage ? 'active' : '';
                $pagination.append(`
                    <button class="mxchat-veo-pagination-btn ${isActive}" data-page="${i}">
                        ${i}
                    </button>
                `);
            }
            
            // Next button
            if (currentPage < totalPages) {
                $pagination.append(`
                    <button class="mxchat-veo-pagination-btn" data-page="${currentPage + 1}">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                `);
            }
        },

        selectImage: function(e) {
            e.preventDefault();
            const $item = $(this);
            const imageId = $item.data('image-id');
            const imageUrl = $item.data('image-url');
            const fullUrl = $item.data('full-url');
            
            // Update the UI
            $('#reference_image_id').val(imageId);
            $('#selected-image').attr('src', fullUrl);
            $('#image-preview').show();
            $('#image-placeholder').hide();
            
            // Close modal
            $('#image-browser-modal').removeClass('active').fadeOut(300);
            $('body').removeClass('modal-open');
            
            VeoAdmin.showNotice('success', 'Reference image selected');
        },

        removeSelectedImage: function(e) {
            e.preventDefault();
            $('#reference_image_id').val('');
            $('#selected-image').attr('src', '');
            $('#image-preview').hide();
            $('#image-placeholder').show();
            
            VeoAdmin.showNotice('success', 'Reference image removed');
        },

        searchImages: function() {
            const search = $('#image-search').val();
            VeoAdmin.loadImages(1, search);
        },

        changePage: function(e) {
            e.preventDefault();
            const page = $(this).data('page');
            const search = $('#image-search').val();
            VeoAdmin.loadImages(page, search);
        },

        generateVideo: function(e) {
            e.preventDefault();

            const $button = $(this);
            const prompt = $('#veo_prompt').val().trim();
            const model = $('#veo_model').val();
            const negativePrompt = $('#veo_negative_prompt').val().trim();
            const aspectRatio = $('#veo_aspect_ratio').val();
            const resolution = $('#veo_resolution').val();
            const duration = $('#veo_duration').val();
            const referenceImageId = $('#reference_image_id').val();

            const config = VeoAdmin.modelConfig[model];
            const maxLength = config ? config.maxPromptLength : 1024;

            if (!prompt) {
                VeoAdmin.showNotice('error', mxchatVeo.strings.promptRequired || 'Please describe your video.');
                $('#veo_prompt').focus();
                return;
            }

            if (prompt.length > maxLength) {
                VeoAdmin.showNotice('error', (mxchatVeo.strings.promptTooLong || 'Description is too long. Maximum {max} characters.').replace('{max}', maxLength));
                $('#veo_prompt').focus();
                return;
            }

            VeoAdmin.startGeneration($button, {
                prompt: prompt,
                model: model,
                negative_prompt: config && config.supportsNegativePrompt ? negativePrompt : '',
                aspect_ratio: aspectRatio,
                resolution: resolution,
                duration_seconds: duration,
                reference_image_id: referenceImageId,
                provider: config ? config.provider : 'google'
            });
        },

        startGeneration: function($button, data) {
            $button.prop('disabled', true).addClass('loading');
            $button.html('<span class="mxchat-veo-spinner"></span> ' + (mxchatVeo.strings.generating || 'Generating...'));
            
            $('#generation-status').fadeIn(300);
            $('#video-result').fadeOut(300);
            
            VeoAdmin.updateProgress(0, mxchatVeo.strings.starting || 'Starting generation...');
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_generate',
                    nonce: mxchatVeo.nonce,
                    ...data
                },
                success: function(response) {
                    if (response.success) {
                        VeoAdmin.currentOperation = response.data.operation_id;
                        VeoAdmin.pollAttempts = 0;
                        VeoAdmin.startPolling();
                        VeoAdmin.updateProgress(10, mxchatVeo.strings.processing || 'Processing...');
                    } else {
                        VeoAdmin.handleGenerationError(response.data || 'Unknown error occurred');
                        VeoAdmin.resetGenerationButton($button);
                    }
                },
                error: function() {
                    VeoAdmin.handleGenerationError(mxchatVeo.strings.networkError || 'Network error occurred');
                    VeoAdmin.resetGenerationButton($button);
                }
            });
        },

        startPolling: function() {
            VeoAdmin.pollInterval = setInterval(VeoAdmin.pollStatus, 5000);
        },

        pollStatus: function() {
            if (!VeoAdmin.currentOperation || VeoAdmin.pollAttempts >= VeoAdmin.maxPollAttempts) {
                VeoAdmin.stopPolling();
                VeoAdmin.handleGenerationError(mxchatVeo.strings.timeout || 'Generation timed out');
                return;
            }
            
            VeoAdmin.pollAttempts++;
            const progress = Math.min(20 + (VeoAdmin.pollAttempts * 2), 90);
            VeoAdmin.updateProgress(progress, mxchatVeo.strings.generating || 'Creating your video...');
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_check_status',
                    nonce: mxchatVeo.nonce,
                    operation_id: VeoAdmin.currentOperation
                },
                success: function(response) {
                    if (response.success) {
                        if (response.data.done) {
                            VeoAdmin.stopPolling();
                            if (response.data.status === 'completed') {
                                VeoAdmin.handleGenerationComplete(response.data);
                            } else {
                                VeoAdmin.handleGenerationError(response.data.error || 'Generation failed');
                            }
                        }
                    } else {
                        //console.error('Polling error:', response.data);
                    }
                },
                error: function() {
                    //console.error('Polling network error');
                }
            });
        },

        stopPolling: function() {
            if (VeoAdmin.pollInterval) {
                clearInterval(VeoAdmin.pollInterval);
                VeoAdmin.pollInterval = null;
            }
        },

        updateProgress: function(percent, message) {
            const $progress = $('.mxchat-veo-progress-bar');
            const $percent = $('.mxchat-veo-progress-percent');
            const $message = $('.mxchat-veo-status-message');
            
            const circumference = 2 * Math.PI * 54;
            const offset = circumference - (percent / 100) * circumference;
            
            $progress.css('stroke-dashoffset', offset);
            $percent.text(percent + '%');
            $message.text(message);
        },

        handleGenerationComplete: function(data) {
            VeoAdmin.updateProgress(100, mxchatVeo.strings.complete || 'Complete!');
            
            setTimeout(function() {
                $('#generation-status').fadeOut(300, function() {
                    VeoAdmin.showVideoResult(data);
                    VeoAdmin.resetGenerationButton($('#generate-video'));
                    // Automatically save the video after generation
                    VeoAdmin.autoSaveVideo();
                });
            }, 1500);
        },

        showVideoResult: function(data) {
                //console.log('Video data received:', data);
                //console.log('Video URL:', data.video_url);
            const $result = $('#video-result');
            const $video = $('#generated-video');
            const $model = $('#video-model');
            const $prompt = $('#video-prompt');
            
            $video.attr('src', data.video_url);
            $model.text($('#veo_model option:selected').text());
            $prompt.text($('#veo_prompt').val());
            
            $result.fadeIn(300);
            
            VeoAdmin.showNotice('success', mxchatVeo.strings.success || 'Video generated successfully!');
        },

        handleGenerationError: function(error) {
            VeoAdmin.stopPolling();
            $('#generation-status').fadeOut(300);
            VeoAdmin.resetGenerationButton($('#generate-video'));
            VeoAdmin.showNotice('error', error);
        },

        resetGenerationButton: function($button) {
            $button.prop('disabled', false).removeClass('loading');
            $button.html('<span class="mxchat-veo-btn-icon">🎬</span> ' + 
                        (mxchatVeo.strings.generate || 'Generate Video'));
        },

        // NEW: Auto-save video after generation
        autoSaveVideo: function() {
            const videoUrl = $('#generated-video').attr('src');
            const prompt = $('#veo_prompt').val().trim();
            const model = $('#veo_model').val();
            
            if (!videoUrl) {
                return;
            }
            
            const $button = $('#save-video');
            $button.prop('disabled', true).addClass('loading');
            $button.html('<span class="mxchat-veo-spinner"></span> ' + 
                        (mxchatVeo.strings.saving || 'Saving...'));
            
            const filename = VeoAdmin.generateFilename(prompt, model);
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_save_video',
                    nonce: mxchatVeo.nonce,
                    video_url: videoUrl,
                    filename: filename,
                    prompt: prompt,
                    model: model
                },
                success: function(response) {
                    $button.prop('disabled', false).removeClass('loading');
                    $button.html('<span class="dashicons dashicons-yes"></span> ' + 
                                (mxchatVeo.strings.saved || 'Saved!'));
                    
                    if (response.success) {
                        VeoAdmin.showNotice('success', mxchatVeo.strings.savedSuccess || 'Video saved to media library automatically!');
                        
                        // Keep "Saved!" text permanently since video is already saved
                        // No need to reset button text
                    } else {
                        VeoAdmin.showNotice('error', response.data || 'Failed to save video');
                        $button.html('<span class="dashicons dashicons-download"></span> ' + 
                                    (mxchatVeo.strings.saveToLibrary || 'Save to Library'));
                    }
                },
                error: function() {
                    $button.prop('disabled', false).removeClass('loading');
                    $button.html('<span class="dashicons dashicons-download"></span> ' + 
                                (mxchatVeo.strings.saveToLibrary || 'Save to Library'));
                    VeoAdmin.showNotice('error', mxchatVeo.strings.networkError || 'Network error occurred while auto-saving');
                }
            });
        },

        saveVideo: function(e) {
            e.preventDefault();
            
            const $button = $(this);
            const videoUrl = $('#generated-video').attr('src');
            const prompt = $('#veo_prompt').val().trim();
            const model = $('#veo_model').val();
            
            if (!videoUrl) {
                VeoAdmin.showNotice('error', mxchatVeo.strings.noVideo || 'No video to save');
                return;
            }
            
            $button.prop('disabled', true).addClass('loading');
            $button.html('<span class="mxchat-veo-spinner"></span> ' + 
                        (mxchatVeo.strings.saving || 'Saving...'));
            
            const filename = VeoAdmin.generateFilename(prompt, model);
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_save_video',
                    nonce: mxchatVeo.nonce,
                    video_url: videoUrl,
                    filename: filename,
                    prompt: prompt,
                    model: model
                },
                success: function(response) {
                    $button.prop('disabled', false).removeClass('loading');
                    $button.html('<span class="dashicons dashicons-yes"></span> ' + 
                                (mxchatVeo.strings.saved || 'Saved!'));
                    
                    if (response.success) {
                        VeoAdmin.showNotice('success', mxchatVeo.strings.savedSuccess || 'Video saved to media library!');
                        
                        setTimeout(function() {
                            $button.html('<span class="dashicons dashicons-download"></span> ' + 
                                        (mxchatVeo.strings.saveToLibrary || 'Save to Library'));
                        }, 2000);
                    } else {
                        VeoAdmin.showNotice('error', response.data || 'Failed to save video');
                        $button.html('<span class="dashicons dashicons-download"></span> ' + 
                                    (mxchatVeo.strings.saveToLibrary || 'Save to Library'));
                    }
                },
                error: function() {
                    $button.prop('disabled', false).removeClass('loading');
                    $button.html('<span class="dashicons dashicons-download"></span> ' + 
                                (mxchatVeo.strings.saveToLibrary || 'Save to Library'));
                    VeoAdmin.showNotice('error', mxchatVeo.strings.networkError || 'Network error occurred');
                }
            });
        },
        
        downloadVideo: function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const $button = $(this);
            const videoUrl = $button.data('video-url');
            const videoId = $button.data('video-id');
            
            // Create a temporary download link
            const link = document.createElement('a');
            link.href = videoUrl;
            link.download = `veo-video-${videoId}.mp4`;
            link.style.display = 'none';
            
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            VeoAdmin.showNotice('success', mxchatVeo.strings.downloadStarted || 'Download started');
        },
        
        deleteVideo: function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const $button = $(this);
            const videoId = $button.data('video-id');
            const $historyItem = $button.closest('.mxchat-veo-history-item');
            
            // Show confirmation dialog
            if (!confirm(mxchatVeo.strings.confirmDelete || 'Are you sure you want to delete this video? This action cannot be undone.')) {
                return;
            }
            
            $button.prop('disabled', true).addClass('loading');
            $button.find('.dashicons').removeClass('dashicons-trash').addClass('dashicons-update');
            
            $.ajax({
                url: mxchatVeo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mxchat_veo_delete_video',
                    nonce: mxchatVeo.nonce,
                    video_id: videoId
                },
                success: function(response) {
                    if (response.success) {
                        $historyItem.fadeOut(300, function() {
                            $historyItem.remove();
                            
                            // Check if history is now empty
                            if ($('.mxchat-veo-history-item').length === 0) {
                                $('.mxchat-veo-history-card').fadeOut(300);
                            }
                        });
                        VeoAdmin.showNotice('success', mxchatVeo.strings.deleteSuccess || 'Video deleted successfully');
                    } else {
                        $button.prop('disabled', false).removeClass('loading');
                        $button.find('.dashicons').removeClass('dashicons-update').addClass('dashicons-trash');
                        VeoAdmin.showNotice('error', response.data || 'Failed to delete video');
                    }
                },
                error: function() {
                    $button.prop('disabled', false).removeClass('loading');
                    $button.find('.dashicons').removeClass('dashicons-update').addClass('dashicons-trash');
                    VeoAdmin.showNotice('error', mxchatVeo.strings.networkError || 'Network error occurred');
                }
            });
        },

        generateFilename: function(prompt, model) {
            const timestamp = new Date().toISOString().slice(0, 19).replace(/:/g, '-');
            const cleanPrompt = prompt.substring(0, 30).replace(/[^a-zA-Z0-9\s]/g, '').replace(/\s+/g, '-').toLowerCase();
            let modelShort = model;

            // Handle different model naming patterns
            if (model.startsWith('veo-')) {
                modelShort = model.replace('veo-', '').replace('.0-generate-preview', '').replace('.0-generate-001', '').replace('-generate-preview', '');
            } else if (model.startsWith('sora-')) {
                modelShort = model; // Keep sora-2 or sora-2-pro as is
            }

            return `${modelShort}-${cleanPrompt}-${timestamp}.mp4`;
        },

        playHistoryVideo: function(e) {
            e.preventDefault();
            const $item = $(this);
            const $video = $item.find('video');
            const videoSrc = $video.attr('src');
            
            if (videoSrc) {
                const $modal = VeoAdmin.createVideoModal(videoSrc);
                $('body').append($modal);
                $modal.fadeIn(300);
            }
        },

        createVideoModal: function(videoSrc) {
            const $modal = $(`
                <div class="mxchat-veo-modal-overlay mxchat-veo-video-modal">
                    <div class="mxchat-veo-modal">
                        <div class="mxchat-veo-modal-header">
                            <button class="mxchat-veo-modal-close">
                                <span class="dashicons dashicons-no-alt"></span>
                            </button>
                        </div>
                        <div class="mxchat-veo-modal-content">
                            <video src="${videoSrc}" controls autoplay></video>
                        </div>
                    </div>
                </div>
            `);
            
            // Close button click
            $modal.on('click', '.mxchat-veo-modal-close', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $modal.fadeOut(300, function() {
                    $modal.remove();
                });
            });
            
            // Overlay click (but not on modal content)
            $modal.on('click', function(e) {
                if (e.target === this) {
                    $modal.fadeOut(300, function() {
                        $modal.remove();
                    });
                }
            });
            
            return $modal;
        },

        showNotice: function(type, message) {
            const $notice = $(`
                <div class="mxchat-veo-notice mxchat-veo-notice-${type}">
                    <div class="mxchat-veo-notice-content">
                        <span class="mxchat-veo-notice-icon">
                            <span class="dashicons ${type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning'}"></span>
                        </span>
                        <span class="mxchat-veo-notice-message">${message}</span>
                        <button type="button" class="mxchat-veo-notice-dismiss">
                            <span class="dashicons dashicons-dismiss"></span>
                        </button>
                    </div>
                </div>
            `);
            
            $('.mxchat-veo-header').after($notice);
            
            $notice.on('click', '.mxchat-veo-notice-dismiss', function() {
                $notice.fadeOut(300, function() {
                    $notice.remove();
                });
            });
            
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $notice.remove();
                });
            }, 5000);
        },

        handleKeyboardShortcuts: function(e) {
            if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
                if ($('#veo_prompt').is(':focus') && !$('#generate-video').prop('disabled')) {
                    $('#generate-video').click();
                }
            }
            
            if (e.key === 'Escape') {
                $('.mxchat-veo-modal-overlay').fadeOut(300, function() {
                    $(this).remove();
                });
                $('body').removeClass('modal-open');
            }
        }
    };

    VeoAdmin.init();
});