<?php

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Veo_Core {
    
    public function __construct() {
        $this->init_ajax_hooks();
    }

    private function init_ajax_hooks() {
        add_action('wp_ajax_mxchat_veo_generate', array($this, 'handle_generate_video'));
        add_action('wp_ajax_mxchat_veo_check_status', array($this, 'handle_check_status'));
        add_action('wp_ajax_mxchat_veo_save_video', array($this, 'handle_save_video'));
        add_action('wp_ajax_mxchat_veo_delete_video', array($this, 'handle_delete_video'));
        add_action('wp_ajax_mxchat_veo_get_media_images', array($this, 'handle_get_media_images'));
    }
    
    /**
     * Determine the provider based on model name
     */
    private function get_provider_from_model($model) {
        if (strpos($model, 'sora-') === 0) {
            return 'openai';
        }
        return 'google';
    }

    public function handle_generate_video() {
        check_ajax_referer('mxchat_veo_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'mxchat-veo'));
        }

        $prompt = sanitize_textarea_field($_POST['prompt']);
        $model = sanitize_text_field($_POST['model']);
        $negative_prompt = sanitize_textarea_field($_POST['negative_prompt'] ?? '');
        $aspect_ratio = sanitize_text_field($_POST['aspect_ratio'] ?? '16:9');
        $resolution = sanitize_text_field($_POST['resolution'] ?? '720p');
        $duration_seconds = intval($_POST['duration_seconds'] ?? 8);
        $reference_image_id = intval($_POST['reference_image_id'] ?? 0);

        if (empty($prompt)) {
            wp_send_json_error(__('Prompt is required', 'mxchat-veo'));
        }

        // Determine provider based on model
        $provider = $this->get_provider_from_model($model);
        $mxchat_options = get_option('mxchat_options', array());

        // Validate API key based on provider
        if ($provider === 'openai') {
            $api_key = $mxchat_options['api_key'] ?? '';
            if (empty($api_key)) {
                wp_send_json_error(__('OpenAI API key not configured. Please add it in MxChat settings.', 'mxchat-veo'));
            }
            $processor = new Sora_Processor();
        } else {
            $api_key = $mxchat_options['gemini_api_key'] ?? '';
            if (empty($api_key)) {
                wp_send_json_error(__('Gemini API key not configured. Please add it in MxChat settings.', 'mxchat-veo'));
            }
            $processor = new Veo_Processor();
        }

        $config = array(
            'negative_prompt' => $negative_prompt,
            'aspect_ratio' => $aspect_ratio,
            'resolution' => $resolution,
            'duration_seconds' => $duration_seconds
        );

        // For Sora, convert resolution and aspect ratio to size format
        if ($provider === 'openai') {
            $config['size'] = $this->get_sora_size($resolution, $aspect_ratio);
        }

        // Handle reference image if provided
        if ($reference_image_id > 0) {
            $image_data = $this->get_image_data($reference_image_id);
            if (is_wp_error($image_data)) {
                wp_send_json_error($image_data->get_error_message());
            }
            $config['image_data'] = $image_data;
            $config['reference_image_id'] = $reference_image_id;
        }

        $result = $processor->generate_video($prompt, $model, $config);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        // Store provider info with operation for status checking
        set_transient('mxchat_veo_op_' . $result['operation_id'], $provider, HOUR_IN_SECONDS);

        wp_send_json_success(array(
            'operation_id' => $result['operation_id'],
            'provider' => $provider,
            'message' => __('Video generation started', 'mxchat-veo')
        ));
    }

    /**
     * Convert resolution and aspect ratio to Sora size format
     */
    private function get_sora_size($resolution, $aspect_ratio) {
        $sizes = array(
            '16:9' => array(
                '480p' => '854x480',
                '720p' => '1280x720',
                '1080p' => '1920x1080'
            ),
            '9:16' => array(
                '480p' => '480x854',
                '720p' => '720x1280',
                '1080p' => '1080x1920'
            ),
            '1:1' => array(
                '480p' => '480x480',
                '720p' => '720x720',
                '1080p' => '1080x1080'
            )
        );

        return $sizes[$aspect_ratio][$resolution] ?? '1280x720';
    }

    public function handle_check_status() {
        check_ajax_referer('mxchat_veo_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'mxchat-veo'));
        }

        $operation_id = sanitize_text_field($_POST['operation_id']);

        if (empty($operation_id)) {
            wp_send_json_error(__('Operation ID is required', 'mxchat-veo'));
        }

        // Get provider from transient or detect from operation ID format
        $provider = get_transient('mxchat_veo_op_' . $operation_id);

        // If no transient, try to detect from operation ID format
        if (!$provider) {
            // Sora video IDs start with "video_", Veo operation IDs contain "operations/"
            if (strpos($operation_id, 'video_') === 0) {
                $provider = 'openai';
            } else {
                $provider = 'google';
            }
        }

        if ($provider === 'openai') {
            $processor = new Sora_Processor();
        } else {
            $processor = new Veo_Processor();
        }

        $result = $processor->check_operation_status($operation_id);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }

    public function handle_save_video() {
        check_ajax_referer('mxchat_veo_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_die(__('Insufficient permissions', 'mxchat-veo'));
        }

        $video_url = sanitize_url($_POST['video_url']);
        $filename = sanitize_file_name($_POST['filename'] ?? 'veo-generated-video.mp4');
        $prompt = sanitize_textarea_field($_POST['prompt'] ?? '');
        $model = sanitize_text_field($_POST['model'] ?? '');

        if (empty($video_url)) {
            wp_send_json_error(__('Video URL is required', 'mxchat-veo'));
        }

        $result = $this->save_video_to_media_library($video_url, $filename, $prompt, $model);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success(array(
            'attachment_id' => $result,
            'message' => __('Video saved to media library', 'mxchat-veo')
        ));
    }
    public function handle_delete_video() {
    check_ajax_referer('mxchat_veo_nonce', 'nonce');

    if (!current_user_can('delete_posts')) {
        wp_die(__('Insufficient permissions', 'mxchat-veo'));
    }

    $video_id = intval($_POST['video_id']);

    if (empty($video_id)) {
        wp_send_json_error(__('Video ID is required', 'mxchat-veo'));
    }

    // Verify this is a Veo-generated video
    $is_veo_video = get_post_meta($video_id, '_mxchat_veo_generated', true);
    if (!$is_veo_video) {
        wp_send_json_error(__('This is not a Veo-generated video', 'mxchat-veo'));
    }

    $result = wp_delete_attachment($video_id, true);

    if ($result === false) {
        wp_send_json_error(__('Failed to delete video', 'mxchat-veo'));
    }

    wp_send_json_success(array(
        'message' => __('Video deleted successfully', 'mxchat-veo')
    ));
}


    private function save_video_to_media_library($video_url, $filename, $prompt = '', $model = '') {
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Determine provider from model or URL
        $provider = $this->get_provider_from_model($model);

        // Download video using the appropriate processor
        if ($provider === 'openai') {
            $processor = new Sora_Processor();
        } else {
            $processor = new Veo_Processor();
        }

        $video_content = $processor->download_video($video_url);

        if (is_wp_error($video_content)) {
            return $video_content;
        }

        // Create temporary file
        $temp_file = wp_tempnam($filename);
        if (!$temp_file) {
            return new WP_Error('temp_file_failed', __('Could not create temporary file', 'mxchat-veo'));
        }

        // Write video content to temp file
        $bytes_written = file_put_contents($temp_file, $video_content);
        if ($bytes_written === false) {
            @unlink($temp_file);
            return new WP_Error('file_write_failed', __('Could not write video file', 'mxchat-veo'));
        }

        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file
        );

        $attachment_id = media_handle_sideload($file_array, 0);

        if (is_wp_error($attachment_id)) {
            @unlink($temp_file);
            return $attachment_id;
        }

        // Add metadata about the video generation
        if (!empty($prompt)) {
            $this->add_video_metadata($attachment_id, $prompt, $model);
        }

        return $attachment_id;
    }

public function get_video_history($limit = 20) {
    global $wpdb;

    $results = $wpdb->get_results($wpdb->prepare("
        SELECT pm.meta_value, p.post_date, p.ID, p.guid
        FROM {$wpdb->postmeta} pm
        INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
        WHERE pm.meta_key = '_mxchat_veo_prompt' 
        AND p.post_type = 'attachment'
        AND p.post_mime_type LIKE 'video%%'
        ORDER BY p.post_date DESC
        LIMIT %d
    ", $limit));

    return $results;
}

public function get_image_data($attachment_id) {
    if (!wp_attachment_is_image($attachment_id)) {
        return new WP_Error('invalid_image', __('Selected file is not a valid image', 'mxchat-veo'));
    }

    $file_path = get_attached_file($attachment_id);
    if (!$file_path || !file_exists($file_path)) {
        return new WP_Error('file_not_found', __('Image file not found', 'mxchat-veo'));
    }

    // Check file size (max 20MB as per Veo docs)
    $file_size = filesize($file_path);
    if ($file_size > 20 * 1024 * 1024) {
        return new WP_Error('file_too_large', __('Image file too large. Maximum size is 20MB', 'mxchat-veo'));
    }

    // Validate image type
    $mime_type = get_post_mime_type($attachment_id);
    $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
    
    if (!in_array($mime_type, $allowed_types)) {
        return new WP_Error('invalid_image_type', __('Unsupported image type. Use JPEG, PNG, GIF, or WebP', 'mxchat-veo'));
    }

    $image_data = file_get_contents($file_path);
    if ($image_data === false) {
        return new WP_Error('read_error', __('Could not read image file', 'mxchat-veo'));
    }

    return $image_data;
}

public function handle_get_media_images() {
    check_ajax_referer('mxchat_veo_nonce', 'nonce');

    if (!current_user_can('upload_files')) {
        wp_die(__('Insufficient permissions', 'mxchat-veo'));
    }

    $search = sanitize_text_field($_POST['search'] ?? '');
    $page = intval($_POST['page'] ?? 1);
    $per_page = 12;

    $args = array(
        'post_type' => 'attachment',
        'post_mime_type' => 'image',
        'post_status' => 'inherit',
        'posts_per_page' => $per_page,
        'paged' => $page,
        'orderby' => 'date',
        'order' => 'DESC'
    );

    if (!empty($search)) {
        $args['s'] = $search;
    }

    $query = new WP_Query($args);
    $images = array();

    foreach ($query->posts as $attachment) {
        $image_url = wp_get_attachment_image_url($attachment->ID, 'medium');
        $full_url = wp_get_attachment_image_url($attachment->ID, 'full');
        
        $images[] = array(
            'id' => $attachment->ID,
            'title' => $attachment->post_title,
            'url' => $image_url,
            'full_url' => $full_url,
            'alt' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true),
            'date' => $attachment->post_date
        );
    }

    wp_send_json_success(array(
        'images' => $images,
        'total_pages' => $query->max_num_pages,
        'current_page' => $page,
        'total_found' => $query->found_posts
    ));
}

    public function add_video_metadata($attachment_id, $prompt, $model, $config = array()) {
        update_post_meta($attachment_id, '_mxchat_veo_prompt', $prompt);
        update_post_meta($attachment_id, '_mxchat_veo_model', $model);
        update_post_meta($attachment_id, '_mxchat_veo_config', $config);
        update_post_meta($attachment_id, '_mxchat_veo_generated', current_time('mysql'));
    }
}
?>