<?php

if (!defined('ABSPATH')) {
    exit;
}

class MxChat_Veo_Settings {

    private $core;

    public function __construct() {
        $this->core = new MxChat_Veo_Core();
    }

    /**
     * Get API key status for all supported providers
     */
    private function get_api_key_statuses() {
        $mxchat_options = get_option('mxchat_options', array());

        return array(
            'gemini' => array(
                'name' => 'Google Gemini',
                'has_key' => !empty($mxchat_options['gemini_api_key'] ?? ''),
                'provider' => 'Veo'
            ),
            'openai' => array(
                'name' => 'OpenAI',
                'has_key' => !empty($mxchat_options['api_key'] ?? ''),
                'provider' => 'Sora'
            )
        );
    }

    public function display_admin_page() {
        $video_history = $this->core->get_video_history(8);
        $api_statuses = $this->get_api_key_statuses();
        $gemini_configured = $api_statuses['gemini']['has_key'];
        $openai_configured = $api_statuses['openai']['has_key'];
        $any_api_configured = $gemini_configured || $openai_configured;
        $mxchat_settings_url = admin_url('admin.php?page=mxchat-settings');
        ?>

        <div class="wrap mxchat-veo-wrap">
            <div class="mxchat-veo-header">
                <div class="mxchat-veo-header-content">
                    <div class="mxchat-veo-header-left">
                        <h1 class="mxchat-veo-gradient-text"><?php _e('Video Generation', 'mxchat-veo'); ?></h1>
                        <p class="mxchat-veo-subtitle"><?php _e('Create stunning videos with AI', 'mxchat-veo'); ?></p>
                    </div>
                    <div class="mxchat-veo-header-right">
                        <!-- API Key Status Indicators -->
                        <div class="mxchat-veo-api-indicators">
                            <?php foreach ($api_statuses as $key => $status): ?>
                            <div class="mxchat-veo-api-indicator <?php echo $status['has_key'] ? 'connected' : 'missing'; ?>" title="<?php echo esc_attr($status['name'] . ' - ' . $status['provider']); ?>">
                                <span class="mxchat-veo-api-dot"></span>
                                <span class="mxchat-veo-api-name"><?php echo esc_html($status['name']); ?></span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="mxchat-veo-container">
                <?php if (!$any_api_configured): ?>
                <!-- API Key Missing Notice -->
                <div class="mxchat-veo-api-notice">
                    <div class="mxchat-veo-api-notice-icon">
                        <span class="dashicons dashicons-warning"></span>
                    </div>
                    <div class="mxchat-veo-api-notice-content">
                        <h3><?php _e('API Keys Required', 'mxchat-veo'); ?></h3>
                        <p><?php _e('To generate videos, please add your API keys in MxChat settings. Google Gemini for Veo models, OpenAI for Sora models.', 'mxchat-veo'); ?></p>
                        <a href="<?php echo esc_url($mxchat_settings_url); ?>" class="mxchat-veo-btn mxchat-veo-btn-primary">
                            <?php _e('Go to MxChat Settings', 'mxchat-veo'); ?>
                        </a>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Video Generator Panel - Always visible -->
                <div class="mxchat-veo-card mxchat-veo-generator-card <?php echo !$any_api_configured ? 'disabled' : ''; ?>">
                    <div class="mxchat-veo-generator-form">
                        <div class="mxchat-veo-form-grid">
                            <div class="mxchat-veo-form-main">
                                <div class="mxchat-veo-form-row">
                                    <label for="veo_prompt" class="mxchat-veo-label">
                                        <?php _e('Describe your video', 'mxchat-veo'); ?>
                                        <span class="mxchat-veo-required">*</span>
                                    </label>
                                    <textarea id="veo_prompt" class="mxchat-veo-textarea"
                                              placeholder="<?php _e('A French bulldog running through a field of flowers...', 'mxchat-veo'); ?>"
                                              rows="4" <?php echo !$any_api_configured ? 'disabled' : ''; ?>></textarea>
                                    <div class="mxchat-veo-char-counter">
                                        <span class="mxchat-veo-char-count">0</span> / <span class="mxchat-veo-char-max">1024</span>
                                    </div>
                                </div>

                                <!-- Reference Image Section -->
                                <div class="mxchat-veo-form-row">
                                    <label class="mxchat-veo-label">
                                        <?php _e('Reference Image (Optional)', 'mxchat-veo'); ?>
                                        <span class="mxchat-veo-help-tooltip" title="<?php _e('Use a reference image to maintain character consistency across videos', 'mxchat-veo'); ?>">?</span>
                                    </label>

                                    <div class="mxchat-veo-image-selector">
                                        <div class="mxchat-veo-image-preview" id="image-preview" style="display: none;">
                                            <img id="selected-image" src="" alt="">
                                            <div class="mxchat-veo-image-overlay">
                                                <button type="button" class="mxchat-veo-image-remove" id="remove-image">
                                                    <span class="dashicons dashicons-no-alt"></span>
                                                </button>
                                            </div>
                                        </div>

                                        <div class="mxchat-veo-image-placeholder" id="image-placeholder">
                                            <div class="mxchat-veo-upload-area">
                                                <span class="mxchat-veo-upload-icon">
                                                    <span class="dashicons dashicons-camera"></span>
                                                </span>
                                                <p class="mxchat-veo-upload-text">
                                                    <?php _e('Select an image for character consistency', 'mxchat-veo'); ?>
                                                </p>
                                                <button type="button" class="mxchat-veo-btn mxchat-veo-btn-secondary" id="select-image" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                                    <?php _e('Choose Image', 'mxchat-veo'); ?>
                                                </button>
                                            </div>
                                        </div>

                                        <input type="hidden" id="reference_image_id" value="">
                                    </div>

                                    <p class="mxchat-veo-help-text">
                                        <?php _e('Select a reference image to animate. This helps maintain character consistency across multiple videos.', 'mxchat-veo'); ?>
                                    </p>
                                </div>
                            </div>

                            <div class="mxchat-veo-form-sidebar">
                                <div class="mxchat-veo-form-row">
                                    <label for="veo_model" class="mxchat-veo-label"><?php _e('Model', 'mxchat-veo'); ?></label>
                                    <select id="veo_model" class="mxchat-veo-select" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                        <optgroup label="<?php _e('Google Veo', 'mxchat-veo'); ?>" <?php echo !$gemini_configured ? 'disabled' : ''; ?>>
                                            <option value="veo-3.1-generate-preview" data-provider="google" <?php echo !$gemini_configured ? 'disabled' : ''; ?>><?php _e('Veo 3.1 (Preview) - 8s with audio', 'mxchat-veo'); ?></option>
                                            <option value="veo-3.1-fast-generate-preview" data-provider="google" <?php echo !$gemini_configured ? 'disabled' : ''; ?>><?php _e('Veo 3.1 Fast (Preview) - Optimized', 'mxchat-veo'); ?></option>
                                            <option value="veo-3.0-generate-001" data-provider="google" <?php echo !$gemini_configured ? 'disabled' : ''; ?>><?php _e('Veo 3 - 8s with audio', 'mxchat-veo'); ?></option>
                                            <option value="veo-3.0-fast-generate-001" data-provider="google" <?php echo !$gemini_configured ? 'disabled' : ''; ?>><?php _e('Veo 3 Fast - Optimized', 'mxchat-veo'); ?></option>
                                            <option value="veo-2.0-generate-001" data-provider="google" <?php echo !$gemini_configured ? 'disabled' : ''; ?>><?php _e('Veo 2 - 5-8s silent', 'mxchat-veo'); ?></option>
                                        </optgroup>
                                        <optgroup label="<?php _e('OpenAI Sora', 'mxchat-veo'); ?>" <?php echo !$openai_configured ? 'disabled' : ''; ?>>
                                            <option value="sora-2" data-provider="openai" <?php echo !$openai_configured ? 'disabled' : ''; ?>><?php _e('Sora 2 - Fast iteration', 'mxchat-veo'); ?></option>
                                            <option value="sora-2-pro" data-provider="openai" <?php echo !$openai_configured ? 'disabled' : ''; ?>><?php _e('Sora 2 Pro - Production quality', 'mxchat-veo'); ?></option>
                                        </optgroup>
                                    </select>
                                </div>

                                <div class="mxchat-veo-advanced-section">
                                    <button type="button" class="mxchat-veo-advanced-toggle" id="toggle-advanced">
                                        <span><?php _e('Advanced Options', 'mxchat-veo'); ?></span>
                                        <span class="mxchat-veo-toggle-icon">
                                            <span class="dashicons dashicons-arrow-down-alt2"></span>
                                        </span>
                                    </button>

                                    <div class="mxchat-veo-advanced-options" id="advanced-options">
                                        <div class="mxchat-veo-form-row mxchat-veo-negative-prompt-row">
                                            <label for="veo_negative_prompt" class="mxchat-veo-label"><?php _e('Negative Prompt', 'mxchat-veo'); ?> <span class="mxchat-veo-provider-badge"><?php _e('Veo only', 'mxchat-veo'); ?></span></label>
                                            <textarea id="veo_negative_prompt" class="mxchat-veo-textarea"
                                                      placeholder="<?php _e('low quality, blurry, distorted...', 'mxchat-veo'); ?>"
                                                      rows="2" <?php echo !$any_api_configured ? 'disabled' : ''; ?>></textarea>
                                        </div>

                                        <div class="mxchat-veo-form-row">
                                            <label for="veo_aspect_ratio" class="mxchat-veo-label"><?php _e('Aspect Ratio', 'mxchat-veo'); ?></label>
                                            <select id="veo_aspect_ratio" class="mxchat-veo-select" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                                <option value="16:9"><?php _e('16:9 Widescreen', 'mxchat-veo'); ?></option>
                                                <option value="9:16"><?php _e('9:16 Portrait', 'mxchat-veo'); ?></option>
                                            </select>
                                        </div>

                                        <div class="mxchat-veo-form-row">
                                            <label for="veo_resolution" class="mxchat-veo-label"><?php _e('Resolution', 'mxchat-veo'); ?></label>
                                            <select id="veo_resolution" class="mxchat-veo-select" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                                <!-- Options populated dynamically by JavaScript based on model -->
                                            </select>
                                        </div>

                                        <div class="mxchat-veo-form-row">
                                            <label for="veo_duration" class="mxchat-veo-label"><?php _e('Duration', 'mxchat-veo'); ?></label>
                                            <select id="veo_duration" class="mxchat-veo-select" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                                <!-- Options populated dynamically by JavaScript based on model -->
                                            </select>
                                        </div>
                                    </div>
                                </div>

                                <div class="mxchat-veo-form-actions">
                                    <button type="button" id="generate-video" class="mxchat-veo-btn mxchat-veo-btn-primary mxchat-veo-btn-lg" <?php echo !$any_api_configured ? 'disabled' : ''; ?>>
                                        <span class="mxchat-veo-btn-icon">🎬</span>
                                        <?php _e('Generate Video', 'mxchat-veo'); ?>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Generation Status Panel -->
                <div class="mxchat-veo-card mxchat-veo-status-card" id="generation-status" style="display: none;">
                    <div class="mxchat-veo-status-content">
                        <div class="mxchat-veo-progress-container">
                            <div class="mxchat-veo-progress-ring">
                                <svg class="mxchat-veo-progress-svg" viewBox="0 0 120 120">
                                    <defs>
                                        <linearGradient id="progress-gradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                            <stop offset="0%" style="stop-color:#fa73e6" />
                                            <stop offset="100%" style="stop-color:#7873f5" />
                                        </linearGradient>
                                    </defs>
                                    <circle class="mxchat-veo-progress-bg" cx="60" cy="60" r="54"></circle>
                                    <circle class="mxchat-veo-progress-bar" cx="60" cy="60" r="54" stroke="url(#progress-gradient)"></circle>
                                </svg>
                                <div class="mxchat-veo-progress-text">
                                    <span class="mxchat-veo-progress-percent">0%</span>
                                </div>
                            </div>
                        </div>

                        <div class="mxchat-veo-status-info">
                            <h3 class="mxchat-veo-status-title"><?php _e('Generating Video', 'mxchat-veo'); ?></h3>
                            <p class="mxchat-veo-status-message"><?php _e('Initializing...', 'mxchat-veo'); ?></p>
                            <p class="mxchat-veo-status-time"><?php _e('Estimated time: 11s - 6min', 'mxchat-veo'); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Video Result Panel -->
                <div class="mxchat-veo-card mxchat-veo-result-card" id="video-result" style="display: none;">
                    <div class="mxchat-veo-card-header">
                        <h2><?php _e('Your Video', 'mxchat-veo'); ?></h2>
                        <div class="mxchat-veo-result-actions">
                            <button type="button" class="mxchat-veo-btn mxchat-veo-btn-secondary" id="save-video">
                                <span class="dashicons dashicons-download"></span>
                                <?php _e('Save to Library', 'mxchat-veo'); ?>
                            </button>
                        </div>
                    </div>

                    <div class="mxchat-veo-video-container">
                        <video id="generated-video" class="mxchat-veo-video" controls>
                            <?php _e('Your browser does not support the video tag.', 'mxchat-veo'); ?>
                        </video>

                        <div class="mxchat-veo-video-info">
                            <div class="mxchat-veo-video-meta">
                                <span class="mxchat-veo-meta-label"><?php _e('Model:', 'mxchat-veo'); ?></span>
                                <span class="mxchat-veo-meta-value" id="video-model"></span>
                            </div>
                            <div class="mxchat-veo-video-meta">
                                <span class="mxchat-veo-meta-label"><?php _e('Prompt:', 'mxchat-veo'); ?></span>
                                <span class="mxchat-veo-meta-value" id="video-prompt"></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Video History Panel -->
                <?php if (!empty($video_history)): ?>
                <div class="mxchat-veo-card mxchat-veo-history-card">
                    <div class="mxchat-veo-card-header">
                        <h2><?php _e('Recent Videos', 'mxchat-veo'); ?></h2>
                        <a href="<?php echo admin_url('upload.php?post_mime_type=video'); ?>" class="mxchat-veo-btn mxchat-veo-btn-text">
                            <?php _e('View All', 'mxchat-veo'); ?>
                        </a>
                    </div>

                    <div class="mxchat-veo-history-grid">
<?php foreach ($video_history as $video): ?>
<div class="mxchat-veo-history-item">
    <div class="mxchat-veo-history-thumbnail">
        <video src="<?php echo esc_url($video->guid); ?>" muted></video>
        <div class="mxchat-veo-play-overlay">
            <span class="dashicons dashicons-controls-play"></span>
        </div>
    </div>
    <div class="mxchat-veo-history-meta">
        <p class="mxchat-veo-history-prompt" title="<?php echo esc_attr($video->meta_value); ?>">
            <?php echo esc_html(wp_trim_words($video->meta_value, 6)); ?>
        </p>
        <span class="mxchat-veo-history-date">
            <?php echo human_time_diff(strtotime($video->post_date), current_time('timestamp')) . ' ' . __('ago', 'mxchat-veo'); ?>
        </span>
        <div class="mxchat-veo-history-actions">
            <button type="button" class="mxchat-veo-history-download" data-video-id="<?php echo $video->ID; ?>" data-video-url="<?php echo esc_url($video->guid); ?>" title="<?php _e('Download', 'mxchat-veo'); ?>">
                <span class="dashicons dashicons-download"></span>
            </button>
            <button type="button" class="mxchat-veo-history-delete" data-video-id="<?php echo $video->ID; ?>" title="<?php _e('Delete', 'mxchat-veo'); ?>">
                <span class="dashicons dashicons-trash"></span>
            </button>
        </div>
    </div>
</div>
<?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Image Browser Modal -->
<div class="mxchat-veo-modal-overlay" id="image-browser-modal">
    <div class="mxchat-veo-modal mxchat-veo-modal-large">
        <div class="mxchat-veo-modal-header">
            <h2><?php _e('Select Reference Image', 'mxchat-veo'); ?></h2>
            <div class="mxchat-veo-modal-controls">
                <div class="mxchat-veo-search-box">
                    <input type="text" id="image-search" placeholder="<?php _e('Search images...', 'mxchat-veo'); ?>" class="mxchat-veo-input">
                    <span class="dashicons dashicons-search"></span>
                </div>
                <button type="button" class="mxchat-veo-modal-close">
                    <span class="dashicons dashicons-no-alt"></span>
                </button>
            </div>
        </div>

        <div class="mxchat-veo-modal-content">
            <div class="mxchat-veo-image-grid" id="image-grid">
                <!-- Images will be loaded here via AJAX -->
            </div>

            <div class="mxchat-veo-loading" id="images-loading" style="display: none;">
                <span class="mxchat-veo-spinner"></span>
                <p><?php _e('Loading images...', 'mxchat-veo'); ?></p>
            </div>

            <div class="mxchat-veo-no-results" id="no-images" style="display: none;">
                <p><?php _e('No images found. Try uploading some images to your media library first.', 'mxchat-veo'); ?></p>
                <a href="<?php echo admin_url('media-new.php'); ?>" class="mxchat-veo-btn mxchat-veo-btn-primary">
                    <?php _e('Upload Images', 'mxchat-veo'); ?>
                </a>
            </div>
        </div>

        <div class="mxchat-veo-modal-footer">
            <div class="mxchat-veo-pagination" id="image-pagination">
                <!-- Pagination will be added here -->
            </div>
            <div class="mxchat-veo-modal-actions">
                <button type="button" class="mxchat-veo-btn mxchat-veo-btn-secondary mxchat-veo-modal-close">
                    <?php _e('Cancel', 'mxchat-veo'); ?>
                </button>
            </div>
        </div>
    </div>
</div>


        </div>

        <?php
    }
}
?>
