<?php

if (!defined('ABSPATH')) {
    exit;
}

class Sora_Processor {

    private $api_key;
    private $base_url = 'https://api.openai.com/v1';

    public function __construct() {
        // Get OpenAI API key from core MxChat plugin
        $mxchat_options = get_option('mxchat_options', array());
        $this->api_key = $mxchat_options['api_key'] ?? '';
    }

    /**
     * Get the OpenAI API key status
     */
    public static function get_api_key_status() {
        $mxchat_options = get_option('mxchat_options', array());
        $openai_key = $mxchat_options['api_key'] ?? '';
        return !empty($openai_key);
    }

    public function generate_video($prompt, $model, $config = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        // Check if we have image data for image-to-video generation
        if (!empty($config['image_data'])) {
            return $this->generate_with_image($prompt, $model, $config['image_data'], $config);
        }

        // Build the request body
        $body = array(
            'model' => $model,
            'prompt' => $prompt
        );

        // Add size (resolution) - Sora uses WxH format
        if (!empty($config['size'])) {
            $body['size'] = $config['size'];
        } elseif (!empty($config['resolution']) && !empty($config['aspect_ratio'])) {
            $body['size'] = $this->get_size_from_resolution($config['resolution'], $config['aspect_ratio']);
        } else {
            $body['size'] = '1280x720'; // Default HD 16:9
        }

        // Add seconds parameter (as string per API docs)
        if (!empty($config['duration_seconds'])) {
            $body['seconds'] = (string) $config['duration_seconds'];
        } else {
            $body['seconds'] = '8'; // Default 8 seconds
        }

        $url = $this->base_url . '/videos';

        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($body)
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 201) {
            $error_message = 'HTTP ' . $status_code . ': ';
            if (isset($data['error']['message'])) {
                $error_message .= $data['error']['message'];
            } elseif (isset($data['error'])) {
                $error_message .= is_string($data['error']) ? $data['error'] : json_encode($data['error']);
            } else {
                $error_message .= 'Unknown API error - ' . $response_body;
            }
            return new WP_Error('api_error', $error_message);
        }

        if (!isset($data['id'])) {
            return new WP_Error('invalid_response', __('Invalid API response - no video ID', 'mxchat-veo'));
        }

        return array(
            'operation_id' => $data['id'],
            'status' => $data['status'] ?? 'queued',
            'progress' => $data['progress'] ?? 0,
            'provider' => 'sora'
        );
    }

    public function check_operation_status($video_id) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        $url = $this->base_url . '/videos/' . $video_id;

        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            $error_message = isset($data['error']['message']) ? $data['error']['message'] : __('Unknown API error', 'mxchat-veo');
            return new WP_Error('api_error', $error_message);
        }

        $result = array(
            'done' => ($data['status'] === 'completed'),
            'operation_id' => $video_id,
            'progress' => $data['progress'] ?? 0,
            'provider' => 'sora'
        );

        if ($data['status'] === 'completed') {
            $result['status'] = 'completed';
            // Video URL will be fetched via download endpoint
            $result['video_url'] = $this->base_url . '/videos/' . $video_id . '/content';
            $result['file_uri'] = $result['video_url'];
        } elseif ($data['status'] === 'failed') {
            $error_msg = isset($data['error']['message']) ? $data['error']['message'] : __('Video generation failed', 'mxchat-veo');
            return new WP_Error('generation_failed', $error_msg);
        } else {
            $result['status'] = $data['status']; // queued, in_progress
        }

        return $result;
    }

    public function download_video($video_id) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        // If it's a full URL, extract the video ID
        if (strpos($video_id, '/videos/') !== false) {
            preg_match('/videos\/([^\/]+)/', $video_id, $matches);
            if (!empty($matches[1])) {
                $video_id = str_replace('/content', '', $matches[1]);
            }
        }

        $url = $this->base_url . '/videos/' . $video_id . '/content';

        $response = wp_remote_get($url, array(
            'timeout' => 120,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        // Handle redirect
        if ($status_code === 302 || $status_code === 301) {
            $redirect_url = wp_remote_retrieve_header($response, 'location');
            if (!empty($redirect_url)) {
                $response = wp_remote_get($redirect_url, array(
                    'timeout' => 120
                ));

                if (is_wp_error($response)) {
                    return $response;
                }
                $status_code = wp_remote_retrieve_response_code($response);
            }
        }

        if ($status_code !== 200) {
            return new WP_Error('download_failed', __('Failed to download video', 'mxchat-veo'));
        }

        return wp_remote_retrieve_body($response);
    }

    public function generate_with_image($prompt, $model, $image_data, $config = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        // Sora uses multipart/form-data for image uploads
        $boundary = wp_generate_password(24, false);

        $body = '';

        // Add prompt
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"prompt\"\r\n\r\n";
        $body .= $prompt . "\r\n";

        // Add model
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"model\"\r\n\r\n";
        $body .= $model . "\r\n";

        // Add size
        $size = '1280x720';
        if (!empty($config['size'])) {
            $size = $config['size'];
        } elseif (!empty($config['resolution']) && !empty($config['aspect_ratio'])) {
            $size = $this->get_size_from_resolution($config['resolution'], $config['aspect_ratio']);
        }
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"size\"\r\n\r\n";
        $body .= $size . "\r\n";

        // Add seconds parameter (as string per API docs)
        $seconds = !empty($config['duration_seconds']) ? (string) $config['duration_seconds'] : '8';
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"seconds\"\r\n\r\n";
        $body .= $seconds . "\r\n";

        // Add input_reference image
        $mime_type = $this->detect_mime_type($image_data);
        $extension = $mime_type === 'image/png' ? 'png' : ($mime_type === 'image/webp' ? 'webp' : 'jpg');
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"input_reference\"; filename=\"reference.{$extension}\"\r\n";
        $body .= "Content-Type: {$mime_type}\r\n\r\n";
        $body .= $image_data . "\r\n";

        $body .= "--{$boundary}--\r\n";

        $url = $this->base_url . '/videos';

        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary
            ),
            'body' => $body
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 201) {
            $error_message = 'HTTP ' . $status_code . ': ';
            if (isset($data['error']['message'])) {
                $error_message .= $data['error']['message'];
            } elseif (isset($data['error'])) {
                $error_message .= is_string($data['error']) ? $data['error'] : json_encode($data['error']);
            } else {
                $error_message .= 'Unknown API error - ' . $response_body;
            }
            return new WP_Error('api_error', $error_message);
        }

        if (!isset($data['id'])) {
            return new WP_Error('invalid_response', __('Invalid API response - no video ID', 'mxchat-veo'));
        }

        return array(
            'operation_id' => $data['id'],
            'status' => $data['status'] ?? 'queued',
            'progress' => $data['progress'] ?? 0,
            'provider' => 'sora'
        );
    }

    private function get_size_from_resolution($resolution, $aspect_ratio) {
        // Sora supported sizes based on aspect ratio
        $sizes = array(
            '16:9' => array(
                '480p' => '854x480',
                '720p' => '1280x720',
                '1080p' => '1920x1080'
            ),
            '9:16' => array(
                '480p' => '480x854',
                '720p' => '720x1280',
                '1080p' => '1080x1920'
            ),
            '1:1' => array(
                '480p' => '480x480',
                '720p' => '720x720',
                '1080p' => '1080x1080'
            )
        );

        if (isset($sizes[$aspect_ratio][$resolution])) {
            return $sizes[$aspect_ratio][$resolution];
        }

        // Default to 720p 16:9
        return '1280x720';
    }

    private function detect_mime_type($image_data) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime_type = $finfo->buffer($image_data);

        $allowed_types = array('image/jpeg', 'image/png', 'image/webp');

        if (in_array($mime_type, $allowed_types)) {
            return $mime_type;
        }

        return 'image/jpeg';
    }

    public function validate_prompt($prompt, $model) {
        $max_length = 2048; // Sora allows longer prompts

        if (strlen($prompt) > $max_length) {
            return new WP_Error('prompt_too_long', sprintf(__('Prompt exceeds maximum length of %d characters', 'mxchat-veo'), $max_length));
        }

        if (empty(trim($prompt))) {
            return new WP_Error('empty_prompt', __('Prompt cannot be empty', 'mxchat-veo'));
        }

        return true;
    }

    public function get_supported_sizes($model) {
        // Sora 2 and Sora 2 Pro support multiple sizes
        return array(
            '1920x1080',
            '1080x1920',
            '1280x720',
            '720x1280',
            '1080x1080'
        );
    }

    public function get_supported_durations($model) {
        // Sora API only supports '4', '8', and '12' seconds (passed as string)
        return array(4, 8, 12);
    }

    public function estimate_generation_time($model) {
        if ($model === 'sora-2') {
            return array('min' => 60, 'max' => 300); // 1-5 minutes
        } elseif ($model === 'sora-2-pro') {
            return array('min' => 120, 'max' => 600); // 2-10 minutes
        }

        return array('min' => 60, 'max' => 300);
    }

    /**
     * Remix a completed video with modifications
     */
    public function remix_video($video_id, $prompt, $config = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        $body = array(
            'prompt' => $prompt
        );

        $url = $this->base_url . '/videos/' . $video_id . '/remix';

        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($body)
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 201) {
            $error_message = 'HTTP ' . $status_code . ': ';
            if (isset($data['error']['message'])) {
                $error_message .= $data['error']['message'];
            } else {
                $error_message .= 'Unknown API error';
            }
            return new WP_Error('api_error', $error_message);
        }

        return array(
            'operation_id' => $data['id'],
            'status' => $data['status'] ?? 'queued',
            'progress' => $data['progress'] ?? 0,
            'provider' => 'sora'
        );
    }

    /**
     * List all videos
     */
    public function list_videos($limit = 20, $after = null) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        $url = $this->base_url . '/videos?limit=' . $limit;
        if (!empty($after)) {
            $url .= '&after=' . urlencode($after);
        }

        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }

    /**
     * Delete a video from OpenAI storage
     */
    public function delete_video($video_id) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key not configured', 'mxchat-veo'));
        }

        $url = $this->base_url . '/videos/' . $video_id;

        $response = wp_remote_request($url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 204) {
            return new WP_Error('delete_failed', __('Failed to delete video from OpenAI', 'mxchat-veo'));
        }

        return true;
    }
}
?>
