<?php

if (!defined('ABSPATH')) {
    exit;
}

class Veo_Models {
    
    private $models;

    public function __construct() {
        $this->init_models();
    }

    private function init_models() {
        $this->models = array(
            'veo-3.1-generate-preview' => array(
                'name' => 'Veo 3.1',
                'display_name' => __('Veo 3.1 (Preview)', 'mxchat-veo'),
                'description' => __('Latest model with enhanced video and audio generation, video extension, and reference images', 'mxchat-veo'),
                'version' => 'preview',
                'status' => 'preview',
                'provider' => 'google',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => true,
                    'sound_effects' => true,
                    'video_extension' => true,
                    'reference_images' => true,
                    'interpolation' => true
                ),
                'specifications' => array(
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '720p',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 6, 8),
                    'default_duration' => 8,
                    'max_prompt_length' => 1024,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16'),
                    'person_generation' => array(
                        'text_to_video' => 'allow_all',
                        'image_to_video' => 'allow_adult',
                        'extension' => 'allow_all',
                        'interpolation' => 'allow_adult',
                        'reference_images' => 'allow_adult'
                    ),
                    'max_reference_images' => 3,
                    'max_extension_length' => 148
                ),
                'performance' => array(
                    'min_time' => 11,
                    'max_time' => 360,
                    'avg_time' => 120
                ),
                'pricing_tier' => 'premium',
                'last_updated' => '2025-09-01'
            ),
            'veo-3.1-fast-generate-preview' => array(
                'name' => 'Veo 3.1 Fast',
                'display_name' => __('Veo 3.1 Fast (Preview)', 'mxchat-veo'),
                'description' => __('Fast generation with high quality, optimized for speed and business use', 'mxchat-veo'),
                'version' => 'preview',
                'status' => 'preview',
                'provider' => 'google',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => true,
                    'sound_effects' => true,
                    'fast_generation' => true,
                    'video_extension' => true,
                    'reference_images' => true,
                    'interpolation' => true
                ),
                'specifications' => array(
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '720p',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 6, 8),
                    'default_duration' => 8,
                    'max_prompt_length' => 1024,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16'),
                    'person_generation' => array(
                        'text_to_video' => 'allow_all',
                        'image_to_video' => 'allow_adult',
                        'extension' => 'allow_all',
                        'interpolation' => 'allow_adult',
                        'reference_images' => 'allow_adult'
                    ),
                    'max_reference_images' => 3,
                    'max_extension_length' => 148
                ),
                'performance' => array(
                    'min_time' => 8,
                    'max_time' => 180,
                    'avg_time' => 60
                ),
                'pricing_tier' => 'premium',
                'last_updated' => '2025-09-01'
            ),
            'veo-3.0-generate-001' => array(
                'name' => 'Veo 3',
                'display_name' => __('Veo 3', 'mxchat-veo'),
                'description' => __('720p/1080p videos with native audio generation', 'mxchat-veo'),
                'version' => 'stable',
                'status' => 'stable',
                'provider' => 'google',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => true,
                    'sound_effects' => true
                ),
                'specifications' => array(
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '720p',
                    'resolution_1080p_aspect' => '16:9',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 6, 8),
                    'default_duration' => 8,
                    'max_prompt_length' => 1024,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16'),
                    'person_generation' => array(
                        'text_to_video' => 'allow_all',
                        'image_to_video' => 'allow_adult'
                    )
                ),
                'performance' => array(
                    'min_time' => 11,
                    'max_time' => 360,
                    'avg_time' => 120
                ),
                'pricing_tier' => 'premium',
                'last_updated' => '2025-07-01'
            ),
            'veo-3.0-fast-generate-001' => array(
                'name' => 'Veo 3 Fast',
                'display_name' => __('Veo 3 Fast', 'mxchat-veo'),
                'description' => __('High-quality videos optimized for speed and business use', 'mxchat-veo'),
                'version' => 'stable',
                'status' => 'stable',
                'provider' => 'google',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => true,
                    'sound_effects' => true,
                    'fast_generation' => true
                ),
                'specifications' => array(
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '720p',
                    'resolution_1080p_aspect' => '16:9',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 6, 8),
                    'default_duration' => 8,
                    'max_prompt_length' => 1024,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16'),
                    'person_generation' => array(
                        'text_to_video' => 'allow_all',
                        'image_to_video' => 'allow_adult'
                    )
                ),
                'performance' => array(
                    'min_time' => 8,
                    'max_time' => 180,
                    'avg_time' => 60
                ),
                'pricing_tier' => 'premium',
                'last_updated' => '2025-07-01'
            ),
            'veo-2.0-generate-001' => array(
                'name' => 'Veo 2',
                'display_name' => __('Veo 2', 'mxchat-veo'),
                'description' => __('5-8 second silent videos with stable performance', 'mxchat-veo'),
                'version' => 'stable',
                'status' => 'stable',
                'provider' => 'google',
                'features' => array(
                    'audio' => false,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => false,
                    'sound_effects' => false
                ),
                'specifications' => array(
                    'resolutions' => array('720p'),
                    'default_resolution' => '720p',
                    'frame_rate' => '24fps',
                    'durations' => array(5, 6, 8),
                    'default_duration' => 8,
                    'max_prompt_length' => 1024,
                    'videos_per_request' => 2,
                    'aspect_ratios' => array('16:9', '9:16'),
                    'person_generation' => array(
                        'text_to_video' => array('allow_all', 'allow_adult', 'dont_allow'),
                        'image_to_video' => array('allow_adult', 'dont_allow')
                    )
                ),
                'performance' => array(
                    'min_time' => 15,
                    'max_time' => 300,
                    'avg_time' => 90
                ),
                'pricing_tier' => 'standard',
                'last_updated' => '2025-04-01'
            ),
            // OpenAI Sora Models - seconds parameter as string ('4', '8', or '12')
            'sora-2' => array(
                'name' => 'Sora 2',
                'display_name' => __('Sora 2', 'mxchat-veo'),
                'description' => __('Fast and flexible video generation for rapid iteration and exploration', 'mxchat-veo'),
                'version' => 'stable',
                'status' => 'stable',
                'provider' => 'openai',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => false,
                    'sound_effects' => true,
                    'fast_generation' => true,
                    'remix' => true
                ),
                'specifications' => array(
                    'sizes' => array('1920x1080', '1080x1920', '1280x720', '720x1280'),
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '720p',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 8, 12),
                    'default_duration' => 8,
                    'max_prompt_length' => 1000,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16')
                ),
                'performance' => array(
                    'min_time' => 60,
                    'max_time' => 300,
                    'avg_time' => 120
                ),
                'pricing_tier' => 'standard',
                'last_updated' => '2025-11-01'
            ),
            'sora-2-pro' => array(
                'name' => 'Sora 2 Pro',
                'display_name' => __('Sora 2 Pro', 'mxchat-veo'),
                'description' => __('Production-quality output with higher fidelity for cinematic footage and marketing assets', 'mxchat-veo'),
                'version' => 'stable',
                'status' => 'stable',
                'provider' => 'openai',
                'features' => array(
                    'audio' => true,
                    'text_to_video' => true,
                    'image_to_video' => true,
                    'dialogue' => false,
                    'sound_effects' => true,
                    'high_quality' => true,
                    'remix' => true
                ),
                'specifications' => array(
                    'sizes' => array('1920x1080', '1080x1920', '1280x720', '720x1280'),
                    'resolutions' => array('720p', '1080p'),
                    'default_resolution' => '1080p',
                    'frame_rate' => '24fps',
                    'durations' => array(4, 8, 12),
                    'default_duration' => 8,
                    'max_prompt_length' => 1000,
                    'videos_per_request' => 1,
                    'aspect_ratios' => array('16:9', '9:16')
                ),
                'performance' => array(
                    'min_time' => 120,
                    'max_time' => 600,
                    'avg_time' => 300
                ),
                'pricing_tier' => 'premium',
                'last_updated' => '2025-11-01'
            )
        );
    }

    public function get_all_models() {
        return $this->models;
    }

    public function get_model($model_id) {
        return isset($this->models[$model_id]) ? $this->models[$model_id] : null;
    }

    public function get_available_models() {
        return array_filter($this->models, function($model) {
            return in_array($model['status'], array('stable', 'preview'));
        });
    }

    public function get_stable_models() {
        return array_filter($this->models, function($model) {
            return $model['status'] === 'stable';
        });
    }

    public function get_preview_models() {
        return array_filter($this->models, function($model) {
            return $model['status'] === 'preview';
        });
    }

    public function get_models_with_audio() {
        return array_filter($this->models, function($model) {
            return $model['features']['audio'] === true;
        });
    }

    public function get_models_with_portrait() {
        return array_filter($this->models, function($model) {
            return in_array('9:16', $model['specifications']['aspect_ratios']);
        });
    }

    public function get_models_by_provider($provider) {
        return array_filter($this->models, function($model) use ($provider) {
            return isset($model['provider']) && $model['provider'] === $provider;
        });
    }

    public function get_google_models() {
        return $this->get_models_by_provider('google');
    }

    public function get_openai_models() {
        return $this->get_models_by_provider('openai');
    }

    public function get_model_provider($model_id) {
        $model = $this->get_model($model_id);
        return $model ? ($model['provider'] ?? 'google') : 'google';
    }

    public function model_supports_feature($model_id, $feature) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return false;
        }

        return isset($model['features'][$feature]) && $model['features'][$feature] === true;
    }

    public function get_supported_aspect_ratios($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array();
        }

        return $model['specifications']['aspect_ratios'];
    }

    public function get_person_generation_options($model_id, $mode = 'text_to_video') {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array();
        }

        $person_gen = $model['specifications']['person_generation'][$mode] ?? 'allow_all';
        
        if (is_array($person_gen)) {
            return $person_gen;
        }
        
        return array($person_gen);
    }

    public function get_default_person_generation($model_id, $mode = 'text_to_video') {
        $options = $this->get_person_generation_options($model_id, $mode);
        return !empty($options) ? $options[0] : 'allow_all';
    }

    public function get_model_capabilities($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array();
        }

        $durations = $model['specifications']['durations'] ?? array(8);

        return array(
            'has_audio' => $model['features']['audio'],
            'supports_text_to_video' => $model['features']['text_to_video'],
            'supports_image_to_video' => $model['features']['image_to_video'],
            'supports_dialogue' => $model['features']['dialogue'] ?? false,
            'supports_sound_effects' => $model['features']['sound_effects'] ?? false,
            'supports_video_extension' => $model['features']['video_extension'] ?? false,
            'supports_reference_images' => $model['features']['reference_images'] ?? false,
            'supports_interpolation' => $model['features']['interpolation'] ?? false,
            'max_duration' => max($durations),
            'min_duration' => min($durations),
            'durations' => $durations,
            'resolutions' => $model['specifications']['resolutions'] ?? array('720p'),
            'default_resolution' => $model['specifications']['default_resolution'] ?? '720p',
            'frame_rate' => $model['specifications']['frame_rate'],
            'aspect_ratios' => $model['specifications']['aspect_ratios'],
            'max_videos_per_request' => $model['specifications']['videos_per_request'],
            'max_reference_images' => $model['specifications']['max_reference_images'] ?? 0
        );
    }

    public function get_supported_resolutions($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array('720p');
        }

        return $model['specifications']['resolutions'] ?? array('720p');
    }

    public function get_supported_durations($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array(8);
        }

        return $model['specifications']['durations'] ?? array(8);
    }

    public function get_generation_time_estimate($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array('min' => 30, 'max' => 300, 'avg' => 120);
        }

        return $model['performance'];
    }

    public function format_time_estimate($seconds) {
        if ($seconds < 60) {
            return sprintf(__('%ds', 'mxchat-veo'), $seconds);
        } elseif ($seconds < 3600) {
            return sprintf(__('%dm', 'mxchat-veo'), ceil($seconds / 60));
        } else {
            return sprintf(__('%dh %dm', 'mxchat-veo'), floor($seconds / 3600), ceil(($seconds % 3600) / 60));
        }
    }

    public function get_model_status_badge($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return '';
        }

        $status = $model['status'];
        $class = 'mxchat-veo-badge mxchat-veo-badge-' . $status;
        $text = ucfirst($status);

        return sprintf('<span class="%s">%s</span>', $class, $text);
    }

    public function get_feature_icons($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array();
        }

        $icons = array();
        
        if ($model['features']['audio']) {
            $icons[] = array(
                'icon' => 'volume-up',
                'title' => __('Audio Generation', 'mxchat-veo'),
                'class' => 'feature-audio'
            );
        }

        if ($model['features']['dialogue'] ?? false) {
            $icons[] = array(
                'icon' => 'format-quote',
                'title' => __('Dialogue Support', 'mxchat-veo'),
                'class' => 'feature-dialogue'
            );
        }

        if ($model['features']['sound_effects'] ?? false) {
            $icons[] = array(
                'icon' => 'controls-play',
                'title' => __('Sound Effects', 'mxchat-veo'),
                'class' => 'feature-sfx'
            );
        }

        if ($model['features']['fast_generation'] ?? false) {
            $icons[] = array(
                'icon' => 'performance',
                'title' => __('Fast Generation', 'mxchat-veo'),
                'class' => 'feature-fast'
            );
        }

        if (count($model['specifications']['aspect_ratios']) > 1) {
            $icons[] = array(
                'icon' => 'screenoptions',
                'title' => __('Multiple Aspect Ratios', 'mxchat-veo'),
                'class' => 'feature-aspect'
            );
        }

        return $icons;
    }

    public function get_models_for_select() {
        $models = $this->get_available_models();
        $options = array();

        foreach ($models as $id => $model) {
            $label = $model['display_name'];
            
            if ($model['features']['audio']) {
                $duration = $model['specifications']['duration'] ?? '5-8';
                $label .= sprintf(' - %ss %s', $duration, __('with audio', 'mxchat-veo'));
            } else {
                $duration_min = $model['specifications']['duration_min'] ?? 5;
                $duration_max = $model['specifications']['duration_max'] ?? 8;
                $label .= sprintf(' - %s-%ss %s', $duration_min, $duration_max, __('silent', 'mxchat-veo'));
            }

            $options[$id] = $label;
        }

        return $options;
    }

    public function validate_model_config($model_id, $config) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return new WP_Error('invalid_model', __('Invalid model selected', 'mxchat-veo'));
        }

        if (!empty($config['aspect_ratio'])) {
            $supported_ratios = $model['specifications']['aspect_ratios'];
            if (!in_array($config['aspect_ratio'], $supported_ratios)) {
                return new WP_Error('invalid_aspect_ratio', 
                    sprintf(__('Aspect ratio %s not supported by %s', 'mxchat-veo'), 
                        $config['aspect_ratio'], $model['display_name']));
            }
        }

        return true;
    }

    public function get_model_pricing_info($model_id) {
        $model = $this->get_model($model_id);
        if (!$model) {
            return array();
        }

        return array(
            'tier' => $model['pricing_tier'],
            'tier_label' => ucfirst($model['pricing_tier'])
        );
    }

    public function get_model_comparison() {
        $models = $this->get_available_models();
        $comparison = array();

        foreach ($models as $id => $model) {
            $comparison[$id] = array(
                'name' => $model['display_name'],
                'status' => $model['status'],
                'audio' => $model['features']['audio'],
                'duration' => $model['specifications']['duration'] ?? $model['specifications']['duration_max'] ?? 8,
                'aspect_ratios' => count($model['specifications']['aspect_ratios']),
                'generation_time' => $model['performance']['avg_time'],
                'pricing' => $model['pricing_tier']
            );
        }

        return $comparison;
    }
}
?>