<?php

if (!defined('ABSPATH')) {
    exit;
}

class Veo_Processor {

    private $api_key;
    private $base_url = 'https://generativelanguage.googleapis.com/v1beta';

    public function __construct() {
        // Get Gemini API key from core MxChat plugin
        $mxchat_options = get_option('mxchat_options', array());
        $this->api_key = $mxchat_options['gemini_api_key'] ?? '';
    }

    /**
     * Get the Gemini API key status
     */
    public static function get_api_key_status() {
        $mxchat_options = get_option('mxchat_options', array());
        $gemini_key = $mxchat_options['gemini_api_key'] ?? '';
        return !empty($gemini_key);
    }

    /**
     * Get the OpenAI API key status (for future Sora support)
     */
    public static function get_openai_api_key_status() {
        $mxchat_options = get_option('mxchat_options', array());
        $openai_key = $mxchat_options['api_key'] ?? '';
        return !empty($openai_key);
    }

    public function generate_video($prompt, $model, $config = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('API key not configured', 'mxchat-veo'));
        }
    
        // Check if we have image data for image-to-video generation
        if (!empty($config['image_data'])) {
            return $this->generate_with_image($prompt, $model, $config['image_data'], $config);
        }
    
        // Standard text-to-video generation - Gemini API REST structure
        $instances = array(
            array(
                'prompt' => $prompt
            )
        );

        $parameters = array();

        if (!empty($config['negative_prompt'])) {
            $parameters['negativePrompt'] = $config['negative_prompt'];
        }

        if (!empty($config['aspect_ratio'])) {
            $parameters['aspectRatio'] = $config['aspect_ratio'];
        }

        if (!empty($config['resolution'])) {
            $parameters['resolution'] = $config['resolution'];
        }

        if (!empty($config['duration_seconds'])) {
            $parameters['durationSeconds'] = (int) $config['duration_seconds'];
        }

        $parameters['personGeneration'] = $this->get_person_generation_setting($model);

        $payload = array(
            'instances' => $instances
        );

        if (!empty($parameters)) {
            $payload['parameters'] = $parameters;
        }

        // Use the correct Gemini API REST endpoint
        $url = $this->base_url . '/models/' . $model . ':predictLongRunning';
    
        //error_log('Veo API Request URL: ' . $url);
        //error_log('Veo API Request Payload: ' . json_encode($payload));
    
        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'x-goog-api-key' => $this->api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($payload)
        ));
    
        if (is_wp_error($response)) {
            //error_log('Veo API WP Error: ' . $response->get_error_message());
            return $response;
        }
    
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $status_code = wp_remote_retrieve_response_code($response);
    
        //error_log('Veo API Response Status: ' . $status_code);
        //error_log('Veo API Response Body: ' . $body);
    
        if ($status_code !== 200) {
            $error_message = 'HTTP ' . $status_code . ': ';
            if (isset($data['error']['message'])) {
                $error_message .= $data['error']['message'];
            } elseif (isset($data['error'])) {
                $error_message .= json_encode($data['error']);
            } else {
                $error_message .= 'Unknown API error - ' . $body;
            }
            return new WP_Error('api_error', $error_message);
        }
    
        if (!isset($data['name'])) {
            return new WP_Error('invalid_response', __('Invalid API response', 'mxchat-veo'));
        }
    
        return array(
            'operation_id' => $data['name'],
            'status' => 'pending'
        );
    }

    public function check_operation_status($operation_id) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('API key not configured', 'mxchat-veo'));
        }

        // Use the correct Gemini API structure for checking status
        $url = $this->base_url . '/' . $operation_id;

        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'x-goog-api-key' => $this->api_key,
                'Content-Type' => 'application/json'
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        // Debug: Log the response
        //error_log('Veo Status Check Response Status: ' . $status_code);
        //error_log('Veo Status Check Response Body: ' . $body);

        if ($status_code !== 200) {
            $error_message = isset($data['error']['message']) ? $data['error']['message'] : __('Unknown API error', 'mxchat-veo');
            return new WP_Error('api_error', $error_message);
        }

        $result = array(
            'done' => isset($data['done']) ? $data['done'] : false,
            'operation_id' => $operation_id
        );

        if ($result['done']) {
            if (isset($data['error'])) {
                return new WP_Error('generation_failed', $data['error']['message']);
            }

            // Gemini API response structure based on the REST documentation
            if (isset($data['response']['generateVideoResponse']['generatedSamples'][0]['video']['uri'])) {
                $file_uri = $data['response']['generateVideoResponse']['generatedSamples'][0]['video']['uri'];
                
                // Add API key as query parameter for video playback
                $video_url = $file_uri . '&key=' . $this->api_key;
                
                $result['video_url'] = $video_url;
                $result['file_uri'] = $file_uri;
                $result['status'] = 'completed';
            } else {
                return new WP_Error('no_video', __('No video in response', 'mxchat-veo'));
            }
        } else {
            $result['status'] = 'processing';
        }

        return $result;
    }

    private function get_download_url($file_uri) {
        // For Gemini API, we need to add the API key as a header, not a query parameter
        return $file_uri;
    }

    public function download_video($file_uri) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('API key not configured', 'mxchat-veo'));
        }

        $response = wp_remote_get($file_uri, array(
            'timeout' => 60,
            'headers' => array(
                'x-goog-api-key' => $this->api_key
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_Error('download_failed', __('Failed to download video', 'mxchat-veo'));
        }

        // Return the video content directly for saving
        return wp_remote_retrieve_body($response);
    }

    public function generate_with_image($prompt, $model, $image_data, $config = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('API key not configured', 'mxchat-veo'));
        }

        // Try Vertex AI format - bytesBase64Encoded instead of imageBytes
        $instances = array(
            array(
                'prompt' => $prompt,
                'image' => array(
                    'bytesBase64Encoded' => base64_encode($image_data),
                    'mimeType' => $this->detect_mime_type($image_data)
                )
            )
        );

        $parameters = array();

        if (!empty($config['negative_prompt'])) {
            $parameters['negativePrompt'] = $config['negative_prompt'];
        }

        if (!empty($config['aspect_ratio'])) {
            $parameters['aspectRatio'] = $config['aspect_ratio'];
        }

        if (!empty($config['resolution'])) {
            $parameters['resolution'] = $config['resolution'];
        }

        if (!empty($config['duration_seconds'])) {
            $parameters['durationSeconds'] = (int) $config['duration_seconds'];
        }

        $parameters['personGeneration'] = $this->get_person_generation_setting($model, true);

        $payload = array(
            'instances' => $instances
        );

        if (!empty($parameters)) {
            $payload['parameters'] = $parameters;
        }

        $url = $this->base_url . '/models/' . $model . ':predictLongRunning';

        //error_log('Veo Image-to-Video API Request URL: ' . $url);
        //error_log('Veo Image-to-Video API Request Payload: ' . json_encode($payload, JSON_PRETTY_PRINT));

        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'x-goog-api-key' => $this->api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($payload)
        ));

        if (is_wp_error($response)) {
            //error_log('Veo Image-to-Video API WP Error: ' . $response->get_error_message());
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $status_code = wp_remote_retrieve_response_code($response);

        //error_log('Veo Image-to-Video API Response Status: ' . $status_code);
        //error_log('Veo Image-to-Video API Response Body: ' . $body);

        if ($status_code !== 200) {
            $error_message = 'HTTP ' . $status_code . ': ';
            if (isset($data['error']['message'])) {
                $error_message .= $data['error']['message'];
            } elseif (isset($data['error'])) {
                $error_message .= json_encode($data['error']);
            } else {
                $error_message .= 'Unknown API error - ' . $body;
            }
            return new WP_Error('api_error', $error_message);
        }

        if (!isset($data['name'])) {
            return new WP_Error('invalid_response', __('Invalid API response', 'mxchat-veo'));
        }

        return array(
            'operation_id' => $data['name'],
            'status' => 'pending'
        );
    }

    private function get_person_generation_setting($model, $has_image = false) {
        if (strpos($model, 'veo-3') !== false) {
            return $has_image ? 'allow_adult' : 'allow_all';
        } elseif (strpos($model, 'veo-2') !== false) {
            return $has_image ? 'allow_adult' : 'allow_all';
        }
        
        return 'allow_all';
    }

    private function detect_mime_type($image_data) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime_type = $finfo->buffer($image_data);
        
        $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
        
        if (in_array($mime_type, $allowed_types)) {
            return $mime_type;
        }
        
        return 'image/png';
    }

    public function validate_prompt($prompt, $model) {
        $max_length = 1024;
        
        if (strlen($prompt) > $max_length) {
            return new WP_Error('prompt_too_long', sprintf(__('Prompt exceeds maximum length of %d characters', 'mxchat-veo'), $max_length));
        }

        if (empty(trim($prompt))) {
            return new WP_Error('empty_prompt', __('Prompt cannot be empty', 'mxchat-veo'));
        }

        return true;
    }

    public function get_supported_aspect_ratios($model) {
        // All current Veo models (2, 3, 3.1) support both aspect ratios
        return array('16:9', '9:16');
    }

    public function get_supported_resolutions($model) {
        if (strpos($model, 'veo-3') !== false) {
            return array('720p', '1080p');
        }
        // Veo 2 only supports 720p
        return array('720p');
    }

    public function get_supported_durations($model) {
        if (strpos($model, 'veo-3') !== false) {
            return array(4, 6, 8);
        }
        // Veo 2 supports 5, 6, 8
        return array(5, 6, 8);
    }

    public function estimate_generation_time($model) {
        if (strpos($model, 'veo-3-fast') !== false) {
            return array('min' => 8, 'max' => 180);
        } elseif (strpos($model, 'veo-3') !== false) {
            return array('min' => 11, 'max' => 360);
        } elseif (strpos($model, 'veo-2') !== false) {
            return array('min' => 15, 'max' => 300);
        }
        
        return array('min' => 11, 'max' => 360);
    }
}
?>