<?php
/**
 * Plugin Name: MxChat Video Generation
 * Plugin URI: https://mxchat.ai/
 * Description: AI video generation add-on for MxChat using Google Veo and OpenAI Sora APIs
 * Version: 1.1.0
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-veo
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MXCHAT_VEO_VERSION', '1.1.0');
define('MXCHAT_VEO_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_VEO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_VEO_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Autoloader for classes
spl_autoload_register(function ($class) {
    $prefix = 'MxChat_Veo_';
    $base_dir = MXCHAT_VEO_PLUGIN_DIR . 'includes/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        // Also handle Veo_Processor, Veo_Models, and Sora_Processor classes
        if (in_array($class, ['Veo_Processor', 'Veo_Models', 'Sora_Processor'])) {
            $file = $base_dir . 'class-' . str_replace('_', '-', strtolower($class)) . '.php';
            if (file_exists($file)) {
                require_once $file;
            }
        }
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-mxchat-veo-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    if (file_exists($file)) {
        require_once $file;
    }
});

/**
 * Check if MxChat is active, supporting multisite environments
 */
function is_mxchat_active_veo() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        if (isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php'])) {
            return true;
        }
    }
    return in_array(
        'mxchat-basic/mxchat-basic.php',
        apply_filters('active_plugins', get_option('active_plugins'))
    );
}

/**
 * Check if MxChat Pro license is active
 */
function is_mxchat_pro_license_active_veo() {
    if (is_multisite()) {
        $network_license_status = get_site_option('mxchat_license_status');
        if ($network_license_status === 'active') {
            return true;
        }
    }
    $license_status = get_option('mxchat_license_status', 'inactive');
    return $license_status === 'active';
}

// Initialize update checker
require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerVeo::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-veo'
]);

/**
 * Main plugin class
 */
final class MxChat_Veo {
    private static $instance = null;
    public $core;
    public $settings;
    public $processor;
    public $models;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        try {
            if (!is_mxchat_active_veo()) {
                add_action('admin_notices', array($this, 'show_mxchat_required_notice'));
                return;
            }

            // Add admin notice for pro requirement
            add_action('admin_notices', array($this, 'show_pro_license_required_notice'));
            
            // Only load features if pro is active
            if (is_mxchat_pro_license_active_veo()) {
                $this->init_hooks();
                $this->init_components();
            }
            
        } catch (Exception $e) {
            add_action('admin_notices', function() use ($e) {
                echo '<div class="notice notice-error"><p>MxChat Video Generation Error: ' . esc_html($e->getMessage()) . '</p></div>';
            });
        }
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'), 20);
        add_action('network_admin_menu', array($this, 'add_network_admin_menu'), 20);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    /**
     * Initialize plugin components
     */
    private function init_components() {
        // Only initialize if classes exist
        if (class_exists('MxChat_Veo_Settings')) {
            $this->settings = new MxChat_Veo_Settings();
        }
        
        if (class_exists('MxChat_Veo_Core')) {
            $this->core = new MxChat_Veo_Core();
        }
        
        if (class_exists('Veo_Processor')) {
            $this->processor = new Veo_Processor();
        }
        
        if (class_exists('Veo_Models')) {
            $this->models = new Veo_Models();
        }
    }

    /**
     * Display notice if MxChat is not active
     */
    public function show_mxchat_required_notice() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><?php _e('<strong>MxChat Video Generation</strong> requires the MxChat plugin to be installed and active.', 'mxchat-veo'); ?></p>
        </div>
        <?php
    }

    /**
     * Display notice if MxChat Pro license is not active
     */
    public function show_pro_license_required_notice() {
        if (!is_mxchat_pro_license_active_veo()) {
            $class = 'notice notice-warning';
            $message = sprintf(
                __('MxChat Video Generation requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-veo'),
                admin_url('admin.php?page=mxchat-activation')
            );
            printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
        }
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our admin page
        if ($hook !== 'mxchat_page_mxchat-veo') {
            return;
        }

        wp_enqueue_style(
            'mxchat-veo-admin',
            MXCHAT_VEO_PLUGIN_URL . 'admin/css/veo-admin.css',
            array(),
            MXCHAT_VEO_VERSION
        );

        wp_enqueue_script(
            'mxchat-veo-admin',
            MXCHAT_VEO_PLUGIN_URL . 'admin/js/veo-admin.js',
            array('jquery'),
            MXCHAT_VEO_VERSION,
            true
        );

        wp_localize_script('mxchat-veo-admin', 'mxchatVeo', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat_veo_nonce'),
            'strings' => array(
                'generating' => __('Generating video...', 'mxchat-veo'),
                'error' => __('Error generating video', 'mxchat-veo'),
                'success' => __('Video generated successfully!', 'mxchat-veo'),
                'saved' => __('Video saved to media library', 'mxchat-veo')
            )
        ));
    }

    /**
     * Add menu to network admin under Settings
     */
    public function add_network_admin_menu() {
        add_submenu_page(
            'settings.php',
            esc_html__('MxChat Video Generation', 'mxchat-veo'),
            esc_html__('Video Generation', 'mxchat-veo'),
            'manage_network_options',
            'mxchat-veo',
            array($this->settings, 'display_admin_page')
        );
    }

    /**
     * Add menu to site admin under MxChat
     */
    public function add_admin_menu() {
        add_submenu_page(
            'mxchat-max',
            esc_html__('MxChat Video Generation', 'mxchat-veo'),
            esc_html__('Video Generation', 'mxchat-veo'),
            'manage_options',
            'mxchat-veo',
            array($this->settings, 'display_admin_page')
        );
    }
}

/**
 * Initialize the plugin
 */
function MxChat_Veo() {
    return MxChat_Veo::instance();
}

add_action('plugins_loaded', 'MxChat_Veo', 20);

/**
 * Activation hook
 */
register_activation_hook(__FILE__, function() {
    // Set default options
    add_option('mxchat_veo_enabled', false);
    add_option('mxchat_veo_model', 'veo-2');
    add_option('mxchat_veo_api_key', '');
});

/**
 * Deactivation hook
 */
register_deactivation_hook(__FILE__, function() {
    // Clean up if needed
});

/**
 * Uninstall hook
 */
register_uninstall_hook(__FILE__, 'mxchat_veo_uninstall');

function mxchat_veo_uninstall() {
    // Remove options on uninstall
    delete_option('mxchat_veo_enabled');
    delete_option('mxchat_veo_model');
    delete_option('mxchat_veo_api_key');
    delete_option('mxchat_veo_settings');
}
?>