<?php
/**
 * Fired during plugin activation
 *
 * @link       https://mxchat.ai/
 * @since      1.0.0
 *
 * @package    MxChat_Woo
 * @subpackage MxChat_Woo/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    MxChat_Woo
 * @subpackage MxChat_Woo/includes
 * @author     MxChat <support@mxchat.ai>
 */
class MxChat_Woo_Activator {

    /**
     * Database version option name
     */
    const DB_VERSION_OPTION = 'mxchat_woo_db_version';

    /**
     * Current database version
     */
    const CURRENT_DB_VERSION = '1.0.0';

    /**
     * Activate the plugin.
     *
     * @since    1.0.0
     */
    public static function activate() {
        self::check_dependencies();
        self::setup_database_tables();
        self::migrate_existing_settings();
        self::set_default_options();
    }

    /**
     * Check if required plugins are active
     *
     * @since    1.0.0
     */
    private static function check_dependencies() {
        if (!is_plugin_active('woocommerce/woocommerce.php')) {
            wp_die(
                esc_html__('This plugin requires WooCommerce to be installed and activated.', 'mxchat-woo'),
                'Plugin Activation Error',
                array('back_link' => true)
            );
        }

        if (!is_plugin_active('mxchat-basic/mxchat-basic.php')) {
            wp_die(
                esc_html__('This plugin requires MxChat to be installed and activated.', 'mxchat-woo'),
                'Plugin Activation Error',
                array('back_link' => true)
            );
        }
    }

    /**
     * Set up any necessary database tables
     *
     * @since    1.0.0
     */
    private static function setup_database_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Table for storing product discussions/mentions
        $product_mentions_table = $wpdb->prefix . 'mxchat_woo_product_mentions';
        
        $sql = "CREATE TABLE IF NOT EXISTS $product_mentions_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            product_id bigint(20) NOT NULL,
            chat_id varchar(50) NOT NULL,
            session_id varchar(50) NOT NULL,
            mention_type varchar(20) NOT NULL,
            mention_context text NOT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY product_id (product_id),
            KEY chat_id (chat_id),
            KEY session_id (session_id)
        ) $charset_collate;";

        // Include WordPress database upgrade functions
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Store the current database version
        update_option(self::DB_VERSION_OPTION, self::CURRENT_DB_VERSION);
    }

    /**
     * Migrate existing WooCommerce settings from core plugin
     *
     * @since    1.0.0
     */
    private static function migrate_existing_settings() {
        // Get existing MxChat options
        $existing_options = get_option('mxchat_options', array());

        // WooCommerce specific options to migrate
        $woo_options = array(
            'enable_woocommerce_integration' => isset($existing_options['enable_woocommerce_integration']) 
                ? $existing_options['enable_woocommerce_integration'] 
                : '',
            'woocommerce_consumer_key' => isset($existing_options['woocommerce_consumer_key']) 
                ? $existing_options['woocommerce_consumer_key'] 
                : '',
            'woocommerce_consumer_secret' => isset($existing_options['woocommerce_consumer_secret']) 
                ? $existing_options['woocommerce_consumer_secret'] 
                : ''
        );

        // Preserve these options in the main MxChat options for backward compatibility
        update_option('mxchat_woo_settings', $woo_options);
        
        // Set migration flag
        update_option('mxchat_woo_migrated', true);
    }

    /**
     * Set default options if they don't exist
     *
     * @since    1.0.0
     */
    private static function set_default_options() {
        // Default settings
        $defaults = array(
            'enable_woocommerce_integration' => 'off',
            'product_mention_tracking' => 'on',
            'cart_integration' => 'on',
            'order_history_access' => 'on',
            'customer_data_access' => 'on'
        );

        // Get existing settings
        $existing = get_option('mxchat_woo_settings', array());

        // Merge with defaults
        $settings = wp_parse_args($existing, $defaults);

        // Update settings
        update_option('mxchat_woo_settings', $settings);
    }

    /**
     * Get the current database version
     *
     * @since    1.0.0
     * @return string
     */
    public static function get_db_version() {
        return get_option(self::DB_VERSION_OPTION, '0');
    }

    /**
     * Check if database needs upgrade
     *
     * @since    1.0.0
     * @return boolean
     */
    public static function needs_db_upgrade() {
        $current_version = self::get_db_version();
        return version_compare($current_version, self::CURRENT_DB_VERSION, '<');
    }
}