<?php
/**
 * The admin-specific functionality of the plugin
 */
class MxChat_Woo_Admin {

    private $loader;
    private $plugin_name;
    private $version;
    private $is_activated;
    private $options;

    public function __construct($loader) {
        $this->loader = $loader;
        $this->plugin_name = 'mxchat-woo';
        $this->version = MXCHAT_WOO_VERSION;
        $this->is_activated = get_option('mxchat_license_status', 'inactive') === 'active';
        // Change to use WooCommerce specific options
        $this->options = get_option('mxchat_woo_options', array());
        
        $this->init();
    }

    private function init() {
        // Add settings init
        $this->loader->add_action('admin_init', $this, 'init_settings');
        
        // Add admin assets
        $this->loader->add_action('admin_enqueue_scripts', $this, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $this, 'enqueue_scripts');
    }

    public function enqueue_styles($hook) {
        if (strpos($hook, 'mxchat-woo') === false) {
            return;
        }

        wp_enqueue_style(
            $this->plugin_name,
            MXCHAT_WOO_PLUGIN_URL . 'admin/css/mxchat-woo-admin.css',
            array(),
            $this->version,
            'all'
        );
    }

    public function enqueue_scripts($hook) {
        if (strpos($hook, 'mxchat-woo') === false) {
            return;
        }

        wp_enqueue_script(
            $this->plugin_name,
            MXCHAT_WOO_PLUGIN_URL . 'admin/js/mxchat-woo-admin.js',
            array('jquery'),
            $this->version,
            true
        );
        
        // Localize script for AJAX
        wp_localize_script(
            $this->plugin_name,
            'mxchatWooAdmin',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mxchat_nonce'),
                'i18n' => array(
                    'testing' => __('Testing...', 'mxchat-woo'),
                    'test_success' => __('Test completed successfully!', 'mxchat-woo'),
                    'test_error' => __('Test failed. Please check your settings.', 'mxchat-woo'),
                )
            )
        );
    }

    public function display_admin_page() {
        require_once MXCHAT_WOO_PLUGIN_DIR . 'admin/partials/mxchat-woo-settings-page.php';
    }

    public function init_settings() {
        // Register setting with the option name
        register_setting(
            'mxchat_woo_options',
            'mxchat_woo_options',
            array($this, 'validate_options')
        );
        
        // Register filtered search settings separately
        register_setting(
            'mxchat_woo_filtered_search_settings',
            'mxchat_woo_filtered_search_settings',
            array($this, 'validate_filtered_search_settings')
        );

        // Product Display Section (empty title since card already has h2)
        add_settings_section(
            'mxchat_product_display_section',
            '',
            array($this, 'render_product_display_section_info'),
            'mxchat-woo'
        );
        
        add_settings_field(
            'add_to_cart_button_text',
            esc_html__('Add to Cart Button Text', 'mxchat-woo'),
            array($this, 'render_button_text_field'),
            'mxchat-woo',
            'mxchat_product_display_section'
        );

        add_settings_field(
            'show_product_prices',
            esc_html__('Show Product Prices', 'mxchat-woo'),
            array($this, 'render_show_prices_field'),
            'mxchat-woo',
            'mxchat_product_display_section'
        );

        add_settings_field(
            'show_add_to_cart_button',
            esc_html__('Show Add to Cart Button', 'mxchat-woo'),
            array($this, 'render_show_cart_button_field'),
            'mxchat-woo',
            'mxchat_product_display_section'
        );
        
        // Filtered Search Section (empty title since card already has h2)
        add_settings_section(
            'mxchat_filtered_search_section',
            '',
            array($this, 'render_filtered_search_section_info'),
            'mxchat-woo-filtered-search'
        );
        
        add_settings_field(
            'filtered_search_enabled',
            esc_html__('Enable Filtered Search', 'mxchat-woo'),
            array($this, 'render_filtered_search_enabled_field'),
            'mxchat-woo-filtered-search',
            'mxchat_filtered_search_section'
        );

        // Note: Trigger phrases and similarity threshold fields removed -
        // triggering is now handled by the core plugin's intent system in MxChat → Intents

        add_settings_field(
            'enable_semantic_rerank',
            esc_html__('Semantic Re-ranking', 'mxchat-woo'),
            array($this, 'render_semantic_rerank_field'),
            'mxchat-woo-filtered-search',
            'mxchat_filtered_search_section'
        );

        add_settings_field(
            'enabled_filters',
            esc_html__('Enabled Filters', 'mxchat-woo'),
            array($this, 'render_enabled_filters_field'),
            'mxchat-woo-filtered-search',
            'mxchat_filtered_search_section'
        );
    }

    public function render_product_display_section_info() {
        echo '<p>' . esc_html__('Control how products are displayed in chat responses. To enable the cart & order history icons, enable the toolbar under MxChat → Toolbar & Components.', 'mxchat-woo') . '</p>';
        echo '<p>' . esc_html__('Note: Product syncing is now managed in the main MxChat settings under Knowledge → Settings tab.', 'mxchat-woo') . '</p>';
    }
    
    public function render_button_text_field() {
        $value = isset($this->options['add_to_cart_button_text']) 
            ? $this->options['add_to_cart_button_text'] 
            : esc_html__('Add to Cart', 'mxchat-woo');
        ?>
        <input type="text" 
               name="mxchat_woo_options[add_to_cart_button_text]" 
               class="regular-text"
               value="<?php echo esc_attr($value); ?>"
               placeholder="<?php esc_attr_e('Add to Cart', 'mxchat-woo'); ?>">
        <p class="description">
            <?php esc_html_e('Customize the text displayed on the add to cart button in product cards.', 'mxchat-woo'); ?>
        </p>
        <?php
    }

    public function render_show_prices_field() {
        $value = isset($this->options['show_product_prices']) 
            ? $this->options['show_product_prices'] 
            : 'on';
        ?>
        <div class="mxchat-setting-field">
            <label class="mxchat-toggle-switch">
                <input type="checkbox" 
                       name="mxchat_woo_options[show_product_prices]" 
                       value="on" <?php checked($value, 'on'); ?>>
                <span class="mxchat-toggle-slider"></span>
            </label>
            <p class="description">
                <?php esc_html_e('Display product prices in chat product cards. Disable this if customers need to contact you for pricing.', 'mxchat-woo'); ?>
            </p>
        </div>
        <?php
    }

    public function render_show_cart_button_field() {
        $value = isset($this->options['show_add_to_cart_button']) 
            ? $this->options['show_add_to_cart_button'] 
            : 'on';
        ?>
        <div class="mxchat-setting-field">
            <label class="mxchat-toggle-switch">
                <input type="checkbox" 
                       name="mxchat_woo_options[show_add_to_cart_button]" 
                       value="on" <?php checked($value, 'on'); ?>>
                <span class="mxchat-toggle-slider"></span>
            </label>
            <p class="description">
                <?php esc_html_e('Show add to cart button in product cards. Disable this if customers cannot purchase directly.', 'mxchat-woo'); ?>
            </p>
        </div>
        <?php
    }
    
    // Filtered Search Settings Renderers
    
    public function render_filtered_search_section_info() {
        echo '<p>' . esc_html__('Configure the filtered product search feature that allows users to search products with specific criteria like price ranges, categories, and attributes.', 'mxchat-woo') . '</p>';
        echo '<div class="notice notice-info inline"><p>';
        echo '<strong>' . esc_html__('How it works:', 'mxchat-woo') . '</strong><br>';
        echo esc_html__('1. User query is compared to trigger phrases using semantic similarity', 'mxchat-woo') . '<br>';
        echo esc_html__('2. If threshold is met, AI extracts structured filters (price, category, attributes)', 'mxchat-woo') . '<br>';
        echo esc_html__('3. Products are filtered using WooCommerce database queries', 'mxchat-woo') . '<br>';
        echo esc_html__('4. Results are optionally re-ranked using semantic similarity', 'mxchat-woo');
        echo '</p></div>';
    }
    
    public function render_filtered_search_enabled_field() {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $value = isset($settings['enabled']) ? $settings['enabled'] : 'off';
        ?>
        <div class="mxchat-setting-field">
            <label class="mxchat-toggle-switch">
                <input type="checkbox" 
                       name="mxchat_woo_filtered_search_settings[enabled]" 
                       value="on" <?php checked($value, 'on'); ?>>
                <span class="mxchat-toggle-slider"></span>
            </label>
            <p class="description">
                <?php esc_html_e('Enable or disable the filtered product search feature. When disabled, the system will fall back to standard product search.', 'mxchat-woo'); ?>
            </p>
        </div>
        <?php
    }
    
    public function render_trigger_phrases_field() {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $default_triggers = array(
            'under $',
            'less than',
            'cheaper than',
            'price range',
            'between $ and $',
            'budget of',
            'within my budget',
            'affordable',
            'in the category',
            'from category',
            'filter by',
            'show me products',
        );
        $triggers = isset($settings['trigger_phrases']) && !empty($settings['trigger_phrases']) 
            ? $settings['trigger_phrases'] 
            : $default_triggers;
        ?>
        <div class="mxchat-trigger-phrases-container">
            <div id="trigger-phrases-list">
                <?php foreach ($triggers as $index => $phrase): ?>
                <div class="trigger-phrase-row">
                    <input type="text" 
                           name="mxchat_woo_filtered_search_settings[trigger_phrases][]" 
                           value="<?php echo esc_attr($phrase); ?>" 
                           class="regular-text">
                    <button type="button" class="button remove-trigger-phrase">
                        <span class="dashicons dashicons-trash"></span>
                    </button>
                </div>
                <?php endforeach; ?>
            </div>
            <button type="button" id="add-trigger-phrase" class="button button-secondary">
                <span class="dashicons dashicons-plus-alt"></span>
                <?php esc_html_e('Add Trigger Phrase', 'mxchat-woo'); ?>
            </button>
            <p class="description">
                <?php esc_html_e('Phrases that will trigger the filtered search. The system uses semantic similarity to match user queries against these phrases.', 'mxchat-woo'); ?>
            </p>
        </div>
        <style>
            .trigger-phrase-row {
                display: flex;
                gap: 10px;
                margin-bottom: 10px;
                align-items: center;
            }
            .trigger-phrase-row input {
                flex: 1;
            }
            .remove-trigger-phrase {
                padding: 5px 10px;
            }
            #add-trigger-phrase {
                margin-top: 10px;
            }
        </style>
        <script>
        jQuery(document).ready(function($) {
            // Add trigger phrase
            $('#add-trigger-phrase').on('click', function() {
                var newRow = $('<div class="trigger-phrase-row">' +
                    '<input type="text" name="mxchat_woo_filtered_search_settings[trigger_phrases][]" value="" class="regular-text">' +
                    '<button type="button" class="button remove-trigger-phrase">' +
                    '<span class="dashicons dashicons-trash"></span>' +
                    '</button>' +
                    '</div>');
                $('#trigger-phrases-list').append(newRow);
            });
            
            // Remove trigger phrase
            $(document).on('click', '.remove-trigger-phrase', function() {
                if ($('.trigger-phrase-row').length > 1) {
                    $(this).closest('.trigger-phrase-row').remove();
                } else {
                    alert('<?php esc_html_e('You must have at least one trigger phrase.', 'mxchat-woo'); ?>');
                }
            });
        });
        </script>
        <?php
    }
    
    public function render_similarity_threshold_field() {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $value = isset($settings['similarity_threshold']) ? $settings['similarity_threshold'] : 0.70;
        ?>
        <input type="number" 
               name="mxchat_woo_filtered_search_settings[similarity_threshold]" 
               value="<?php echo esc_attr($value); ?>" 
               step="0.01" 
               min="0" 
               max="1" 
               class="small-text">
        <p class="description">
            <?php esc_html_e('Minimum similarity score (0-1) required to trigger filtered search. Higher values mean stricter matching. Recommended: 0.65-0.75', 'mxchat-woo'); ?>
        </p>
        <?php
    }
    
    public function render_semantic_rerank_field() {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $value = isset($settings['enable_semantic_rerank']) ? $settings['enable_semantic_rerank'] : 'off';
        ?>
        <div class="mxchat-setting-field">
            <label class="mxchat-toggle-switch">
                <input type="checkbox" 
                       name="mxchat_woo_filtered_search_settings[enable_semantic_rerank]" 
                       value="on" <?php checked($value, 'on'); ?>>
                <span class="mxchat-toggle-slider"></span>
            </label>
            <p class="description">
                <?php esc_html_e('Re-rank filtered results using semantic similarity to the user query. This improves relevance but increases processing time and API costs.', 'mxchat-woo'); ?>
            </p>
        </div>
        <?php
    }
    
    public function render_enabled_filters_field() {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $enabled = isset($settings['enabled_filters']) ? $settings['enabled_filters'] : array('price', 'category', 'attributes');
        ?>
        <fieldset>
            <label>
                <input type="checkbox" 
                       name="mxchat_woo_filtered_search_settings[enabled_filters][]" 
                       value="price" 
                       <?php checked(in_array('price', $enabled)); ?>>
                <?php esc_html_e('Price Filters (min/max price)', 'mxchat-woo'); ?>
            </label><br>
            <label>
                <input type="checkbox" 
                       name="mxchat_woo_filtered_search_settings[enabled_filters][]" 
                       value="category" 
                       <?php checked(in_array('category', $enabled)); ?>>
                <?php esc_html_e('Category Filters', 'mxchat-woo'); ?>
            </label><br>
            <label>
                <input type="checkbox" 
                       name="mxchat_woo_filtered_search_settings[enabled_filters][]" 
                       value="attributes" 
                       <?php checked(in_array('attributes', $enabled)); ?>>
                <?php esc_html_e('Product Attributes (color, size, etc.)', 'mxchat-woo'); ?>
            </label>
        </fieldset>
        <p class="description">
            <?php esc_html_e('Select which types of filters the AI can extract and apply from user queries.', 'mxchat-woo'); ?>
        </p>
        <?php
    }

    public function validate_options($input) {
        $valid = array();
        
        // Handle product display options
        $valid['show_product_prices'] = isset($input['show_product_prices']) ? 'on' : 'off';
        $valid['show_add_to_cart_button'] = isset($input['show_add_to_cart_button']) ? 'on' : 'off';
        
        // Validate button text
        if (isset($input['add_to_cart_button_text'])) {
            $button_text = sanitize_text_field($input['add_to_cart_button_text']);
            $valid['add_to_cart_button_text'] = !empty($button_text) ? $button_text : esc_html__('Add to Cart', 'mxchat-woo');
        } else {
            $valid['add_to_cart_button_text'] = esc_html__('Add to Cart', 'mxchat-woo');
        }
        
        return $valid;
    }
    
    public function validate_filtered_search_settings($input) {
        $valid = array();
        
        // Validate enabled status
        $valid['enabled'] = isset($input['enabled']) ? 'on' : 'off';
        
        // Validate trigger phrases
        if (isset($input['trigger_phrases']) && is_array($input['trigger_phrases'])) {
            $valid['trigger_phrases'] = array_filter(array_map('sanitize_text_field', $input['trigger_phrases']));
        } else {
            // Provide default if empty
            $valid['trigger_phrases'] = array('under $', 'less than', 'price range');
        }
        
        // Validate similarity threshold
        if (isset($input['similarity_threshold'])) {
            $threshold = floatval($input['similarity_threshold']);
            $valid['similarity_threshold'] = max(0, min(1, $threshold));
        } else {
            $valid['similarity_threshold'] = 0.70;
        }
        
        // Validate semantic rerank
        $valid['enable_semantic_rerank'] = isset($input['enable_semantic_rerank']) ? 'on' : 'off';
        
        // Validate enabled filters
        if (isset($input['enabled_filters']) && is_array($input['enabled_filters'])) {
            $valid['enabled_filters'] = array_map('sanitize_text_field', $input['enabled_filters']);
        } else {
            $valid['enabled_filters'] = array();
        }
        
        // Clear cached trigger embeddings when settings change
        delete_transient('mxchat_woo_filtered_search_trigger_embeddings_' . md5(serialize($valid['trigger_phrases'])));
        
        return $valid;
    }
}
