<?php
/**
 * Handles WooCommerce cart operations for MxChat
 */
class MxChat_Woo_Cart_Handler {

    private $loader;
    private $options;
    private $api_key;

    public function __construct($loader, $options = array()) {
        $this->loader = $loader;
        
        // Get the core plugin options directly
        $core_options = get_option('mxchat_options', array());
        
        // Set options and API key
        $this->options = $core_options;
        $this->api_key = isset($core_options['api_key']) ? $core_options['api_key'] : '';
        
        //error_log("🔍 MXCHAT DEBUG: Cart Handler initialized with API key: " . (empty($this->api_key) ? 'Not found' : 'Found'));
        
        $this->init();
    }

private function init() {
    // Add AJAX handlers for both logged in and non-logged in users
    $this->loader->add_action('wp_ajax_mxchat_add_to_cart', $this, 'add_to_cart');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_add_to_cart', $this, 'add_to_cart');
    
    $this->loader->add_action('wp_ajax_mxchat_update_cart', $this, 'update_cart');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_update_cart', $this, 'update_cart');
    
    $this->loader->add_action('wp_ajax_mxchat_remove_from_cart', $this, 'remove_from_cart');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_remove_from_cart', $this, 'remove_from_cart');
    
    $this->loader->add_action('wp_ajax_mxchat_get_cart', $this, 'get_cart');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_get_cart', $this, 'get_cart');

    // New AJAX handlers for cart icon functionality
    $this->loader->add_action('wp_ajax_mxchat_woo_get_cart_count', $this, 'get_cart_count');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_woo_get_cart_count', $this, 'get_cart_count');
    
    $this->loader->add_action('wp_ajax_mxchat_woo_get_cart_contents', $this, 'get_cart_contents');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_woo_get_cart_contents', $this, 'get_cart_contents');
    
    $this->loader->add_action('wp_ajax_mxchat_woo_remove_cart_item', $this, 'remove_cart_item');
    $this->loader->add_action('wp_ajax_nopriv_mxchat_woo_remove_cart_item', $this, 'remove_cart_item');

    // Add script enqueuing
    $this->loader->add_action('wp_enqueue_scripts', $this, 'enqueue_cart_scripts');
    
    // Add filter to inject cart icon into chatbot toolbar
    $this->loader->add_filter('mxchat_chatbot_toolbar_end', $this, 'inject_cart_icon', 10, 1);
    
    // Add action to enqueue cart styles
    $this->loader->add_action('wp_enqueue_scripts', $this, 'enqueue_cart_styles');
}

public function enqueue_cart_scripts() {
    if (!class_exists('WooCommerce')) {
        return;
    }

    // Ensure WooCommerce's add to cart script is loaded
    if (!wp_script_is('wc-add-to-cart', 'enqueued')) {
        wp_enqueue_script('wc-add-to-cart');
    }

    wp_enqueue_script(
        'mxchat-woo-cart',
        plugin_dir_url(dirname(__FILE__)) . 'public/js/mxchat-woo-cart.js',
        array('jquery', 'wc-add-to-cart'),
        MXCHAT_WOO_VERSION,
        true
    );

wp_localize_script(
    'mxchat-woo-cart',
    'mxchatWooCart',
    array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('mxchat_nonce'),
        'cart_url' => function_exists('wc_get_cart_url') ? wc_get_cart_url() : '',
        'checkout_url' => function_exists('wc_get_checkout_url') ? wc_get_checkout_url() : '',
        'i18n' => array(
            'loading_text' => __('Loading cart...', 'mxchat-woo'),
            'empty_cart_text' => __('Your cart is empty.', 'mxchat-woo'),
            'error_text' => __('Error loading cart.', 'mxchat-woo'),
            'quantity_text' => __('Qty', 'mxchat-woo'),
            'subtotal_text' => __('Subtotal', 'mxchat-woo'),
            'remove_text' => __('Remove item', 'mxchat-woo'),
            'cart_title' => __('Your Cart', 'mxchat-woo'),
            'your_cart' => __('Your Cart', 'mxchat-woo'),
            'close' => __('Close', 'mxchat-woo'),
            'view_cart' => __('View Cart', 'mxchat-woo'),
            'checkout' => __('Checkout', 'mxchat-woo')
        )
    )
);
}

public function enqueue_cart_styles() {
    if (!class_exists('WooCommerce')) {
        return;
    }
    
    wp_enqueue_style(
        'mxchat-woo-cart',
        plugin_dir_url(dirname(__FILE__)) . 'public/css/mxchat-woo-public.css',
        array(),
        MXCHAT_WOO_VERSION
    );
}
    /**
     * Inject cart icon into chatbot toolbar
     * 
     * @param string $toolbar_html Current toolbar HTML
     * @return string Modified toolbar HTML with cart icon
     */
    public function inject_cart_icon($toolbar_html) {
        if (!class_exists('WooCommerce')) {
            return $toolbar_html;
        }
        
        ob_start();
        
        // Get cart item count from WooCommerce
        $cart_item_count = is_object(WC()->cart) ? WC()->cart->get_cart_contents_count() : 0;
        $cart_count_class = ($cart_item_count > 0) ? '' : 'empty-cart';
        
        ?>
        <button id="woo-cart-btn" class="toolbar-btn" title="<?php echo esc_attr__('Your Cart', 'mxchat-woo'); ?>">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512" stroke="currentColor">
                <path d="M0 24C0 10.7 10.7 0 24 0H69.5c22 0 41.5 12.8 50.6 32h411c26.3 0 45.5 25 38.6 50.4l-41 152.3c-8.5 31.4-37 53.3-69.5 53.3H170.7l5.4 28.5c2.2 11.3 12.1 19.5 23.6 19.5H488c13.3 0 24 10.7 24 24s-10.7 24-24 24H199.7c-34.6 0-64.3-24.6-70.7-58.5L77.4 54.5c-.7-3.8-4-6.5-7.9-6.5H24C10.7 48 0 37.3 0 24zM128 464a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm336-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z"/>
            </svg>
            <span id="woo-cart-count" class="woo-cart-count <?php echo esc_attr($cart_count_class); ?>"><?php echo esc_html($cart_item_count); ?></span>
        </button>
        
        <div id="woo-cart-dropdown" class="woo-cart-dropdown">
            <div class="woo-cart-header">
                <h4><?php echo esc_html__('Your Cart', 'mxchat-woo'); ?></h4>
                <button class="woo-cart-close-btn" title="<?php echo esc_attr__('Close', 'mxchat-woo'); ?>">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                        <line x1="18" y1="6" x2="6" y2="18"></line>
                        <line x1="6" y1="6" x2="18" y2="18"></line>
                    </svg>
                </button>
            </div>
            <div class="woo-cart-items-container">
                <div id="woo-cart-items" class="woo-cart-items">
                    <div class="woo-cart-loading"><?php echo esc_html__('Loading cart...', 'mxchat-woo'); ?></div>
                </div>
            </div>
            <div class="woo-cart-footer">
                <div id="woo-cart-subtotal" class="woo-cart-subtotal"></div>
                <a href="<?php echo esc_url(wc_get_cart_url()); ?>" class="woo-cart-view-btn"><?php echo esc_html__('View Cart', 'mxchat-woo'); ?></a>
                <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="woo-cart-checkout-btn"><?php echo esc_html__('Checkout', 'mxchat-woo'); ?></a>
            </div>
        </div>
        <?php
        
        $cart_icon_html = ob_get_clean();
        
        // Add the cart icon just before the end of the toolbar div
        return str_replace('</div><!-- End .chat-toolbar -->', $cart_icon_html . '</div><!-- End .chat-toolbar -->', $toolbar_html);
    }

    /**
     * Add to cart AJAX handler
     */
    public function add_to_cart() {
        //error_log("🛒 MXCHAT: Processing add to cart request");
        //error_log("🛒 MXCHAT: Received nonce: " . (isset($_POST['nonce']) ? 'yes' : 'no'));

        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            //error_log("❌ MXCHAT: Add to cart nonce verification failed");
            wp_send_json_error('Invalid nonce.');
            return;
        }

        $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        
        if (!$product_id) {
            //error_log("❌ MXCHAT: Invalid product ID");
            wp_send_json_error('Invalid product ID.');
            return;
        }

        try {
            $added = WC()->cart->add_to_cart($product_id, 1);
            
            if ($added) {
                $product = wc_get_product($product_id);
                //error_log("✅ MXCHAT: Successfully added product to cart");
                wp_send_json_success([
                    'message' => sprintf(__('Added %s to your cart.', 'mxchat-woo'), $product->get_name()),
                    'fragments' => apply_filters('woocommerce_add_to_cart_fragments', []),
                    'cart_hash' => WC()->cart->get_cart_hash()
                ]);
            } else {
                //error_log("❌ MXCHAT: Failed to add product to cart");
                wp_send_json_error(__('Could not add product to cart.', 'mxchat-woo'));
            }
        } catch (Exception $e) {
            //error_log("❌ MXCHAT: Cart error - " . $e->getMessage());
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Update cart AJAX handler
     */
    public function update_cart() {
        //error_log("🛒 MXCHAT: Processing cart update request");

        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            //error_log("❌ MXCHAT: Update cart nonce verification failed");
            wp_send_json_error('Invalid nonce.');
            return;
        }

        $cart_item_key = isset($_POST['cart_item_key']) ? sanitize_text_field($_POST['cart_item_key']) : '';
        $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 0;

        if (!$cart_item_key) {
            //error_log("❌ MXCHAT: Invalid cart item key");
            wp_send_json_error('Invalid cart item key.');
            return;
        }

        try {
            WC()->cart->set_quantity($cart_item_key, $quantity);
            $cart_data = $this->get_cart_data();
            //error_log("✅ MXCHAT: Cart updated successfully");
            wp_send_json_success(array(
                'message' => __('Cart updated.', 'mxchat-woo'),
                'cart' => $cart_data
            ));
        } catch (Exception $e) {
            //error_log("❌ MXCHAT: Cart update error - " . $e->getMessage());
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Remove from cart AJAX handler
     */
    public function remove_from_cart() {
        //error_log("🛒 MXCHAT: Processing remove from cart request");

        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            //error_log("❌ MXCHAT: Remove from cart nonce verification failed");
            wp_send_json_error('Invalid nonce.');
            return;
        }

        $cart_item_key = isset($_POST['cart_item_key']) ? sanitize_text_field($_POST['cart_item_key']) : '';

        if (!$cart_item_key) {
            //error_log("❌ MXCHAT: Invalid cart item key");
            wp_send_json_error('Invalid cart item key.');
            return;
        }

        try {
            WC()->cart->remove_cart_item($cart_item_key);
            $cart_data = $this->get_cart_data();
            //error_log("✅ MXCHAT: Item removed from cart successfully");
            wp_send_json_success(array(
                'message' => __('Item removed from cart.', 'mxchat-woo'),
                'cart' => $cart_data
            ));
        } catch (Exception $e) {
            //error_log("❌ MXCHAT: Remove from cart error - " . $e->getMessage());
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Get cart AJAX handler
     */
    public function get_cart() {
        //error_log("🛒 MXCHAT: Processing get cart request");

        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            //error_log("❌ MXCHAT: Get cart nonce verification failed");
            wp_send_json_error('Invalid nonce.');
            return;
        }

        $cart_data = $this->get_cart_data();
        //error_log("✅ MXCHAT: Cart data retrieved successfully");
        wp_send_json_success($cart_data);
    }

    /**
     * Get cart count AJAX handler (for the cart icon)
     */
    public function get_cart_count() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'mxchat-woo')));
        }

        $count = 0;
        if (function_exists('WC') && isset(WC()->cart)) {
            $count = WC()->cart->get_cart_contents_count();
        }

        wp_send_json_success(array('count' => $count));
    }

    /**
     * Get cart contents AJAX handler (for the cart dropdown)
     */
    public function get_cart_contents() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'mxchat-woo')));
        }

        if (!function_exists('WC') || !isset(WC()->cart)) {
            wp_send_json_error(array('message' => __('WooCommerce cart not available.', 'mxchat-woo')));
        }

        $cart_items = array();
        foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
            $product = $cart_item['data'];
            $product_id = $cart_item['product_id'];
            
            // Get product image
            $image = wp_get_attachment_image_src(get_post_thumbnail_id($product_id), 'thumbnail');
            $image_url = $image ? $image[0] : wc_placeholder_img_src();
            
            $cart_items[] = array(
                'key' => $cart_item_key,
                'name' => $product->get_name(),
                'price' => WC()->cart->get_product_price($product),
                'quantity' => $cart_item['quantity'],
                'image' => $image_url,
                'product_id' => $product_id,
                'variation_id' => $cart_item['variation_id'],
                'total' => WC()->cart->get_product_subtotal($product, $cart_item['quantity'])
            );
        }

        wp_send_json_success(array(
            'items' => $cart_items,
            'subtotal' => WC()->cart->get_cart_subtotal(),
            'total' => WC()->cart->get_cart_total()
        ));
    }

    /**
     * Remove cart item AJAX handler (for the cart dropdown)
     */
    public function remove_cart_item() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'mxchat-woo')));
        }

        if (!function_exists('WC') || !isset(WC()->cart)) {
            wp_send_json_error(array('message' => __('WooCommerce cart not available.', 'mxchat-woo')));
        }

        $cart_item_key = isset($_POST['cart_key']) ? sanitize_text_field($_POST['cart_key']) : '';
        
        if (!$cart_item_key) {
            wp_send_json_error(array('message' => __('No item specified.', 'mxchat-woo')));
        }

        $removed = WC()->cart->remove_cart_item($cart_item_key);
        
        if ($removed) {
            WC()->cart->calculate_totals();
            wp_send_json_success(array(
                'message' => __('Item removed.', 'mxchat-woo'),
                'cart_count' => WC()->cart->get_cart_contents_count()
            ));
        } else {
            wp_send_json_error(array('message' => __('Error removing item.', 'mxchat-woo')));
        }
    }

    /**
     * Get formatted cart data
     */
    private function get_cart_data() {
        $cart = WC()->cart;
        $items = array();

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $product = $cart_item['data'];
            $items[] = array(
                'key' => $cart_item_key,
                'product_id' => $product->get_id(),
                'variation_id' => $cart_item['variation_id'],
                'variation' => $cart_item['variation'],
                'name' => $product->get_name(),
                'quantity' => $cart_item['quantity'],
                'price' => WC()->cart->get_product_price($product),
                'line_total' => WC()->cart->get_product_subtotal($product, $cart_item['quantity']),
                'permalink' => get_permalink($product->get_id())
            );
        }

        return array(
            'items' => $items,
            'item_count' => $cart->get_cart_contents_count(),
            'subtotal' => $cart->get_cart_subtotal(),
            'total' => $cart->get_cart_total(),
            'needs_shipping' => $cart->needs_shipping(),
            'cart_url' => wc_get_cart_url(),
            'checkout_url' => wc_get_checkout_url()
        );
    }

    /**
     * Check if cart has items
     */
    public function cart_has_items() {
        return WC()->cart && WC()->cart->get_cart_contents_count() > 0;
    }

    /**
     * Get cart URL
     */
    public function get_cart_url() {
        return wc_get_cart_url();
    }

    /**
     * Get checkout URL
     */
    public function get_checkout_url() {
        return wc_get_checkout_url();
    }
    
    /**
     * Handle checkout intent from chat
     */
public function mxchat_handle_checkout_intent($default, $message, $user_id, $session_id = null) {
    //error_log('🔍 MXCHAT DEBUG: Checkout Intent Triggered');
    //error_log("🔍 MXCHAT DEBUG: Message: '$message', User ID: $user_id, Session: $session_id");

    if (!class_exists('WooCommerce')) {
        //error_log('❌ MXCHAT DEBUG: WooCommerce not available');

        // Use AI for multilingual "checkout unavailable" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The checkout feature is currently unavailable because WooCommerce is not active.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and inform them that the checkout feature isn't available at the moment.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return [
            'text' => $ai_response['text'] ?? esc_html__("I apologize, but the checkout feature isn't available at the moment.", 'mxchat-woo'),
            'html' => ''
        ];
    }

    // Check if cart has items
    if (WC()->cart->is_empty()) {
        //error_log('❌ MXCHAT DEBUG: Cart is empty');

        // Use AI for multilingual "empty cart" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The user wants to checkout, but their shopping cart is currently empty.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Inform them their cart is empty and ask if they would like to see products.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return [
            'text' => $ai_response['text'] ?? esc_html__("Your cart is empty at the moment. Would you like to see our products?", 'mxchat-woo'),
            'html' => ''
        ];
    }

    // Get cart summary
    $cart_count = WC()->cart->get_cart_contents_count();
    $cart_total = WC()->cart->get_total();
    //error_log("🔍 MXCHAT DEBUG: Cart Summary - Items: $cart_count, Total: " . strip_tags($cart_total));

    // Get and validate checkout URL
    $checkout_url = wc_get_checkout_url();
    if (!$checkout_url) {
        //error_log('❌ MXCHAT DEBUG: Failed to get checkout URL');

        // Use AI for multilingual "checkout URL error" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: There's a technical issue - the checkout page URL cannot be accessed right now.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and ask them to try again in a moment.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return [
            'text' => $ai_response['text'] ?? esc_html__("I'm having trouble accessing the checkout page. Please try again in a moment.", 'mxchat-woo'),
            'html' => ''
        ];
    }

    //error_log("✅ MXCHAT DEBUG: Got checkout URL: $checkout_url");

    // Use AI for multilingual "redirecting to checkout" response
    $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
    $ai_prompt .= "Context: The user has {$cart_count} item" . ($cart_count > 1 ? 's' : '') . " in their cart totaling " . strip_tags($cart_total) . ". ";
    $ai_prompt .= "I am redirecting them to the checkout page now.\n\n";
    $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
    $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
    $ai_prompt .= "Task: Inform them about their cart contents and that they're being redirected to checkout.";

    $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

    $response_text = $ai_response['text'] ?? sprintf(
        esc_html__("You have %d item%s in your cart totaling %s. Redirecting you to checkout now...", 'mxchat-woo'),
        $cart_count,
        $cart_count > 1 ? esc_html__('s', 'mxchat-woo') : '',
        strip_tags($cart_total)
    );

    // Create JavaScript to redirect immediately
    $redirect_script = '<script>
        setTimeout(function() {
            window.location.href = "' . esc_url($checkout_url) . '";
        }, 1000); // 1 second delay to show the message
    </script>';

    //error_log("✅ MXCHAT DEBUG: Sending checkout response with immediate redirect");

    return [
        'text' => $response_text,
        'html' => $redirect_script
    ];
}

/**
 * Process markdown formatting
 */
private function process_markdown($text) {
    // Process markdown links [text](url) first
    $text = preg_replace('/\[([^\]]+)\]\(([^)]+)\)/', '<a href="$2" target="_blank">$1</a>', $text);
    
    // Process bold and italic
    $text = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $text);
    $text = preg_replace('/\*(.*?)\*/', '<em>$1</em>', $text);
    
    // Process standalone URLs to make them clickable (but avoid double-processing already linked URLs)
    $text = preg_replace('/(?<!href=")(?<!href=\')(?<!>)(https?:\/\/[^\s<>"\']+)/i', '<a href="$1" target="_blank">$1</a>', $text);
    
    // Don't process newlines here - let the main chat handler do it
    return $text;
}
public function mxchat_handle_add_to_cart_intent($default, $message, $user_id, $session_id) {
    error_log("🛒 MXCHAT DEBUG: ADD TO CART START - Message: '$message', User ID: $user_id");

    if (!class_exists('WooCommerce')) {
        error_log("❌ MXCHAT DEBUG: WooCommerce not available");

        // Use AI for multilingual "WooCommerce unavailable" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The store system (WooCommerce) is currently unavailable, so I cannot add items to the cart.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and inform them that the cart functionality is temporarily unavailable.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return array(
            'text' => $ai_response['text'] ?? esc_html__("I can't add items to your cart right now. The store system seems to be unavailable.", 'mxchat-woo'),
            'html' => ''
        );
    }

    $sanitized_user_id = sanitize_key($user_id);
    $product_id = null;

    // First, always check for the last discussed product
    $product_id = get_transient('mxchat_last_discussed_product_' . $sanitized_user_id);
    error_log("🔍 MXCHAT DEBUG: Last discussed product ID from transient: " . ($product_id ?: 'none'));

    // Check if this is a generic "add to cart" message
    $message_lower = strtolower(trim($message));
    $generic_cart_phrases = ['add to cart', 'add it to cart', 'add this to cart', 'cart', 'add it', 'add this', 'add that'];
    $is_generic_add = false;
    
    foreach ($generic_cart_phrases as $phrase) {
        if ($message_lower === $phrase || $message_lower === $phrase . '.' || $message_lower === $phrase . '!') {
            $is_generic_add = true;
            break;
        }
    }

    // Only search for a different product if:
    // 1. We don't have a product_id from transient, OR
    // 2. The message is NOT generic (contains specific product name)
    if (!$product_id || (!$is_generic_add && strlen($message_lower) > 15)) {
        error_log("🔍 MXCHAT DEBUG: Message appears to contain product name, searching...");
        
        // Clean the message by removing common add-to-cart phrases
        $search_query = str_ireplace(['add', 'to', 'cart', 'please', 'can you', 'i want', 'buy', 'purchase', '?', '!', '.', ','], '', $message);
        $search_query = trim(preg_replace('/\s+/', ' ', $search_query));
        
        if (!empty($search_query) && strlen($search_query) > 2) {
            error_log("🔍 MXCHAT DEBUG: Cleaned search query: '$search_query'");
            
            // Search for products
            $args = array(
                'post_type'      => 'product',
                'posts_per_page' => 5,
                's'              => $search_query,
                'post_status'    => 'publish'
            );
            
            $query = new WP_Query($args);
            
            if ($query->have_posts()) {
                $best_match = null;
                $best_score = 0;
                
                while ($query->have_posts()) {
                    $query->the_post();
                    $title = get_the_title();
                    $id = get_the_ID();
                    
                    // Calculate relevance score
                    similar_text(strtolower($search_query), strtolower($title), $score);
                    
                    error_log("🔍 MXCHAT DEBUG: Found product: '$title' (ID: $id) with score: $score");
                    
                    if ($score > $best_score) {
                        $best_score = $score;
                        $best_match = $id;
                    }
                }
                
                wp_reset_postdata();
                
                // Only override the transient product if we have a very good match
                // This prevents replacing the correct product with a wrong one
                if ($best_score > 50) {
                    $product_id = $best_match;
                    error_log("✅ MXCHAT DEBUG: Found specific product match with score: $best_score");
                }
            }
        }
    } else if ($is_generic_add && $product_id) {
        error_log("✅ MXCHAT DEBUG: Using last discussed product for generic 'add to cart' request");
    }

    // Validate the product
    if (!$product_id) {
        error_log("❌ MXCHAT DEBUG: No product ID found");

        // Use AI for multilingual "product not found" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The user wants to add a product to their cart, but I couldn't determine which specific product they're referring to.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Politely ask them to specify the product name or discuss a specific product first.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return array(
            'text' => $ai_response['text'] ?? esc_html__("I'm not sure which product you want to add to your cart. Could you please specify the product name, or discuss a specific product first?", 'mxchat-woo'),
            'html' => ''
        );
    }

    $product = wc_get_product($product_id);
    if (!$product || !$product->is_purchasable()) {
        error_log("❌ MXCHAT DEBUG: Product not found or not purchasable: $product_id");

        // Use AI for multilingual "product not available" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The product they're trying to add is not available for purchase at the moment.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and inform them that the product is not available for purchase at the moment.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        return array(
            'text' => $ai_response['text'] ?? esc_html__("Sorry, that product is not available for purchase at the moment.", 'mxchat-woo'),
            'html' => ''
        );
    }

    // Check if product is already in cart
    $cart = WC()->cart->get_cart();
    $already_in_cart = false;
    
    foreach ($cart as $cart_item) {
        if ($cart_item['product_id'] == $product_id) {
            $already_in_cart = true;
            break;
        }
    }
    
    if ($already_in_cart) {
        error_log("⚠️ MXCHAT DEBUG: Product already in cart: " . $product->get_name());
        
        // Use AI for multilingual response about already in cart
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: The product '{$product->get_name()}' is already in their shopping cart.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Politely inform them the product is already in their cart and ask if they'd like to continue shopping or checkout.";
        
        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
        
        return array(
            'text' => $ai_response['text'] ?? sprintf(__("%s is already in your cart.", 'mxchat-woo'), $product->get_name()),
            'html' => ''
        );
    }

    // Attempt to add the product to the cart
    $added = WC()->cart->add_to_cart($product_id);
    
    // Make sure the cart is saved to the session
    WC()->cart->set_session();
    
    if ($added) {
        error_log("✅ MXCHAT DEBUG: Successfully added to cart: " . $product->get_name());
        
        // Save this product as the last discussed product
        set_transient('mxchat_last_discussed_product_' . $sanitized_user_id, $product_id, HOUR_IN_SECONDS);

        // Build AI prompt with correct product name and price
        $product_name = $product->get_name();
        $product_price = strip_tags($product->get_price_html());
        $cart_count = WC()->cart->get_cart_contents_count();

        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: I successfully added '{$product_name}' (price: {$product_price}) to their cart. ";
        $ai_prompt .= "They now have {$cart_count} item" . ($cart_count > 1 ? 's' : '') . " in their cart.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Confirm the addition in a friendly, conversational tone. ";
        $ai_prompt .= "Be specific about the product name '{$product_name}' - do not mention any other products.";

        // Call the AI API to generate a multilingual response
        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
        error_log("🔍 MXCHAT: AI response received for product: " . $product_name);

        // Process the markdown before returning
        $processed_text = $this->process_markdown($ai_response['text']);

        // Add JavaScript to trigger cart update events
        $script = '<script>
            jQuery(document).ready(function($) {
                // Trigger WooCommerce cart update events
                $(document.body).trigger("added_to_cart");
                $(document.body).trigger("wc_fragment_refresh");
                
                // Update the cart count directly
                if ($("#woo-cart-count").length) {
                    $("#woo-cart-count").text(' . $cart_count . ').removeClass("empty-cart");
                }
            });
        </script>';

        return array(
            'text' => $processed_text,
            'html' => $script
        );
    } else {
        error_log("❌ MXCHAT DEBUG: Failed to add to cart: " . $product->get_name());
        
        // Use AI for error message too
        $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
        $ai_prompt .= "Context: They tried to add '{$product->get_name()}' to cart but it failed.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and suggest they try again or contact support.";
        
        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
        
        return array(
            'text' => $ai_response['text'] ?? sprintf(__("Sorry, I couldn't add %s to your cart.", 'mxchat-woo'), $product->get_name()),
            'html' => ''
        );
    }
}


}