<?php
/**
 * Handles WooCommerce customer operations for MxChat
 */
class MxChat_Woo_Customer_Handler {

    private $loader;
    private $options;

    public function __construct($loader) {
        $this->loader = $loader;
        $this->options = get_option('mxchat_options', array());
        $this->init();
    }

    private function init() {
        $this->loader->add_action('wp_ajax_mxchat_get_customer_data', $this, 'get_customer_data');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_get_customer_data', $this, 'get_customer_data');
        
        $this->loader->add_action('wp_ajax_mxchat_update_customer_data', $this, 'update_customer_data');
        
        $this->loader->add_action('wp_ajax_mxchat_get_saved_addresses', $this, 'get_saved_addresses');
        
        $this->loader->add_action('woocommerce_created_customer', $this, 'track_new_customer', 10, 1);
        $this->loader->add_action('woocommerce_update_customer', $this, 'track_customer_update', 10, 1);
    }

    public function get_customer_data() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $customer_id = $this->get_current_customer_id();
        if (!$customer_id) {
            wp_send_json_error('Customer not found.');
        }

        $customer = new WC_Customer($customer_id);
        wp_send_json_success($this->format_customer_data($customer));
    }

    public function update_customer_data() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        if (!is_user_logged_in()) {
            wp_send_json_error('User not logged in.');
        }

        $customer_id = get_current_user_id();
        $customer = new WC_Customer($customer_id);

        $fields = array(
            'billing_first_name', 'billing_last_name', 'billing_company',
            'billing_address_1', 'billing_address_2', 'billing_city',
            'billing_state', 'billing_postcode', 'billing_country',
            'billing_phone', 'billing_email',
            'shipping_first_name', 'shipping_last_name', 'shipping_company',
            'shipping_address_1', 'shipping_address_2', 'shipping_city',
            'shipping_state', 'shipping_postcode', 'shipping_country'
        );

        foreach ($fields as $field) {
            if (isset($_POST[$field])) {
                $setter = "set_" . $field;
                if (method_exists($customer, $setter)) {
                    $customer->$setter(sanitize_text_field($_POST[$field]));
                }
            }
        }

        $customer->save();
        wp_send_json_success($this->format_customer_data($customer));
    }

    public function get_saved_addresses() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $customer_id = $this->get_current_customer_id();
        if (!$customer_id) {
            wp_send_json_error('Customer not found.');
        }

        $customer = new WC_Customer($customer_id);
        
        $addresses = array(
            'billing' => $this->format_address($customer, 'billing'),
            'shipping' => $this->format_address($customer, 'shipping')
        );

        wp_send_json_success($addresses);
    }

    private function format_customer_data($customer) {
        return array(
            'id' => $customer->get_id(),
            'first_name' => $customer->get_first_name(),
            'last_name' => $customer->get_last_name(),
            'email' => $customer->get_email(),
            'role' => $customer->get_role(),
            'date_created' => $customer->get_date_created() ? $customer->get_date_created()->date('Y-m-d H:i:s') : '',
            'date_modified' => $customer->get_date_modified() ? $customer->get_date_modified()->date('Y-m-d H:i:s') : '',
            'billing' => $this->format_address($customer, 'billing'),
            'shipping' => $this->format_address($customer, 'shipping'),
            'is_paying_customer' => $customer->get_is_paying_customer(),
            'order_count' => $customer->get_order_count(),
            'total_spent' => $customer->get_total_spent(),
            'avatar_url' => get_avatar_url($customer->get_id()),
            'last_order' => $this->get_last_order_data($customer->get_id())
        );
    }

    private function format_address($customer, $type = 'billing') {
        $address = array();
        $fields = array(
            'first_name', 'last_name', 'company', 'address_1', 
            'address_2', 'city', 'state', 'postcode', 'country',
            'email', 'phone'
        );

        foreach ($fields as $field) {
            $getter = "get_{$type}_{$field}";
            if (method_exists($customer, $getter)) {
                $address[$field] = $customer->$getter();
            }
        }

        return $address;
    }

    private function get_last_order_data($customer_id) {
        $args = array(
            'customer' => $customer_id,
            'limit' => 1,
            'orderby' => 'date',
            'order' => 'DESC',
        );

        $orders = wc_get_orders($args);
        if (!empty($orders)) {
            $order = $orders[0];
            return array(
                'id' => $order->get_id(),
                'status' => $order->get_status(),
                'total' => $order->get_total(),
                'date_created' => $order->get_date_created()->date('Y-m-d H:i:s')
            );
        }
        return null;
    }

    private function get_current_customer_id() {
        $user_id = get_current_user_id();
        if ($user_id) {
            return $user_id;
        }

        // Handle guest users with session
        if (isset(WC()->session)) {
            return WC()->session->get_customer_id();
        }

        return false;
    }

    public function track_new_customer($customer_id) {
        $this->log_customer_event($customer_id, 'created');
    }

    public function track_customer_update($customer_id) {
        $this->log_customer_event($customer_id, 'updated');
    }

    private function log_customer_event($customer_id, $event_type) {
        if (!$customer_id) return;

        $customer = new WC_Customer($customer_id);
        
        $event_data = array(
            'customer_id' => $customer_id,
            'event_type' => $event_type,
            'timestamp' => current_time('mysql'),
            'email' => $customer->get_email()
        );

        do_action('mxchat_customer_event', $event_data);
    }

    public function get_customer_purchase_history($customer_id = null) {
        if (!$customer_id) {
            $customer_id = $this->get_current_customer_id();
        }

        if (!$customer_id) {
            return array();
        }

        $customer = new WC_Customer($customer_id);
        return array(
            'order_count' => $customer->get_order_count(),
            'total_spent' => $customer->get_total_spent(),
            'avg_order_value' => $customer->get_order_count() ? 
                ($customer->get_total_spent() / $customer->get_order_count()) : 0,
            'first_order_date' => $this->get_first_order_date($customer_id),
            'last_order_date' => $this->get_last_order_date($customer_id)
        );
    }

    private function get_first_order_date($customer_id) {
        $args = array(
            'customer' => $customer_id,
            'limit' => 1,
            'orderby' => 'date',
            'order' => 'ASC',
        );

        $orders = wc_get_orders($args);
        return !empty($orders) ? $orders[0]->get_date_created()->date('Y-m-d H:i:s') : null;
    }

    private function get_last_order_date($customer_id) {
        $args = array(
            'customer' => $customer_id,
            'limit' => 1,
            'orderby' => 'date',
            'order' => 'DESC',
        );

        $orders = wc_get_orders($args);
        return !empty($orders) ? $orders[0]->get_date_created()->date('Y-m-d H:i:s') : null;
    }
}