<?php
/**
 * Handles filtered search operations for WooCommerce products in MxChat
 * 
 * This class implements a hybrid search approach that combines:
 * - Vector-based trigger detection (cosine similarity)
 * - LLM-based filter extraction (price, category, attributes)
 * - Structured SQL queries with extracted filters
 * - Optional semantic re-ranking of results
 * 
 * @package MxChat_Woo
 * @since 1.2.0
 */
class MxChat_Woo_Filtered_Search_Handler {
    private $options;
    private $loader;
    private $api_key;
    private $trigger_phrases;
    private $trigger_embeddings;
    private $similarity_threshold;

    public function __construct($loader, $options = array()) {
        $this->loader = $loader;
        
        // Get the core plugin options
        $core_options = get_option('mxchat_options', array());
        
        // Get WooCommerce specific options
        $this->options = get_option('mxchat_woo_options', array());
        
        // Get API key from core options
        $this->api_key = isset($core_options['api_key']) ? $core_options['api_key'] : '';
        
        // Initialize filtered search settings
        $this->init_filtered_search_settings();
        
        // Only initialize if loader is valid
        if ($this->loader && method_exists($this->loader, 'add_action')) {
            $this->init();
        }
    }

    private function init() {
        // Add AJAX handlers for admin configuration
        $this->loader->add_action('wp_ajax_mxchat_save_filtered_search_settings', $this, 'save_filtered_search_settings');
        $this->loader->add_action('wp_ajax_mxchat_test_filtered_search', $this, 'test_filtered_search');
    }

    /**
     * Initialize filtered search settings with defaults
     */
    private function init_filtered_search_settings() {
        // Get saved settings or use defaults
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        
        // Default trigger phrases for filtered search
        $default_triggers = array(
            'under $',
            'less than',
            'cheaper than',
            'price range',
            'between $ and $',
            'budget of',
            'within my budget',
            'affordable',
            'in the category',
            'from category',
            'filter by',
            'show me products',
        );
        
        $this->trigger_phrases = isset($settings['trigger_phrases']) && !empty($settings['trigger_phrases']) 
            ? $settings['trigger_phrases'] 
            : $default_triggers;
        
        // Default similarity threshold (0.7 = 70%)
        $this->similarity_threshold = isset($settings['similarity_threshold']) 
            ? (float) $settings['similarity_threshold'] 
            : 0.70;
        
        // Generate embeddings for trigger phrases if not already cached
        $this->trigger_embeddings = $this->get_or_generate_trigger_embeddings();
    }

    /**
     * Get or generate embeddings for trigger phrases
     * 
     * @return array Array of embeddings for each trigger phrase
     */
    private function get_or_generate_trigger_embeddings() {
        $cache_key = 'mxchat_woo_filtered_search_trigger_embeddings_' . md5(serialize($this->trigger_phrases));
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $embeddings = array();
        foreach ($this->trigger_phrases as $phrase) {
            $embedding = MxChat_Woo_Util::generate_embedding($phrase, $this->api_key);
            if ($embedding) {
                $embeddings[] = array(
                    'phrase' => $phrase,
                    'embedding' => $embedding
                );
            }
        }
        
        // Cache for 24 hours
        set_transient($cache_key, $embeddings, DAY_IN_SECONDS);
        
        return $embeddings;
    }

    /**
     * Check if user query should trigger filtered search
     * 
     * @param string $query User's query
     * @return array|false Array with trigger info or false if no match
     */
    private function should_trigger_filtered_search($query) {
        if (empty($this->api_key) || empty($this->trigger_embeddings)) {
            return false;
        }
        
        // Generate embedding for user query
        $query_embedding = MxChat_Woo_Util::generate_embedding($query, $this->api_key);
        if (!$query_embedding) {
            return false;
        }
        
        // Compare with trigger phrase embeddings
        $best_match = null;
        $highest_similarity = 0;
        
        foreach ($this->trigger_embeddings as $trigger) {
            $similarity = MxChat_Woo_Util::mxchat_calculate_cosine_similarity(
                $query_embedding, 
                $trigger['embedding']
            );
            
            if ($similarity > $highest_similarity) {
                $highest_similarity = $similarity;
                $best_match = $trigger['phrase'];
            }
        }
        
        // Check if best match exceeds threshold
        if ($highest_similarity >= $this->similarity_threshold) {
            return array(
                'triggered' => true,
                'matched_phrase' => $best_match,
                'similarity' => $highest_similarity,
                'query_embedding' => $query_embedding
            );
        }
        
        return false;
    }

    /**
     * Extract filters from user query using LLM
     * 
     * @param string $query User's query
     * @return array Extracted filters
     */
    private function extract_filters_from_query($query) {
        $prompt = "You are a product search filter extraction system. Analyze the user's query and extract structured filters.\n\n";
        $prompt .= "User Query: \"{$query}\"\n\n";
        $prompt .= "Extract the following information if present:\n";
        $prompt .= "1. Price filters (min_price, max_price) - extract numeric values only\n";
        $prompt .= "2. Category - extract category name if mentioned\n";
        $prompt .= "3. Product attributes - extract any mentioned attributes (color, size, brand, etc.)\n";
        $prompt .= "4. Search terms - general keywords to search for\n\n";
        $prompt .= "Respond ONLY with a valid JSON object in this exact format:\n";
        $prompt .= "{\n";
        $prompt .= "  \"price_min\": null,\n";
        $prompt .= "  \"price_max\": 50.00,\n";
        $prompt .= "  \"category\": \"electronics\",\n";
        $prompt .= "  \"attributes\": {\"color\": \"red\", \"size\": \"large\"},\n";
        $prompt .= "  \"search_terms\": \"laptop\"\n";
        $prompt .= "}\n\n";
        $prompt .= "Use null for any filter not mentioned. Extract only what is explicitly stated.";
        
        $response = MxChat_Woo_Util::mxchat_call_ai_api($prompt, $this->api_key);
        
        if (empty($response['text'])) {
            error_log('MxChat Filtered Search: Empty response from AI API');
            return array();
        }
        
        // Clean up response - extract JSON from markdown code blocks if present
        $text = $response['text'];
        if (preg_match('/```(?:json)?\s*(\{.*?\})\s*```/s', $text, $matches)) {
            $text = $matches[1];
        }
        
        error_log('MxChat Filtered Search: Raw AI response for filter extraction: ' . $text);
        
        // Try to parse JSON
        $filters = json_decode($text, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('MxChat Filtered Search: Failed to parse filters JSON - ' . json_last_error_msg());
            error_log('MxChat Filtered Search: Raw text was: ' . $text);
            return array();
        }
        
        return $filters;
    }

    /**
     * Execute filtered WooCommerce product query
     * 
     * @param array $filters Extracted filters
     * @return array Array of WC_Product objects
     */
    private function execute_filtered_query($filters) {
        $args = array(
            'status' => 'publish',
            'limit' => -1, // Get all products to filter manually
            'orderby' => 'date',
            'order' => 'DESC',
            'return' => 'objects'
        );
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Starting query execution');
        error_log('🔍 MXCHAT FILTERED SEARCH: Initial args: ' . json_encode($args));
        
        // We'll handle price filtering manually after the query
        $price_min = isset($filters['price_min']) && $filters['price_min'] !== null ? floatval($filters['price_min']) : null;
        $price_max = isset($filters['price_max']) && $filters['price_max'] !== null ? floatval($filters['price_max']) : null;
        
        if ($price_min !== null) {
            error_log('🔍 MXCHAT FILTERED SEARCH: Will filter for price_min: ' . $price_min);
        }
        if ($price_max !== null) {
            error_log('🔍 MXCHAT FILTERED SEARCH: Will filter for price_max: ' . $price_max);
        }
        
        // Apply category filter
        if (isset($filters['category']) && !empty($filters['category'])) {
            // Try to find category by slug or name
            $category = get_term_by('slug', sanitize_title($filters['category']), 'product_cat');
            if (!$category) {
                $category = get_term_by('name', $filters['category'], 'product_cat');
            }
            
            if ($category) {
                $args['category'] = array($category->slug);
                error_log('🔍 MXCHAT FILTERED SEARCH: Added category filter: ' . $category->slug);
            } else {
                error_log('⚠️ MXCHAT FILTERED SEARCH: Category not found: ' . $filters['category']);
            }
        }
        
        // Apply search terms (but ignore generic terms that would match everything or nothing)
        if (isset($filters['search_terms']) && !empty($filters['search_terms'])) {
            $search_term = sanitize_text_field($filters['search_terms']);
            // Ignore overly generic search terms that don't add value
            $generic_terms = array('products', 'product', 'items', 'item', 'things', 'stuff', 'goods', 'merchandise');
            if (!in_array(strtolower(trim($search_term)), $generic_terms)) {
                $args['s'] = $search_term;
                error_log('🔍 MXCHAT FILTERED SEARCH: Added search terms: ' . $search_term);
            } else {
                error_log('🔍 MXCHAT FILTERED SEARCH: Ignoring generic search term: ' . $search_term);
            }
        }
        
        // Apply attribute filters
        if (isset($filters['attributes']) && is_array($filters['attributes'])) {
            foreach ($filters['attributes'] as $attr_name => $attr_value) {
                $taxonomy = 'pa_' . sanitize_title($attr_name);
                if (taxonomy_exists($taxonomy)) {
                    $term = get_term_by('slug', sanitize_title($attr_value), $taxonomy);
                    if (!$term) {
                        $term = get_term_by('name', $attr_value, $taxonomy);
                    }
                    
                    if ($term) {
                        $args['tax_query'][] = array(
                            'taxonomy' => $taxonomy,
                            'field' => 'term_id',
                            'terms' => $term->term_id
                        );
                        error_log('🔍 MXCHAT FILTERED SEARCH: Added attribute filter: ' . $attr_name . ' = ' . $attr_value);
                    }
                }
            }
        }
        
        // Set relation for tax_query if multiple conditions
        if (isset($args['tax_query']) && count($args['tax_query']) > 1) {
            $args['tax_query']['relation'] = 'AND';
        }
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Final query args: ' . json_encode($args));
        
        // Execute query
        $products = wc_get_products($args);
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Query returned ' . count($products) . ' products before price filtering');
        
        // NOW filter by price manually
        $filtered_products = array();
        foreach ($products as $product) {
            $product_price = (float) $product->get_price();
            
            // Skip products with no price
            if (empty($product_price) && $product_price !== 0.0) {
                error_log('⚠️ MXCHAT FILTERED SEARCH: Skipping product with no price: ' . $product->get_name());
                continue;
            }
            
            // Check price_min
            if ($price_min !== null && $product_price < $price_min) {
                error_log('⚠️ MXCHAT FILTERED SEARCH: Skipping ' . $product->get_name() . ' ($' . $product_price . ') - below min price');
                continue;
            }
            
            // Check price_max
            if ($price_max !== null && $product_price > $price_max) {
                error_log('⚠️ MXCHAT FILTERED SEARCH: Skipping ' . $product->get_name() . ' ($' . $product_price . ') - above max price');
                continue;
            }
            
            $filtered_products[] = $product;
        }
        
        error_log('🔍 MXCHAT FILTERED SEARCH: After price filtering: ' . count($filtered_products) . ' products remain');
        
        // Limit to 20 products
        $filtered_products = array_slice($filtered_products, 0, 20);
        
        // Log the first few products for debugging
        $product_summary = array();
        foreach (array_slice($filtered_products, 0, 5) as $product) {
            $product_summary[] = $product->get_name() . ' ($' . $product->get_price() . ')';
        }
        if (!empty($product_summary)) {
            error_log('🔍 MXCHAT FILTERED SEARCH: Sample products: ' . implode(', ', $product_summary));
        }
        
        return $filtered_products;
    }

    /**
     * Optionally re-rank products using semantic similarity
     * 
     * @param array $products Array of WC_Product objects
     * @param array $query_embedding User query embedding
     * @param string $original_query Original user query
     * @return array Re-ranked array of products
     */
    private function semantic_rerank_products($products, $query_embedding, $original_query) {
        $settings = get_option('mxchat_woo_filtered_search_settings', array());
        $enable_rerank = isset($settings['enable_semantic_rerank']) && $settings['enable_semantic_rerank'] === 'on';
        
        if (!$enable_rerank || empty($query_embedding)) {
            error_log('🔍 MXCHAT FILTERED SEARCH: Semantic reranking is disabled or no query embedding');
            return $products;
        }
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Starting semantic reranking');
        
        $scored_products = array();
        
        foreach ($products as $product) {
            // Create text representation of product for embedding
            $product_text = $product->get_name() . ' ' . 
                          $product->get_short_description() . ' ' . 
                          strip_tags($product->get_description());
            
            // Generate embedding for product
            $product_embedding = MxChat_Woo_Util::generate_embedding($product_text, $this->api_key);
            
            if ($product_embedding) {
                $similarity = MxChat_Woo_Util::mxchat_calculate_cosine_similarity(
                    $query_embedding, 
                    $product_embedding
                );
                
                $scored_products[] = array(
                    'product' => $product,
                    'score' => $similarity
                );
            } else {
                // If embedding fails, add with neutral score
                $scored_products[] = array(
                    'product' => $product,
                    'score' => 0.5
                );
            }
        }
        
        // Sort by semantic similarity score
        usort($scored_products, function($a, $b) {
            return $b['score'] <=> $a['score'];
        });
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Reranking complete');
        
        // Return only the product objects
        return array_map(function($item) {
            return $item['product'];
        }, $scored_products);
    }

    /**
     * Format filtered search results for display
     * 
     * @param array $products Array of WC_Product objects
     * @param array $filters Applied filters
     * @param string $original_query Original user query
     * @return array Response with text and HTML
     */
    private function format_filtered_results($products, $filters, $original_query) {
        if (empty($products)) {
            error_log('🔍 MXCHAT FILTERED SEARCH: No products found, generating no-results response');
            
            // Generate AI response for no results
            $ai_prompt = "User searched for: '{$original_query}'\n\n";
            $ai_prompt .= "No products matched their search criteria.\n\n";
            $ai_prompt .= "CRITICAL: Respond in the SAME language as the user's query.\n\n";
            $ai_prompt .= "Task: Apologize and suggest they try:\n";
            $ai_prompt .= "- Adjusting their price range\n";
            $ai_prompt .= "- Trying different keywords\n";
            $ai_prompt .= "- Browsing all products\n";
            $ai_prompt .= "Keep response brief and friendly (2-3 sentences).";
            
            $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
            
            return array(
                'text' => $ai_response['text'] ?? __("I couldn't find any products matching your criteria. Try adjusting your search!", 'mxchat-woo'),
                'html' => ''
            );
        }
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Formatting results for ' . count($products) . ' products');
        
        // Limit to top 6 products
        $products = array_slice($products, 0, 6);
        
        // Build filter summary
        $filter_summary = array();
        if (isset($filters['price_min']) && $filters['price_min'] !== null) {
            $filter_summary[] = sprintf(__('Min price: $%s', 'mxchat-woo'), number_format($filters['price_min'], 2));
        }
        if (isset($filters['price_max']) && $filters['price_max'] !== null) {
            $filter_summary[] = sprintf(__('Max price: $%s', 'mxchat-woo'), number_format($filters['price_max'], 2));
        }
        if (isset($filters['category']) && !empty($filters['category'])) {
            $filter_summary[] = sprintf(__('Category: %s', 'mxchat-woo'), $filters['category']);
        }
        
        // Build product list with pricing information for AI
        $products_info = array();
        foreach ($products as $product) {
            $price = $product->get_price();
            $products_info[] = sprintf(
                "- %s (Price: $%s)",
                $product->get_name(),
                number_format((float)$price, 2)
            );
        }
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Products being sent to AI: ' . implode(' | ', $products_info));
        
        // Generate AI introduction WITH product details
        $ai_prompt = "User searched for: '{$original_query}'\n\n";
        $ai_prompt .= "Applied filters:\n";
        if (!empty($filter_summary)) {
            $ai_prompt .= implode("\n", $filter_summary) . "\n";
        }
        $ai_prompt .= "\nFound " . count($products) . " matching products:\n";
        $ai_prompt .= implode("\n", $products_info) . "\n\n";
        $ai_prompt .= "CRITICAL: Respond in the SAME language as the user's query.\n\n";
        $ai_prompt .= "Task: Write a brief introduction (1-2 sentences) acknowledging their search and confirming you found products that match their criteria. ";
        $ai_prompt .= "You can reference the price range of the products found if relevant to their query. Be conversational and helpful.";
        
        error_log('🔍 MXCHAT FILTERED SEARCH: Sending prompt to AI for response generation');
        
        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
        
        $text_response = $ai_response['text'] ?? sprintf(
            __("I found %d products matching your search!", 'mxchat-woo'),
            count($products)
        );
        
        error_log('🔍 MXCHAT FILTERED SEARCH: AI generated response: ' . substr($text_response, 0, 100) . '...');
        
        // Build HTML for product grid
        $html = '<div class="mxchat-filtered-products">';
        
        // Add filter summary
        if (!empty($filter_summary)) {
            $html .= '<div class="mxchat-filter-summary">';
            $html .= '<strong>' . __('Filters Applied:', 'mxchat-woo') . '</strong> ';
            $html .= implode(' | ', array_map('esc_html', $filter_summary));
            $html .= '</div>';
        }
        
        $html .= '<div class="mxchat-product-grid">';
        
        foreach ($products as $product) {
            $html .= $this->build_product_card($product);
        }
        
        $html .= '</div></div>';
        
        // Add inline styles with high specificity to override theme defaults (no !important for user overridability)
        $html .= '<style>
            div.mxchat-filtered-products.mxchat-filtered-products {
                margin: 15px 0;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
                line-height: 1.5;
            }
            div.mxchat-filtered-products div.mxchat-filter-summary {
                background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                padding: 12px 15px;
                border-radius: 8px;
                margin-bottom: 15px;
                font-size: 0.85em;
                color: #495057;
                border: 1px solid #dee2e6;
            }
            div.mxchat-filtered-products div.mxchat-filter-summary strong {
                color: #212529;
            }
            div.mxchat-filtered-products div.mxchat-product-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(160px, 1fr));
                gap: 12px;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card {
                border: 1px solid #e9ecef;
                border-radius: 12px;
                padding: 12px;
                text-align: center;
                transition: all 0.2s ease;
                background: #ffffff;
                box-shadow: 0 2px 4px rgba(0,0,0,0.04);
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card:hover {
                box-shadow: 0 8px 16px rgba(0,0,0,0.1);
                transform: translateY(-2px);
                border-color: #dee2e6;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href] {
                text-decoration: none;
                color: inherit;
                border: none;
                border-bottom: none;
                box-shadow: none;
                outline: none;
                background-image: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a:hover,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a:focus,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a:visited,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a:active,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href]:hover,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href]:focus,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href]:visited,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href]:active {
                text-decoration: none;
                color: inherit;
                border: none;
                border-bottom: none;
                box-shadow: none;
                background-image: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card img {
                max-width: 100%;
                width: 100%;
                height: 120px;
                object-fit: cover;
                border-radius: 8px;
                margin-bottom: 10px;
                background: #f8f9fa;
                border: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card h4,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a[href] h4,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a h4 {
                font-size: 0.85em;
                font-weight: 600;
                margin: 8px 0;
                color: #212529;
                line-height: 1.3;
                min-height: 2.6em;
                display: -webkit-box;
                -webkit-line-clamp: 2;
                -webkit-box-orient: vertical;
                overflow: hidden;
                text-decoration: none;
                border-bottom: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card div.price {
                font-weight: 700;
                font-size: 1em;
                color: #212529;
                margin: 8px 0;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card div.price del {
                color: #6c757d;
                font-weight: 400;
                font-size: 0.85em;
                margin-right: 5px;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card div.price ins {
                text-decoration: none;
                background: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button[href] {
                display: inline-block;
                padding: 8px 16px;
                background: linear-gradient(135deg, #4a6cf7 0%, #3b5ae5 100%);
                color: #ffffff;
                text-decoration: none;
                border-radius: 6px;
                font-size: 0.8em;
                font-weight: 500;
                margin-top: 8px;
                border: none;
                border-bottom: none;
                box-shadow: 0 2px 4px rgba(74, 108, 247, 0.25);
                transition: all 0.2s ease;
                cursor: pointer;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button:visited,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button[href]:visited {
                color: #ffffff;
                text-decoration: none;
                border-bottom: none;
            }
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button:hover,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button:focus,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button:active,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button[href]:hover,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button[href]:focus,
            div.mxchat-filtered-products div.mxchat-product-grid div.mxchat-filtered-product-card a.view-button[href]:active {
                background: linear-gradient(135deg, #3b5ae5 0%, #2c4ad4 100%);
                color: #ffffff;
                text-decoration: none;
                border-bottom: none;
                box-shadow: 0 4px 8px rgba(74, 108, 247, 0.35);
                transform: translateY(-1px);
            }
        </style>';
        
        return array(
            'text' => $text_response,
            'html' => $html
        );
    }

    /**
     * Build HTML card for a single product
     * 
     * @param WC_Product $product
     * @return string HTML
     */
    private function build_product_card($product) {
        $name = esc_html($product->get_name());
        $price = $product->get_price_html();
        $url = esc_url(get_permalink($product->get_id()));
        $image = wp_get_attachment_url($product->get_image_id());
        $image = $image ? esc_url($image) : wc_placeholder_img_src();
        
        $html = '<div class="mxchat-filtered-product-card">';
        $html .= '<a href="' . $url . '" target="_blank">';
        $html .= '<img src="' . $image . '" alt="' . $name . '" />';
        $html .= '<h4>' . $name . '</h4>';
        $html .= '</a>';
        $html .= '<div class="price">' . $price . '</div>';
        $html .= '<a href="' . $url . '" target="_blank" class="view-button">' . __('View Product', 'mxchat-woo') . '</a>';
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Main handler for filtered search intent
     * 
     * @param mixed $default Default response
     * @param string $message User message
     * @param int $user_id User ID
     * @param string $session_id Session ID
     * @return array Response array with text and html
     */
    public function mxchat_handle_filtered_search($default, $message, $user_id, $session_id) {
        try {
            error_log('🔍 MXCHAT FILTERED SEARCH: Starting filtered search handler');
            error_log('🔍 MXCHAT FILTERED SEARCH: Message: ' . $message);

            // Note: Trigger detection is already handled by the core plugin's intent system.
            // This handler is only called when the intent has already matched.
            // We skip the redundant trigger check and go straight to filter extraction.

            // Generate query embedding for potential semantic reranking later
            $query_embedding = null;
            $settings = get_option('mxchat_woo_filtered_search_settings', array());
            $enable_rerank = isset($settings['enable_semantic_rerank']) && $settings['enable_semantic_rerank'] === 'on';

            if ($enable_rerank && !empty($this->api_key)) {
                $query_embedding = MxChat_Woo_Util::generate_embedding($message, $this->api_key);
            }

            // Extract filters from query
            $filters = $this->extract_filters_from_query($message);
            error_log('🔍 MXCHAT FILTERED SEARCH: Extracted filters: ' . json_encode($filters));
            
            if (empty($filters)) {
                error_log('⚠️ MXCHAT FILTERED SEARCH: No filters extracted, falling back to default');
                return $default;
            }
            
            // Check if at least one valid filter was extracted
            $has_valid_filter = false;
            if ((isset($filters['price_min']) && $filters['price_min'] !== null) ||
                (isset($filters['price_max']) && $filters['price_max'] !== null) ||
                (isset($filters['category']) && !empty($filters['category'])) ||
                (isset($filters['search_terms']) && !empty($filters['search_terms'])) ||
                (isset($filters['attributes']) && !empty($filters['attributes']))) {
                $has_valid_filter = true;
            }
            
            if (!$has_valid_filter) {
                error_log('⚠️ MXCHAT FILTERED SEARCH: No valid filters found, falling back to default');
                return $default;
            }
            
            // Execute filtered query
            $products = $this->execute_filtered_query($filters);
            error_log('🔍 MXCHAT FILTERED SEARCH: Found ' . count($products) . ' products');
            
            // Optionally re-rank results
            if (!empty($products) && $query_embedding) {
                $products = $this->semantic_rerank_products(
                    $products,
                    $query_embedding,
                    $message
                );
                error_log('✅ MXCHAT FILTERED SEARCH: Products re-ranked semantically');
            }
            
            // Format and return results
            $response = $this->format_filtered_results($products, $filters, $message);
            error_log('✅ MXCHAT FILTERED SEARCH: Response generated successfully');
            
            return $response;
            
        } catch (Exception $e) {
            error_log('❌ MXCHAT FILTERED SEARCH ERROR: ' . $e->getMessage());
            error_log('❌ MXCHAT FILTERED SEARCH ERROR TRACE: ' . $e->getTraceAsString());
            return array(
                'text' => __('An error occurred while searching for products. Please try again.', 'mxchat-woo'),
                'html' => ''
            );
        }
    }

    /**
     * AJAX handler to save filtered search settings
     */
    public function save_filtered_search_settings() {
        check_ajax_referer('mxchat_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $settings = array(
            'trigger_phrases' => isset($_POST['trigger_phrases']) 
                ? array_map('sanitize_text_field', $_POST['trigger_phrases']) 
                : array(),
            'similarity_threshold' => isset($_POST['similarity_threshold']) 
                ? floatval($_POST['similarity_threshold']) 
                : 0.70,
            'enable_semantic_rerank' => isset($_POST['enable_semantic_rerank']) 
                ? sanitize_text_field($_POST['enable_semantic_rerank']) 
                : 'off',
            'enabled_filters' => isset($_POST['enabled_filters']) 
                ? array_map('sanitize_text_field', $_POST['enabled_filters']) 
                : array('price', 'category', 'attributes')
        );
        
        update_option('mxchat_woo_filtered_search_settings', $settings);
        
        // Clear cached trigger embeddings
        delete_transient('mxchat_woo_filtered_search_trigger_embeddings_' . md5(serialize($settings['trigger_phrases'])));
        
        wp_send_json_success(array(
            'message' => __('Filtered search settings saved successfully!', 'mxchat-woo')
        ));
    }

    /**
     * AJAX handler to test filtered search
     */
    public function test_filtered_search() {
        check_ajax_referer('mxchat_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $test_query = isset($_POST['test_query']) ? sanitize_text_field($_POST['test_query']) : '';
        
        if (empty($test_query)) {
            wp_send_json_error('No test query provided');
        }
        
        // Run the filtered search
        $result = $this->mxchat_handle_filtered_search(null, $test_query, 0, 'test');
        
        wp_send_json_success($result);
    }
}
