<?php
/**
 * Handles WooCommerce order operations for MxChat
 */
class MxChat_Woo_Order_Handler {
    private $loader;
    private $options;
    private $fallbackResponse;

    public function __construct($loader, $options) {
        $this->loader = $loader;
        $this->options = $options;
        $this->fallbackResponse = array('text' => '');
        $this->init();
    }

        private function init() {
            // Keep existing AJAX handlers
            $this->loader->add_action('wp_ajax_mxchat_fetch_orders', $this, 'fetch_orders');
            $this->loader->add_action('wp_ajax_nopriv_mxchat_fetch_orders', $this, 'fetch_orders');
            
            $this->loader->add_action('wp_ajax_mxchat_get_order_details', $this, 'get_order_details');
            $this->loader->add_action('wp_ajax_nopriv_mxchat_get_order_details', $this, 'get_order_details');
            
            $this->loader->add_action('wp_ajax_mxchat_track_order', $this, 'track_order');
            $this->loader->add_action('wp_ajax_nopriv_mxchat_track_order', $this, 'track_order');
            
            // Add new AJAX handlers for order history button
            $this->loader->add_action('wp_ajax_mxchat_woo_get_recent_orders', $this, 'get_recent_orders_ajax');
            $this->loader->add_action('wp_ajax_nopriv_mxchat_woo_get_recent_orders', $this, 'get_recent_orders_ajax');
            
            // Add filter to inject order history button
            $this->loader->add_filter('mxchat_chatbot_toolbar_end', $this, 'inject_order_history_button', 20, 1);
        }


    public function fetch_orders() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $user_id = get_current_user_id();
        if (!$user_id && isset(WC()->session)) {
            $user_id = WC()->session->get_customer_id();
        }

        if (!$user_id) {
            wp_send_json_error('User not logged in.');
        }

        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $per_page = isset($_POST['per_page']) ? intval($_POST['per_page']) : 10;

        $args = array(
            'customer_id' => $user_id,
            'limit' => $per_page,
            'paged' => $page,
            'orderby' => 'date',
            'order' => 'DESC',
        );

        $orders = wc_get_orders($args);
        $formatted_orders = array_map(array($this, 'format_order_data'), $orders);

        wp_send_json_success(array(
            'orders' => $formatted_orders,
            'total_pages' => ceil(wc_get_customer_order_count($user_id) / $per_page)
        ));
    }

    public function get_order_details() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        
        if (!$order_id) {
            wp_send_json_error('Invalid order ID.');
        }

        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_send_json_error('Order not found.');
        }

        // Verify user has access to this order
        if (!$this->user_can_access_order($order)) {
            wp_send_json_error('Access denied.');
        }

        wp_send_json_success($this->format_order_data($order, true));
    }

    public function track_order() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        $order_email = isset($_POST['order_email']) ? sanitize_email($_POST['order_email']) : '';

        if (!$order_id || !$order_email) {
            wp_send_json_error('Invalid tracking information.');
        }

        $order = wc_get_order($order_id);

        if (!$order) {
            wp_send_json_error('Order not found.');
        }

        // Verify order email matches
        if ($order->get_billing_email() !== $order_email) {
            wp_send_json_error('Invalid order information.');
        }

        wp_send_json_success($this->format_order_data($order, true));
    }

    private function format_order_data($order, $include_details = false) {
        $data = array(
            'id' => $order->get_id(),
            'status' => $order->get_status(),
            'status_name' => wc_get_order_status_name($order->get_status()),
            'date_created' => $order->get_date_created()->date('Y-m-d H:i:s'),
            'total' => $order->get_total(),
            'formatted_total' => $order->get_formatted_order_total(),
            'payment_method' => $order->get_payment_method_title(),
            'order_number' => $order->get_order_number()
        );

        if ($include_details) {
            $items = array();
            foreach ($order->get_items() as $item_id => $item) {
                $product = $item->get_product();
                $items[] = array(
                    'id' => $item_id,
                    'product_id' => $product ? $product->get_id() : null,
                    'name' => $item->get_name(),
                    'quantity' => $item->get_quantity(),
                    'total' => $order->get_formatted_line_subtotal($item),
                    'sku' => $product ? $product->get_sku() : '',
                    'meta_data' => $this->format_item_meta_data($item)
                );
            }

            $data['items'] = $items;
            $data['billing_address'] = $this->format_address_data($order, 'billing');
            $data['shipping_address'] = $this->format_address_data($order, 'shipping');
            
            // Add shipping info if available
            if ($order->get_shipping_method()) {
                $data['shipping_method'] = $order->get_shipping_method();
                $tracking_info = $this->get_order_tracking_info($order);
                if ($tracking_info) {
                    $data['tracking_info'] = $tracking_info;
                }
            }

            // Add order notes
            $data['customer_notes'] = $this->get_customer_order_notes($order->get_id());
        }

        return $data;
    }

    private function format_item_meta_data($item) {
        $formatted_meta = array();
        $meta_data = $item->get_formatted_meta_data('_');

        foreach ($meta_data as $meta) {
            $formatted_meta[] = array(
                'key' => $meta->key,
                'value' => $meta->display_value
            );
        }

        return $formatted_meta;
    }

    private function format_address_data($order, $type = 'billing') {
        $address = array();
        $fields = array(
            'first_name', 'last_name', 'company', 'address_1', 
            'address_2', 'city', 'state', 'postcode', 'country'
        );

        foreach ($fields as $field) {
            $getter = "get_{$type}_{$field}";
            if (is_callable(array($order, $getter))) {
                $address[$field] = $order->$getter();
            }
        }

        return $address;
    }

    private function get_order_tracking_info($order) {
        // This can be extended to support various shipping tracking plugins
        $tracking_info = array();
        
        // Example: Support for WooCommerce Shipment Tracking
        if (function_exists('wc_get_order_tracking_items')) {
            $tracking_items = wc_get_order_tracking_items($order->get_id());
            if (!empty($tracking_items)) {
                foreach ($tracking_items as $tracking_item) {
                    $tracking_info[] = array(
                        'provider' => $tracking_item['tracking_provider'],
                        'number' => $tracking_item['tracking_number'],
                        'link' => $tracking_item['tracking_link']
                    );
                }
            }
        }

        return $tracking_info;
    }

    private function get_customer_order_notes($order_id) {
        $args = array(
            'post_id' => $order_id,
            'type' => 'order_note',
            'status' => 'approve',
            'meta_query' => array(
                array(
                    'key' => 'is_customer_note',
                    'value' => 1
                )
            )
        );

        $notes = wc_get_order_notes($args);
        $formatted_notes = array();

        foreach ($notes as $note) {
            $formatted_notes[] = array(
                'id' => $note->id,
                'content' => $note->content,
                'date' => $note->date_created->date('Y-m-d H:i:s')
            );
        }

        return $formatted_notes;
    }

    private function user_can_access_order($order) {
        $user_id = get_current_user_id();
        
        if ($user_id && $order->get_user_id() === $user_id) {
            return true;
        }

        // Check if guest user with matching session
        if (!$user_id && isset(WC()->session)) {
            $session_customer_id = WC()->session->get_customer_id();
            if ($session_customer_id && $order->get_customer_id() === $session_customer_id) {
                return true;
            }
        }

        return false;
    }

    public function get_customer_last_order($customer_id = null) {
        if (!$customer_id) {
            $customer_id = get_current_user_id();
        }

        if (!$customer_id) {
            return false;
        }

        $args = array(
            'customer_id' => $customer_id,
            'limit' => 1,
            'orderby' => 'date',
            'order' => 'DESC',
        );

        $orders = wc_get_orders($args);
        return !empty($orders) ? $orders[0] : false;
    }
    
    
public function mxchat_handle_order_history($default, $message, $user_id, $session_id) {
    error_log('🔍 MXCHAT DEBUG: Order History Intent Triggered');
    error_log("🔍 MXCHAT DEBUG: Message: '$message', User ID: $user_id, Session: $session_id");

    // Get API key for AI calls
    $api_key = get_option('mxchat_options')['api_key'] ?? '';

    // If message is empty or very short, provide a default context
    $user_request = !empty($message) && strlen(trim($message)) > 2 ? $message : 'show me my order history';
    error_log("🔍 MXCHAT DEBUG: User request used for AI: '$user_request'");

    // Get browser language as fallback for language detection
    $browser_language = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ?
        substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2) : 'en';

    if (!class_exists('WooCommerce')) {
        //error_log('❌ MXCHAT DEBUG: WooCommerce not available');

        // Use AI for multilingual "WooCommerce unavailable" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$user_request}'\n\n";
        $ai_prompt .= "Context: The order system (WooCommerce) is currently unavailable, so I cannot access their order information.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Apologize and inform them that the order system is temporarily unavailable.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $api_key);

        $response_text = $ai_response['text'] ?? esc_html__("I can't access order information right now. The order system seems to be unavailable.", 'mxchat-woo');

        return array(
            'text' => $this->process_markdown($response_text)
        );
    }

    if (!is_user_logged_in()) {
        //error_log('❌ MXCHAT DEBUG: User not logged in');

        // Use AI for multilingual "not logged in" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$user_request}'\n\n";
        $ai_prompt .= "Context: The user wants to view their order history, but they are not currently logged in to their account.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Politely inform them they need to log in to view their order history.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $api_key);

        $response_text = $ai_response['text'] ?? esc_html__("Please log in to view your order history.", 'mxchat-woo');

        return array(
            'text' => $this->process_markdown($response_text)
        );
    }

    $orderDetails = $this->fetch_user_orders_details('all');
    //error_log('🔍 MXCHAT DEBUG: Order Details Fetched - Count: ' . count($orderDetails));

    if (empty($orderDetails)) {
        //error_log('❌ MXCHAT DEBUG: No orders found for user');

        // Use AI for multilingual "no orders found" response
        $ai_prompt = "IMPORTANT: The user's message is: '{$user_request}'\n\n";
        $ai_prompt .= "Context: The user asked about their orders, but there are no orders associated with their account.\n\n";
        $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
        $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
        $ai_prompt .= "Task: Politely inform them that there are no orders found on their account.";

        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $api_key);

        $response_text = $ai_response['text'] ?? esc_html__("I don't see any orders associated with your account.", 'mxchat-woo');

        return array(
            'text' => $this->process_markdown($response_text)
        );
    }

    // Build the AI prompt with order data
    $ai_prompt = "IMPORTANT: The user's message is: '{$user_request}'\n\n";
    $ai_prompt .= "Context: Here is the user's order information:\n\n";

    foreach ($orderDetails as $order) {
        $items_list = array_map(function($item) {
            return "{$item['name']} ({$item['quantity']})";
        }, $order['items']);

        $ai_prompt .= "Order #{$order['order_id']}: {$order['formatted_total']} on {$order['date']}\n";
        $ai_prompt .= "Items: " . implode(', ', $items_list) . "\n\n";

        //error_log("🔍 MXCHAT DEBUG: Processing Order #{$order['order_id']} with " . count($order['items']) . " items");
    }

    $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
    $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
    $ai_prompt .= "Task: Provide a natural, conversational response about their orders. ";
    $ai_prompt .= "If they ask about a specific order or detail, provide just that information. ";
    $ai_prompt .= "If they ask about license keys or sensitive information, inform them to check their email or contact support. ";
    $ai_prompt .= "Keep your response friendly and concise.\n\n";
    $ai_prompt .= "FORMATTING: Do NOT use markdown headings (# or ## or ###). Use bold formatting (**text**) for emphasis instead.";

    error_log('🔍 MXCHAT DEBUG: Full AI Prompt being sent:');
    error_log($ai_prompt);

    // Call the AI API
    $response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $api_key);
    error_log('✅ MXCHAT DEBUG: Raw AI Response: ' . print_r($response, true));

    if (!empty($response['text'])) {
        // Process markdown formatting before returning
        $processed_text = $this->process_markdown($response['text']);

        return array(
            'text' => $processed_text
        );
    }

    // Fallback response if AI call fails
    return array(
        'text' => esc_html__("I found your orders but am having trouble processing them right now. Here's a summary:", 'mxchat-woo') . "\n\n" .
                 implode("\n", array_map(function($order) {
                     return sprintf(
                         __("Order #%s: %s on %s", 'mxchat-woo'),
                         $order['order_id'],
                         $order['formatted_total'],
                         $order['date']
                     );
                 }, $orderDetails))
    );
}


     /**
     * Fetch user order details
     * 
     * @param string $status Order status to fetch
     * @return array
     */
    private function fetch_user_orders_details($status = 'all') {
        if (!is_user_logged_in()) {
            return array();
        }

        $orders = wc_get_orders(array(
            'customer_id' => get_current_user_id(),
            'status' => $status,
            'limit' => -1
        ));

        if (empty($orders)) {
            return array();
        }

        $orderDetails = array();
        foreach ($orders as $order) {
            $items = array();
            foreach ($order->get_items() as $item) {
                $items[] = array(
                    'name' => $item->get_name(),
                    'quantity' => $item->get_quantity()
                );
            }

            $orderDetails[] = array(
                'order_id' => $order->get_id(),
                'date' => $order->get_date_created()->date_i18n(get_option('date_format')),
                'formatted_total' => $order->get_formatted_order_total(),
                'items' => $items
            );
        }

        return $orderDetails;
    }
    
    
    
    /**
 * Inject order history button into chatbot toolbar
 * 
 * @param string $toolbar_html Current toolbar HTML
 * @return string Modified toolbar HTML with order history button
 */
public function inject_order_history_button($toolbar_html) {
    if (!class_exists('WooCommerce')) {
        return $toolbar_html;
    }
    
    // Generate button HTML
    ob_start();
    ?>
    <button id="woo-orders-btn" class="toolbar-btn" title="<?php echo esc_attr__('Your Orders', 'mxchat-woo'); ?>">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512" stroke="currentColor">
            <path d="M64 0C28.7 0 0 28.7 0 64V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V160H256c-17.7 0-32-14.3-32-32V0H64zM256 0V128H384L256 0zM80 64h64c8.8 0 16 7.2 16 16s-7.2 16-16 16H80c-8.8 0-16-7.2-16-16s7.2-16 16-16zm0 64h64c8.8 0 16 7.2 16 16s-7.2 16-16 16H80c-8.8 0-16-7.2-16-16s7.2-16 16-16zm16 96h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H96c-8.8 0-16-7.2-16-16V240c0-8.8 7.2-16 16-16zm112 16c0-8.8 7.2-16 16-16h32c8.8 0 16 7.2 16 16v32c0 8.8-7.2 16-16 16H224c-8.8 0-16-7.2-16-16V240zM80 320h64c8.8 0 16 7.2 16 16s-7.2 16-16 16H80c-8.8 0-16-7.2-16-16s7.2-16 16-16zm144 0h64c8.8 0 16 7.2 16 16s-7.2 16-16 16H224c-8.8 0-16-7.2-16-16s7.2-16 16-16z"/>
        </svg>
    </button>
    
    <div id="woo-orders-dropdown" class="woo-orders-dropdown">
        <div class="woo-orders-header">
            <h4><?php echo esc_html__('Your Orders', 'mxchat-woo'); ?></h4>
            <button class="woo-orders-close-btn" title="<?php echo esc_attr__('Close', 'mxchat-woo'); ?>">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>
        </div>
        <div class="woo-orders-container">
            <div id="woo-orders-list" class="woo-orders-list">
                <div class="woo-orders-loading"><?php echo esc_html__('Loading orders...', 'mxchat-woo'); ?></div>
            </div>
        </div>
        <div class="woo-orders-footer">
            <a href="<?php echo esc_url(wc_get_account_endpoint_url('orders')); ?>" class="woo-orders-view-all-btn"><?php echo esc_html__('View All Orders', 'mxchat-woo'); ?></a>
        </div>
    </div>
    <?php
    
    $orders_button_html = ob_get_clean();
    
    // Add the button to the toolbar
    return str_replace('</div><!-- End .chat-toolbar -->', $orders_button_html . '</div><!-- End .chat-toolbar -->', $toolbar_html);
}

/**
 * AJAX handler to get recent orders
 */
public function get_recent_orders_ajax() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
        wp_send_json_error(['message' => __('Security check failed', 'mxchat-woo')]);
    }
    
    // Check if WooCommerce is available
    if (!class_exists('WooCommerce')) {
        wp_send_json_error(['message' => __('Order system is not available', 'mxchat-woo')]);
    }
    
    // Check if user is logged in
    if (!is_user_logged_in()) {
        wp_send_json_error([
            'message' => __('Please log in to view your orders', 'mxchat-woo'),
            'login_required' => true
        ]);
    }
    
    // Get user ID
    $user_id = get_current_user_id();
    
    // Get recent orders
    $args = array(
        'customer_id' => $user_id,
        'limit' => 5,
        'orderby' => 'date',
        'order' => 'DESC',
    );
    
    $orders = wc_get_orders($args);
    
    if (empty($orders)) {
        wp_send_json_success([
            'orders' => [],
            'count' => 0,
            'view_all_url' => wc_get_account_endpoint_url('orders')
        ]);
        return;
    }
    
    // Format orders for display
    $formatted_orders = [];
    foreach ($orders as $order) {
        $order_id = $order->get_id();
        $status = $order->get_status();
        $status_name = wc_get_order_status_name($status);
        
        // Get order items (limited to first 2 for display)
        $items = [];
        $item_count = 0;
        foreach ($order->get_items() as $item) {
            if ($item_count < 2) {
                $items[] = $item->get_name() . ' × ' . $item->get_quantity();
            }
            $item_count++;
        }
        
        // If there are more items, add a note
        if ($item_count > 2) {
            $items[] = sprintf(__('and %d more items', 'mxchat-woo'), $item_count - 2);
        }
        
        $formatted_orders[] = [
            'id' => $order_id,
            'number' => $order->get_order_number(),
            'status' => $status,
            'status_name' => $status_name,
            'date' => $order->get_date_created()->date_i18n(get_option('date_format')),
            'total' => $order->get_formatted_order_total(),
            'items' => $items,
            'item_count' => $item_count,
            'view_url' => $order->get_view_order_url()
        ];
    }
    
    // Get order count
    $order_count = wc_get_customer_order_count($user_id);
    
    wp_send_json_success([
        'orders' => $formatted_orders,
        'count' => $order_count,
        'view_all_url' => wc_get_account_endpoint_url('orders')
    ]);
}

/**
 * Process markdown formatting
 */
private function process_markdown($text) {
    // Process markdown headings (# H1, ## H2, ### H3)
    $text = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $text);
    $text = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $text);
    $text = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $text);

    // Process markdown links [text](url) first
    $text = preg_replace('/\[([^\]]+)\]\(([^)]+)\)/', '<a href="$2" target="_blank">$1</a>', $text);

    // Process bold and italic
    $text = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $text);
    $text = preg_replace('/\*(.*?)\*/', '<em>$1</em>', $text);

    // Process standalone URLs to make them clickable (but avoid double-processing already linked URLs)
    $text = preg_replace('/(?<!href=")(?<!href=\')(?<!>)(https?:\/\/[^\s<>"\']+)/i', '<a href="$1" target="_blank">$1</a>', $text);

    // Don't process newlines here - let the main chat handler do it
    return $text;
}

}