<?php
/**
 * Handles WooCommerce product operations for MxChat
 */
class MxChat_Woo_Product_Handler {
    private $options;
    private $loader;
    private $api_key;

    public function __construct($loader, $options = array()) {
        $this->loader = $loader;
        
        // Get the core plugin options directly
        $core_options = get_option('mxchat_options', array());
        
        // Get WooCommerce specific options
        $this->options = get_option('mxchat_woo_options', array());
        
        // Get API key from core options
        $this->api_key = isset($core_options['api_key']) ? $core_options['api_key'] : '';
        
        // Only initialize if loader is valid
        if ($this->loader && method_exists($this->loader, 'add_action')) {
            $this->init();
        }
        
        $this->loader->add_action('wp_enqueue_scripts', $this, 'enqueue_cart_scripts');
    }

    private function init() {
        // Only add actions if loader is valid
        if (!$this->loader || !method_exists($this->loader, 'add_action')) {
            return;
        }

        // Add product sync hooks if enabled
        if (isset($this->options['enable_woocommerce_integration']) &&
            ($this->options['enable_woocommerce_integration'] === '1' ||
             $this->options['enable_woocommerce_integration'] === 'on')) {
            
            $this->loader->add_action('wp_insert_post', $this, 'mxchat_handle_product_change', 10, 3);
            $this->loader->add_action('wp_trash_post', $this, 'mxchat_handle_product_delete');
            $this->loader->add_action('before_delete_post', $this, 'mxchat_handle_product_delete');
        }

        // Standard product operations
        $this->loader->add_action('wp_ajax_mxchat_fetch_products', $this, 'fetch_products');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_fetch_products', $this, 'fetch_products');
        $this->loader->add_action('wp_ajax_mxchat_product_search', $this, 'product_search');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_product_search', $this, 'product_search');
        
        // Product tracking
        $this->loader->add_action('woocommerce_update_product', $this, 'track_product_update');
        $this->loader->add_action('woocommerce_delete_product', $this, 'track_product_deletion');
        
        // Search improvements
        $this->loader->add_filter('posts_search', $this, 'improve_product_search', 10, 2);
        $this->loader->add_filter('posts_clauses', $this, 'order_by_relevance', 10, 2);
        
        // Add these new AJAX handlers for variations
        $this->loader->add_action('wp_ajax_mxchat_get_variation_data', $this, 'get_variation_data');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_get_variation_data', $this, 'get_variation_data');
        $this->loader->add_action('wp_ajax_mxchat_add_variation_to_cart', $this, 'add_variation_to_cart');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_add_variation_to_cart', $this, 'add_variation_to_cart');
        
        // Add these AJAX handlers for simple products
        $this->loader->add_action('wp_ajax_mxchat_woo_add_to_cart', $this, 'add_simple_product_to_cart');
        $this->loader->add_action('wp_ajax_nopriv_mxchat_woo_add_to_cart', $this, 'add_simple_product_to_cart');
    }
    
    
public function enqueue_cart_scripts() {
    // First check if WooCommerce is active and we're on a page where cart functionality is needed
    if (!class_exists('WooCommerce') || !is_woocommerce() && !is_cart() && !is_checkout() && !is_product()) {
        return;
    }

    // Ensure WooCommerce's add to cart script is loaded
    if (!wp_script_is('wc-add-to-cart', 'enqueued')) {
        wp_enqueue_script('wc-add-to-cart');
    }

    // Register and enqueue our cart handling script
    wp_register_script(
        'mxchat-woo-cart',
        MXCHAT_WOO_PLUGIN_URL . 'public/js/mxchat-woo-cart.js',
        array('jquery', 'wc-add-to-cart'),
        MXCHAT_WOO_VERSION,
        true
    );

    // Get custom button text from options
    $button_text = isset($this->options['add_to_cart_button_text']) && !empty($this->options['add_to_cart_button_text'])
        ? esc_html($this->options['add_to_cart_button_text'])
        : esc_html__('Add to Cart', 'mxchat-woo');

    // Add any localized data if needed
    wp_localize_script(
        'mxchat-woo-cart',
        'mxchatWooCart',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mxchat-woo-cart'),
            'i18n' => array(
                'addToCart' => $button_text,
                'addToCartError' => __('Error adding product to cart', 'mxchat-woo'),
                'addToCartSuccess' => __('Product added to cart', 'mxchat-woo'),
                'selectAllOptions' => __('Please select all product options', 'mxchat-woo'),
                'variationNotAvailable' => __('This variation is not available', 'mxchat-woo'),
                'adding' => __('Adding...', 'mxchat-woo')
            )
        )
    );

    wp_enqueue_script('mxchat-woo-cart');

    // Enqueue associated styles if needed
    wp_enqueue_style(
        'mxchat-woo-cart',
        MXCHAT_WOO_PLUGIN_URL . 'public/css/mxchat-woo-public.css',
        array(),
        MXCHAT_WOO_VERSION
    );
}

    public function fetch_products() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $args = array(
            'status' => 'publish',
            'limit' => 10,
            'orderby' => 'date',
            'order' => 'DESC',
        );

        if (isset($_POST['category'])) {
            $args['category'] = sanitize_text_field($_POST['category']);
        }

        $products = wc_get_products($args);
        $formatted_products = array();

        foreach ($products as $product) {
            $formatted_products[] = $this->format_product_data($product);
        }

        wp_send_json_success($formatted_products);
    }

    public function product_search() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
            wp_send_json_error('Invalid nonce.');
        }

        $search_term = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $product_id = $this->extract_product_id_from_message($search_term);

        if ($product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
                wp_send_json_success($this->format_product_data($product));
            }
        }

        wp_send_json_error('Product not found.');
    }

    private function extract_product_id_from_message($message) {
        if (!function_exists('wc_get_products')) {
            return null;
        }

        $message = sanitize_text_field(strtolower($message));
        $products = wc_get_products([
            'status' => 'publish',
            'limit' => -1,
            'return' => 'objects'
        ]);

        $matches = [];
        $search_terms = explode(' ', $message);

        foreach ($products as $product) {
            $score = 0;
            $product_name = strtolower($product->get_name());
            $product_sku = strtolower($product->get_sku());

            $matched_terms = 0;
            $consecutive_matches = 0;
            $max_consecutive = 0;

            foreach ($search_terms as $term) {
                if (strlen($term) <= 2) continue;

                if (strpos($product_name, $term) !== false || 
                    strpos($product_sku, $term) !== false) {
                    $matched_terms++;
                    $consecutive_matches++;
                    $max_consecutive = max($max_consecutive, $consecutive_matches);
                } else {
                    $consecutive_matches = 0;
                }
            }

            if ($matched_terms > 0) {
                $score = $matched_terms / count(array_filter($search_terms, function($term) {
                    return strlen($term) > 2;
                }));

                $score += ($max_consecutive > 1) ? 0.3 : 0;

                if (strpos($product_name, implode(' ', $search_terms)) !== false) {
                    $score += 0.5;
                }

                if ($score > 0) {
                    $matches[] = [
                        'product_id' => $product->get_id(),
                        'score' => $score
                    ];
                }
            }
        }

        if (!empty($matches)) {
            usort($matches, function($a, $b) {
                return $b['score'] <=> $a['score'];
            });
            return $matches[0]['product_id'];
        }

        return null;
    }

    private function format_product_data($product) {
        return array(
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'price' => $product->get_price(),
            'regular_price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'type' => $product->get_type(),
            'status' => $product->get_status(),
            'featured' => $product->get_featured(),
            'catalog_visibility' => $product->get_catalog_visibility(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'sku' => $product->get_sku(),
            'stock_status' => $product->get_stock_status(),
            'stock_quantity' => $product->get_stock_quantity(),
            'permalink' => get_permalink($product->get_id()),
            'categories' => wp_list_pluck(get_the_terms($product->get_id(), 'product_cat'), 'name'),
            'attributes' => $this->get_product_attributes($product)
        );
    }

    private function get_product_attributes($product) {
        $attributes = array();
        $product_attributes = $product->get_attributes();

        foreach ($product_attributes as $attribute) {
            if ($attribute->is_taxonomy()) {
                $attribute_name = wc_attribute_label($attribute->get_name());
                $attribute_terms = wp_list_pluck(wc_get_product_terms($product->get_id(), $attribute->get_name()), 'name');
                $attributes[$attribute_name] = implode(', ', $attribute_terms);
            } else {
                $attributes[$attribute->get_name()] = $attribute->get_options();
            }
        }

        return $attributes;
    }
    
private function mxchat_generate_embedding($text, $api_key) {
    // Get main plugin options for model selection
    $main_plugin_options = get_option('mxchat_options', array());
    $selected_model = $main_plugin_options['embedding_model'] ?? 'text-embedding-ada-002';
    
    // Determine endpoint and API key based on model
    if (strpos($selected_model, 'voyage') === 0) {
        $endpoint = 'https://api.voyageai.com/v1/embeddings';
        $api_key = $main_plugin_options['voyage_api_key'] ?? '';
    } elseif (strpos($selected_model, 'gemini-embedding') === 0) {
        $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/' . $selected_model . ':embedContent';
        $api_key = $main_plugin_options['gemini_api_key'] ?? '';
    } else {
        $endpoint = 'https://api.openai.com/v1/embeddings';
        // Use the passed $api_key parameter for OpenAI
    }
    
    // Prepare request body and headers based on provider
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        $body = wp_json_encode([
            'model' => 'models/' . $selected_model,
            'content' => [
                'parts' => [
                    ['text' => $text]
                ]
            ],
            'outputDimensionality' => 1536
        ]);
        
        $endpoint .= '?key=' . $api_key;
        $headers = [
            'Content-Type' => 'application/json'
        ];
    } else {
        $request_body = [
            'input' => $text,
            'model' => $selected_model
        ];
        
        if ($selected_model === 'voyage-3-large') {
            $request_body['output_dimension'] = 2048;
        }
        
        $body = wp_json_encode($request_body);
        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $api_key
        ];
    }
    
    $args = [
        'body' => $body,
        'headers' => $headers,
        'timeout' => 60,
        'redirection' => 5,
        'blocking' => true,
        'httpversion' => '1.0',
        'sslverify' => true,
    ];
    
    $response = wp_remote_post($endpoint, $args);
    
    if (is_wp_error($response)) {
        //error_log('MxChat Addon: Error generating embedding - ' . $response->get_error_message());
        return null;
    }
    
    $response_body = json_decode(wp_remote_retrieve_body($response), true);
    
    // Handle different response formats
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        if (isset($response_body['embedding']['values']) && is_array($response_body['embedding']['values'])) {
            return $response_body['embedding']['values'];
        }
    } else {
        if (isset($response_body['data'][0]['embedding']) && is_array($response_body['data'][0]['embedding'])) {
            return $response_body['data'][0]['embedding'];
        }
    }
    
    //error_log('MxChat Addon: Invalid response from API - ' . wp_json_encode($response_body));
    return null;
}

private function mxchat_generate_ai_recommendation_prompt($recommendations, $user_message, $user_context = []) {
    // Get user's browser language as fallback
    $browser_language = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? 
        substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2) : 'en';
    
    // Build the complete prompt with language detection first
    $prompt = "User said: '{$user_message}'\n\n";
    $prompt .= "CRITICAL: Analyze the language of their message and respond in that EXACT same language. ";
    $prompt .= "If the message language is unclear or generic, use {$browser_language} as the default language. ";
    $prompt .= "Do not translate or use random languages. Use only the detected language or {$browser_language}.\n\n";

    // Format the user's context
    if (!empty($user_context)) {
        if (!empty($user_context['order_history'])) {
            $prompt .= "Their recent orders include:\n";
            foreach ($user_context['order_history'] as $order) {
                $prompt .= "- {$order['name']} ({$order['date']})\n";
            }
            $prompt .= "\n";
        }
        if (!empty($user_context['cart_items'])) {
            $prompt .= "They currently have in their cart:\n";
            foreach ($user_context['cart_items'] as $item) {
                $prompt .= "- {$item['name']}\n";
            }
            $prompt .= "\n";
        }
    }

    // Format the product recommendations
    $prompt .= "Based on their request, here are recommended products:\n\n";
    foreach ($recommendations as $index => $rec) {
        $number = $index + 1;
        $name = $rec['name'];
        $price = $rec['price'];
        $prompt .= "{$number}. {$name} (\${$price})\n";
    }

    $prompt .= "\nRespond in their language with:";
    $prompt .= "\n1. A brief personalized introduction acknowledging their specific request";
    $prompt .= "\n2. Numbered product recommendations matching the order above";
    $prompt .= "\n3. Clear justification for each recommendation";
    $prompt .= "\n4. Reference their shopping history or cart items when relevant";
    $prompt .= "\nUse the store's currency format and keep it conversational and friendly.";
    $prompt .= "\nDo not include any links, images, or markdown formatting in your response. Just provide clean text.";

    return $prompt;
}

private function mxchat_generate_recommendations($user_id, $message) {
    // 1. Gather user context
    $user_context = $this->mxchat_get_user_context($user_id);

    // Get cart item IDs for filtering
    $cart_item_ids = [];
    if (!empty($user_context['cart_items'])) {
        $cart_item_ids = array_map(function($item) {
            return $item['product_id'];
        }, $user_context['cart_items']);
    }

    // 2. Get AI recommendation based on context
    $ai_suggestion = $this->mxchat_get_ai_shopping_suggestion($message, $user_context);
    //error_log("🔍 MXCHAT DEBUG: AI Shopping Suggestion: " . $ai_suggestion);

    // 3. Generate embedding for the AI suggestion
    if (empty($this->api_key)) {
        //error_log("❌ MXCHAT DEBUG: No API key found in settings");
        return ['recommendations' => [], 'sources' => []];
    }

    $suggestion_embedding = $this->mxchat_generate_embedding($ai_suggestion, $this->api_key);
    if (!$suggestion_embedding) {
        //error_log("❌ MXCHAT DEBUG: Could not generate embedding for AI suggestion");
        return ['recommendations' => [], 'sources' => []];
    }

    // 4. Use existing relevant content function to find matches
    $relevant_content = MxChat_Woo_Util::mxchat_find_relevant_content_woo($suggestion_embedding);

    // 5. Extract product URLs from the relevant content
    preg_match_all('/https?:\/\/[^\s<>"\']+?\/product\/[^\s<>"\']+/', $relevant_content, $matches);

    $found_products = [];
    if (!empty($matches[0])) {
        foreach ($matches[0] as $url) {
            // Clean the URL
            $url = rtrim($url, '/."\']');

            // Get the product slug
            $path = parse_url($url, PHP_URL_PATH);
            $slug = basename(rtrim($path, '/'));

            // Find product by slug
            $args = array(
                'post_type' => 'product',
                'post_status' => 'publish',
                'name' => $slug,
                'posts_per_page' => 1
            );

            $products = get_posts($args);

            if (!empty($products)) {
                $product_id = $products[0]->ID;

                // Skip if product is in cart
                if (in_array($product_id, $cart_item_ids)) {
                    //error_log("Skipping product ID $product_id - already in cart");
                    continue;
                }

                $product = wc_get_product($product_id);

                if ($product && $product->is_purchasable() && $product->is_in_stock() && !in_array($product_id, array_map(function($p) {
                    return $p->get_id();
                }, $found_products))) {
                    $found_products[] = $product;
                }
            }
        }
    }

    // If no products found by URLs, look for product names in the content
    if (empty($found_products)) {
        $products = wc_get_products([
            'status' => 'publish',
            'limit' => -1,
            'return' => 'objects'
        ]);

        foreach ($products as $product) {
            // Skip if product is in cart
            if (in_array($product->get_id(), $cart_item_ids)) {
                continue;
            }

            $name = $product->get_name();
            if (stripos($relevant_content, $name) !== false) {
                if ($product->is_purchasable() && $product->is_in_stock() && !in_array($product->get_id(), array_map(function($p) {
                    return $p->get_id();
                }, $found_products))) {
                    $found_products[] = $product;
                }
            }
        }
    }

    // Keep searching until we have 4 products or run out of options
    $formatted_recommendations = [];
    foreach ($found_products as $product) {
        if (count($formatted_recommendations) >= 4) break;

        $formatted_recommendations[] = [
            'name' => $product->get_name(),
            'price' => $product->get_price(),
            'url' => get_permalink($product->get_id()),
            'image' => wp_get_attachment_url($product->get_image_id())
        ];
    }

    return [
        'recommendations' => $formatted_recommendations,
        'sources' => [__('AI-powered personalized recommendations', 'mxchat')]
    ];
}

private function mxchat_get_ai_shopping_suggestion($message, $user_context) {
    $prompt = esc_html__("As a shopping assistant, analyze this user's context and generate a specific product search suggestion. ", 'mxchat');
    $prompt .= esc_html__("User's Question: \"{$message}\"\n\n", 'mxchat');

    if (!empty($user_context['order_history'])) {
        $prompt .= esc_html__("Their recent orders include:\n", 'mxchat');
        foreach ($user_context['order_history'] as $order) {
            $prompt .= esc_html__("- {$order['name']} ({$order['date']})\n", 'mxchat');
        }
    }

    if (!empty($user_context['cart_items'])) {
        $prompt .= esc_html__("\nThey currently have in their cart:\n", 'mxchat');
        foreach ($user_context['cart_items'] as $item) {
            $prompt .= esc_html__("- {$item['name']}\n", 'mxchat');
        }
    }

    $prompt .= esc_html__("\nBased on their question and history, suggest a specific search query that would help find the most relevant products. ", 'mxchat');
    $prompt .= esc_html__("Respond with ONLY the search query, nothing else.", 'mxchat');

    $response = MxChat_Woo_Util::mxchat_call_ai_api($prompt, $this->api_key);
    return isset($response['text']) ? trim($response['text']) : $message;
}

public function mxchat_handle_product_recommendations($default, $message, $user_id, $session_id) {
    try {
        //error_log('🔍 MXCHAT DEBUG: Product Recommendations Intent Triggered in Add-on');
        //error_log("🔍 MXCHAT DEBUG: Message: '$message', User ID: $user_id, Session: $session_id");

        // Get user context first
        $user_context = $this->mxchat_get_user_context($user_id);
        //error_log('🔍 MXCHAT DEBUG: User context: ' . wp_json_encode($user_context));

        // Pass the message to get context-aware recommendations
        $recommendation_data = $this->mxchat_generate_recommendations($user_id, $message);
        //error_log('🔍 MXCHAT DEBUG: Generated recommendation data: ' . wp_json_encode($recommendation_data));

        if (empty($recommendation_data['recommendations'])) {
            //error_log('❌ MXCHAT DEBUG: No recommendations found for user: ' . $user_id);

            // Use AI for multilingual "no recommendations" response
            $ai_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
            $ai_prompt .= "Context: The user asked for product recommendations, but I couldn't find any suitable products to recommend at the moment.\n\n";
            $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
            $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
            $ai_prompt .= "Task: Apologize politely and ask them to try again or provide more details about what they're looking for.";

            $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

            return [
                'text' => $ai_response['text'] ?? __("I couldn't find any product recommendations for you right now. Please try again later!", 'mxchat'),
                'html' => ''
            ];
        }

        // Process recommendations: remove duplicates and limit to 4
        $unique_recommendations = [];
        foreach ($recommendation_data['recommendations'] as $rec) {
            $unique_recommendations[$rec['url']] = $rec;
        }
        $unique_recommendations = array_slice($unique_recommendations, 0, 4);

        $recommendations_summary = [];
        foreach ($unique_recommendations as $rec) {
            $recommendations_summary[] = [
                'name'  => $rec['name'],
                'price' => strip_tags($rec['price']),
                'url'   => $rec['url'],
                'image' => $rec['image'],
            ];
        }

        // Generate AI prompt
        $ai_prompt = $this->mxchat_generate_ai_recommendation_prompt(
            $recommendations_summary,
            $message,
            $user_context
        );

        // Call the AI API using the static util function
        $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);

        if (empty($ai_response) || empty($ai_response['text'])) {
            // Use AI for multilingual error response
            $error_prompt = "IMPORTANT: The user's message is: '{$message}'\n\n";
            $error_prompt .= "Context: The user asked for product recommendations, but there was a technical error while generating them.\n\n";
            $error_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
            $error_prompt .= "Detect the language from their message and use only that language in your response.\n\n";
            $error_prompt .= "Task: Apologize for the technical issue and ask them to try again in a moment.";

            $error_response = MxChat_Woo_Util::mxchat_call_ai_api($error_prompt, $this->api_key);

            return [
                'text' => $error_response['text'] ?? __('An unexpected error occurred while generating recommendations. Please try again later.', 'mxchat'),
                'html' => ''
            ];
        }

        // Process AI response: split the response into lines using the response text
        $ai_lines = preg_split('/\r\n|\r|\n/', $ai_response['text']);

        $formatted_response = '';
        $justifications = [];
        $current_number = 0;
        $in_introduction = true;
        $introduction = '';

        // Parse response lines
        foreach ($ai_lines as $line) {
            if (preg_match('/^\s*(\d+)\.\s*(.*)$/', $line, $matches)) {
                $current_number = intval($matches[1]) - 1;
                $justifications[$current_number] = $matches[2];
                $in_introduction = false;
            } elseif ($in_introduction) {
                $introduction .= $line . ' ';
            } else {
                if (isset($justifications[$current_number])) {
                    $justifications[$current_number] .= ' ' . $line;
                }
            }
        }

        // Build final formatted response
        if (!empty($introduction)) {
            $formatted_response .= esc_html(trim($introduction)) . "<br><br>";
        }

        foreach ($recommendations_summary as $index => $rec) {
            $name = esc_html($rec['name']);
            // Format price as a float to ensure correct number formatting
            $price = number_format((float)$rec['price'], 2, '.', '');
            $url = esc_url($rec['url']);
            $image = esc_url($rec['image']);

            $formatted_response .= ($index + 1) . ". <a href=\"{$url}\" target=\"_blank\" rel=\"noopener noreferrer\" style=\"text-decoration: none; color: inherit;\"><strong>" . $name . "</strong></a> - <strong>$" . $price . "</strong><br>";
            $formatted_response .= "<a href=\"{$url}\" target=\"_blank\" rel=\"noopener noreferrer\">";
            $formatted_response .= "<img src=\"{$image}\" alt=\"{$name}\" style=\"max-width: 200px; height: auto; display: block; margin: 10px 0;\" />";
            $formatted_response .= "</a><br>";

            if (isset($justifications[$index])) {
                $formatted_response .= "<em>" . esc_html(trim($justifications[$index])) . "</em><br><br>";
            }
        }

        //error_log('✅ MXCHAT DEBUG: Final formatted response length: ' . strlen($formatted_response));

        // Process the final response through markdown processor to handle any remaining markdown
        $processed_response = $this->process_markdown($formatted_response);

        return [
            'text' => $processed_response,
            'html' => ''
        ];
    } catch (Exception $e) {
        //error_log('❌ MXCHAT DEBUG: Error in product recommendations add-on: ' . $e->getMessage());
        return [
            'text' => __('An unexpected error occurred while generating recommendations. Please try again later.', 'mxchat'),
            'html' => ''
        ];
    }
}

private function mxchat_get_user_context($user_id) {
    $context = [
        'order_history' => [],
        'cart_items' => [],
        'recently_viewed' => [],
    ];

    // Get order history
    if (is_user_logged_in() && $user_id) {
        $orders = wc_get_orders([
            'customer_id' => $user_id,
            'limit' => 5,  // Last 5 orders
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                $context['order_history'][] = [
                    'name' => $item->get_name(),
                    'product_id' => $item->get_product_id(),
                    'date' => $order->get_date_created()->format('Y-m-d')
                ];
            }
        }
    }

    // Get cart items
    if (WC()->cart && WC()->cart->get_cart_contents_count() > 0) {
        foreach (WC()->cart->get_cart() as $cart_item) {
            $product = wc_get_product($cart_item['product_id']);
            if ($product) {
                $context['cart_items'][] = [
                    'name' => $product->get_name(),
                    'product_id' => $product->get_id()
                ];
            }
        }
    }

    // Get recently viewed items from session
    $viewed_products = WC()->session->get('recently_viewed_products', []);
    foreach ($viewed_products as $product_id) {
        $product = wc_get_product($product_id);
        if ($product) {
            $context['recently_viewed'][] = [
                'name' => $product->get_name(),
                'product_id' => $product->get_id()
            ];
        }
    }

    return $context;
}

    public function track_product_update($product_id) {
        update_post_meta($product_id, '_mxchat_last_updated', current_time('mysql'));
    }

    public function track_product_deletion($product_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mxchat_woo_product_mentions';
        $wpdb->delete($table_name, ['product_id' => $product_id], ['%d']);
    }
    
private function process_markdown($text) {
    // Only process bold and italic, let the main chat handler deal with line breaks
    $text = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $text);
    $text = preg_replace('/\*(.*?)\*/', '<em>$1</em>', $text);
    
    // Don't process newlines here - let the main chat handler do it
    return $text;
}

public function mxchat_handle_product_inquiry($default, $message, $user_id, $session_id) {
    //error_log("🔍 MXCHAT: Starting product search for message: '$message'");

    // Sanitize user ID and find product
    $sanitized_user_id = sanitize_key($user_id);
    $product_id = $this->find_product_in_message($message);
    
    if (!$product_id) {
        $product_id = get_transient('mxchat_last_discussed_product_' . $sanitized_user_id);
        //error_log("🔍 MXCHAT: Using transient product ID: " . ($product_id ?: 'None found'));
    }

    // Handle product inquiries
    if ($product_id && class_exists('WooCommerce')) {
        $product = wc_get_product($product_id);
        
        if ($product) {
            //error_log("✅ MXCHAT: Found product: " . $product->get_name() . " (ID: $product_id)");

            // Prepare product details
            $product_name = esc_html($product->get_name());
            $product_price = $product->get_price_html();
            $product_image_url = esc_url(wp_get_attachment_url($product->get_image_id()));
            $product_url = esc_url(get_permalink($product_id));
            $product_id_attr = esc_attr($product_id);
            $product_description = $product->get_description() ?: $product->get_short_description();

            // Check if prices should be shown
            $show_prices = isset($this->options['show_product_prices']) && $this->options['show_product_prices'] === 'on';
            // Default to true for backward compatibility
            if (!isset($this->options['show_product_prices'])) {
                $show_prices = true;
            }

            // Check if add to cart button should be shown
            $show_add_to_cart = isset($this->options['show_add_to_cart_button']) && $this->options['show_add_to_cart_button'] === 'on';
            // Default to true for backward compatibility
            if (!isset($this->options['show_add_to_cart_button'])) {
                $show_add_to_cart = true;
            }

            // Build AI prompt - include pricing info conditionally AND language detection
            $ai_prompt = "IMPORTANT: The user's current message is: '{$message}'\n\n";
            $ai_prompt .= "CRITICAL INSTRUCTION: You MUST respond in the EXACT SAME language that the user used in their message above. ";
            $ai_prompt .= "Detect the language from their message and use only that language in your response.\n\n";

            $ai_prompt .= "CONTEXT: You are continuing an ongoing conversation with this customer. ";
            $ai_prompt .= "Do NOT introduce yourself or start with greetings like 'Hello!' or 'Hi there!' - you're already mid-conversation. ";
            $ai_prompt .= "Simply answer their question naturally and directly.\n\n";

            $ai_prompt .= "PRODUCT INFORMATION:\n";
            $ai_prompt .= "- Name: '{$product_name}'\n";

            if ($show_prices) {
                $ai_prompt .= "- Price: " . strip_tags($product_price) . "\n";
            }

            if ($product_description) {
                $ai_prompt .= "- Description: {$product_description}\n\n";
            }

            $ai_prompt .= "\nINSTRUCTIONS:\n";
            $ai_prompt .= "1. Continue the conversation naturally - do not restart with introductions\n";
            $ai_prompt .= "2. Address their specific question directly and concisely\n";
            $ai_prompt .= "3. Keep your response brief and appropriate for a chatbot (2-3 sentences max)\n";
            $ai_prompt .= "4. Use a friendly, conversational tone\n";
            $ai_prompt .= "5. REMEMBER: Respond in their language\n";

            if (!$show_prices) {
                $ai_prompt .= "6. If they ask about pricing, let them know pricing details are available but don't mention specific contact instructions\n";
            }

            // Get AI response and log length for debugging
            $ai_response = MxChat_Woo_Util::mxchat_call_ai_api($ai_prompt, $this->api_key);
            //error_log("🔍 MXCHAT: AI response received, length: " . strlen($ai_response['text']));

            // Process the markdown before returning
            $processed_text = $this->process_markdown($ai_response['text']);

            // Build product card HTML based on settings
            $nonce = wp_create_nonce('mxchat_nonce');
            $product_card_html = '<div class="mxchat-product-card" data-nonce="' . $nonce . '">';
            
            // Product link and image (always shown)
            $product_card_html .= '<a href="' . $product_url . '" target="_blank" class="mxchat-product-link">';
            $product_card_html .= '<img src="' . $product_image_url . '" alt="' . $product_name . '" class="mxchat-product-image" />';
            $product_card_html .= '<h3 class="mxchat-product-name">' . $product_name . '</h3>';
            $product_card_html .= '</a>';
            
            // Price (conditional)
            if ($show_prices) {
                $product_card_html .= '<div class="mxchat-product-price" id="mxchat-price-' . $product_id_attr . '">' . $product_price . '</div>';
            }
            
            // Check if product has variations
            if ($product->is_type('variable')) {
                $product_card_html .= $this->build_variation_selectors($product);
            }
            
            // Add to cart button (conditional)
            if ($show_add_to_cart) {
                // Get custom button text from options
                $button_text = isset($this->options['add_to_cart_button_text']) && !empty($this->options['add_to_cart_button_text'])
                    ? esc_html($this->options['add_to_cart_button_text'])
                    : esc_html__('Add to Cart', 'mxchat-woo');
            
                $product_card_html .= '<button type="button" ';
                $product_card_html .= 'class="mxchat-add-to-cart-button" ';
                $product_card_html .= 'data-product-id="' . $product_id_attr . '" ';
                $product_card_html .= 'data-nonce="' . $nonce . '" ';
                $product_card_html .= 'data-product-type="' . esc_attr($product->get_type()) . '" ';
                $product_card_html .= 'data-original-text="' . esc_attr($button_text) . '" ';
                $product_card_html .= 'data-mxchat-action="add-to-cart">' . $button_text . '</button>';
            }
            
            $product_card_html .= '</div>';

            // Set transient and return response
            set_transient('mxchat_last_discussed_product_' . $sanitized_user_id, $product_id, HOUR_IN_SECONDS);
            //error_log("✅ MXCHAT: Successfully generated product card for ID: $product_id");
            
            return [
                'text' => $processed_text,
                'html' => $product_card_html
            ];
        }
        //error_log("❌ MXCHAT: Failed to retrieve product for ID: $product_id");
    }

    //error_log("❌ MXCHAT: No matching product found");
    return [
        'text' => esc_html__("I couldn't find that specific product. Could you please be more specific?", 'mxchat'),
        'html' => ''
    ];
}

private function build_variation_selectors($product) {
    if (!$product->is_type('variable')) {
        return '';
    }

    $available_variations = $product->get_available_variations();
    $variation_attributes = $product->get_variation_attributes();
    
    if (empty($variation_attributes)) {
        return '';
    }

    $html = '<div class="mxchat-variation-selectors">';
    
    foreach ($variation_attributes as $attribute_name => $options) {
        $attribute_label = wc_attribute_label($attribute_name);
        $sanitized_name = sanitize_title($attribute_name);
        
        $html .= '<div class="mxchat-variation-option">';
        $html .= '<label for="mxchat-' . esc_attr($sanitized_name) . '-' . esc_attr($product->get_id()) . '">';
        $html .= esc_html($attribute_label) . ':</label>';
        $html .= '<select id="mxchat-' . esc_attr($sanitized_name) . '-' . esc_attr($product->get_id()) . '" ';
        $html .= 'class="mxchat-variation-select" ';
        $html .= 'data-attribute="' . esc_attr($attribute_name) . '" ';
        $html .= 'data-product-id="' . esc_attr($product->get_id()) . '">';
        $html .= '<option value="">' . esc_html__('Choose an option', 'mxchat-woo') . '</option>';
        
        foreach ($options as $option) {
            if (empty($option)) continue;
            
            $html .= '<option value="' . esc_attr($option) . '">';
            $html .= esc_html($option);
            $html .= '</option>';
        }
        
        $html .= '</select>';
        $html .= '</div>';
    }
    
    $html .= '</div>';
    
    // Add variations data as JSON for JavaScript
    $html .= '<script type="application/json" class="mxchat-variations-data" data-product-id="' . esc_attr($product->get_id()) . '">';
    $html .= wp_json_encode($available_variations);
    $html .= '</script>';
    
    return $html;
}

private function get_variation_by_attributes($product_id, $attributes) {
    if (!$attributes || !is_array($attributes)) {
        return null;
    }

    $product = wc_get_product($product_id);
    if (!$product || !$product->is_type('variable')) {
        return null;
    }

    $available_variations = $product->get_available_variations();
    
    foreach ($available_variations as $variation_data) {
        $variation_attributes = $variation_data['attributes'];
        $match = true;
        
        foreach ($attributes as $attr_name => $attr_value) {
            $variation_attr_key = 'attribute_' . sanitize_title($attr_name);
            
            if (!isset($variation_attributes[$variation_attr_key])) {
                $match = false;
                break;
            }
            
            // Check if variation attribute matches (empty means any value is acceptable)
            if ($variation_attributes[$variation_attr_key] !== '' && 
                $variation_attributes[$variation_attr_key] !== $attr_value) {
                $match = false;
                break;
            }
        }
        
        if ($match) {
            return $variation_data;
        }
    }
    
    return null;
}

public function get_variation_data() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
        wp_send_json_error('Invalid nonce.');
    }

    $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
    $attributes = isset($_POST['attributes']) ? $_POST['attributes'] : array();

    if (!$product_id) {
        wp_send_json_error('Invalid product ID.');
    }

    $variation_data = $this->get_variation_by_attributes($product_id, $attributes);
    
    if ($variation_data) {
        wp_send_json_success($variation_data);
    } else {
        wp_send_json_error('Variation not found.');
    }
}

public function add_variation_to_cart() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
        wp_send_json_error('Invalid nonce.');
    }

    $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
    $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;
    $attributes = isset($_POST['attributes']) ? $_POST['attributes'] : array();
    $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;

    if (!$product_id || !$variation_id) {
        wp_send_json_error('Invalid product or variation ID.');
    }

    // Sanitize attributes
    $sanitized_attributes = array();
    foreach ($attributes as $key => $value) {
        $sanitized_key = sanitize_text_field($key);
        $sanitized_value = sanitize_text_field($value);
        $sanitized_attributes[$sanitized_key] = $sanitized_value;
    }

    try {
        $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $sanitized_attributes);
        
        if ($cart_item_key) {
            wp_send_json_success(array(
                'message' => __('Product added to cart successfully!', 'mxchat-woo'),
                'cart_item_key' => $cart_item_key
            ));
        } else {
            wp_send_json_error('Failed to add product to cart.');
        }
    } catch (Exception $e) {
        wp_send_json_error('Error: ' . $e->getMessage());
    }
}

/**
 * Add simple product to cart via AJAX
 */
public function add_simple_product_to_cart() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mxchat_nonce')) {
        wp_send_json_error('Invalid nonce.');
    }

    $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
    $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;

    if (!$product_id) {
        wp_send_json_error('Invalid product ID.');
    }

    // Verify product exists and is purchasable
    $product = wc_get_product($product_id);
    if (!$product || !$product->is_purchasable()) {
        wp_send_json_error('Product is not available for purchase.');
    }

    try {
        $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity);
        
        if ($cart_item_key) {
            wp_send_json_success(array(
                'message' => __('Product added to cart successfully!', 'mxchat-woo'),
                'cart_item_key' => $cart_item_key,
                'product_id' => $product_id,
                'quantity' => $quantity
            ));
        } else {
            wp_send_json_error('Failed to add product to cart.');
        }
    } catch (Exception $e) {
        wp_send_json_error('Error: ' . $e->getMessage());
    }
}

private function find_product_in_message($message) {
    //error_log("🔍 MXCHAT DEBUG: Starting product search for: '$message'");

    if (!function_exists('wc_get_products')) {
        //error_log("❌ MXCHAT DEBUG: WooCommerce function not available");
        return null;
    }

    // Get all published products first
    $all_products = wc_get_products([
        'status' => 'publish',
        'limit' => -1,
        'return' => 'objects'
    ]);

    //error_log("🔍 MXCHAT DEBUG: Found " . count($all_products) . " total products");

    // Clean up search query
    $search_query = strtolower(trim(str_replace(['?', '!', '.', ','], '', $message)));
    $search_words = array_filter(explode(' ', $search_query));

    //error_log("🔍 MXCHAT DEBUG: Searching for words: " . implode(', ', $search_words));

    $matches = [];
    foreach ($all_products as $product) {
        $score = 0;
        $product_name = strtolower($product->get_name());
        $product_sku = strtolower($product->get_sku());
        $product_desc = strtolower($product->get_short_description() . ' ' . $product->get_description());

        // Calculate name match score
        similar_text($search_query, $product_name, $name_percentage);
        $score += $name_percentage;

        // Check each word from search query
        foreach ($search_words as $word) {
            if (strlen($word) < 3) continue;

            // Exact product name matches
            if (strpos($product_name, $word) !== false) {
                $score += 30;
                //error_log("🔍 MXCHAT DEBUG: Word '$word' found in product name: $product_name");
            }

            // SKU matches
            if (!empty($product_sku) && strpos($product_sku, $word) !== false) {
                $score += 20;
            }

            // Description matches
            if (strpos($product_desc, $word) !== false) {
                $score += 10;
            }
        }

        // Boost score for products that are purchasable and in stock
        if ($product->is_purchasable() && $product->is_in_stock()) {
            $score += 15;
        }

        if ($score > 0) {
            $matches[] = [
                'product_id' => $product->get_id(),
                'name' => $product->get_name(),
                'score' => $score
            ];
            //error_log("✅ MXCHAT DEBUG: Product '{$product->get_name()}' scored: $score");
        }
    }

    // Sort matches by score
    if (!empty($matches)) {
        usort($matches, function($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        // Log all matches for debugging
        foreach ($matches as $index => $match) {
            //error_log("🔍 MXCHAT DEBUG: Match #" . ($index + 1) . ": {$match['name']} (Score: {$match['score']})");
        }

        // Use the best match if it has a good enough score
        if ($matches[0]['score'] >= 30) {
            //error_log("✅ MXCHAT DEBUG: Selected best match: {$matches[0]['name']} with score {$matches[0]['score']}");
            return $matches[0]['product_id'];
        }
    }

    //error_log("❌ MXCHAT DEBUG: No qualifying matches found");
    return null;
}
    // And add these methods to your class:
public function improve_product_search($search, $wp_query) {
    if (!$wp_query->is_main_query() || !$wp_query->is_search() || 'product' !== $wp_query->get('post_type')) {
        return $search;
    }
    
    global $wpdb;
    
    $search_term = $wp_query->get('s');
    if (empty($search_term)) {
        return $search;
    }

    $search = '';
    $searchand = '';
    $terms = explode(' ', $search_term);
    
    foreach ($terms as $term) {
        $term = esc_sql($wpdb->esc_like($term));
        $search .= "{$searchand}(
            ($wpdb->posts.post_title LIKE '%{$term}%')
            OR ($wpdb->posts.post_content LIKE '%{$term}%')
            OR EXISTS (
                SELECT * FROM $wpdb->postmeta
                WHERE post_id = $wpdb->posts.ID
                AND (
                    (meta_key = '_sku' AND meta_value LIKE '%{$term}%')
                    OR (meta_key = '_product_attributes' AND meta_value LIKE '%{$term}%')
                )
            )
        )";
        $searchand = ' AND ';
    }

    if (!empty($search)) {
        $search = " AND ({$search}) ";
    }

    return $search;
}

public function order_by_relevance($clauses, $wp_query) {
    if (!$wp_query->is_main_query() || !$wp_query->is_search() || 'product' !== $wp_query->get('post_type')) {
        return $clauses;
    }

    global $wpdb;

    $search_term = $wp_query->get('s');
    if (empty($search_term)) {
        return $clauses;
    }

    $search_terms = explode(' ', $search_term);
    $relevance_terms = array();

    foreach ($search_terms as $term) {
        $term = esc_sql($wpdb->esc_like($term));
        $relevance_terms[] = "
            (
                (CASE WHEN $wpdb->posts.post_title LIKE '%{$term}%' THEN 10 ELSE 0 END)
                + (CASE WHEN $wpdb->posts.post_content LIKE '%{$term}%' THEN 5 ELSE 0 END)
                + (CASE WHEN EXISTS (
                    SELECT * FROM $wpdb->postmeta
                    WHERE post_id = $wpdb->posts.ID
                    AND meta_key = '_sku'
                    AND meta_value LIKE '%{$term}%'
                ) THEN 8 ELSE 0 END)
            )
        ";
    }

    $clauses['fields'] .= ", (" . implode(' + ', $relevance_terms) . ") AS relevance_score";
    $clauses['orderby'] = "relevance_score DESC, " . $clauses['orderby'];

    return $clauses;
}



public function mxchat_handle_product_change($post_id, $post, $update) {
    if ($post->post_type !== 'product') {
        return;
    }

    if ($post->post_status === 'publish') {
        add_action('shutdown', function() use ($post_id) {
            $product = wc_get_product($post_id);
            if ($product) {
                $this->mxchat_store_product_embedding($product);
            }
        });
    }
}

private function mxchat_store_product_embedding($product) {
    // First check WooCommerce addon's own setting
    if (!isset($this->options['enable_woocommerce_integration']) ||
        !in_array($this->options['enable_woocommerce_integration'], ['1', 'on'])) {
        return;
    }

    $source_url = get_permalink($product->get_id());

    // Build description
    $regular_price = $product->get_regular_price();
    $sale_price = $product->get_sale_price();
    $description = $product->get_description() . "\n\n" .
                  esc_html__('Short Description:', 'mxchat-woo') . " " . $product->get_short_description() . "\n" .
                  esc_html__('Price:', 'mxchat-woo') . " " . $regular_price . "\n" .
                  esc_html__('Sale Price:', 'mxchat-woo') . " " . ($sale_price ?: esc_html__('N/A', 'mxchat-woo')) . "\n" .
                  esc_html__('SKU:', 'mxchat-woo') . " " . $product->get_sku();

    // Generate embedding using the core plugin's API key
    $embedding_vector = MxChat_Woo_Util::generate_embedding($description, $this->api_key);
    if (!is_array($embedding_vector)) {
        return;
    }

    // Use the same Pinecone checking logic from your util file
    $pinecone_settings = get_option('mxchat_pinecone_addon_options');
    $use_pinecone = false;

    if ($pinecone_settings && is_array($pinecone_settings)) {
        $use_pinecone = (
            isset($pinecone_settings['mxchat_use_pinecone']) &&
            $pinecone_settings['mxchat_use_pinecone'] === '1' &&
            !empty($pinecone_settings['mxchat_pinecone_api_key']) &&
            !empty($pinecone_settings['mxchat_pinecone_host']) &&
            !empty($pinecone_settings['mxchat_pinecone_index']) &&
            !empty($pinecone_settings['mxchat_pinecone_environment'])
        );
    }

    if ($use_pinecone) {
        $result = MxChat_Woo_Util::store_in_pinecone_woocommerce(
            $embedding_vector,
            $description,
            $source_url,
            $pinecone_settings['mxchat_pinecone_api_key'],
            $pinecone_settings['mxchat_pinecone_environment'],
            $pinecone_settings['mxchat_pinecone_index']
        );

        if (!$result['success']) {
            // Fallback to WordPress DB if Pinecone storage fails
            MxChat_Woo_Util::store_in_wordpress_db_woocommerce($description, $source_url, $embedding_vector);
        }
    } else {
        // Use WordPress DB storage
        MxChat_Woo_Util::store_in_wordpress_db_woocommerce($description, $source_url, $embedding_vector);
    }
}

public function mxchat_handle_product_delete($post_id) {
    if (get_post_type($post_id) !== 'product') {
        return;
    }

    $source_url = get_permalink($post_id);

    // Check for Pinecone addon using the same logic as above
    $pinecone_settings = get_option('mxchat_pinecone_addon_options');
    $use_pinecone = false;

    if ($pinecone_settings && is_array($pinecone_settings)) {
        $use_pinecone = (
            isset($pinecone_settings['mxchat_use_pinecone']) &&
            $pinecone_settings['mxchat_use_pinecone'] === '1' &&
            !empty($pinecone_settings['mxchat_pinecone_api_key']) &&
            !empty($pinecone_settings['mxchat_pinecone_host']) &&
            !empty($pinecone_settings['mxchat_pinecone_index']) &&
            !empty($pinecone_settings['mxchat_pinecone_environment'])
        );
    }

    if ($use_pinecone) {
        MxChat_Woo_Util::delete_from_pinecone_woocommerce(
            array($source_url),
            $pinecone_settings['mxchat_pinecone_api_key'],
            $pinecone_settings['mxchat_pinecone_environment'],
            $pinecone_settings['mxchat_pinecone_index']
        );
    } else {
        // Fallback to WordPress DB deletion
        global $wpdb;
        $table_name = $wpdb->prefix . 'mxchat_system_prompt_content';
        $wpdb->delete($table_name, array('source_url' => $source_url), array('%s'));
    }
}


    
}