<?php
/**
 * Utility functions for MxChat WooCommerce integration
 */
class MxChat_Woo_Util {
    
public static function generate_embedding($text, $api_key) {
    // Get main plugin options for model selection
    $main_plugin_options = get_option('mxchat_options', array());
    $selected_model = $main_plugin_options['embedding_model'] ?? 'text-embedding-ada-002';
    
    // Determine endpoint and API key based on model
    if (strpos($selected_model, 'voyage') === 0) {
        $endpoint = 'https://api.voyageai.com/v1/embeddings';
        $api_key = $main_plugin_options['voyage_api_key'] ?? '';
    } elseif (strpos($selected_model, 'gemini-embedding') === 0) {
        $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/' . $selected_model . ':embedContent';
        $api_key = $main_plugin_options['gemini_api_key'] ?? '';
    } else {
        $endpoint = 'https://api.openai.com/v1/embeddings';
        // Use the passed $api_key parameter for OpenAI
    }
    
    // Prepare request body and headers based on provider
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        // Gemini API format
        $body = wp_json_encode([
            'model' => 'models/' . $selected_model,
            'content' => [
                'parts' => [
                    ['text' => $text]
                ]
            ],
            'outputDimensionality' => 1536
        ]);
        
        // Gemini uses API key as query parameter
        $endpoint .= '?key=' . $api_key;
        $headers = [
            'Content-Type' => 'application/json'
        ];
    } else {
        // OpenAI/Voyage API format
        $request_body = [
            'input' => $text,
            'model' => $selected_model
        ];
        
        // Add output_dimension for voyage-3-large
        if ($selected_model === 'voyage-3-large') {
            $request_body['output_dimension'] = 2048;
        }
        
        $body = wp_json_encode($request_body);
        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $api_key
        ];
    }
    
    $args = [
        'body' => $body,
        'headers' => $headers,
        'timeout' => 60,
        'redirection' => 5,
        'blocking' => true,
        'httpversion' => '1.0',
        'sslverify' => true,
    ];
    
    $response = wp_remote_post($endpoint, $args);
    
    if (is_wp_error($response)) {
        error_log('MxChat Addon: Error generating embedding - ' . $response->get_error_message());
        return null;
    }
    
    $response_body = json_decode(wp_remote_retrieve_body($response), true);
    
    // Handle different response formats based on provider
    if (strpos($selected_model, 'gemini-embedding') === 0) {
        // Gemini API response format
        if (isset($response_body['embedding']['values']) && is_array($response_body['embedding']['values'])) {
            return $response_body['embedding']['values'];
        } else {
            error_log('MxChat Addon: Invalid response from Gemini API - ' . wp_json_encode($response_body));
            return null;
        }
    } else {
        // OpenAI/Voyage API response format
        if (isset($response_body['data'][0]['embedding']) && is_array($response_body['data'][0]['embedding'])) {
            return $response_body['data'][0]['embedding'];
        } else {
            error_log('MxChat Addon: Invalid response from API - ' . wp_json_encode($response_body));
            return null;
        }
    }
}

public static function mxchat_find_relevant_content_woo($user_embedding) {
    // Retrieve the add-on settings from the database.
    $addon_options = get_option('mxchat_pinecone_addon_options', array());

    // Determine whether Pinecone is enabled.
    $use_pinecone = (isset($addon_options['mxchat_use_pinecone']) && $addon_options['mxchat_use_pinecone'] === '1') ? 1 : 0;

    if ($use_pinecone === 1) {
        return self::find_relevant_content_pinecone_woo($user_embedding);
    } else {
        return self::find_relevant_content_wordpress_woo($user_embedding);
    }
}

private static function find_relevant_content_wordpress_woo($user_embedding) {
   error_log('🔍 MXCHAT DEBUG: Starting WordPress content search');
   global $wpdb;
   $system_prompt_table = $wpdb->prefix . 'mxchat_system_prompt_content';
   $cache_key = 'mxchat_system_prompt_embeddings';
   $batch_size = 500;

   // Get similarity threshold
   $similarity_threshold = ((int) get_option('mxchat_similarity_threshold', 80)) / 100;
   error_log('🔍 MXCHAT DEBUG: Using similarity threshold: ' . $similarity_threshold);

   // Retrieve embeddings from cache or database
   $embeddings = wp_cache_get($cache_key, 'mxchat_system_prompts');
   if ($embeddings === false) {
       error_log('🔍 MXCHAT DEBUG: Cache miss - loading embeddings from database');
       $embeddings = [];
       $offset = 0;

       // Load in batches and build cache
       do {
           $query = $wpdb->prepare(
               "SELECT id, embedding_vector 
               FROM {$system_prompt_table}
               LIMIT %d OFFSET %d",
               $batch_size,
               $offset
           );

           $batch = $wpdb->get_results($query);
           if (empty($batch)) {
               break;
           }

           $embeddings = array_merge($embeddings, $batch);
           $offset += $batch_size;
           error_log('🔍 MXCHAT DEBUG: Loaded batch of ' . count($batch) . ' embeddings, total: ' . count($embeddings));
           unset($batch);
       } while (true);

       if (empty($embeddings)) {
           error_log('❌ MXCHAT DEBUG: No embeddings found in database');
           return '';
       }
       wp_cache_set($cache_key, $embeddings, 'mxchat_system_prompts', 3600);
   } else {
       error_log('✅ MXCHAT DEBUG: Using cached embeddings, count: ' . count($embeddings));
   }

   $all_results = [];
   $highest_similarity = 0;
   $lowest_similarity = 1;
   $similarity_sum = 0;
   $similarity_count = 0;

   foreach ($embeddings as $embedding) {
       $database_embedding = $embedding->embedding_vector
           ? unserialize($embedding->embedding_vector, ['allowed_classes' => false])
           : null;

       if (is_array($database_embedding) && is_array($user_embedding)) {
           $similarity = self::mxchat_calculate_cosine_similarity($user_embedding, $database_embedding);
           
           // Track similarity stats
           $highest_similarity = max($highest_similarity, $similarity);
           $lowest_similarity = min($lowest_similarity, $similarity);
           $similarity_sum += $similarity;
           $similarity_count++;

           $all_results[] = [
               'id' => $embedding->id,
               'similarity' => $similarity
           ];
       }
       unset($database_embedding);
   }

   // Log similarity statistics
   if ($similarity_count > 0) {
       $avg_similarity = $similarity_sum / $similarity_count;
   }

   // Sort all results by similarity (highest first)
   usort($all_results, function ($a, $b) {
       return $b['similarity'] <=> $a['similarity'];
   });

   // Get top 5 results regardless of threshold
   $top_results = array_slice($all_results, 0, 5);
   
   error_log('🔍 MXCHAT DEBUG: Using top ' . count($top_results) . ' results');

   // Log matches and indicate if they met threshold
   foreach ($top_results as $index => $result) {

   }

   $content = '';
   foreach ($top_results as $result) {
       $chunk_content = self::fetch_content_with_product_links($result['id']);
       if (!empty($chunk_content)) {
           if (strpos($chunk_content, '{"document_type":"pdf"') !== false) {
               $surrounding_content = $wpdb->get_results($wpdb->prepare(
                   "SELECT article_content FROM {$system_prompt_table}
                   WHERE id IN (
                       (SELECT id FROM {$system_prompt_table} WHERE id < %d ORDER BY id DESC LIMIT 1),
                       (SELECT id FROM {$system_prompt_table} WHERE id > %d ORDER BY id ASC LIMIT 1)
                   )",
                   $result['id'],
                   $result['id']
               ));
               if (!empty($surrounding_content[0])) {
                   $content .= $surrounding_content[0]->article_content . "\n\n";
               }
               $content .= $chunk_content . "\n\n";
               if (!empty($surrounding_content[1])) {
                   $content .= $surrounding_content[1]->article_content . "\n\n";
               }
           } else {
               $content .= $chunk_content . "\n\n";
           }
       }
   }

   error_log('✅ MXCHAT DEBUG: Final content length: ' . strlen(trim($content)));
   return trim($content);
}


private static function fetch_content_with_product_links($most_relevant_id) {
    global $wpdb;
    $system_prompt_table = $wpdb->prefix . 'mxchat_system_prompt_content';

    // Fetch the article content and associated product URL
    $query = $wpdb->prepare(
        "SELECT article_content, source_url 
        FROM {$system_prompt_table} 
        WHERE id = %d",
        $most_relevant_id
    );
    $result = $wpdb->get_row($query);

    if ($result && !empty($result->article_content)) {
        // Append the product link to the content if available
        $content = $result->article_content;
        if (!empty($result->source_url)) {
            $content .= "\n\nFor more details, check out this product: " . esc_url($result->source_url);
        }
        return $content;
    }

    return ''; 
}

private static function find_relevant_content_pinecone_woo($user_embedding) {
    error_log('🔍 MXCHAT DEBUG: Starting Pinecone content search');
    
    $options = get_option('mxchat_pinecone_addon_options', array());
    $api_key = $options['mxchat_pinecone_api_key'] ?? '';
    $host = $options['mxchat_pinecone_host'] ?? '';

    if (empty($host) || empty($api_key)) {
        error_log('❌ MXCHAT DEBUG: Pinecone credentials not properly configured');
        return '';
    }

    // Get similarity threshold (for logging purposes only)
    $similarity_threshold = ((int) get_option('mxchat_similarity_threshold', 80)) / 100;
    error_log('🔍 MXCHAT DEBUG: Reference similarity threshold: ' . $similarity_threshold);

    // Prepare the query request for Pinecone
    $api_endpoint = "https://{$host}/query";
    $request_body = array(
        'vector' => $user_embedding,
        'topK' => 5, // Get top 5 matches regardless of score
        'includeMetadata' => true,
        'includeValues' => true
    );

    error_log('🔍 MXCHAT DEBUG: Querying Pinecone for top 5 matches');

    $response = wp_remote_post($api_endpoint, array(
        'headers' => array(
            'Api-Key' => $api_key,
            'accept' => 'application/json',
            'content-type' => 'application/json'
        ),
        'body' => wp_json_encode($request_body),
        'timeout' => 30
    ));

    if (is_wp_error($response)) {
        error_log('❌ MXCHAT DEBUG: Pinecone query error: ' . $response->get_error_message());
        return '';
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        error_log('❌ MXCHAT DEBUG: Pinecone API error: ' . wp_remote_retrieve_body($response));
        return '';
    }

    $results = json_decode(wp_remote_retrieve_body($response), true);
    if (empty($results['matches'])) {
        error_log('❌ MXCHAT DEBUG: No matches returned from Pinecone');
        return '';
    }

    error_log('✅ MXCHAT DEBUG: Found ' . count($results['matches']) . ' matches from Pinecone');

    // Initialize the final content
    $content = '';
    
    // Process each match - no threshold filtering
    foreach ($results['matches'] as $index => $match) {


        if (!empty($match['metadata']['text']) && !empty($match['metadata']['source_url'])) {
            // Add content with citation
            $content .= $match['metadata']['text'] . "\n";
            $content .= "Source: " . $match['metadata']['source_url'] . "\n\n";
            error_log('✅ MXCHAT DEBUG: Added content from: ' . $match['metadata']['source_url']);
        }
    }

    $final_content = trim($content);
    error_log('✅ MXCHAT DEBUG: Final content length: ' . strlen($final_content));
    
    return $final_content;
}

public static function mxchat_calculate_cosine_similarity($vectorA, $vectorB) {
    if (!is_array($vectorA) || !is_array($vectorB) || empty($vectorA) || empty($vectorB)) {
        return 0;
    }

    $dotProduct = array_sum(array_map(function ($a, $b) {
        return $a * $b;
    }, $vectorA, $vectorB));
    $normA = sqrt(array_sum(array_map(function ($a) {
        return $a * $a;
    }, $vectorA)));
    $normB = sqrt(array_sum(array_map(function ($b) {
        return $b * $b;
    }, $vectorB)));

    if ($normA == 0 || $normB == 0) {
        return 0;
    }

    return $dotProduct / ($normA * $normB);
}

public static function mxchat_call_ai_api($prompt, $api_key) {
    error_log("🔍 MXCHAT DEBUG: Attempting AI API call");
    
    // Check if prompt is an array and convert it
    if (is_array($prompt)) {
        error_log("⚠️ MXCHAT DEBUG: Prompt is an array, converting to string");
        error_log("⚠️ MXCHAT DEBUG: Prompt array: " . print_r($prompt, true));
        $prompt = json_encode($prompt); // or use implode() or serialize() depending on your needs
    }
    
    error_log("🔍 MXCHAT DEBUG: Prompt length: " . strlen($prompt));

    if (empty($api_key)) {
        error_log("❌ MXCHAT DEBUG: API key is not set");
        return ['text' => esc_html__('API key is not set.', 'mxchat')];
    }

    $main_plugin_options = get_option('mxchat_options', array());
    $selected_model = $main_plugin_options['model'] ?? 'gpt-4o';
    
    error_log("🔍 MXCHAT DEBUG: Using model: " . $selected_model);

    // Get all API keys
    $xai_api_key = $main_plugin_options['xai_api_key'] ?? '';
    $claude_api_key = $main_plugin_options['claude_api_key'] ?? '';
    $deepseek_api_key = $main_plugin_options['deepseek_api_key'] ?? '';
    $gemini_api_key = $main_plugin_options['gemini_api_key'] ?? '';

    // Extract model prefix to determine the provider
    $model_parts = explode('-', $selected_model);
    $provider = strtolower($model_parts[0]);
    error_log("🔍 MXCHAT DEBUG: Detected provider: " . $provider);

    try {
        switch ($provider) {
            case 'gemini':
                if (empty($gemini_api_key)) {
                    throw new Exception(esc_html__('Google Gemini API key is not configured', 'mxchat'));
                }
                return self::mxchat_call_gemini_api($prompt, $gemini_api_key, $selected_model);

            case 'claude':
                if (empty($claude_api_key)) {
                    throw new Exception(esc_html__('Claude API key is not configured', 'mxchat'));
                }
                return self::mxchat_call_claude_api($prompt, $claude_api_key, $selected_model);

            case 'grok':
                if (empty($xai_api_key)) {
                    throw new Exception(esc_html__('X.AI API key is not configured', 'mxchat'));
                }
                return self::mxchat_call_xai_api($prompt, $xai_api_key, $selected_model);

            case 'deepseek':
                if (empty($deepseek_api_key)) {
                    throw new Exception(esc_html__('DeepSeek API key is not configured', 'mxchat'));
                }
                return self::mxchat_call_deepseek_api($prompt, $deepseek_api_key, $selected_model);

            case 'gpt':
            default:
                return self::mxchat_call_openai_api($prompt, $api_key, $selected_model);
        }
    } catch (Exception $e) {
        error_log("❌ MXCHAT DEBUG: API Error: " . $e->getMessage());
        error_log("❌ MXCHAT DEBUG: Stack trace: " . $e->getTraceAsString());
        return ['text' => esc_html__('Error communicating with AI API.', 'mxchat')];
    }
}

private static function mxchat_call_openai_api($prompt, $api_key, $model) {
    $url = 'https://api.openai.com/v1/chat/completions';
    $messages = [
        [
            'role'    => 'system',
            'content' => 'You are a helpful assistant. Respond in the same language as the user.',
        ],
        [
            'role'    => 'user',
            'content' => $prompt,
        ],
    ];

    // Check if the model supports temperature parameter
    // GPT-5 and some other models only support default temperature
    $request_body = [
        'model'    => $model,
        'messages' => $messages,
    ];
    
    // Only add temperature for models that support it
    // GPT-5 doesn't support custom temperature
    if (strpos($model, 'gpt-5') === false) {
        $request_body['temperature'] = 0.7;
    }

    $args = [
        'headers' => [
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type'  => 'application/json',
        ],
        'body'    => wp_json_encode($request_body),
        'timeout' => 10,
        'method'  => 'POST',
    ];

    error_log("🔍 MXCHAT DEBUG: Sending OpenAI API request with model: " . $model);
    if (isset($request_body['temperature'])) {
        error_log("🔍 MXCHAT DEBUG: Using temperature: " . $request_body['temperature']);
    } else {
        error_log("🔍 MXCHAT DEBUG: Using default temperature (model doesn't support custom temperature)");
    }
    
    $response = wp_remote_post($url, $args);

    if (is_wp_error($response)) {
        error_log("❌ MXCHAT DEBUG: OpenAI API Error: " . $response->get_error_message());
        throw new Exception($response->get_error_message());
    }

    $body = wp_remote_retrieve_body($response);
    $decoded_body = json_decode($body, true);

    if (isset($decoded_body['choices'][0]['message']['content'])) {
        error_log("✅ MXCHAT DEBUG: OpenAI API call successful");
        return ['text' => $decoded_body['choices'][0]['message']['content']];
    } else {
        error_log("❌ MXCHAT DEBUG: Unexpected OpenAI API response: " . wp_json_encode($decoded_body));
        throw new Exception('No response received from OpenAI API.');
    }
}

private static function mxchat_call_claude_api($prompt, $api_key, $model) {
   error_log("🔍 MXCHAT DEBUG: Starting Claude API call");

   // Create conversation history array with the prompt
   $messages = [
       [
           'role' => 'user',
           'content' => $prompt
       ]
   ];

   // Build request body
   $body = json_encode([
       'model' => $model,
       'max_tokens' => 1000,
       'temperature' => 0.8,
       'messages' => $messages,
       'system' => 'You are a helpful assistant. Respond in the same language as the user.'
   ]);

   // Set up API request
   $args = [
       'body' => $body,
       'headers' => [
           'Content-Type' => 'application/json',
           'x-api-key' => $api_key,
           'anthropic-version' => '2023-06-01'
       ],
       'timeout' => 60,
       'redirection' => 5,
       'blocking' => true,
       'httpversion' => '1.0',
       'sslverify' => true,
   ];

   error_log("🔍 MXCHAT DEBUG: Sending Claude API request");

   // Make API request
   $response = wp_remote_post('https://api.anthropic.com/v1/messages', $args);

   // Check for WordPress errors
   if (is_wp_error($response)) {
       error_log("❌ MXCHAT DEBUG: Claude API request error: " . $response->get_error_message());
       throw new Exception('Error connecting to Claude API');
   }

   // Check HTTP response code
   $http_code = wp_remote_retrieve_response_code($response);
   if ($http_code !== 200) {
       $error_body = wp_remote_retrieve_body($response);
       error_log("❌ MXCHAT DEBUG: Claude API HTTP error: " . $http_code . " - " . $error_body);
       
       // Try to extract error message from response
       $error_data = json_decode($error_body, true);
       $error_message = isset($error_data['error']['message']) ? 
           $error_data['error']['message'] : 
           "HTTP error " . $http_code;
           
       throw new Exception($error_message);
   }

   // Parse response
   $response_body = json_decode(wp_remote_retrieve_body($response), true);
   
   // Check for JSON decode errors
   if (json_last_error() !== JSON_ERROR_NONE) {
       error_log("❌ MXCHAT DEBUG: Claude API JSON decode error: " . json_last_error_msg());
       throw new Exception('Error processing Claude API response');
   }

   // Extract and validate response content
   if (isset($response_body['content']) && 
       is_array($response_body['content']) && 
       !empty($response_body['content']) && 
       isset($response_body['content'][0]['text'])) {
       
       error_log("✅ MXCHAT DEBUG: Claude API call successful");
       return ['text' => trim($response_body['content'][0]['text'])];
   }

   // Log unexpected response format
   error_log("❌ MXCHAT DEBUG: Claude API unexpected response format: " . print_r($response_body, true));
   throw new Exception('Unexpected response format from Claude API');
}

private static function mxchat_call_xai_api($prompt, $api_key, $model) {
   error_log("🔍 MXCHAT DEBUG: Starting X.AI API call");

   // Create conversation history with system prompt and user message
   $conversation_history = [
       [
           'role' => 'system',
           'content' => 'You are a helpful assistant. Respond in the same language as the user.'
       ],
       [
           'role' => 'user',
           'content' => $prompt
       ]
   ];

   // Build the request body
   $body = json_encode([
       'model' => $model,
       'messages' => $conversation_history,
       'temperature' => 0.8,
       'stream' => false
   ]);

   // Set up the API request
   $args = [
       'body' => $body,
       'headers' => [
           'Content-Type' => 'application/json',
           'Authorization' => 'Bearer ' . $api_key,
       ],
       'timeout' => 60,
       'redirection' => 5,
       'blocking' => true,
       'httpversion' => '1.0',
       'sslverify' => true,
   ];

   error_log("🔍 MXCHAT DEBUG: Sending X.AI API request");

   // Make the API request
   $response = wp_remote_post('https://api.x.ai/v1/chat/completions', $args);

   // Process the response
   if (is_wp_error($response)) {
       error_log("❌ MXCHAT DEBUG: X.AI API error: " . $response->get_error_message());
       throw new Exception('Error connecting to X.AI API');
   }

   $response_body = json_decode(wp_remote_retrieve_body($response), true);
   
   if (isset($response_body['choices'][0]['message']['content'])) {
       error_log("✅ MXCHAT DEBUG: X.AI API call successful");
       return ['text' => trim($response_body['choices'][0]['message']['content'])];
   } else {
       error_log("❌ MXCHAT DEBUG: Unexpected X.AI API response: " . wp_json_encode($response_body));
       throw new Exception('Unexpected response from X.AI API');
   }
}

private static function mxchat_call_deepseek_api($prompt, $api_key, $model) {
   error_log("🔍 MXCHAT DEBUG: Starting DeepSeek API call");

   // Create formatted conversation array
   $formatted_conversation = [
       [
           'role' => 'system',
           'content' => 'You are a helpful assistant. Respond in the same language as the user.'
       ],
       [
           'role' => 'user',
           'content' => $prompt
       ]
   ];

   $body = json_encode([
       'model' => $model,
       'messages' => $formatted_conversation,
       'temperature' => 0.8,
       'stream' => false
   ]);

   $args = [
       'body' => $body,
       'headers' => [
           'Content-Type' => 'application/json',
           'Authorization' => 'Bearer ' . $api_key,
       ],
       'timeout' => 60,
       'redirection' => 5,
       'blocking' => true,
       'httpversion' => '1.0',
       'sslverify' => true,
   ];

   error_log("🔍 MXCHAT DEBUG: Sending DeepSeek API request");

   $response = wp_remote_post('https://api.deepseek.com/v1/chat/completions', $args);

   if (is_wp_error($response)) {
       error_log("❌ MXCHAT DEBUG: DeepSeek API Error: " . $response->get_error_message());
       throw new Exception('Error connecting to DeepSeek API');
   }

   $response_body = wp_remote_retrieve_body($response);
   $decoded_response = json_decode($response_body, true);

   if (isset($decoded_response['choices'][0]['message']['content'])) {
       error_log("✅ MXCHAT DEBUG: DeepSeek API call successful");
       return ['text' => trim($decoded_response['choices'][0]['message']['content'])];
   } else {
       error_log("❌ MXCHAT DEBUG: DeepSeek API Response Format Error: " . print_r($decoded_response, true));
       throw new Exception('Unexpected response from DeepSeek API');
   }
}

private static function mxchat_call_gemini_api($prompt, $api_key, $model) {
    error_log("🔍 MXCHAT DEBUG: Starting Gemini API call");

    // Format messages for Gemini API
    $formatted_messages = [];
    
    // Add system message as the first user message with role prefix
    // Note: Gemini doesn't have a dedicated system role, so we use a prefixed user message
    $formatted_messages[] = [
        'role' => 'user',
        'parts' => [
            ['text' => "[System Instructions] You are a helpful assistant. Respond in the same language as the user."]
        ]
    ];

    // Add model response to acknowledge system instructions
    $formatted_messages[] = [
        'role' => 'model',
        'parts' => [
            ['text' => "I understand and will respond in the user's language."]
        ]
    ];
    
    // Add the actual user prompt
    $formatted_messages[] = [
        'role' => 'user',
        'parts' => [
            ['text' => $prompt]
        ]
    ];

    // Build the request body
    $body = json_encode([
        'contents' => $formatted_messages,
        'generationConfig' => [
            'temperature' => 0.7,
            'topP' => 0.95,
            'topK' => 40,
            'maxOutputTokens' => 1000,
        ],
        'safetySettings' => [
            [
                'category' => 'HARM_CATEGORY_HARASSMENT',
                'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
            ],
            [
                'category' => 'HARM_CATEGORY_HATE_SPEECH',
                'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
            ],
            [
                'category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT',
                'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
            ],
            [
                'category' => 'HARM_CATEGORY_DANGEROUS_CONTENT',
                'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
            ]
        ]
    ]);
    
    // Prepare the API endpoint
    $api_endpoint = 'https://generativelanguage.googleapis.com/v1/models/' . $model . ':generateContent?key=' . $api_key;
    
    // Set up the API request
    $args = [
        'body'        => $body,
        'headers'     => [
            'Content-Type' => 'application/json',
        ],
        'timeout'     => 60,
        'redirection' => 5,
        'blocking'    => true,
        'httpversion' => '1.0',
        'sslverify'   => true,
    ];
    
    error_log("🔍 MXCHAT DEBUG: Sending Gemini API request");
    
    // Make the API request
    $response = wp_remote_post($api_endpoint, $args);
    
    // Check for WordPress errors
    if (is_wp_error($response)) {
        error_log("❌ MXCHAT DEBUG: Gemini API request error: " . $response->get_error_message());
        throw new Exception('Error connecting to Gemini API');
    }
    
    // Check HTTP response code
    $http_code = wp_remote_retrieve_response_code($response);
    if ($http_code !== 200) {
        $error_body = wp_remote_retrieve_body($response);
        error_log("❌ MXCHAT DEBUG: Gemini API HTTP error: " . $http_code . " - " . $error_body);
        
        // Try to extract error message from response
        $error_data = json_decode($error_body, true);
        $error_message = isset($error_data['error']['message']) ? 
            $error_data['error']['message'] : 
            "HTTP error " . $http_code;
            
        throw new Exception($error_message);
    }
    
    // Parse response
    $response_body = json_decode(wp_remote_retrieve_body($response), true);
    
    // Check for JSON decode errors
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("❌ MXCHAT DEBUG: Gemini API JSON decode error: " . json_last_error_msg());
        throw new Exception('Error processing Gemini API response');
    }
    
    // Extract the response text
    if (isset($response_body['candidates'][0]['content']['parts'][0]['text'])) {
        error_log("✅ MXCHAT DEBUG: Gemini API call successful");
        return ['text' => trim($response_body['candidates'][0]['content']['parts'][0]['text'])];
    } else {
        error_log("❌ MXCHAT DEBUG: Unexpected Gemini API response format: " . wp_json_encode($response_body));
        throw new Exception('Unexpected response format from Gemini API');
    }
}

public static function store_in_pinecone_woocommerce($embedding_vector, $content, $url, $api_key, $environment, $index_name, $vector_id = null, $product = null) {
    // Get vector ID or create from URL
    $vector_id = $vector_id ?: md5($url);

    // Get Pinecone settings
    $options = get_option('mxchat_pinecone_addon_options');
    $host = $options['mxchat_pinecone_host'] ?? '';

    // Validate host
    if (empty($host)) {
        return array(
            'success' => false,
            'message' => esc_html__('Pinecone host is not configured. Please set the host in your settings.', 'mxchat-woo')
        );
    }

    // Build metadata with proper WooCommerce product info
    $metadata = array(
        'text' => $content,
        'source_url' => $url,
        'type' => 'product',
        'last_updated' => time()
    );

    // Add WooCommerce specific metadata if product object is available
    if ($product instanceof WC_Product) {
        $metadata['woocommerce'] = array(
            'product_type' => $product->get_type(),
            'product_id' => $product->get_id(),
            'sku' => $product->get_sku()
        );
    }

    $api_endpoint = "https://{$host}/vectors/upsert";
    $request_body = array(
        'vectors' => array(
            array(
                'id' => $vector_id,
                'values' => $embedding_vector,
                'metadata' => $metadata
            )
        )
    );

    $response = wp_remote_post($api_endpoint, array(
        'headers' => array(
            'Api-Key' => $api_key,
            'accept' => 'application/json',
            'content-type' => 'application/json'
        ),
        'body' => wp_json_encode($request_body),
        'timeout' => 30,
        'data_format' => 'body'
    ));

    if (is_wp_error($response)) {
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        $body = wp_remote_retrieve_body($response);
        return array(
            'success' => false,
            'message' => sprintf(
                /* translators: 1: HTTP response code, 2: Error message */
                esc_html__('Pinecone API error (HTTP %1$d): %2$s', 'mxchat-woo'),
                $response_code,
                $body
            )
        );
    }

    return array(
        'success' => true,
        'message' => esc_html__('Successfully stored WooCommerce product in Pinecone', 'mxchat-woo')
    );
}

public static function delete_from_pinecone_woocommerce($urls, $api_key, $environment, $index_name) {
    // Get the Pinecone host from options
    $options = get_option('mxchat_pinecone_addon_options');
    $host = $options['mxchat_pinecone_host'] ?? '';
    
    if (empty($host)) {
        return array(
            'success' => false,
            'message' => esc_html__('Pinecone host is not configured. Please set the host in your settings.', 'mxchat-woo')
        );
    }

    // Build API endpoint using the configured host
    $api_endpoint = "https://{$host}/vectors/delete";

    // Create vector IDs from URLs
    $vector_ids = array_map('md5', $urls);

    // Simplified request body without metadata filtering
    $request_body = array(
        'ids' => $vector_ids
    );

    // Make the deletion request
    $response = wp_remote_post($api_endpoint, array(
        'headers' => array(
            'Api-Key' => $api_key,
            'accept' => 'application/json',
            'content-type' => 'application/json'
        ),
        'body' => wp_json_encode($request_body),
        'timeout' => 30,
        'data_format' => 'body'
    ));

    // Handle WordPress HTTP API errors
    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {

        }
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }

    // Check response status
    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        $body = wp_remote_retrieve_body($response);
        if (defined('WP_DEBUG') && WP_DEBUG) {
            
        }
        return array(
            'success' => false,
            'message' => sprintf(
                /* translators: 1: HTTP response code, 2: Error message */
                esc_html__('Pinecone API error (HTTP %1$d): %2$s', 'mxchat-woo'),
                $response_code,
                $body
            )
        );
    }

    return array(
        'success' => true,
        'message' => sprintf(
            /* translators: %d: Number of vectors deleted */
            esc_html__('Successfully deleted %d vectors from Pinecone', 'mxchat-woo'),
            count($vector_ids)
        )
    );
}

public static function store_in_wordpress_db_woocommerce($content, $url, $embedding_vector, $product = null) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'mxchat_system_prompt_content';
    
    // Add WooCommerce metadata to content if product is provided
    if ($product instanceof WC_Product) {
        $metadata = array(
            'product_id' => $product->get_id(),
            'product_type' => $product->get_type(),
            'sku' => $product->get_sku(),
            'type' => 'woocommerce_product'
        );
        $content = wp_json_encode($metadata) . "\n---\n" . $content;
    }

    $existing_entry = $wpdb->get_row(
        $wpdb->prepare("SELECT id FROM $table_name WHERE source_url = %s", $url)
    );

    if ($existing_entry) {
        $result = $wpdb->update(
            $table_name,
            array(
                'article_content'  => $content,
                'embedding_vector' => serialize($embedding_vector),
                'timestamp'        => current_time('mysql')
            ),
            array('id' => $existing_entry->id),
            array('%s', '%s', '%s')
        );
    } else {
        $result = $wpdb->insert(
            $table_name,
            array(
                'article_content'  => $content,
                'source_url'       => $url,
                'embedding_vector' => serialize($embedding_vector),
                'timestamp'        => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s')
        );
    }

    if ($result === false) {
        // Log the error if debugging is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {

        }
        return false;
    }

    return true;
}


}
