<?php
/**
 * Plugin Name: MxChat WooCommerce Integration
 * Plugin URI: https://mxchat.ai/
 * Description: WooCommerce integration add-on for MxChat (Pro only)
 * Version: 1.2.0
 * Author: MxChat
 * Author URI: https://mxchat.ai/
 * Text Domain: mxchat-woo
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.0
 *
 * @package MxChat_Woo
 *
 * VERSION UPDATE INSTRUCTIONS:
 * To update the plugin version, you only need to update TWO places in this file:
 * 1. Line 6 above: Version: X.X.X (in the plugin header)
 * 2. Line 31 below: define('MXCHAT_WOO_VERSION', 'X.X.X');
 *
 * Also update the readme.txt file:
 * 3. Stable tag: X.X.X
 * 4. Add changelog entry
 *
 * All other version references in the code will automatically use the constant.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Declare WooCommerce feature compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('remote_logging', __FILE__, true);
    }
});

// VERSION: Update this constant when releasing a new version (also update line 6 above)
define('MXCHAT_WOO_VERSION', '1.2.0');
define('MXCHAT_WOO_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MXCHAT_WOO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MXCHAT_WOO_PLUGIN_BASENAME', plugin_basename(__FILE__));

spl_autoload_register(function ($class) {
    $prefix = 'MxChat_Woo_';
    $base_dir = MXCHAT_WOO_PLUGIN_DIR . 'includes/';

    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }

    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-mxchat-woo-' . str_replace('_', '-', strtolower($relative_class)) . '.php';

    if (file_exists($file)) {
        require $file;
    }
});

function is_mxchat_woo_requirements_met() {
    if (is_multisite()) {
        $active_sitewide_plugins = get_site_option('active_sitewide_plugins');
        $mxchat_active = isset($active_sitewide_plugins['mxchat-basic/mxchat-basic.php']);
        $woo_active = isset($active_sitewide_plugins['woocommerce/woocommerce.php']);
    }
    
    // Check site-specific plugins if not found in network
    if (!isset($mxchat_active) || !isset($woo_active)) {
        $active_plugins = apply_filters('active_plugins', get_option('active_plugins'));
        $mxchat_active = in_array('mxchat-basic/mxchat-basic.php', $active_plugins);
        $woo_active = in_array('woocommerce/woocommerce.php', $active_plugins);
    }
    
    return $mxchat_active && $woo_active;
}

/**
 * Handle migration for compatibility - simplified for 1.1.6+
 * This function now only handles the enable_woocommerce_integration setting
 * Consumer keys have been removed
 */
function mxchat_woo_handle_migration() {
    $existing_options = get_option('mxchat_options', array());
    $woo_specific_options = array(
        'enable_woocommerce_integration' => isset($existing_options['enable_woocommerce_integration']) ? $existing_options['enable_woocommerce_integration'] : ''
        // Consumer key and secret removed in version 1.1.6
    );

    if (!get_option('mxchat_woo_migrated')) {
        update_option('mxchat_woo_migrated', true);
    }

    return $woo_specific_options;
}

require_once dirname(__FILE__) . '/update-checker.php';
PluginUpdateCheckerWooCommerce::init([
    'user_id' => 'wp_4_SGKQ2sGzkIGB',
    'plugin_slug' => 'mxchat-woo'
]);

final class MxChat_Woo {
    private static $instance = null;
    public $loader;
    public $admin;
    public $product_handler;
    public $order_handler;
    public $cart_handler;
    public $customer_handler;
    public $filtered_search_handler;
    private $migrated_options;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Run version 1.1.6 migration to clean up unused API keys
        $this->migrate_to_1_1_6();
        
        // Run standard migration
        mxchat_woo_migrate_settings();

        $this->migrated_options = mxchat_woo_handle_migration();
        $this->check_dependencies();
        $this->init_components();
        register_deactivation_hook(__FILE__, array($this, 'handle_deactivation'));
    }

/**
 * Migration for version 1.1.6 - Remove unused WooCommerce API keys and product sync setting
 * 
 * @since 1.1.6
 */
private function migrate_to_1_1_6() {
    $current_version = get_option('mxchat_woo_plugin_version', '1.0.0');
    
    // Only run if upgrading from a version before 1.1.6
    if (version_compare($current_version, '1.1.6', '<')) {
        // Get the current WooCommerce addon options
        $woo_options = get_option('mxchat_woo_options', array());
        
        // Remove the consumer key and secret if they exist
        if (isset($woo_options['woocommerce_consumer_key'])) {
            unset($woo_options['woocommerce_consumer_key']);
        }
        if (isset($woo_options['woocommerce_consumer_secret'])) {
            unset($woo_options['woocommerce_consumer_secret']);
        }
        
        // Remove product sync setting (now in main plugin)
        if (isset($woo_options['enable_woocommerce_integration'])) {
            unset($woo_options['enable_woocommerce_integration']);
        }
        
        // Update the options without the API keys and sync setting
        update_option('mxchat_woo_options', $woo_options);
        
        // Also clean up from main MxChat options if they're still there
        $main_options = get_option('mxchat_options', array());
        if (isset($main_options['woocommerce_consumer_key'])) {
            unset($main_options['woocommerce_consumer_key']);
        }
        if (isset($main_options['woocommerce_consumer_secret'])) {
            unset($main_options['woocommerce_consumer_secret']);
        }
        if (isset($main_options['enable_woocommerce_integration'])) {
            unset($main_options['enable_woocommerce_integration']);
        }
        update_option('mxchat_options', $main_options);
        
        // Clean up from legacy mxchat_woo_settings if it exists
        $legacy_settings = get_option('mxchat_woo_settings', array());
        if (!empty($legacy_settings)) {
            if (isset($legacy_settings['woocommerce_consumer_key'])) {
                unset($legacy_settings['woocommerce_consumer_key']);
            }
            if (isset($legacy_settings['woocommerce_consumer_secret'])) {
                unset($legacy_settings['woocommerce_consumer_secret']);
            }
            if (isset($legacy_settings['enable_woocommerce_integration'])) {
                unset($legacy_settings['enable_woocommerce_integration']);
            }
            update_option('mxchat_woo_settings', $legacy_settings);
        }
        
        // Update the plugin version using the constant
        update_option('mxchat_woo_plugin_version', MXCHAT_WOO_VERSION);

        // Add migration flag for 1.1.6
        update_option('mxchat_woo_migrated_1_1_6', true);
    }
}

    private function check_dependencies() {
        if (!is_mxchat_woo_requirements_met()) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php _e('<strong>MxChat WooCommerce Integration</strong> requires both MxChat and WooCommerce plugins to be installed and active.', 'mxchat-woo'); ?></p>
                </div>
                <?php
            });
            return;
        }
    }

    private function init_components() {
        require_once MXCHAT_WOO_PLUGIN_DIR . 'includes/class-mxchat-woo-loader.php';
        $this->loader = new MxChat_Woo_Loader();

        if ($this->is_license_active()) {
            // Initialize admin first
            $this->admin = new MxChat_Woo_Admin($this->loader);
            
            // Initialize components with loader
            $this->product_handler = new MxChat_Woo_Product_Handler($this->loader, $this->migrated_options);
            $this->order_handler = new MxChat_Woo_Order_Handler($this->loader, $this->migrated_options);
            $this->cart_handler = new MxChat_Woo_Cart_Handler($this->loader, $this->migrated_options);
            $this->customer_handler = new MxChat_Woo_Customer_Handler($this->loader, $this->migrated_options);
            $this->filtered_search_handler = new MxChat_Woo_Filtered_Search_Handler($this->loader, $this->migrated_options);

            // Add WooCommerce callbacks to the main plugin
            add_filter('mxchat_available_callbacks', array($this, 'add_woo_callbacks'));
            add_filter('mxchat_chatbot_html', 'mxchat_woo_add_toolbar_filter', 10, 1);

            // Register existing handlers
            add_filter('mxchat_handle_product_recommendations', array($this->product_handler, 'mxchat_handle_product_recommendations'), 10, 4);
            add_filter('mxchat_handle_product_inquiry', array($this->product_handler, 'mxchat_handle_product_inquiry'), 10, 4);
            add_filter('mxchat_handle_checkout_intent', array($this->cart_handler, 'mxchat_handle_checkout_intent'), 10, 4);
            add_filter('mxchat_handle_order_history', array($this->order_handler, 'mxchat_handle_order_history'), 10, 4);
            add_filter('mxchat_handle_add_to_cart_intent', array($this->cart_handler, 'mxchat_handle_add_to_cart_intent'), 10, 4);
            
            // Register new filtered search handler
            add_filter('mxchat_handle_filtered_search', array($this->filtered_search_handler, 'mxchat_handle_filtered_search'), 10, 4);
            
            add_filter('mxchat_generate_embedding', array($this, 'mxchat_generate_embedding'), 10, 2);
            
            // Add hooks for cart functionality
            add_action('wp_enqueue_scripts', array($this, 'enqueue_cart_assets'));
            add_filter('mxchat_chatbot_html', 'mxchat_woo_add_toolbar_filter', 10, 1);

            // Add AJAX handlers for cart operations
            add_action('wp_ajax_mxchat_woo_get_cart_count', array($this->cart_handler, 'get_cart_count'));
            add_action('wp_ajax_nopriv_mxchat_woo_get_cart_count', array($this->cart_handler, 'get_cart_count'));
            add_action('wp_ajax_mxchat_woo_get_cart_contents', array($this->cart_handler, 'get_cart_contents'));
            add_action('wp_ajax_nopriv_mxchat_woo_get_cart_contents', array($this->cart_handler, 'get_cart_contents'));
            add_action('wp_ajax_mxchat_woo_remove_cart_item', array($this->cart_handler, 'remove_cart_item'));
            add_action('wp_ajax_nopriv_mxchat_woo_remove_cart_item', array($this->cart_handler, 'remove_cart_item'));

            // Add admin menu with higher priority to ensure it loads after main menu
            add_action('admin_menu', array($this, 'add_admin_menu'), 100);

            // Run the loader
            $this->loader->run();
        } else {
            add_action('admin_notices', array($this, 'show_pro_requirement_notice'));
        }
    }

    public function add_woo_callbacks($callbacks) {
        $woo_callbacks = array(
            'mxchat_handle_product_inquiry' => array(
                'label'       => __('Show Product Card', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'products',
                'description' => __('Display product information in the chat', 'mxchat-woo'),
            ),
            'mxchat_handle_order_history' => array(
                'label'       => __('Order History', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'clipboard',
                'description' => __('Allow customers to check their order status', 'mxchat-woo'),
            ),
            'mxchat_handle_add_to_cart_intent' => array(
                'label'       => __('Add to Cart', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'plus-alt',
                'description' => __('Add products to cart directly from chat', 'mxchat-woo'),
            ),
            'mxchat_handle_checkout_intent' => array(
                'label'       => __('Proceed to Checkout', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'arrow-right-alt',
                'description' => __('Redirect customer to checkout page', 'mxchat-woo'),
            ),
            'mxchat_handle_product_recommendations' => array(
                'label'       => __('Product Recommendations', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'cart',
                'description' => __('Suggest products based on customer preferences', 'mxchat-woo'),
            ),
            'mxchat_handle_filtered_search' => array(
                'label'       => __('Filtered Product Search', 'mxchat-woo'),
                'pro_only'    => true,
                'group'       => __('WooCommerce Features', 'mxchat-woo'),
                'icon'        => 'filter',
                'description' => __('Search products with price, category, and attribute filters using natural language', 'mxchat-woo'),
            ),
        );
        return array_merge($callbacks, $woo_callbacks);
    }

    public function add_admin_menu() {
        if (is_multisite() && is_network_admin()) {
            add_submenu_page(
                'settings.php',  // Network settings menu
                __('WooCommerce Integration', 'mxchat-woo'),
                __('WooCommerce', 'mxchat-woo'),
                'manage_network_options',
                'mxchat-woo',
                array($this->admin, 'display_admin_page')
            );
        } else {
            add_submenu_page(
                'mxchat-max',
                __('WooCommerce Integration', 'mxchat-woo'),
                __('WooCommerce', 'mxchat-woo'),
                'manage_options',
                'mxchat-woo',
                array($this->admin, 'display_admin_page')
            );
        }
    }

    private function is_license_active() {
        if (is_multisite()) {
            $network_license_status = get_site_option('mxchat_license_status');
            if ($network_license_status === 'active') {
                return true;
            }
        }
        $license_status = get_option('mxchat_license_status', 'inactive');
        return $license_status === 'active';
    }

    public function handle_deactivation() {
        delete_option('mxchat_woo_migrated');
    }

    public function show_pro_requirement_notice() {
        $class = 'notice notice-warning';
        $message = sprintf(
            __('MxChat WooCommerce Integration requires MxChat Pro to be activated. Please <a href="%s">activate your pro license</a> to use this addon.', 'mxchat-woo'),
            admin_url('admin.php?page=mxchat-activation')
        );
        printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), wp_kses_post($message));
    }

    public function __clone() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-woo'), MXCHAT_WOO_VERSION);
    }

    public function __wakeup() {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'mxchat-woo'), MXCHAT_WOO_VERSION);
    }

    public function enqueue_cart_assets() {
        if (!class_exists('WooCommerce')) {
            return;
        }
        
        // Enqueue the CSS
        wp_enqueue_style(
            'mxchat-woo-public',
            MXCHAT_WOO_PLUGIN_URL . 'public/css/mxchat-woo-public.css',
            array(),
            MXCHAT_WOO_VERSION
        );
        
        // Enqueue the cart JS
        wp_enqueue_script(
            'mxchat-woo-cart',
            MXCHAT_WOO_PLUGIN_URL . 'public/js/mxchat-woo-cart.js',
            array('jquery'),
            MXCHAT_WOO_VERSION,
            true
        );
        
        // Enqueue the orders JS
        wp_enqueue_script(
            'mxchat-woo-orders',
            MXCHAT_WOO_PLUGIN_URL . 'public/js/mxchat-woo-orders.js',
            array('jquery'),
            MXCHAT_WOO_VERSION,
            true
        );
        
        // Localize the cart script
        wp_localize_script(
            'mxchat-woo-cart',
            'mxchatWooCart',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mxchat_nonce'),
                'cart_url' => function_exists('wc_get_cart_url') ? wc_get_cart_url() : '',
                'checkout_url' => function_exists('wc_get_checkout_url') ? wc_get_checkout_url() : '',
                'i18n' => array(
                    'loading_text' => __('Loading cart...', 'mxchat-woo'),
                    'empty_cart_text' => __('Your cart is empty.', 'mxchat-woo'),
                    'error_text' => __('Error loading cart.', 'mxchat-woo'),
                    'quantity_text' => __('Qty', 'mxchat-woo'),
                    'subtotal_text' => __('Subtotal', 'mxchat-woo'),
                    'remove_text' => __('Remove item', 'mxchat-woo'),
                    'cart_title' => __('Your Cart', 'mxchat-woo'),
                    'your_cart' => __('Your Cart', 'mxchat-woo'),
                    'close' => __('Close', 'mxchat-woo'),
                    'view_cart' => __('View Cart', 'mxchat-woo'),
                    'checkout' => __('Checkout', 'mxchat-woo')
                )
            )
        );
        
        // Localize the orders script
        wp_localize_script(
            'mxchat-woo-orders',
            'mxchatWooOrders',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('mxchat_nonce'),
                'account_url' => function_exists('wc_get_account_endpoint_url') ? wc_get_account_endpoint_url('orders') : '',
                'i18n' => array(
                    'your_orders' => __('Your Orders', 'mxchat-woo'),
                    'loading_text' => __('Loading orders...', 'mxchat-woo'),
                    'no_orders_text' => __('You have no orders yet.', 'mxchat-woo'),
                    'error_text' => __('Error loading orders.', 'mxchat-woo'),
                    'login_required' => __('Please log in to view your orders.', 'mxchat-woo'),
                    'view_order' => __('View Order', 'mxchat-woo'),
                    'view_all_orders' => __('View All Orders', 'mxchat-woo'),
                    'close' => __('Close', 'mxchat-woo')
                )
            )
        );
    }
}

/**
 * Function to modify chatbot HTML to add our toolbar filter hook
 */
function mxchat_woo_add_toolbar_filter($html) {
    // Pattern to find the toolbar
    $pattern = '/<div\s+(?:[^>]*?\s+)?class=(["\']).*?chat-toolbar.*?\1[^>]*>(.*?)<\/div>/s';
    
    $replacement = function($matches) {
        $toolbar_opening_tag = $matches[0];
        $toolbar_content = $matches[2];
        
        // Strip the closing div from the match
        $toolbar_opening_tag = str_replace($toolbar_content . '</div>', '', $toolbar_opening_tag);
        
        // Generate the cart icon HTML
        $cart_icon_html = mxchat_woo_generate_cart_icon_html();
        
        // Combine everything
        return $toolbar_opening_tag . $toolbar_content . $cart_icon_html . '</div>';
    };
    
    // Apply the replacement
    $modified_html = preg_replace_callback($pattern, $replacement, $html);
    
    // If replacement failed, try a simpler approach
    if ($modified_html === null || $modified_html === $html) {
        // Try a simpler pattern
        $simple_pattern = '/<\/div>(\s*)<\/div>(\s*)<div class="chatbot-footer">/s';
        $simple_replacement = function($matches) {
            $cart_icon_html = mxchat_woo_generate_cart_icon_html();
            return $cart_icon_html . '</div>' . $matches[1] . '</div>' . $matches[2] . '<div class="chatbot-footer">';
        };
        
        $modified_html = preg_replace_callback($simple_pattern, $simple_replacement, $html);
    }
    
    return $modified_html ?: $html;
}

/**
 * Generate the cart icon HTML
 */
function mxchat_woo_generate_cart_icon_html() {
    if (!class_exists('WooCommerce')) {
        return '';
    }
    
    ob_start();
    
    // Get cart item count
    $cart_item_count = is_object(WC()->cart) ? WC()->cart->get_cart_contents_count() : 0;
    $cart_count_class = ($cart_item_count > 0) ? '' : 'empty-cart';
    
    ?>
    <!-- MxChat WooCommerce Cart Start -->
    <button id="woo-cart-btn" class="toolbar-btn" title="<?php echo esc_attr__('Your Cart', 'mxchat-woo'); ?>">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512" stroke="currentColor">
            <path d="M0 24C0 10.7 10.7 0 24 0H69.5c22 0 41.5 12.8 50.6 32h411c26.3 0 45.5 25 38.6 50.4l-41 152.3c-8.5 31.4-37 53.3-69.5 53.3H170.7l5.4 28.5c2.2 11.3 12.1 19.5 23.6 19.5H488c13.3 0 24 10.7 24 24s-10.7 24-24 24H199.7c-34.6 0-64.3-24.6-70.7-58.5L77.4 54.5c-.7-3.8-4-6.5-7.9-6.5H24C10.7 48 0 37.3 0 24zM128 464a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm336-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z"/>
        </svg>
        <span id="woo-cart-count" class="woo-cart-count <?php echo esc_attr($cart_count_class); ?>"><?php echo esc_html($cart_item_count); ?></span>
    </button>
    
    <div id="woo-cart-dropdown" class="woo-cart-dropdown">
        <div class="woo-cart-header">
            <h4><?php echo esc_html__('Your Cart', 'mxchat-woo'); ?></h4>
            <button class="woo-cart-close-btn" title="<?php echo esc_attr__('Close', 'mxchat-woo'); ?>">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>
        </div>
        <div class="woo-cart-items-container">
            <div id="woo-cart-items" class="woo-cart-items">
                <div class="woo-cart-loading"><?php echo esc_html__('Loading cart...', 'mxchat-woo'); ?></div>
            </div>
        </div>
        <div class="woo-cart-footer">
            <div id="woo-cart-subtotal" class="woo-cart-subtotal"></div>
            <a href="<?php echo esc_url(wc_get_cart_url()); ?>" class="woo-cart-view-btn"><?php echo esc_html__('View Cart', 'mxchat-woo'); ?></a>
            <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="woo-cart-checkout-btn"><?php echo esc_html__('Checkout', 'mxchat-woo'); ?></a>
        </div>
    </div>
    <!-- MxChat WooCommerce Cart End -->
    <?php
    
    return ob_get_clean();
}

/**
 * Migrate settings from main plugin options to separate WooCommerce addon options
 * Updated for version 1.1.6 to exclude consumer keys
 */
function mxchat_woo_migrate_settings() {
    if (!get_option('mxchat_woo_migrated_v2')) {
        $main_options = get_option('mxchat_options', array());
        $woo_options = array();

        // Only migrate the enable_woocommerce_integration setting
        // Consumer keys are no longer migrated as of version 1.1.6
        if (isset($main_options['enable_woocommerce_integration'])) {
            $woo_options['enable_woocommerce_integration'] = $main_options['enable_woocommerce_integration'];
            unset($main_options['enable_woocommerce_integration']);
        }

        // Save both options
        update_option('mxchat_woo_options', $woo_options);
        update_option('mxchat_options', $main_options);
        update_option('mxchat_woo_migrated_v2', true);
    }
}

function MxChat_Woo() {
    return MxChat_Woo::instance();
}

add_action('plugins_loaded', 'MxChat_Woo', 20);

register_activation_hook(__FILE__, function() {
    require_once MXCHAT_WOO_PLUGIN_DIR . 'includes/class-mxchat-woo-activator.php';
    MxChat_Woo_Activator::activate();
});
