/**
 * MxChat WooCommerce Cart Handler
 * Handles cart functionality in the MxChat chatbot
 */
(function($) {
    'use strict';

    //console.log('MxChat WooCommerce cart script loaded');

    // Cart variables
    let cartDropdownVisible = false;
    let cartItemsLoaded = false;
    
// Add this at the top with your other variables
let cartInjectionInProgress = false;

/**
 * Function to inject the cart icon into the toolbar
 */
function injectCartIcon() {
    // Check if the cart icon already exists
    if ($('#woo-cart-btn').length > 0) {
        //console.log('Cart icon already exists');
        return;
    }
    
    // Check if injection is already in progress
    if (cartInjectionInProgress) {
        //console.log('Cart icon injection already in progress');
        return;
    }
    
    // Set flag to prevent multiple simultaneous injections
    cartInjectionInProgress = true;
    
    // Find the chat toolbar with multiple selectors for better compatibility
    const toolbar = $('.chat-toolbar, #mxchat-chatbot .chat-toolbar, #mxchat-chatbot-wrapper .chat-toolbar');
    
    //console.log('Attempting to inject cart icon');
    //console.log('Toolbar found:', toolbar.length > 0);
    
    if (toolbar.length === 0) {
        // If toolbar isn't ready yet, try again in a moment
        cartInjectionInProgress = false; // Reset flag
        setTimeout(injectCartIcon, 500);
        return;
    }

    // Get cart count from WooCommerce REST API
    $.ajax({
        url: mxchatWooCart.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_woo_get_cart_count',
            nonce: mxchatWooCart.nonce
        },
        success: function(response) {
            // Double-check that the cart icon doesn't exist yet
            if ($('#woo-cart-btn').length > 0) {
                //console.log('Cart icon added by another process');
                cartInjectionInProgress = false;
                return;
            }
            
            //console.log('Cart count response:', response);
            
            if (!response.success) {
                //console.log('Failed to get cart count');
                cartInjectionInProgress = false;
                return;
            }
            
            const count = parseInt(response.data.count);
            const cartCountClass = count > 0 ? '' : 'empty-cart';
            
            // Create cart button
            const cartButton = $(`
                <button id="woo-cart-btn" class="toolbar-btn" title="${mxchatWooCart.i18n.cart_title || 'Your Cart'}">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512" stroke="currentColor">
                        <path d="M0 24C0 10.7 10.7 0 24 0H69.5c22 0 41.5 12.8 50.6 32h411c26.3 0 45.5 25 38.6 50.4l-41 152.3c-8.5 31.4-37 53.3-69.5 53.3H170.7l5.4 28.5c2.2 11.3 12.1 19.5 23.6 19.5H488c13.3 0 24 10.7 24 24s-10.7 24-24 24H199.7c-34.6 0-64.3-24.6-70.7-58.5L77.4 54.5c-.7-3.8-4-6.5-7.9-6.5H24C10.7 48 0 37.3 0 24zM128 464a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm336-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z"/>
                    </svg>
                    <span id="woo-cart-count" class="woo-cart-count ${cartCountClass}">${count}</span>
                </button>
            `);
            
            // Create dropdown
            const cartDropdown = $(`
                <div id="woo-cart-dropdown" class="woo-cart-dropdown">
                    <div class="woo-cart-header">
                        <h4>${mxchatWooCart.i18n.your_cart || 'Your Cart'}</h4>
                        <button class="woo-cart-close-btn" title="${mxchatWooCart.i18n.close || 'Close'}">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"></line>
                                <line x1="6" y1="6" x2="18" y2="18"></line>
                            </svg>
                        </button>
                    </div>
                    <div class="woo-cart-items-container">
                        <div id="woo-cart-items" class="woo-cart-items">
                            <div class="woo-cart-loading">${mxchatWooCart.i18n.loading_text || 'Loading cart...'}</div>
                        </div>
                    </div>
                    <div class="woo-cart-footer">
                        <div id="woo-cart-subtotal" class="woo-cart-subtotal"></div>
                        <a href="${mxchatWooCart.cart_url || '#'}" class="woo-cart-view-btn">${mxchatWooCart.i18n.view_cart || 'View Cart'}</a>
                        <a href="${mxchatWooCart.checkout_url || '#'}" class="woo-cart-checkout-btn">${mxchatWooCart.i18n.checkout || 'Checkout'}</a>
                    </div>
                </div>
            `);
            
            // Append to DOM
            toolbar.append(cartButton);
            $('body').append(cartDropdown);
            
            //console.log('Cart icon and dropdown injected');
            
            // Initialize cart functionality
            initCartFunctionality();
            
            // Reset flag
            cartInjectionInProgress = false;
        },
        error: function(xhr, status, error) {
            //console.log('AJAX error:', error);
            // Reset flag on error
            cartInjectionInProgress = false;
        }
    });
}
    
    /**
     * Initialize the cart functionality after elements are injected
     */
    function initCartFunctionality() {
        const cartBtn = $('#woo-cart-btn');
        const cartCountEl = $('#woo-cart-count');
        const cartDropdown = $('#woo-cart-dropdown');
        const cartItemsContainer = $('#woo-cart-items');
        
        // Add click event for cart button
        cartBtn.on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            toggleCartDropdown();
        });
        
        // Close cart when clicking outside
        $(document).on('click', function(e) {
            if (cartDropdownVisible && 
                !$(e.target).closest('#woo-cart-dropdown').length && 
                !$(e.target).closest('#woo-cart-btn').length) {
                closeCartDropdown();
            }
        });
        
        // Close button event
        $('.woo-cart-close-btn').on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            closeCartDropdown();
        });
        
        // Listen for cart updates from other parts of the site
        $(document.body).on('added_to_cart removed_from_cart updated_cart_totals', function() {
            updateCartCount();
            if (cartDropdownVisible) {
                loadCartContents();
            }
        });
        
        // Handle remove item clicks within cart dropdown
        $(document).on('click', '.woo-cart-item-remove', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const cartItemKey = $(this).data('cart-key');
            removeCartItem(cartItemKey);
        });
    }
    
    /**
     * Toggle cart dropdown visibility
     */
    function toggleCartDropdown() {
        //console.log('Toggling cart dropdown');
        if (cartDropdownVisible) {
            closeCartDropdown();
        } else {
            openCartDropdown();
        }
    }
    
    /**
     * Open the cart dropdown and load contents
     */
    function openCartDropdown() {
        //console.log('Opening cart dropdown');
        $('#woo-cart-dropdown').addClass('active');
        cartDropdownVisible = true;
        
        // Load cart contents
        loadCartContents();
    }
    
    /**
     * Close the cart dropdown
     */
    function closeCartDropdown() {
        //console.log('Closing cart dropdown');
        $('#woo-cart-dropdown').removeClass('active');
        cartDropdownVisible = false;
    }
    
    /**
     * Update the cart count badge
     */
    function updateCartCount() {
        //console.log('Updating cart count');
        $.ajax({
            url: mxchatWooCart.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_woo_get_cart_count',
                nonce: mxchatWooCart.nonce
            },
            success: function(response) {
                if (response.success) {
                    const count = parseInt(response.data.count);
                    const cartCountEl = $('#woo-cart-count');
                    
                    cartCountEl.text(count);
                    
                    if (count > 0) {
                        cartCountEl.removeClass('empty-cart');
                    } else {
                        cartCountEl.addClass('empty-cart');
                        
                        // If cart is empty and dropdown is visible, update contents
                        if (cartDropdownVisible) {
                            loadCartContents();
                        }
                    }
                }
            }
        });
    }
    
    /**
     * Load the cart contents via AJAX
     */
    function loadCartContents() {
        //console.log('Loading cart contents');
        const cartItemsContainer = $('#woo-cart-items');
        
        cartItemsContainer.html('<div class="woo-cart-loading">' + (mxchatWooCart.i18n.loading_text || 'Loading cart...') + '</div>');
        cartItemsLoaded = false;
        
        $.ajax({
            url: mxchatWooCart.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_woo_get_cart_contents',
                nonce: mxchatWooCart.nonce
            },
            success: function(response) {
                //console.log('Cart contents response:', response);
                if (response.success) {
                    renderCartContents(response.data);
                    cartItemsLoaded = true;
                } else {
                    cartItemsContainer.html('<div class="woo-cart-error">' + (response.data.message || mxchatWooCart.i18n.error_text || 'Error loading cart') + '</div>');
                }
            },
            error: function(xhr, status, error) {
                //console.log('AJAX error loading cart contents:', error);
                cartItemsContainer.html('<div class="woo-cart-error">' + (mxchatWooCart.i18n.error_text || 'Error loading cart') + '</div>');
            }
        });
    }
    
    /**
     * Render the cart contents in the dropdown
     */
    function renderCartContents(data) {
        //console.log('Rendering cart contents');
        const cartItemsContainer = $('#woo-cart-items');
        
        if (!data.items || data.items.length === 0) {
            cartItemsContainer.html('<div class="woo-cart-empty">' + (mxchatWooCart.i18n.empty_cart_text || 'Your cart is empty') + '</div>');
            $('#woo-cart-subtotal').hide();
            $('.woo-cart-checkout-btn').hide();
            return;
        }
        
        let itemsHtml = '';
        
        data.items.forEach(function(item) {
            itemsHtml += `
                <div class="woo-cart-item">
                    <div class="woo-cart-item-image" style="background-image: url('${item.image}');"></div>
                    <div class="woo-cart-item-details">
                        <div class="woo-cart-item-title">${item.name}</div>
                        <div class="woo-cart-item-price">${item.price}</div>
                        <div class="woo-cart-item-quantity">${mxchatWooCart.i18n.quantity_text || 'Qty'}: ${item.quantity}</div>
                    </div>
                    <button class="woo-cart-item-remove" data-cart-key="${item.key}" title="${mxchatWooCart.i18n.remove_text || 'Remove item'}">&times;</button>
                </div>
            `;
        });
        
        cartItemsContainer.html(itemsHtml);
        
        // Update subtotal
        $('#woo-cart-subtotal').show().html(`
            <span>${mxchatWooCart.i18n.subtotal_text || 'Subtotal'}:</span>
            <span>${data.subtotal}</span>
        `);
        
        // Show checkout button
        $('.woo-cart-checkout-btn').show();
    }
    
    /**
     * Remove an item from the cart
     */
    function removeCartItem(cartKey) {
    //console.log('Removing item from cart:', cartKey);
    $.ajax({
        url: mxchatWooCart.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_woo_remove_cart_item',  // Make sure this matches your PHP handler
            cart_key: cartKey,
            nonce: mxchatWooCart.nonce
        },
        success: function(response) {
            //console.log('Remove item response:', response);
            if (response.success) {
                updateCartCount();
                loadCartContents();
                
                // Trigger WooCommerce events to update other cart elements on the page
                $(document.body).trigger('removed_from_cart', [null, null, $(`.woo-cart-item-remove[data-cart-key="${cartKey}"]`)]);
                
                // Force a cart update on the WooCommerce side
                if (typeof wc_cart_fragments_params !== 'undefined') {
                    $(document.body).trigger('wc_fragment_refresh');
                }
            } else {
                console.error('Failed to remove item:', response.data ? response.data.message : 'Unknown error');
            }
        },
        error: function(xhr, status, error) {
            //console.log('AJAX error removing item:', error);
        }
    });
}


/**
 * Handle add to cart button clicks
 */
function handleAddToCart() {
    $(document).on('click', '.mxchat-add-to-cart-button', function(e) {
        e.preventDefault();
        
        const button = $(this);
        const productId = button.data('product-id');
        const productType = button.data('product-type');
        const nonce = button.data('nonce');
        
        // Disable button during processing
        button.prop('disabled', true);
        const originalText = button.text();
        // Store original text as data attribute for later use
        if (!button.data('original-text')) {
            button.data('original-text', originalText);
        }
        button.text(mxchatWooCart.i18n.adding || 'Adding...');
        
        if (productType === 'variable') {
            handleVariableProductAddToCart(button, productId, nonce);
        } else {
            handleSimpleProductAddToCart(button, productId, nonce);
        }
    });
    
    // Handle variation dropdown changes
    $(document).on('change', '.mxchat-variation-select', function() {
        const productId = $(this).data('product-id');
        updateVariationData(productId);
    });
}

/**
 * Handle simple product add to cart
 */
function handleSimpleProductAddToCart(button, productId, nonce) {
    $.ajax({
        url: mxchatWooCart.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_woo_add_to_cart',
            product_id: productId,
            quantity: 1,
            nonce: nonce
        },
        success: function(response) {
            if (response.success) {
                // 1. Button success state with checkmark
                button.text('✓ Added').addClass('mxchat-success-state');
                
                // 2. Animate cart count with bounce
                updateCartCount();
                const cartCount = $('#woo-cart-count');
                cartCount.addClass('mxchat-bounce-animation');
                setTimeout(() => cartCount.removeClass('mxchat-bounce-animation'), 600);
                
                // 3. Brief notification dot on cart icon
                showCartNotification();
                
                // 4. If cart dropdown is open, refresh it to show new item
                if (cartDropdownVisible) {
                    loadCartContents();
                }
                
                // Trigger WooCommerce events
                $(document.body).trigger('added_to_cart', [response.data, null, button]);
                
                // Reset button after delay
                setTimeout(function() {
                    button.prop('disabled', false);
                    button.removeClass('mxchat-success-state');
                    button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
                }, 2500);
            } else {
                button.text(mxchatWooCart.i18n.addToCartError);
                setTimeout(function() {
                    button.prop('disabled', false);
                    button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
                }, 2000);
            }
        },
        error: function() {
            button.text(mxchatWooCart.i18n.addToCartError);
            setTimeout(function() {
                button.prop('disabled', false);
                button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
            }, 2000);
        }
    });
}

/**
 * Handle variable product add to cart
 */
function handleVariableProductAddToCart(button, productId, nonce) {
    const productCard = button.closest('.mxchat-product-card');
    const selects = productCard.find('.mxchat-variation-select');
    const attributes = {};
    let allSelected = true;
    
    // Collect selected attributes
    selects.each(function() {
        const select = $(this);
        const attributeName = select.data('attribute');
        const value = select.val();
        
        if (!value) {
            allSelected = false;
            return false; // Break out of loop
        }
        
        attributes[attributeName] = value;
    });
    
    if (!allSelected) {
        alert(mxchatWooCart.i18n.selectAllOptions || 'Please select all product options');
        button.prop('disabled', false);
        button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
        return;
    }
    
    // Get variation data first
    $.ajax({
        url: mxchatWooCart.ajaxurl,
        type: 'POST',
        data: {
            action: 'mxchat_get_variation_data',
            product_id: productId,
            attributes: attributes,
            nonce: nonce
        },
        success: function(response) {
            if (response.success && response.data.variation_id) {
                // Add variation to cart
                $.ajax({
                    url: mxchatWooCart.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'mxchat_add_variation_to_cart',
                        product_id: productId,
                        variation_id: response.data.variation_id,
                        attributes: attributes,
                        quantity: 1,
                        nonce: nonce
                    },
                    success: function(cartResponse) {
                        if (cartResponse.success) {
                            // 1. Button success state with checkmark
                            button.text('✓ Added').addClass('mxchat-success-state');
                            
                            // 2. Animate cart count with bounce
                            updateCartCount();
                            const cartCount = $('#woo-cart-count');
                            cartCount.addClass('mxchat-bounce-animation');
                            setTimeout(() => cartCount.removeClass('mxchat-bounce-animation'), 600);
                            
                            // 3. Brief notification dot on cart icon
                            showCartNotification();
                            
                            // 4. If cart dropdown is open, refresh it to show new item
                            if (cartDropdownVisible) {
                                loadCartContents();
                            }
                            
                            // Trigger WooCommerce events
                            $(document.body).trigger('added_to_cart', [cartResponse.data, null, button]);
                            
                            // Reset button after delay
                            setTimeout(function() {
                                button.prop('disabled', false);
                                button.removeClass('mxchat-success-state');
                                button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
                            }, 2500);
                        } else {
                            button.text(mxchatWooCart.i18n.addToCartError);
                            setTimeout(function() {
                                button.prop('disabled', false);
                                button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
                            }, 2000);
                        }
                    },
                    error: function() {
                        button.text(mxchatWooCart.i18n.addToCartError);
                        setTimeout(function() {
                            button.prop('disabled', false);
                            button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
                        }, 2000);
                    }
                });
            } else {
                alert(mxchatWooCart.i18n.variationNotAvailable || 'This variation is not available');
                button.prop('disabled', false);
                button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
            }
        },
        error: function() {
            button.text(mxchatWooCart.i18n.addToCartError);
            setTimeout(function() {
                button.prop('disabled', false);
                button.text(button.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart');
            }, 2000);
        }
    });
}

/**
 * Update variation data when dropdowns change - with proper validation
 */
function updateVariationData(productId) {
    const productCard = $(`.mxchat-product-card`).has(`[data-product-id="${productId}"]`);
    const selects = productCard.find('.mxchat-variation-select');
    const addToCartBtn = productCard.find('.mxchat-add-to-cart-button');
    
    // Get variations data from the JSON script tag
    const variationsData = productCard.find(`.mxchat-variations-data[data-product-id="${productId}"]`);
    
    if (variationsData.length === 0) {
        return;
    }
    
    let variations;
    try {
        variations = JSON.parse(variationsData.text());
    } catch (e) {
        console.error('Error parsing variations data:', e);
        return;
    }
    
    // Get currently selected attributes
    const selectedAttributes = {};
    selects.each(function() {
        const select = $(this);
        const attributeName = select.data('attribute');
        const value = select.val();
        if (value) {
            selectedAttributes[attributeName] = value;
        }
    });
    
    // Update each dropdown to disable invalid options
    selects.each(function() {
        const currentSelect = $(this);
        const currentAttribute = currentSelect.data('attribute');
        const currentValue = currentSelect.val();
        
        // Get attributes without current selection for testing
        const testAttributes = { ...selectedAttributes };
        delete testAttributes[currentAttribute];
        
        // Check each option in this dropdown
        currentSelect.find('option').each(function() {
            const option = $(this);
            const optionValue = option.val();
            
            if (!optionValue) {
                return; // Skip empty "Choose an option"
            }
            
            // Test if this option value creates a valid combination
            const testCombination = { ...testAttributes };
            testCombination[currentAttribute] = optionValue;
            
            const isValidCombination = variations.some(function(variation) {
                const varAttributes = variation.attributes;
                
                // Check if this variation matches our test combination
                for (const attrName in testCombination) {
                    const varAttrKey = 'attribute_' + attrName.toLowerCase().replace('pa_', '');
                    
                    if (varAttributes[varAttrKey] && 
                        varAttributes[varAttrKey] !== '' && 
                        varAttributes[varAttrKey] !== testCombination[attrName]) {
                        return false;
                    }
                }
                return variation.is_in_stock; // Only valid if in stock
            });
            
            // Enable/disable the option
            if (isValidCombination) {
                option.prop('disabled', false);
                option.removeClass('disabled-option');
            } else {
                option.prop('disabled', true);
                option.addClass('disabled-option');
                
                // If this was selected but is now invalid, clear it
                if (optionValue === currentValue) {
                    currentSelect.val('');
                }
            }
        });
    });
    
    // Update price and button state based on current selection
    const allSelected = selects.length === Object.keys(selectedAttributes).length;
    
    if (allSelected) {
        // Find exact matching variation
        const matchingVariation = variations.find(function(variation) {
            const varAttributes = variation.attributes;
            
            for (const attrName in selectedAttributes) {
                const varAttrKey = 'attribute_' + attrName.toLowerCase().replace('pa_', '');
                
                if (varAttributes[varAttrKey] && 
                    varAttributes[varAttrKey] !== '' && 
                    varAttributes[varAttrKey] !== selectedAttributes[attrName]) {
                    return false;
                }
            }
            return true;
        });
        
        if (matchingVariation) {
            // Update price
            const priceElement = productCard.find(`#mxchat-price-${productId}`);
            if (priceElement.length > 0 && matchingVariation.price_html) {
                priceElement.html(matchingVariation.price_html);
            }
            
            // Update button state
            if (matchingVariation.is_in_stock) {
                addToCartBtn.prop('disabled', false);
                const originalText = addToCartBtn.data('original-text') || mxchatWooCart.i18n.addToCart || 'Add to Cart';
                addToCartBtn.text(originalText);
                // Store it for future use if not already stored
                if (!addToCartBtn.data('original-text')) {
                    addToCartBtn.data('original-text', originalText);
                }
            } else {
                addToCartBtn.prop('disabled', true);
                addToCartBtn.text('Out of Stock');
            }
        } else {
            // No matching variation found
            addToCartBtn.prop('disabled', true);
            addToCartBtn.text('Unavailable');
        }
    } else {
        // Not all options selected
        addToCartBtn.prop('disabled', true);
        addToCartBtn.text('Select Options');
    }
}

/**
 * Show cart notification dot
 */
function showCartNotification() {
    const cartBtn = $('#woo-cart-btn');
    
    // Remove existing notification
    cartBtn.find('.mxchat-notification-dot').remove();
    
    // Add notification dot
    cartBtn.append('<span class="mxchat-notification-dot"></span>');
    
    // Auto-remove after 3 seconds
    setTimeout(() => {
        cartBtn.find('.mxchat-notification-dot').fadeOut(300, function() {
            $(this).remove();
        });
    }, 3000);
}
    
// Update the document ready section to include this line
$(document).ready(function() {
    //console.log('Document ready, attempting to inject cart icon');
    
    // Initialize add to cart functionality
    handleAddToCart();
    
    // First attempt immediately
    injectCartIcon();
    
    // Just one more retry after a delay
    setTimeout(injectCartIcon, 1500);
    
    // Try when chatbot is opened or minimized
    $(document).on('click', '#floating-chatbot-button, .exit-chat', function() {
        //console.log('Chatbot state changed, attempting to inject cart icon');
        setTimeout(injectCartIcon, 500);
    });
});

    
})(jQuery);