/**
 * MxChat WooCommerce Order History Handler
 * Handles order history functionality in the MxChat chatbot
 */
(function($) {
    'use strict';

    //console.log('MxChat WooCommerce orders script loaded');

    // Variables
    let ordersDropdownVisible = false;
    let ordersLoaded = false;
    let orderInjectionInProgress = false;
    
    /**
     * Function to inject the order history button into the toolbar if not already present
     */
    function injectOrderHistoryButton() {
        // Check if the button already exists
        if ($('#woo-orders-btn').length > 0) {
            //console.log('Order history button already exists');
            return;
        }
        
        // Check if injection is already in progress
        if (orderInjectionInProgress) {
            //console.log('Order history button injection already in progress');
            return;
        }
        
        // Set flag to prevent multiple simultaneous injections
        orderInjectionInProgress = true;
        
        // Find the chat toolbar with multiple selectors for better compatibility
        const toolbar = $('.chat-toolbar, #mxchat-chatbot .chat-toolbar, #mxchat-chatbot-wrapper .chat-toolbar');
        
        //console.log('Attempting to inject order history button');
        //console.log('Toolbar found:', toolbar.length > 0);
        
        if (toolbar.length === 0) {
            // If toolbar isn't ready yet, try again in a moment
            orderInjectionInProgress = false; // Reset flag
            setTimeout(injectOrderHistoryButton, 500);
            return;
        }

        // Only inject if there's no button with this ID already
        if ($('#woo-orders-btn').length === 0) {
const orderButton = $(`
    <button id="woo-orders-btn" class="toolbar-btn" title="${mxchatWooOrders.i18n.your_orders || 'Your Orders'}">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512" stroke="currentColor" style="fill: #212121; stroke: #212121; color: #212121;">
            <path d="M192 0c-41.8 0-77.4 26.7-90.5 64H64C28.7 64 0 92.7 0 128V448c0 35.3 28.7 64 64 64H320c35.3 0 64-28.7 64-64V128c0-35.3-28.7-64-64-64H282.5C269.4 26.7 233.8 0 192 0zm0 64a32 32 0 1 1 0 64 32 32 0 1 1 0-64zM305 273L177 401c-9.4 9.4-24.6 9.4-33.9 0L79 337c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l47 47L271 239c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9z"/>
        </svg>
    </button>
`);
            
            // Create dropdown for order history
            const ordersDropdown = $(`
                <div id="woo-orders-dropdown" class="woo-orders-dropdown">
                    <div class="woo-orders-header">
                        <h4>${mxchatWooOrders.i18n.your_orders || 'Your Orders'}</h4>
                        <button class="woo-orders-close-btn" title="${mxchatWooOrders.i18n.close || 'Close'}">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"></line>
                                <line x1="6" y1="6" x2="18" y2="18"></line>
                            </svg>
                        </button>
                    </div>
                    <div class="woo-orders-container">
                        <div id="woo-orders-list" class="woo-orders-list">
                            <div class="woo-orders-loading">${mxchatWooOrders.i18n.loading_text || 'Loading orders...'}</div>
                        </div>
                    </div>
                    <div class="woo-orders-footer">
                        <a href="${mxchatWooOrders.account_url || '#'}" class="woo-orders-view-all-btn">${mxchatWooOrders.i18n.view_all_orders || 'View All Orders'}</a>
                    </div>
                </div>
            `);
            
            // Append to DOM
            toolbar.append(orderButton);
            $('body').append(ordersDropdown);
            
            //console.log('Order history button and dropdown injected');
            
            // Initialize functionality
            initOrderFunctionality();
        }
        
        // Reset flag
        orderInjectionInProgress = false;
    }
    
    /**
     * Initialize the order history functionality
     */
    function initOrderFunctionality() {
        const orderBtn = $('#woo-orders-btn');
        const ordersDropdown = $('#woo-orders-dropdown');
        
        // Add click event for order button
        orderBtn.on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            toggleOrdersDropdown();
        });
        
        // Close dropdown when clicking outside
        $(document).on('click', function(e) {
            if (ordersDropdownVisible && 
                !$(e.target).closest('#woo-orders-dropdown').length && 
                !$(e.target).closest('#woo-orders-btn').length) {
                closeOrdersDropdown();
            }
        });
        
        // Close button event
        $('.woo-orders-close-btn').on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            closeOrdersDropdown();
        });
    }
    
    /**
     * Toggle order history dropdown visibility
     */
    function toggleOrdersDropdown() {
        //console.log('Toggling orders dropdown');
        if (ordersDropdownVisible) {
            closeOrdersDropdown();
        } else {
            openOrdersDropdown();
        }
    }
    
    /**
     * Open the order history dropdown and load orders
     */
    function openOrdersDropdown() {
        //console.log('Opening orders dropdown');
        $('#woo-orders-dropdown').addClass('active');
        ordersDropdownVisible = true;
        
        // Load orders if not already loaded
        if (!ordersLoaded) {
            loadOrders();
        }
    }
    
    /**
     * Close the order history dropdown
     */
    function closeOrdersDropdown() {
        //console.log('Closing orders dropdown');
        $('#woo-orders-dropdown').removeClass('active');
        ordersDropdownVisible = false;
    }
    
    /**
     * Load orders via AJAX
     */
    function loadOrders() {
        //console.log('Loading orders');
        const ordersContainer = $('#woo-orders-list');
        
        ordersContainer.html('<div class="woo-orders-loading">' + (mxchatWooOrders.i18n.loading_text || 'Loading orders...') + '</div>');
        ordersLoaded = false;
        
        $.ajax({
            url: mxchatWooOrders.ajaxurl,
            type: 'POST',
            data: {
                action: 'mxchat_woo_get_recent_orders',
                nonce: mxchatWooOrders.nonce
            },
            success: function(response) {
                //console.log('Orders response:', response);
                if (response.success) {
                    renderOrders(response.data);
                    ordersLoaded = true;
                } else {
                    // Handle error, including login required
                    if (response.data && response.data.login_required) {
                        ordersContainer.html('<div class="woo-orders-login-required">' + 
                            (mxchatWooOrders.i18n.login_required || 'Please log in to view your orders.') + '</div>');
                    } else {
                        ordersContainer.html('<div class="woo-orders-error">' + 
                            (response.data.message || mxchatWooOrders.i18n.error_text || 'Error loading orders.') + '</div>');
                    }
                }
            },
            error: function(xhr, status, error) {
                //console.log('AJAX error loading orders:', error);
                ordersContainer.html('<div class="woo-orders-error">' + 
                    (mxchatWooOrders.i18n.error_text || 'Error loading orders.') + '</div>');
            }
        });
    }
    
    /**
     * Render orders in the dropdown
     */
    function renderOrders(data) {
        //console.log('Rendering orders');
        const ordersContainer = $('#woo-orders-list');
        
        if (!data.orders || data.orders.length === 0) {
            ordersContainer.html('<div class="woo-orders-empty">' + 
                (mxchatWooOrders.i18n.no_orders_text || 'You have no orders yet.') + '</div>');
            return;
        }
        
        let ordersHtml = '';
        
        data.orders.forEach(function(order) {
            // Generate status class
            const statusClass = 'status-' + order.status.replace('wc-', '');
            
            // Build items list
            let itemsList = '';
            if (order.items && order.items.length > 0) {
                itemsList = '<div class="woo-order-items">' + order.items.join(', ') + '</div>';
            }
            
            ordersHtml += `
                <div class="woo-order-item">
                    <div class="woo-order-header">
                        <div class="woo-order-number">Order #${order.number}</div>
                        <div class="woo-order-status ${statusClass}">${order.status_name}</div>
                    </div>
                    <div class="woo-order-date">${order.date}</div>
                    ${itemsList}
                    <div class="woo-order-footer">
                        <div class="woo-order-total">${order.total}</div>
                        <a href="${order.view_url}" class="woo-order-view-btn" target="_blank">${mxchatWooOrders.i18n.view_order || 'View Order'}</a>
                    </div>
                </div>
            `;
        });
        
        ordersContainer.html(ordersHtml);
        
        // Update view all link
        $('.woo-orders-view-all-btn').attr('href', data.view_all_url || mxchatWooOrders.account_url || '#');
    }
    
    // Attempt to inject on document ready with reasonable retries
    $(document).ready(function() {
        //console.log('Document ready, attempting to inject order history button');
        
        // First attempt immediately
        injectOrderHistoryButton();
        
        // One more retry after a delay
        setTimeout(injectOrderHistoryButton, 1500);
        
        // Try when chatbot is opened or minimized
        $(document).on('click', '#floating-chatbot-button, .exit-chat', function() {
            //console.log('Chatbot state changed, attempting to inject order history button');
            setTimeout(injectOrderHistoryButton, 500);
        });
    });
    
})(jQuery);