<?php
/**
 * QuickDeploy WordPress Plugin Update Checker
 * 
 * A generic update checker for WordPress plugins using QuickDeploy's infrastructure
 */

if (!defined('ABSPATH')) {
    exit;
}

class PluginUpdateCheckerWooCommerce {
    private static $instance;
    private $plugin_file;
    private $plugin_basename;
    private $current_version;
    private $firebase_url;
    private $user_id;
    private $plugin_slug;

    /**
     * Initialize the update checker
     * 
     * @param array $args {
     *     Optional. Array of arguments.
     *     @type string $plugin_file     The main plugin file path
     *     @type string $firebase_url    Custom Firebase function URL
     *     @type string $user_id         QuickDeploy user ID
     *     @type string $plugin_slug     Plugin slug as defined in QuickDeploy
     * }
     * @return PluginUpdateChecker
     */
    public static function init($args = []) {
        $default_firebase_url = 'https://us-central1-quickdeploywp.cloudfunctions.net';

        // If no plugin file specified, detect it from the calling file
        if (empty($args['plugin_file'])) {
            $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 1);
            $args['plugin_file'] = $backtrace[0]['file'];
        }

        // Merge with defaults
        $config = wp_parse_args($args, [
            'plugin_file' => '',
            'firebase_url' => $default_firebase_url,
            'user_id' => '',
            'plugin_slug' => ''
        ]);

        // Create or return existing instance
        if (!self::$instance) {
            self::$instance = new self($config);
        }
        return self::$instance;
    }

    private function __construct($config) {
        $this->plugin_file = $config['plugin_file'];
        $this->plugin_basename = plugin_basename($this->plugin_file);
        $this->firebase_url = $config['firebase_url'];
        $this->user_id = $config['user_id'];
        $this->plugin_slug = $config['plugin_slug'];
        
        // Get version from plugin file headers
        if (!function_exists('get_file_data')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        $plugin_data = get_file_data($this->plugin_file, ['Version' => 'Version']);
        $this->current_version = $plugin_data['Version'];

        // Hook into WordPress update mechanisms
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_update']);
        add_filter('plugins_api', [$this, 'get_plugin_info'], 20, 3);
        
        // IMPROVED: Better post-update handling
        add_action('upgrader_process_complete', [$this, 'on_plugin_update'], 10, 2);
        add_action('activated_plugin', [$this, 'clear_update_cache_on_activation']);
        
        // ADDED: Clear cache when plugin is loaded after update
        add_action('plugins_loaded', [$this, 'check_version_change'], 1);
    }

    /**
     * Check if version changed and clear cache accordingly
     */
    public function check_version_change() {
        $stored_version = get_option($this->plugin_slug . '_version');
        $current_version = $this->get_current_version();
        
        if ($stored_version !== $current_version) {
            // Version changed, clear all update caches
            $this->clear_update_transients();
            update_option($this->plugin_slug . '_version', $current_version);
            
            // Force WordPress to recheck updates
            wp_clean_plugins_cache(true);
        }
    }

    /**
     * Get current version with better file reading
     */
    private function get_current_version() {
        // Clear file cache to ensure we read fresh data
        clearstatcache();
        
        if (!function_exists('get_file_data')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        $plugin_data = get_file_data($this->plugin_file, ['Version' => 'Version']);
        return $plugin_data['Version'];
    }

    /**
     * Validate QuickDeploy user ID format
     */
    private function validate_user_id($user_id) {
        return !empty($user_id) && is_string($user_id) && 
               preg_match('/^wp_\d+_[a-zA-Z0-9]+$/', $user_id);
    }

    /**
     * IMPROVED: More thorough transient clearing
     */
    public function clear_update_transients() {
        // Clear all update-related transients
        delete_site_transient('update_plugins');
        delete_transient('update_plugins');
        delete_site_transient('update_core');
        delete_site_transient('update_themes');
        
        // Clear plugin-specific transients
        delete_transient('plugin_slugs');
        delete_site_transient('plugin_slugs');
        
        // Clear WordPress.org API cache
        wp_clean_plugins_cache(true);
        
        // Force clear any cached plugin data
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
    }

    /**
     * Clear cache when plugin is activated after update
     */
    public function clear_update_cache_on_activation($plugin) {
        if ($plugin === $this->plugin_basename) {
            $this->clear_update_transients();
        }
    }

    /**
     * IMPROVED: Better post-update cleanup with delayed execution
     */
    public function on_plugin_update($upgrader_object, $options) {
        // Check if this is a plugin update and if it's our plugin
        if ($options['action'] === 'update' && 
            $options['type'] === 'plugin' && 
            isset($options['plugins']) && 
            in_array($this->plugin_basename, $options['plugins'])) {
            
            // Clear the update transients immediately
            $this->clear_update_transients();
            
            // Schedule delayed cleanup to handle race conditions
            wp_schedule_single_event(time() + 2, 'quickdeploy_delayed_cleanup', [$this->plugin_slug]);
            
            // Store the new version
            $new_version = $this->get_current_version();
            update_option($this->plugin_slug . '_version', $new_version);
        }
    }

    /**
     * ADDED: Delayed cleanup hook
     */
    public function delayed_cleanup() {
        $this->clear_update_transients();
        wp_clean_plugins_cache(true);
    }

    /**
     * IMPROVED: Better version comparison and caching
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        if (!$this->validate_user_id($this->user_id)) {
            return $transient;
        }

        // Get fresh version data
        $current_version = $this->get_current_version();
        $this->current_version = $current_version;

        // Check if we already have this plugin marked for update
        if (isset($transient->response[$this->plugin_basename])) {
            $existing_update = $transient->response[$this->plugin_basename];
            // If current version is greater than or equal to update version, remove the update notice
            if (version_compare($current_version, $existing_update->new_version, '>=')) {
                unset($transient->response[$this->plugin_basename]);
                // Also remove from no_update if it exists there
                if (isset($transient->no_update[$this->plugin_basename])) {
                    unset($transient->no_update[$this->plugin_basename]);
                }
                return $transient;
            }
        }

        // Check for updates from server
        $response = wp_remote_get(
            add_query_arg([
                'userId' => $this->user_id,
                'slug' => $this->plugin_slug,
                'current_version' => $current_version // Send current version to server
            ], $this->firebase_url . '/checkUpdate'),
            [
                'timeout' => 10,
                'headers' => ['Accept' => 'application/json']
            ]
        );

        if (is_wp_error($response)) {
            return $transient;
        }

        $update_data = json_decode(wp_remote_retrieve_body($response), true);

        if ($update_data && isset($update_data['version']) && 
            version_compare($current_version, $update_data['version'], '<')) {
            
            $transient->response[$this->plugin_basename] = (object) [
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_basename,
                'new_version' => $update_data['version'],
                'package' => $update_data['download_url'] ?? '',
                'tested' => $update_data['tested_wp_version'] ?? '',
                'icons' => [],
                'banners' => []
            ];
            
            // Remove from no_update if it exists there
            if (isset($transient->no_update[$this->plugin_basename])) {
                unset($transient->no_update[$this->plugin_basename]);
            }
        } else {
            // No update available - ensure we remove any existing update notices
            if (isset($transient->response[$this->plugin_basename])) {
                unset($transient->response[$this->plugin_basename]);
            }
            
            // Add to no_update to prevent repeated checks
            $transient->no_update[$this->plugin_basename] = (object) [
                'id' => $this->plugin_basename,
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_basename,
                'new_version' => $current_version,
                'url' => '',
                'package' => ''
            ];
        }

        return $transient;
    }

    /**
     * Provide plugin information for the WordPress updates screen
     */
    public function get_plugin_info($res, $action, $args) {
        // Test logging to confirm the method is called
        error_log('get_plugin_info called for slug: ' . $this->plugin_slug);

        if ('plugin_information' !== $action || $this->plugin_slug !== $args->slug) {
            error_log('get_plugin_info early return: action or slug mismatch');
            return $res;
        }

        $response = wp_remote_get(
            add_query_arg([
                'userId' => $this->user_id,
                'slug' => $this->plugin_slug
            ], $this->firebase_url . '/pluginInfo'),
            [
                'timeout' => 10,
                'headers' => ['Accept' => 'application/json']
            ]
        );

        if (is_wp_error($response)) {
            error_log('Plugin info request failed: ' . $response->get_error_message());
            return $res;
        }

        $info = json_decode(wp_remote_retrieve_body($response), true);
        error_log('API response from pluginInfo: ' . print_r($info, true));

        if ($info) {
            // Initialize the plugin info object with all required fields
            $plugin_info = (object) [
                'name' => $info['plugin_name'] ?? '',
                'slug' => $this->plugin_slug,
                'version' => $info['version'] ?? '',
                'tested' => $info['tested_wp_version'] ?? '',
                'requires' => $info['requires_wp_version'] ?? '',
                'download_link' => $info['download_url'] ?? '',
                'trunk' => $info['download_url'] ?? '',
                'requires_php' => $info['requires_php'] ?? '',
                'last_updated' => $info['last_updated'] ?? '',
                'sections' => [],
                'banners' => [],
                'icons' => [],
                'screenshots' => [], // Optional, but included for completeness
                'rating' => 0,
                'num_ratings' => 0,
            ];

            // Dynamically set 'tested' to the current WordPress version if higher
            $current_wp_version = get_bloginfo('version');
            error_log('Detected WordPress version: ' . $current_wp_version);
            error_log('Plugin tested version: ' . $plugin_info->tested);
            if (version_compare($current_wp_version, $plugin_info->tested, '>')) {
                $plugin_info->tested = $current_wp_version;
                error_log('Updated tested version to current WP version: ' . $current_wp_version);
            } else {
                error_log('No update to tested version needed');
            }

            // Process the description (convert Markdown/readme syntax to HTML)
            if (!empty($info['description'])) {
                $description = $info['description'];
                $description = preg_replace('/==\s*([^=]+)\s*==/', '<h2>$1</h2>', $description);
                $description = preg_replace('/###\s*([^#]+)\s*###/', '<h3>$1</h3>', $description);
                $description = preg_replace('/=\s*([^=]+)\s*=/', '<h4>$1</h4>', $description);
                $description = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $description);
                $description = preg_replace('/^\s*\*\s*(.+)$/m', '<li>$1</li>', $description);
                $description = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $description);
                $description = nl2br($description);
                $description = wpautop($description);
                $plugin_info->sections['description'] = $description;
            }

            // Process the changelog (convert to HTML)
            if (!empty($info['changelog'])) {
                $changelog = $info['changelog'];
                $changelog = preg_replace('/=\s*([\d\.]+)\s*=/', '<h4>Version $1</h4>', $changelog);
                $changelog = preg_replace('/^\s*\*\s*(.+)$/m', '<li>$1</li>', $changelog);
                $changelog = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $changelog);
                $changelog = nl2br($changelog);
                $plugin_info->sections['changelog'] = $changelog;
            }

            // Add screenshots section
            if (!empty($info['screenshots']) && is_array($info['screenshots'])) {
                $screenshots_html = '<ol>';
                foreach ($info['screenshots'] as $index => $screenshot) {
                    $url = isset($screenshot['url']) ? esc_url($screenshot['url']) : '';
                    $caption = isset($screenshot['caption']) ? esc_html($screenshot['caption']) : "Screenshot " . ($index + 1);
                    if ($url) {
                        $screenshots_html .= "<li><img src='$url' alt='$caption' style='max-width: 100%;' /><p>$caption</p></li>";
                    }
                }
                $screenshots_html .= '</ol>';
                if ($screenshots_html !== '<ol></ol>') {
                    $plugin_info->sections['screenshots'] = $screenshots_html;
                }
            }

            // Add banner images using cropped versions from Firestore
            if (!empty($info['banner_low_url']) && !empty($info['banner_high_url'])) {
                $low_url = esc_url($info['banner_low_url']);
                $high_url = esc_url($info['banner_high_url']);
                $plugin_info->banners = [
                    'low' => $low_url,
                    'high' => $high_url,
                ];
                error_log('Banner URLs set: low=' . $low_url . ', high=' . $high_url);

                // Verify URLs are accessible
                $low_response = wp_remote_head($low_url);
                if (is_wp_error($low_response)) {
                    error_log('Low banner URL inaccessible: ' . $low_response->get_error_message());
                } else {
                    error_log('Low banner URL response code: ' . wp_remote_retrieve_response_code($low_response));
                }

                $high_response = wp_remote_head($high_url);
                if (is_wp_error($high_response)) {
                    error_log('High banner URL inaccessible: ' . $high_response->get_error_message());
                } else {
                    error_log('High banner URL response code: ' . wp_remote_retrieve_response_code($high_response));
                }
            } elseif (!empty($info['banner_url'])) {
                $banner_url = esc_url($info['banner_url']);
                $plugin_info->banners = [
                    'low' => $banner_url,
                    'high' => $banner_url,
                ];
                error_log('Using original banner URL: ' . $banner_url);
            } else {
                error_log('No banner_url or banner_low_url/banner_high_url found in API response');
                // Fallback banner for testing
                $plugin_info->banners = [
                    'low' => 'https://mxchat.ai/wp-content/uploads/2025/03/banner-772x250-1.png',
                    'high' => 'https://mxchat.ai/wp-content/uploads/2025/03/banner-1544x500-1.png',
                ];
                error_log('Using fallback banner URLs');
            }

            // Optionally add ratings if available in Firestore
            if (!empty($info['ratings'])) {
                $plugin_info->rating = isset($info['ratings']['average']) ? (float) $info['ratings']['average'] * 20 : 0;
                $plugin_info->num_ratings = isset($info['ratings']['count']) ? (int) $info['ratings']['count'] : 0;
            }

            // Log the final plugin info for debugging
            error_log('Plugin info response: ' . print_r($plugin_info, true));

            return $plugin_info;
        }

        error_log('No valid info returned from API');
        return $res;
    }
}

// ADDED: Register the delayed cleanup hook
add_action('quickdeploy_delayed_cleanup', function($plugin_slug) {
    if (class_exists('PluginUpdateCheckerWooCommerce')) {
        $checker = PluginUpdateCheckerWooCommerce::$instance;
        if ($checker) {
            $checker->delayed_cleanup();
        }
    }
});