define(['cla_angular', 'ng.cla.core.growl'], function (angular)
{
	var moduleName = 'cla.buttons.manage.service';

	// Manual singleton as the .service() appears to return multiple instances across Angular apps
	var instance = null;
	if (instance === null)
		instance = new ButtonsService();

	angular.module(moduleName, ['cla.core.growl', 'cla.button_editor.service'])
		.service('buttonsService', ['$http', 'growl.service', 'buttonEditorService', function($http, growl, editorService)
		{
			instance.setHttp($http);
			instance.setGrowl(growl);
			instance.setEditorService(editorService);
			return instance;
		}]);

	function decodeHtml(html) {
		var txt;

		if (typeof(html) === 'object')
		{
			for (var key in html)
			{
				if (html.hasOwnProperty(key))
				{
					txt = document.createElement("textarea");
					txt.innerHTML = html[key];
					html[key] = txt.value;
				}
			}

			return html;
		}

		txt = document.createElement("textarea");
		txt.innerHTML = html;
		return txt.value;
	}

	function ButtonsService()
	{
		var service = this;

		service.BASE_URL = '/api/buttons/v1/';
		service.$http = null;
		service.growl = null;
		service.editorService = null;
		service.editCallback = null;
		service.enableCallback = null;
		service.filteredCallback = null;

		service.filterTerm = '';

		service.buttons = [];
		service.filteredButtons = [];
		service.newButtonData = {};

		service.setHttp = function($http)
		{
			service.$http = $http;
		};

		service.setGrowl = function(growl)
		{
			service.growl = growl;
		};

		service.setEditorService = function(editorService)
		{
			service.editorService = editorService;
		};

		service.init = function(data)
		{
			service.buttons.length = 0;
			service.filteredButtons.length = 0;
			for (var i = 0; i < data.length; i++)
			{
				// Text fields will be encoded to allow special characters so decode them first
				data[i].title = decodeHtml(data[i].title);
				data[i].url = decodeHtml(data[i].url);
				data[i].tooltip = decodeHtml(data[i].tooltip);
				data[i].tags = decodeHtml(data[i].tags);
				for (var j = 0; j < data[i].perms.owners.length; j++)
				{
					data[i].perms.owners[j].name = decodeHtml(data[i].perms.owners[j].name);
				}
				for (j = 0; j < data[i].tags.length; j++)
				{
					data[i].tags[j] = decodeHtml(data[i].tags[j]);
				}

				service.buttons.push(data[i]);
				service.filteredButtons.push(data[i]);
			}
		};

		service.getButtons = function()
		{
			return service.filteredButtons;
		};

		service.getTotalButtonCount = function()
		{
			return service.buttons.length;
		};

		service.getStatus = function()
		{
			return service.editorService.status;
		};

		service.getButton = function(id)
		{
			for (var i = 0; i < service.buttons.length; i++)
			{
				if (service.buttons[i].id === id)
					return service.buttons[i];
			}

			return null;
		};

		service.registerEditCallback = function(callback, enable)
		{
			service.editCallback = callback;
			service.enableCallback = enable;
		};

		service.registerFilteredCallback = function(callback)
		{
			service.filteredCallback = callback;
		};

		service.select = function(id)
		{
			if (service.editCallback !== null)
			{
				service.editorService.setSelectedId(id);

				for (var i = 0; i < service.buttons.length; i++)
				{
					if (service.buttons[i].id === id)
					{
						service.editCallback(service.buttons[i]);
						service.enableCallback();
					}
				}
			}
		};

		service.unselect = function()
		{
			service.editorService.setSelectedId(0);
		};

		service.newButton = function()
		{
			if (service.editCallback !== null)
			{
				service.editorService.setSelectedId(-1);
				service.newButtonData = {
					id: 0,
					title: '',
					type: 'icon',
					icon: '',
					image: {
						id: 0,
						url: '',
						new_image: '',
						new_url: ''
					},
					url: '',
					open_in_new_tab: true,
					icon_colour: '000000',
					background_colour: 'ffffff',
					tooltip: '',
					tags: [],
					perms: {
						owners: [{
								name: lmsg('common.all_registered'),
								value: '4_0'
							}]
					}
				};
				service.editCallback(service.newButtonData);
				service.enableCallback();
			}
		};

		service.findButton = function(id)
		{
			for (var i = 0; i < service.buttons.length; i++)
			{
				if (service.buttons[i].id === id)
				{
					return i;
				}
			}

			return false;
		};

		service.saveButton = function(button)
		{
			if (!button.showTooltip)
				button.tooltip = '';
			if (button.image.new_url === '' && button.image.url === '')
				button.type = 'icon';

			if (button.id === 0)
			{
				service.$http.post(service.BASE_URL + 'button/new', {
					button: button
				}).then(function(result)
				{
					if (result.data.error)
					{
						if (service.enableCallback !== null)
							service.enableCallback();

						service.growl.showError(result.data.message);
					} else
					{
						var image = button.image;
						var type = button.type;

						if (image.new_url !== '')
						{
							image = {
								id: result.data.id,
								isImage: true,
								url: '/buttons/image/' + result.data.id,
								new_image: '',
								new_url: ''
							};
						} else
						{
							image = {
								isImage: false,
								id: 0,
								url: '',
								new_image: '',
								new_url: ''
							};
							type = 'icon';
						}

						service.newButtonData.type = result.data.type;
						service.newButtonData.id = result.data.id;
						service.newButtonData.title = button.title;
						service.newButtonData.url = button.url;
						service.newButtonData.icon = button.icon;
						service.newButtonData.url = button.url;
						service.newButtonData.type = type;
						service.newButtonData.icon = button.icon;
						service.newButtonData.image = image;
						service.newButtonData.url = button.url;
						service.newButtonData.open_in_new_tab = button.open_in_new_tab;
						service.newButtonData.icon_colour = button.icon_colour.substring(1);
						service.newButtonData.background_colour = button.background_colour.substring(1);
						service.newButtonData.tooltip = button.tooltip;
						service.newButtonData.perms = button.perms;
						service.newButtonData.tags = button.tags.split(', ');

						service.editorService.setSelectedId(0);

						service.buttons.push(service.newButtonData);

						service.sort();
						service.filter(service.filterTerm);

						service.newButtonData = {};

						service.growl.showSuccess(result.data.message);
					}
				}, function(result)
				{
					if (service.enableCallback !== null)
						service.enableCallback();

					var message = result.statusText;
					if ((typeof result.data === 'object') &&
						(result.data.error))
					{
						message = result.data.message;
					}

					service.growl.showError('', message);
				});
			} else
			{
				var index = service.findButton(button.id);

				if (index === false)
					return;

				service.$http.put(service.BASE_URL + 'button/' + button.id, {
					button: button
				}).then(function(result)
				{
					if (result.data.error)
					{
						if (service.enableCallback !== null)
							service.enableCallback();

						service.growl.showError(result.data.message);
					} else
					{
						var index = service.findButton(button.id);

						if (index !== false)
						{
							var targetButton = service.buttons[index];

							var image = button.image;
							var type = button.type;

							if ((button.type === 'image') &&
								(button.image.url === '') &&
								(button.image.new_url === ''))
							{
								button.type = 'icon';
							}

							targetButton.type = button.type;
							targetButton.title = button.title;
							targetButton.url = button.url;
							targetButton.icon = button.icon;
							targetButton.image.id = button.image.id;
							targetButton.image.url = button.image.url;
							targetButton.image.new_image = button.image.new_image;
							targetButton.image.new_url = button.image.new_url;
							targetButton.open_in_new_tab = button.open_in_new_tab;
							targetButton.icon_colour = button.icon_colour.substring(1);
							targetButton.background_colour = button.background_colour.substring(1);
							targetButton.tooltip = button.tooltip;
							targetButton.perms = button.perms;
							targetButton.tags = button.tags.split(', ');
						}

						// The title may have changed so re-sort
						service.sort();

						service.growl.showSuccess(result.data.message);
						service.editorService.setSelectedId(0);
						service.filter(service.filterTerm);
					}
				}, function(result)
				{
					if (service.enableCallback !== null)
						service.enableCallback();

					var message = result.statusText;
					if ((typeof result.data === 'object') &&
						(result.data.error))
					{
						message = result.data.message;
					}

					service.growl.showError('', message);
				});
			}
		};

		service.deleteButton = function(button)
		{
			service.$http.delete(service.BASE_URL + 'button/' + button.id, {
				button: button
			}).then(function(result){

				if (result.data.error)
				{
					service.growl.showError(result.data.message);
				} else
				{
					// Remove the entry from our local list of buttons
					var buttons = [];
					for (var i = 0; i < service.buttons.length; i++)
					{
						if (service.buttons[i].id !== button.id)
							buttons.push(service.buttons[i]);
					}
					service.buttons.length = 0;
					for (i = 0; i < buttons.length; i++)
					{
						service.buttons.push(buttons[i]);
					}

					service.editorService.setSelectedId(0);
					service.filter(service.filterTerm);

					service.growl.showSuccess(result.data.message);
				}
			}, function(result){
				var message = result.statusText;
				if ((typeof result.data === 'object') &&
					(result.data.error))
				{
					message = result.data.message;
				}

				service.growl.showError('', message);
			});
		};

		service.cloneButton = function(button)
		{
			service.$http.put(service.BASE_URL + 'button/' + button.id + '/clone', {}).then(function(result)
			{
				if (result.data.error)
				{
					service.growl.showError(result.data.message);
				} else
				{
					// Add the entry to our local list of buttons
					var newButton = {
						id: result.data.id,
						title: result.data.new_title,
						type: button.type,
						icon: button.icon,
						image: button.image,
						url: button.url,
						open_in_new_tab: button.open_in_new_tab,
						icon_colour: button.icon_colour,
						background_colour: button.background_colour,
						tooltip: button.tooltip,
						tags: button.tags,
						perms: button.perms
					};
					service.buttons.push(newButton);
					service.sort();

					service.filter(service.filterTerm);

					service.growl.showSuccess(result.data.message);
				}
			}, function(result)
			{
				service.growl.showError('', result.statusText);
			});
		};

		service.filter = function(term)
		{
			service.select(0);

			service.filterTerm = term;

			var lowerTerms = term.toLowerCase().split(' ');
			if ((lowerTerms.length === 1) &&
				(lowerTerms[0] === ''))
			{
				lowerTerms = [];
			}

			service.filteredButtons.length = 0;
			for (var i = 0; i < service.buttons.length; i++)
			{
				var button = service.buttons[i];
				var found = lowerTerms.length === 0;

				for (var j = 0; j < lowerTerms.length; j++)
				{
					if (button.title.toLowerCase().indexOf(lowerTerms[j]) !== -1)
						found = true;

					for (var k = 0; k < button.tags.length; k++)
					{
						if (button.tags[k].toLowerCase().indexOf(lowerTerms[j]) !== -1)
							found = true;
					}
				}

				if (found)
					service.filteredButtons.push(button);
			}

			if (service.filteredCallback !== null)
				service.filteredCallback();
		};

		service.sort = function()
		{
			function alpha_compare(a, b)
			{
				if (a.title.toLowerCase() < b.title.toLowerCase())
					return -1;
				if (a.title.toLowerCase() > b.title.toLowerCase())
					return 1;
				return 0;
			}

			service.buttons.sort(alpha_compare);
		};
	}
});
