---
title: CI/CD
---

# Claromentis CI/CD

Claromentis uses [GitLab CI/CD](https://docs.gitlab.com/ee/ci/) to facilitate
continuous integration and continuous delivery: [CI/CD](https://en.wikipedia.org/wiki/CI/CD).
This GitLab feature offers us a powerful means to automate the delivery and
deployment of our software.

## Overview

Software projects will define and use different [CI/CD pipelines](https://docs.gitlab.com/ee/ci/pipelines/)
depending on their needs, and pipelines are defined by a
[`.gitlab-ci.yml` configuration file](https://docs.gitlab.com/ee/ci/yaml/README.html).
These files describe pipeline **stages** that each contain a number of **jobs**,
which can run automatically under different circumstances. For example,
automated test jobs can run for every Git commit push, or a release job can run
for every Git tag push.

CI/CD pipelines are commonly used to:

- Lint the codebase to ensure a consistent code style
- Run unit tests, end-to-end tests, and other automated tests
- Build software as packaged distribution archives or as Docker images
- Publish software releases
- Deploy software to a server infrastructure

<!--
    TODO: List notable/critical CI/CD pipelines for the Claromentis ecosystem
          e.g. Core, Satisfactory, Documentis, developer.claromentis.com, etc
-->

## Claromentis CI/CD pipeline

![Directed acyclic graph of a successful Claromentis CI/CD pipeline](./images/cicd-dag.png)

The [Claromentis project](https://gitlab.com/claromentis/product/claromentis)
defines a CI/CD pipeline that performs the following jobs for every commit or
tag pushed to the project's Git repository.

1. **Setup stage**
   1. `setup:composer`: Install project dependencies using [Composer](https://getcomposer.org/)
   2. `setup:variables`: Dynamically define derived CI/CD variables for later
      pipeline stages
2. **Build stage**
   1. `build:docker`: Build a production Claromentis Docker image
   2. `build:docker-dev`: Build a development Claromentis Docker image
3. **Test stage**
   1. `test:phpcs`: Use [PHPCS](https://github.com/squizlabs/PHP_CodeSniffer)
      to scan for any PHP incompatibilities or usage of deprecated PHP features
   2. `test:phpunit`: Run [PHPUnit](https://phpunit.de/) automated test suites
   3. `test:install`: Test the production Claromentis installation process
   4. `test:install-dev`: Test the development Claromentis installation process
   5. `test:health`: Run health checks against a Claromentis instance
4. **Release stage**
   1. **[Tag push]** `release:package`: Build and upload a Claromentis
      distribution to the [GitLab Generic Packages Registry](https://docs.gitlab.com/ee/user/packages/generic_packages/)
   2. **[Tag push]** `release:gitlab`: Create a [GitLab release](https://docs.gitlab.com/ee/user/project/releases/)
      and associate its corresponding packages
   3. `release:docker`: Tag and push the production Docker image to the
      [GitLab Container Registry](https://docs.gitlab.com/ee/user/packages/container_registry/)
      and [Google Container Registry](https://cloud.google.com/container-registry)
   4. `release:docker-dev`: Tag and push the development Docker image to GitLab
      Container Registry
5. **Deploy stage**
   - Placeholder stage for extending projects, such as
     [Distros](https://gitlab.com/claromentis/product/distros), to deploy
     Claromentis instances

### Services

The Claromentis CI/CD pipeline utilises [GitLab CI/CD services](https://docs.gitlab.com/ee/ci/services/) to run
supporting services alongside Claromentis, such as [Elasticsearch](https://gitlab.com/claromentis/infrastructure/docker/elasticsearch),
during installation and healthcheck tests.

### Services

The Claromentis CI/CD pipeline utilises [GitLab CI/CD services](https://docs.gitlab.com/ee/ci/services/) to run
supporting services alongside Claromentis, such as [Elasticsearch](https://gitlab.com/claromentis/infrastructure/docker/elasticsearch),
during installation and healthcheck tests.

### Variable reference

The Claromentis CI/CD pipeline uses a several [CI variables](https://docs.gitlab.com/ee/ci/variables/)
to maintain consistent information throughout its lifecycle. Below is a
reference for those variables.

- **[Configurable]** variables can be configured by the GitLab project's CI/CD
  settings:
  <br/>See **Settings > CI/CD > Variables** as a project maintainer
- **[Derived]** variables are variables that have been defined dynamically at
  runtime by a CI/CD job
  <br/>See the `setup:variables` job defined in [`.gitlab-ci.yml`](../.gitlab-ci.yml)
- All other variables have static values, some of which are provided by
  [predefined GitLab CI variables](https://docs.gitlab.com/ee/ci/variables/predefined_variables.html)
  <br/>Variables are considered predefined if they are not marked as **[Configurable]** or **[Derived]**

| CI/CD variable name       | Example values                                                                                                           | Description                                                                                                                                                           |
|---------------------------|--------------------------------------------------------------------------------------------------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `CACHE_FALLBACK_KEY`      | `9-2`                                                                                                                    | **[Derived]** [Global fallback key](https://docs.gitlab.com/ee/ci/caching/#global-fallback-key) for caching Composer dependencies, defaulting to `$CI_DEFAULT_BRANCH` |
| `CI_COMPOSER_COMMAND`     | `composer install`                                                                                                       | Composer command used to install Composer dependencies                                                                                                                |
| `CI_PACKAGE_VERSION`      | `9.2.0`                                                                                                                  | **[Derived]** Version name to use for the GitLab Generic Package release                                                                                              |
| `CI_PACKAGE_FILE`         | `claromentis-9.2.0`                                                                                                      | **[Derived]** File name to use for publishing a GitLab Generic Package, excluding the file extension                                                                  |
| `CI_PACKAGE_FILE_ZIP`     | `claromentis-9.2.0.zip`                                                                                                  | **[Derived]** File name to use for publishing a GitLab Generic Package, including the file extension                                                                  |
| `CI_PACKAGE_REGISTRY_URL` |                                                                                                                          | **[Derived]** GitLab API URL to use for publishing a GitLab Generic Package                                                                                           |
| `CI_RELEASE_NAME`         | `Claromentis 9.2.0`                                                                                                      | **[Derived]** GitLab release name, defaulting to `"Claromentis $CI_COMMIT_TAG"`                                                                                       |
| `CI_PHP_VERSION`          | `7.4`                                                                                                                    | PHP version to use for PHP CI/CD jobs, and to use as the basis for Claromentis Docker images                                                                          |
| `CI_PHP_IMAGE`            | <details><summary>Show/hide</summary>`registry.gitlab.com/claromentis/infrastructure/docker/php:7.4-alpine-ci`</details> | PHP Docker image name to use for PHP CI/CD jobs                                                                                                                       |
| `CI_DOCKER_BASE_IMAGE`    | <details><summary>Show/hide</summary>`registry.gitlab.com/claromentis/infrastructure/docker/php`</details>               | Base Docker image name to use for Docker image builds                                                                                                                 |
| `CI_DOCKER_BASE_TAG`      | `7.4-fpm-alpine`                                                                                                         | Base Docker image tag to use for production Docker image builds                                                                                                       |
| `CI_DOCKER_BASE_TAG_DEV`  | `7.4-fpm-alpine-dev`                                                                                                     | Base Docker image tag to use for development Docker image builds                                                                                                      |
| `CI_DOCKER_BUILD_TAG`     | `build-123`                                                                                                              | Docker image tag to use for intermediate production Docker images                                                                                                     |
| `CI_DOCKER_BUILD_TAG_DEV` | `build-123-dev`                                                                                                          | Docker image tag to use for intermediate development Docker images                                                                                                    |
| `CI_DOCKER_DEFAULT_TAG`   | `latest`                                                                                                                 | Docker image tag to use as a fallback for build caching                                                                                                               |
| `CI_DOCKER_TAG`           | `9.2.0`, `9-edge`                                                                                                        | **[Derived]** Docker image tag to use for production Docker images                                                                                                    |
| `CI_DOCKER_TAG_DEV`       | `9.2.0-dev`                                                                                                              | **[Derived]** Docker image tag to use for development Docker images                                                                                                   |
| `CI_DOCKER_TAG_MAJOR`     | `9`                                                                                                                      | **[Derived]** Docker image tag to use for major versions of production Docker images                                                                                  |
| `CI_DOCKER_TAG_MINOR`     | `9.2`                                                                                                                    | **[Derived]** Docker image tag to use for minor versions of production Docker images                                                                                  |
| `CI_DOCKER_TAG_MAJOR_DEV` | `9-dev`                                                                                                                  | **[Derived]** Docker image tag to use for major versions of development Docker images                                                                                 |
| `CI_DOCKER_TAG_MINOR_DEV` | `9.2-dev`                                                                                                                | **[Derived]** Docker image tag to use for minor versions of development Docker images                                                                                 |
| `CI_DOCKER_VERSION`       | `27.5.1`                                                                                                                 | Docker version to use for CI/CD jobs that use Docker                                                                                                                  |
| `CI_GITLAB_IMAGE`         | <details><summary>Show/hide</summary>`registry.gitlab.com/claromentis/product/claromentis`</details>                     | GitLab Container Registry URL to push Docker images to                                                                                                                |
| `CI_GCR_IMAGE`            |                                                                                                                          | Google Container Registry URL to push Docker images to                                                                                                                |
| `GCR_PROJECT`             | `claro-registry-dev`                                                                                                     | **[Configurable]** Google Container Registry project name                                                                                                             |
| `GCR_REGION`              | `us`                                                                                                                     | **[Configurable]** Google Container Registry region                                                                                                                   |
| `GCR_SA`                  |                                                                                                                          | **[Configurable]** JSON service account credentials for Google Container Registry                                                                                     |

## Downstream CI/CD pipelines

The canonical Claromentis CI/CD pipeline is defined by the
[**default branch of the Claromentis project**](https://gitlab.com/claromentis/product/claromentis).

Downstream CI/CD pipelines depend on the canonical pipeline:

- [Claromentis project branches](https://gitlab.com/claromentis/product/claromentis/-/branches)
- [Claromentis Distros](https://gitlab.com/claromentis/product/distros)

Downstream projects can extend this pipeline in their `.gitlab-ci.yml` file.
They can then alter the behaviour of the pipeline by overriding CI/CD variables
and/or overriding specific jobs.

See GitLab's [Keyword reference for the `.gitlab-ci.yml` file](https://docs.gitlab.com/ee/ci/yaml/index.html#include)
for more information on how to extend CI/CD pipelines.

Claromentis branches extend the pipeline quite simply, typically only overriding
the PHP version that's used for the CI/CD jobs and the base PHP Docker image.

```yaml
include:
  - project: $CI_PROJECT_PATH
    file: .gitlab-ci.yml
    ref: $CI_DEFAULT_BRANCH

variables:
  CI_PHP_VERSION: '7.4'
```

Claromentis Distros extend similarly, but override the `variables` configuration
and the `setup:variables` job to define [variables](#variable-reference)
differently.

```yaml
# The Distros CI/CD pipeline extends the Claromentis CI/CD pipeline
# Visit https://gitlab.com/claromentis/product/claromentis and read the
# `.gitlab-ci.yml` file for more information
include:
  - project: 'claromentis/product/claromentis'
    ref: '9'
    file: .gitlab-ci.yml

variables:
  CI_PHP_VERSION: '7.4'
  CI_DOCKER_BASE_IMAGE: registry.gitlab.com/claromentis/product/claromentis
  CI_DOCKER_BASE_TAG: 9-edge

setup:variables:
  # Redacted for brevity
```
