<?php

namespace Claromentis\Composer;

use Composer\EventDispatcher\Event;
use Composer\Installer\InstallerEvent;
use Composer\IO\IOInterface;
use Composer\Package\AliasPackage;
use Composer\Package\BasePackage;
use Composer\Package\Link;
use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Package\RootPackageInterface;
use Composer\Plugin\PrePoolCreateEvent;
use Composer\Repository\RepositoryManager;
use Composer\Semver\Comparator;
use Composer\Semver\Constraint\Constraint;
use Composer\Semver\VersionParser;

/**
 * Package dependency processing service.
 *
 * Processes dependencies for legacy Composer usage, compatibility or convenience.
 *
 * - Restores `require`s for Core 7.x & 8.x dependencies installed from source
 * - Strips Core from root-project Claromentis Module `require`s so they can `composer install`
 *   comfortably and run automated tests in isolation
 * - Normalizes Claromentis Composer package types to improve Composer's update behaviour
 *
 * TODO: Unit tests
 *
 * @author Chris Andrew <chris.andrew@claromentis.com>
 */
class DependencyProcessor
{
	/**
	 * Composer package names of Claromentis Core.
	 *
	 * @var string[]
	 */
	const CORE_PACKAGE_NAMES = [
		'claromentis/core', // Future-proofing for project name change
		'claromentis/framework'
	];

	/**
	 * @var IOInterface
	 */
	private $io;

	/**
	 * @var RepositoryManager
	 */
	private $repositoryManager;

	/**
	 * @var CoreInstaller
	 */
	private $coreInstaller;

	/**
	 * @var ModuleInstaller
	 */
	private $moduleInstaller;

	public function __construct(IOInterface $io, RepositoryManager $repositoryManager, CoreInstaller $coreInstaller, ModuleInstaller $moduleInstaller)
	{
		$this->io                = $io;
		$this->repositoryManager = $repositoryManager;
		$this->coreInstaller     = $coreInstaller;
		$this->moduleInstaller   = $moduleInstaller;
	}

	/**
	 * Pre-process root package dependencies.
	 *
	 * - Allows source installations of Claromentis 8 with dependencies
	 * - Excludes Core from module root-project installs for unit tests and CI
	 *
	 * This method _should_ be idempotent for the Claromentis Composer Installer, and will exhibit undefined behaviour
	 * if it's not. It needs to be called multiple times throughout the Composer `install`/`update` event lifecycle.
	 *
	 * @param RootPackageInterface $rootPackage Root package to process dependencies for
	 * @param Event|null           $event       Optional event from a Composer event handler
	 */
	public function processRootPackage(RootPackageInterface $rootPackage, Event $event = null): void
	{
		// Backwards-compatibility; restore `require`s from `replace`s for Claromentis Core 7.x/8.x dependencies
		$this->restoreCoreDependencyRequires($rootPackage);

		// Remove Core dependency if the root package is a module; mainly for CI
		// TODO: Perhaps this should be removed or configurable
//		if ($this->moduleInstaller->supports($rootPackage->getType())) {
			// Remove Core as a requirement of the module
//			$this->removeCoreRequire($rootPackage);

			// Explicitly request removal of the package as part of dependency solving
//			if ($event instanceof InstallerEvent) {
//				$request = $event->getRequest();
//
//				foreach (self::CORE_PACKAGE_NAMES as $corePackageName) {
//					$request->remove($corePackageName);
//				}
//			}
//		}
	}

	/**
	 * Normalizes Claromentis Composer package types to their canonical values.
	 *
	 * Processes Claromentis packages from the installed package repository and installation pool to ensure that their
	 * package types match canonically to avoid a bug in Composer. See the linked GitLab issue for more information.
	 *
	 * @link https://gitlab.com/claromentis/product/composer-installer/-/issues/14
	 * @param PrePoolCreateEvent $event
	 */
	public function normalizePackageTypes(PrePoolCreateEvent $event): void
	{
		// Grab packages from the pre-solve event and installed repository
		$eventPackages = $event->getPackages();
		$localPackages = $this->repositoryManager->getLocalRepository()->getPackages();

		$packages = array_merge($eventPackages, $localPackages);

		// Count and log each set of packages
		$eventPackagesCount = count($eventPackages);
		$localPackagesCount = count($localPackages);
		$totalPackagesCount = count($packages);

		$this->io->write("<info>Normalizing package types</info>");
		$this->io->write([
			"<info>  $eventPackagesCount PrePoolCreateEvent packages</info>",
			"<info>  $localPackagesCount local packages</info>",
			"<info>  $totalPackagesCount total packages</info>"
		], true, IOInterface::VERBOSE);

		$normalizedPackagesCount = 0;

		foreach ($packages as $package) {
			// Skip unsupported packages nice and early
			if (!($this->coreInstaller->supports($package->getType()) || $this->moduleInstaller->supports($package->getType()))) {
				continue;
			}

			$this->io->write("Normalizing {$package->getPrettyString()}", true, IOInterface::DEBUG);

			// Resolve aliased packages
			if ($package instanceof AliasPackage) {
				$package = $package->getAliasOf();
				$this->io->write("  Resolved alias to {$package->getPrettyString()}", true, IOInterface::DEBUG);
			}

			// Let the Core & Module installers to normalize the package
			if ($package instanceof Package) {
				$originalType = $package->getType();
				$this->coreInstaller->normalizePackageType($package);
				$this->moduleInstaller->normalizePackageType($package);
				$newType = $package->getType();

				$typeChanged = $originalType !== $newType;

				if ($typeChanged) {
					$this->io->write("  <info>Normalized {$package->getPrettyString()} type from $originalType to $newType</info>", true, IOInterface::DEBUG);
					$normalizedPackagesCount++;
				}
			} else {
				$packageClass = get_class($package);
				$this->io->write("  <info>Skipped normalizing {$package->getPrettyString()} of class $packageClass</info>", true, IOInterface::DEBUG);
			}
		}

		if ($normalizedPackagesCount > 0) {
			$this->io->write("<info>Normalized package types of $normalizedPackagesCount packages</info>");
		} else {
			$this->io->write("<info>No package types needed normalizing</info>");
		}
	}

	/**
	 * Restore the dependencies of a Claromentis Core dependency of the given root package.
	 *
	 * Claromentis Core 7.x and 8.x distributions bundle their `composer.lock`ed `require` dependencies and mark them as
	 * `replace`s in their `composer.json` and in Composer repositories.
	 *
	 * This method undoes this for source installations by moving all non-Core `replace`s back to `require`s in-memory
	 * at installation time, for all matching Core packages found.
	 *
	 * The affected packages include those found in the local (`vendor/composer/installed.json`) repository and any
	 * remote repositories, matching the version constraints of the `require`s in the given root project.
	 *
	 * @param RootPackageInterface $rootPackage Root package that may have a Claromentis Core dependency that needs its dependencies restoring
	 */
	protected function restoreCoreDependencyRequires(RootPackageInterface $rootPackage)
	{
		$io = $this->io;

		// TODO: Extract findCoreRequires($rootPackage)
		//       Search beyond just the root package requires, for example if the root package requires claromentis/claromentis
		$io->write('<info>Searching for Claromentis Core dependency</info>', true, IOInterface::VERBOSE);

		$rootRequires = $rootPackage->getRequires();
		$coreLink     = null;

		foreach ($rootRequires as $rootRequire) {
			if ($this->packageNameIsCore($rootRequire->getTarget())) {
				$coreLink = $rootRequire;
				break;
			}
		}

		if (!isset($coreLink)) {
			$io->write('<info>Claromentis Core dependency not found</info>', true, IOInterface::VERBOSE);
			return;
		}

		$io->write(sprintf('<info>Processing dependencies of Claromentis Core %s</info>', $coreLink->getPrettyConstraint()));

		$repositories = $this->repositoryManager->getRepositories();

		$this->logRepositories($repositories);

		$corePackages = $this->findLocalAndCandidatePackages($coreLink);

		$io->write(sprintf('<info>%s Core packages found</info>', count($corePackages)), true, IOInterface::VERBOSE);

		/**
		 * Check whether **any** of the packages have a 'source' installation source.
		 *
		 * If so, we'll process the dependencies of every <9.x or dev package we can find.
		 *
		 * TODO: A custom DownloadManager or InstalledRepository may allow us to find or set the installation preference
		 *       for non-local packages. Until then, the information simply isn't there.
		 */
		$fromSource = false;

		foreach ($corePackages as $corePackage) {
			if ($corePackage->getInstallationSource() === 'source') {
				$fromSource = true;
				break;
			}
		}

		if (!$fromSource) {
			$io->write("<info>No 'source' packages found, skipping processing</info>", true, IOInterface::VERBOSE);
			return;
		}

		// TODO: Refactor/extract the below if possible, it's chunky
		foreach ($corePackages as $corePackage) {
			$corePrettyString       = $corePackage->getPrettyString();
			$coreInstallationSource = $corePackage->getInstallationSource();

			if (!empty($coreInstallationSource)) {
				$corePrettyString .= " from $coreInstallationSource";
			}

			$io->write(sprintf('<info>Checking %s</info>', $corePrettyString), true, IOInterface::VERBOSE);

			if (!$corePackage instanceof Package) {
				$io->write(sprintf("<info>Package %s is not an instance of %s, skipping</info>", $corePrettyString, Package::class), true, IOInterface::VERBOSE);
				continue;
			}

			if (empty($corePackage->getReplaces())) {
				$io->write(sprintf("<info>No replaces found in package %s, skipping</info>", $corePrettyString), true, IOInterface::VERBOSE);
				continue;
			}

			$earlierThanNine = Comparator::lessThan($corePackage->getVersion(), '9.0.0-dev');
			$developmentVersion = VersionParser::parseStability($corePackage->getVersion()) === 'dev';

			if ($earlierThanNine || $developmentVersion) {
				$io->write(sprintf('<info>Merging replaces into requires for %s</info>', $corePrettyString));

				// Extract its replaces, merge them into the requires, and empty the replaces
				$requires = $corePackage->getRequires();
				$replaces = $corePackage->getReplaces();
				$io->write(sprintf('<info>Found:      %s requires, %s replaces</info>', count($requires), count($replaces)), true, IOInterface::VERBOSE);

				$requires = array_merge($requires, $replaces);
				$replaces = [];

				// Retain Core package replaces
				foreach (self::CORE_PACKAGE_NAMES as $packageName) {
					if (isset($requires[$packageName])) {
						$replaces[$packageName] = $requires[$packageName];
						unset($requires[$packageName]);
					}
				}

				$corePackage->setRequires($requires);
				$corePackage->setReplaces($replaces);
				$io->write(sprintf('<info>Updated to: %s requires, %s replaces</info>', count($requires), count($replaces)), true, IOInterface::VERBOSE);
			} else {
				$io->write(sprintf("<info>Skipping replace merging for %s</info>", $corePrettyString), true, IOInterface::VERBOSE);
			}
		}
	}

	/**
	 * Remove Claromentis Core dependencies from the given package.
	 *
	 * @param RootPackageInterface $package Root package that may have a Claromentis Core dependency that needs removing
	 */
	protected function removeCoreRequire(RootPackageInterface $package)
	{
		$this->io->write(sprintf('<info>Removing Claromentis Core requirement from %s</info>', $package->getPrettyString()));

		$requires = $package->getRequires();

		foreach (self::CORE_PACKAGE_NAMES as $packageName) {
			unset($requires[$packageName]);
		}

		$this->io->write(sprintf('<info>New requires: %s</info>', json_encode($requires, JSON_PRETTY_PRINT)), true, IOInterface::VERBOSE);

		$package->setRequires($requires);
	}

	/**
	 * Find Core packages matching the given package link.
	 *
	 * - Finds the installed package that match the link's package name only
	 * - Finds the repository packages that match the link's package name version constraint
	 * - Merge both lists into a single array
	 *
	 * @param Link $coreLink The link to match
	 * @return PackageInterface[] Local packages matching the link's package name, and any repository packages matching the link's package name and version constraint
	 */
	protected function findLocalAndCandidatePackages(Link $coreLink): array
	{
		// Find ANY package that Composer has installed, not just those matching the constraint
		// We could be setting up a reinstallation instead of an update, otherwise
		$corePackages = $this->repositoryManager->getLocalRepository()->findPackages(
			$coreLink->getTarget()
		);

		// Find any candidate packages that Composer may install
		// This doesn't really have an effect for Composer 1, because repo packages won't have an installation
		// source set (it'll be empty string prior to downloading), but it should hopefully help for Composer 2
		return array_merge(
			$corePackages,
			$this->repositoryManager->findPackages(
				$coreLink->getTarget(), $coreLink->getConstraint()
			)
		);
	}

	/**
	 * Determine whether a given package name is Claromentis Core.
	 *
	 * @param string $name
	 * @return bool
	 */
	protected function packageNameIsCore(string $name): bool
	{
		return in_array($name, self::CORE_PACKAGE_NAMES);
	}

	protected function logRepositories(array $repositories)
	{
		$this->io->write('<info>Repositories:</info>', true, IOInterface::VERBOSE);

		foreach ($repositories as $repository) {
			$repositoryClass = get_class($repository);
			$repositoryName = method_exists($repository, 'getRepoName') ? $repository->getRepoName() : '';

			$this->io->write("$repositoryClass $repositoryName", true, IOInterface::VERBOSE);
		}
	}
}