<?php

namespace Claromentis\Composer;

use Composer\Composer;
use Composer\EventDispatcher\Event;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\Installer\InstallerEvent;
use Composer\Installer\InstallerEvents;
use Composer\IO\IOInterface;
use Composer\Plugin\CommandEvent;
use Composer\Plugin\PluginEvents;
use Composer\Plugin\PluginInterface;
use Composer\Plugin\PrePoolCreateEvent;
use Composer\Repository\ComposerRepository;
use Composer\Repository\FilterRepository;
use Composer\Repository\RepositoryManager;
use Composer\Util\Filesystem;

/**
 * Claromentis Composer Installer Plugin.
 *
 * Registers Composer installers for Claromentis Core & Modules.
 *
 * @author Chris Andrew <chris@claromentis.com>
 */
class InstallerPlugin implements PluginInterface, EventSubscriberInterface
{
	/**
	 * Package name.
	 */
	public const NAME = 'claromentis/composer-installer-plugin';

	/**
	 * Package version.
	 */
	public const VERSION = '2.1.0';

	/**
	 * Package aliases.
	 */
	public const ALIASES = [
		'claromentis/installer-composer-plugin',
		'claromentis/installer_composer_plugin'
	];

	/**
	 * Canonical Claromentis Composer repository URL.
	 */
	public const REPOSITORY_URL = 'https://packages.claromentis.net';

	/**
	 * @var Composer
	 */
	protected $composer;

	/**
	 * @var IOInterface
	 */
	protected $io;

	/**
	 * @var CoreInstaller
	 */
	protected $coreInstaller;

	/**
	 * @var ModuleInstaller
	 */
	protected $moduleInstaller;

	/**
	 * @var DependencyProcessor
	 */
	protected $dependencyProcessor;

	/**
	 * Command event dispatched from a Composer command.
	 *
	 * @var CommandEvent|null
	 */
	protected $commandEvent;

	/**
	 * @inheritDoc
	 *
	 * @see onInit()
	 * @see onDependencySolve()
	 * @link https://getcomposer.org/upgrade/UPGRADE-2.0.md
	 * @return array[]
	 */
	public static function getSubscribedEvents(): array
	{
		// Retro-compatible! https://stackoverflow.com/a/50697766/1744006
		$composer1PreSolveEvent = InstallerEvents::class . "::PRE_DEPENDENCIES_SOLVING";
		$composer2PreSolveEvent = PluginEvents::class . "::PRE_POOL_CREATE";

		$preSolveEvent = defined($composer1PreSolveEvent)
			? $composer1PreSolveEvent
			: $composer2PreSolveEvent;

		return [
			PluginEvents::INIT       => ['onInit', -1],
			constant($preSolveEvent) => ['onDependencySolve', -1],
		];
	}

	/**
	 * Activate the plugin by extending Composer.
	 *
	 * - Adds Core & Module installers
	 * - Initializes the Core dependency processor for Core 8.x support
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function activate(Composer $composer, IOInterface $io)
	{
		$this->composer = $composer;
		$this->io       = $io;

		$filesystem        = new Filesystem();
		$repositoryManager = $composer->getRepositoryManager();

		$this->coreInstaller       = new CoreInstaller($io, $composer, $filesystem);
		$this->moduleInstaller     = new ModuleInstaller($io, $composer, $filesystem);
		$this->dependencyProcessor = new DependencyProcessor($io, $repositoryManager, $this->coreInstaller, $this->moduleInstaller);

		$installationManager = $composer->getInstallationManager();
		$installationManager->addInstaller($this->coreInstaller);
		$installationManager->addInstaller($this->moduleInstaller);

		$io->write(sprintf('Composer %s', $composer::getVersion()), true, IOInterface::DEBUG);
		$io->write(sprintf('Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);
	}

	/**
	 * Deactivate the plugin.
	 *
	 * Removes Core & Module installers.
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function deactivate(Composer $composer, IOInterface $io)
	{
		$io->write(sprintf('Deactivating Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);

		$installationManager = $composer->getInstallationManager();
		$installationManager->removeInstaller($this->coreInstaller);
		$installationManager->removeInstaller($this->moduleInstaller);
	}

	/**
	 * Prepare the plugin for uninstallation.
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function uninstall(Composer $composer, IOInterface $io)
	{
		$io->write(sprintf('Uninstalling Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);
	}

	/**
	 * Initialise Composer once it's ready.
	 *
	 * @see prepareComposer()
	 * @param Event $event
	 */
	public function onInit(Event $event)
	{
		$this->io->write(__METHOD__, true, IOInterface::VERBOSE);
		$this->prepareComposer($event);
	}

	/**
	 * Process dependencies of the root package.
	 *
	 * @see prepareComposer()
	 * @param InstallerEvent|PrePoolCreateEvent $event
	 */
	public function onDependencySolve(Event $event)
	{
		$this->io->write(__METHOD__, true, IOInterface::VERBOSE);
		$this->prepareComposer($event);

		if ($event instanceof PrePoolCreateEvent) {
			$this->dependencyProcessor->normalizePackageTypes($event);
		}
	}

	/**
	 * Prepares Composer for Claromentis installations.
	 *
	 * - Prepends the Claromentis Composer repository (https://packages.claromentis.net).
	 * - Normalizes Claromentis Composer package types to canonical values
	 * - Preprocesses Core dependencies if the root package requires it
	 *
	 * @param Event|null $event Optional event that instigated the preparation
	 */
	protected function prepareComposer(Event $event = null)
	{
		$this->prependClaromentisRepository($this->composer->getRepositoryManager());
		$this->dependencyProcessor->processRootPackage($this->composer->getPackage(), $event);
	}

	/**
	 * Add the Claromentis Composer repository to the given repository manager.
	 *
	 * @param RepositoryManager $repositoryManager The repository manager to add the Claromentis Composer repository to
	 */
	protected function prependClaromentisRepository(RepositoryManager $repositoryManager)
	{
		// Bail if the repository is already present
		foreach ($repositoryManager->getRepositories() as $repository) {
			// Unravel filtered repositories (Composer 2)
			if ($repository instanceof FilterRepository) {
				$repository = $repository->getRepository();
			}

			if (
				$repository instanceof ComposerRepository
				&& isset($repository->getRepoConfig()['url'])
				&& $repository->getRepoConfig()['url'] === self::REPOSITORY_URL
			) {
				$this->io->write('<info>Claromentis Packages repository already present, no need to prepend</info>', true, IOInterface::VERBOSE);
				return;
			}
		}

		// Prepend the repository so that we get in front of Packagist
		$this->io->write('<info>Prepending Claromentis Packages repository</info>');

		$repositoryManager->prependRepository(
			$repositoryManager->createRepository('composer', ['url' => self::REPOSITORY_URL], 'Packages')
		);
	}
}
