<?php

namespace Claromentis\Composer;

use Composer\Composer;
use Composer\Downloader\DownloadManager;
use Composer\EventDispatcher\EventDispatcher;
use Composer\Installer\BinaryInstaller;
use Composer\Installer\InstallerInterface;
use Composer\IO\IOInterface;
use Composer\Package\CompletePackageInterface;
use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Repository\InstalledRepositoryInterface;
use Composer\Script\ScriptEvents;
use Composer\Util\Filesystem;
use InvalidArgumentException;
use React\Promise\PromiseInterface;
use ReflectionClass;

/**
 * Base class for the common behaviours of Core and Module installers.
 *
 * Most installation logic for safely installing and upgrading potentially stateful packages is laid out in this class.
 *
 * @author Chris Andrew <chris.andrew@claromentis.com>
 */
abstract class BaseInstaller implements InstallerInterface
{
	/**
	 * @var Composer
	 */
	protected $composer;

	/**
	 * @var DownloadManager
	 */
	protected $downloadManager;

	/**
	 * @var IOInterface
	 */
	protected $io;

	/**
	 * @var Filesystem
	 */
	protected $filesystem;

	/**
	 * @var BinaryInstaller|null
	 */
	protected $binaryInstaller;

	/**
	 * @var Util
	 */
	protected $util;

	public function __construct(IOInterface $io, Composer $composer, Filesystem $filesystem = null, BinaryInstaller $binaryInstaller = null)
	{
		$this->composer        = $composer;
		$this->downloadManager = $composer->getDownloadManager();
		$this->io              = $io;

		$this->filesystem      = $filesystem ?: new Filesystem();
		$this->util            = new Util($this->filesystem);
		$this->binaryInstaller = $binaryInstaller ?: $this->util->createBinaryInstaller($io, $composer->getConfig());
	}

	public function download(PackageInterface $package, PackageInterface $prevPackage = null): ?PromiseInterface
	{
		// Download the package to the Composer cache
		$downloadPath = $this->getInstallPath($package);

		return $this->downloadManager->download($package, $downloadPath, $prevPackage);
	}

	public function prepare($type, PackageInterface $package, PackageInterface $prevPackage = null)
	{
		// Register pre-install and pre-update scripts for the previous package, if there is one
		if ($prevPackage instanceof CompletePackageInterface) {
			$this->registerPackageScripts($prevPackage, [ScriptEvents::PRE_INSTALL_CMD, ScriptEvents::PRE_UPDATE_CMD]);
		}

		// We check for ANY stateful paths here. This applies for Core & Modules.
		// If the following holds true, then we can safely call prepare() (async-delete the installation directory).
		//
		// - The installation path does not contain the vendor directory
		// - There are no stateful files in the target directory, according to this installer
		// - The installation path is not within another installation (i.e. Core)
		//
		// Otherwise, we allow `update()` -> `updateStatefulCode()` to take care of everything synchronously.
		if ($this->shouldBackup($package)) {
			if ($this->backupsEnabled()) {
				//$this->waitUntilSynchronous(); // TODO: May not be needed for modules

				$this->backup($package); // TODO: $prevPackage?
			}

			return $this->util->resolve();
		} else {
			if ($this->util->composer1Api()) {
				return null;
			}

			// Async-delete the package
			$installPath = $this->getInstallPath($package);

			$this->logDebug("Running downloader preparation for {$package->getPrettyString()} (emptying \"$installPath\" or cleaning changes)");

			return $this->downloadManager->prepare($type, $package, $installPath, $prevPackage);
		}
	}

	/**
	 * Backup a package.
	 *
	 * Backs up stateful package files when preparing to install or update a package.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	public function backup(PackageInterface $package): void
	{
		$this->backupStatefulPaths($package);
	}

	/**
	 * Backup a package in its entirety.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	protected function backupFullPackage(PackageInterface $package): void
	{
		$installPath = $this->getInstallPath($package);
		$backupPath  = $this->getBackupPath($package);

		$this->logVerbose("<info>Backing up {$package->getPrettyString()} fully from $installPath to $backupPath</info>");

		$this->filesystem->copy($installPath, $backupPath);
	}

	/**
	 * Backup the stateful paths of a package, including the vendor directory if it resides within the package
	 * installation directory.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	protected function backupStatefulPaths(PackageInterface $package): void
	{
		$installPath = $this->getInstallPath($package);
		$backupPath  = $this->getBackupPath($package);

		$this->logVerbose("<info>Backing up stateful paths of {$package->getPrettyString()} from $installPath to $backupPath</info>");

		$this->copyStatefulPackageFiles($package, $installPath, $backupPath);
		$this->copyVendor($package, $backupPath);
	}

	/**
	 * Remove a package's backup if backups are disabled.
	 *
	 * Also clears the backup path stored with the given package.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	public function removeBackup(PackageInterface $package)
	{
		$backupPath = $this->getBackupPath($package);

		if (!$this->backupsEnabled() && is_dir($backupPath)) {
			$this->logVerbose("<info>Removing backup of package {$package->getPrettyString()} from $backupPath</info>");
			$this->filesystem->removeDirectory($backupPath);
		}

		$this->unsetBackupPath($package);
	}

	/**
	 * Remove the downloaded copy of a package, if it exists.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	public function removeDownload(PackageInterface $package)
	{
		$downloadPath = $this->getDownloadPath($package);

		if (is_dir($downloadPath)) {
			$this->logVerbose("<info>Removing download of package {$package->getPrettyString()} from $downloadPath</info>");
			$this->filesystem->removeDirectory($downloadPath);
		}
	}

	public function isInstalled(InstalledRepositoryInterface $repo, PackageInterface $package): bool
	{
		return $repo->hasPackage($package) && is_readable($this->getInstallPath($package));
	}

	public function install(InstalledRepositoryInterface $repo, PackageInterface $package)
	{
		$this->logDebug(__METHOD__);

		$binaryInstaller = $this->binaryInstaller;
		$installPath = $this->getInstallPath($package);

		return $this->util->resolve($this->installCode($repo, $package))
			->then(function () use ($binaryInstaller, $installPath, $repo, $package) {
				$binaryInstaller->installBinaries($package, $installPath);

				if (!$repo->hasPackage($package)) {
					$installedPackage = clone $package;
					$this->unsetBackupPath($installedPackage);
					$repo->addPackage($installedPackage);
				}

				if ($package instanceof CompletePackageInterface) {
					$this->registerPackageScripts($package);
				}
			});
	}

	public function update(InstalledRepositoryInterface $repo, PackageInterface $initial, PackageInterface $target)
	{
		$this->logDebug(__METHOD__);

		$binaryInstaller = $this->binaryInstaller;
		$installPath = $this->getInstallPath($target);

		$binaryInstaller->removeBinaries($initial);

		return $this->util->resolve($this->updateCode($repo, $initial, $target))
			->then(function () use ($binaryInstaller, $installPath, $repo, $initial, $target) {
				$this->binaryInstaller->installBinaries($target, $installPath);

				$repo->removePackage($initial);

				if (!$repo->hasPackage($target)) {
					$installedPackage = clone $target;
					$this->unsetBackupPath($installedPackage);
					$repo->addPackage($installedPackage);
				}

				if ($target instanceof CompletePackageInterface) {
					$this->registerPackageScripts($target);
				}
			});
	}

	public function uninstall(InstalledRepositoryInterface $repo, PackageInterface $package)
	{
		$this->logDebug(__METHOD__);

		$this->verifyInstalledPackage($repo, $package);

		$binaryInstaller = $this->binaryInstaller;

		return $this->util->resolve($this->removeCode($package))
			->then(function () use ($binaryInstaller, $repo, $package) {
				$binaryInstaller->removeBinaries($package);
				$repo->removePackage($package);
			});
	}

	public function cleanup($type, PackageInterface $package, PackageInterface $prevPackage = null): ?PromiseInterface
	{
		$this->logDebug(__METHOD__ . " $type {$package->getPrettyString()}");

		// TODO: Track changes attempted, reverse them and cleanup wherever possible
		//       Restoring from backups would be ideal
		$installPath = $this->getInstallPath($package);

		$this->removeDownload($package);

		// TODO: Check whether install/update/uninstall was successful
		//       We will want to keep the backup on failure
		$this->removeBackup($package);

		// Bail early for Composer 1, cleanup() only exists for Composer 2
		if ($this->util->composer1Api()) {
			return null;
		}

		return $this->util->resolve(
			$this->downloadManager->cleanup($type, $package, $installPath, $prevPackage)
		);
	}

	public function installCode(
		InstalledRepositoryInterface $installedRepository,
		PackageInterface             $package
	): ?PromiseInterface {
		$installPath = $this->getInstallPath($package);

		$this->filesystem->ensureDirectoryExists($installPath);

		// If the installation path is empty, we can install to it directly
		if ($this->filesystem->isDirEmpty($installPath)) {
			$this->logVerbose("<info>Installing $package directly to $installPath</info>");
			$this->filesystem->emptyDirectory($installPath);

			return $this->downloadInstallOrUpdate($package, $package, $installPath);
		} else {
			$this->logVerbose("<info>Install path $installPath is not empty; safely installing code</info>");

			return $this->updateCode($installedRepository, $package, $package);
		}
	}

	/**
	 * Update the code of the given package.
	 *
	 * If `$initial === $target`, this resolves to an installation operation.
	 *
	 * @param InstalledRepositoryInterface      $repo
	 * @param PackageInterface                  $initial
	 * @param PackageInterface                  $target
	 * @return PromiseInterface|null
	 */
	public function updateCode(
		InstalledRepositoryInterface $repo,
		PackageInterface             $initial,
		PackageInterface             $target
	): ?PromiseInterface {
		$install = $initial === $target;

		if (!$install) {
			$this->verifyInstalledPackage($repo, $initial);
		}

		$installPath = $this->getInstallPath($target);

		$gitDirectoryExists = is_readable($this->util->joinPath($installPath, '.git'));
		$gitInstallations = $initial->getInstallationSource() === 'source'
			&& $initial->getSourceType() === 'git'
			&& $target->getSourceType() === 'git';

		// Try to update using Git first, otherwise perform a stateful update
		// TODO: Check for stateful paths; it may be safe to update normally
		//       CoreInstaller::getStatefulPaths() needs to include modules
		//       before this is safe to implement
		if ($gitInstallations && $gitDirectoryExists) {
			$this->logVerbose("<info>Updating using Git</info>");
			return $this->downloadInstallOrUpdate($initial, $target, $installPath, 'update');
		} else {
			$this->logVerbose("<info>Updating stateful installation</info>");
			return $this->updateStatefulCode($repo, $initial, $target);
		}
	}

	/**
	 * Update potentially-stateful package code safely using the following algorithm:
	 *
	 * 1. Download the new package to a new temporary path
	 * 2. Copy stateful files from the installed package to the newly downloaded package
	 * 3. Replace the installed package with the new package
	 *
	 * This also handles the legacy `installer/composer.json` case, where we're running Composer from the directory
	 * we're replacing. In this case, the algorithm is altered to avoid filesystem errors:
	 *
	 * 1. Download the new package to a new temporary path
	 * 2. Copy stateful files from the installed package to the newly downloaded package
	 * 3. Change working directory to the installation path
	 * 4. Empty the installation path
	 * 5. Recreate the original working directory
	 * 6. Copy the new package into the installation path
	 *
	 * @param InstalledRepositoryInterface $installedRepository Repository of currently installed packages.
	 * @param PackageInterface             $initial             Currently installed package.
	 * @param PackageInterface             $package             Package to update to.
	 * @return PromiseInterface|null
	 */
	public function updateStatefulCode(
		InstalledRepositoryInterface $installedRepository,
		PackageInterface $initial,
		PackageInterface $package
	): ?PromiseInterface {
		// Wait until any other operations have finished; we could be about to back up or remove a directory that
		// another operation is modifying
		$this->waitUntilSynchronous();

		$installPath = $this->getInstallPath($package);
		$downloadPath = $this->getDownloadPath($package);

		$this->logDebug([
			"Install path:  $installPath",
			"Download path: $downloadPath"
		]);

		// Composer 1 doesn't call prepare(), so we call it here synchronously at install time
		if ($this->util->composer1Api()) {
			// Load the promise class before it's wiped out; we may be about to load it for the first time in the
			// asynchronous callback below!
			// TODO: Can we move this?
			class_exists(Promise::class);

			// Backup
			// If this is Core, it will be partially modified already,
			// but this is the best we can do with Composer 1
			$operation = $this->resolveOperationType($initial, $package);
			$this->prepare($operation, $package, $initial);
		}

		// Install the new package to the temporary download directory asynchronously
		$this->filesystem->emptyDirectory($downloadPath);
		$promise = $this->downloadInstallOrUpdate($initial, $package, $downloadPath);

		return $this->util->resolve($promise)
			->then(function () use ($installedRepository, $initial, $package, $installPath, $downloadPath) {
				// Copy any stateful files from the backup, and the vendor directory from the current installation,
				// if any, into the newly downloaded package
				$this->waitUntilSynchronous();

				$this->copyStateAndReplace($installPath, $downloadPath, $package, $initial);

				// Phew!
				// TODO: Verify that critical files are in place, maybe even verify a run of CLC to ensure things are
				//       running as expected
				//       Revert and throw an exception if anything is wrong

				// Composer 1 doesn't call cleanup(), so we invoke it here instead
				if ($this->util->composer1Api()) {
					$operation = $this->resolveOperationType($initial, $package);

					$this->cleanup($operation, $package, $initial);
				}
			});
	}

	/**
	 * Helper method that calls the appropriate Composer Download Manager method based on:
	 *
	 * - Whether we're running Composer 1
	 * - Whether the "initial" and "target" packages are the same (install operation)
	 *
	 * Calling DownloadManager methods incorrectly can cause Composer 1 & 2 to behave undesirably in certain situations.
	 *
	 * For example, calling `$downloadManager->update()` for a freshly installed package using Composer 2
	 * can cause it to think the package is already installed, when it isn't, or without interactivity it
	 * will simply fail to install, complaining about a missing .git directory.
	 *
	 * @param PackageInterface $initial The currently installed package.
	 * @param PackageInterface $target  The package to download, install, or update to.
	 * @param string           $path    The path to install the package to.
	 * @param string|null      $type    Optional operation type. Pass `'update'` to force an update.
	 * @return PromiseInterface|null
	 */
	protected function downloadInstallOrUpdate(
		PackageInterface $initial,
		PackageInterface $target,
		string           $path,
		string           $type = null
	): ?PromiseInterface {
		$install = isset($type) ? $type !== 'update' : $initial === $target;
		$message = "<info>%s {$target->getPrettyString()} to $path</info>";

		if ($this->util->composer1Api()) {
			$verb = $install ? 'Downloading' : 'Updating';
			$this->logVerbose(sprintf($message, $verb));

			return $install
				? $this->downloadManager->download($target, $path)
				: $this->downloadManager->update($initial, $target, $path);
		}

		$verb = $install ? 'Installing' : 'Updating';
		$this->logVerbose(sprintf($message, $verb));

		return $install
			? $this->downloadManager->install($target, $path)
			: $this->downloadManager->update($initial, $target, $path);
	}

	/**
	 * Remove a package and its installation directory while retaining any stateful paths.
	 *
	 * @param PackageInterface $package
	 * @return PromiseInterface|null
	 */
	public function removeCode(PackageInterface $package): ?PromiseInterface
	{
		$installPath = $this->getInstallPath($package);

		// TODO: May not be necessary to be synchronous for modules?
		$this->waitUntilSynchronous();

		// Backup synchronously per-package for Composer 1; prepare() won't be called otherwise
		if ($this->util->composer1Api()) {
			$this->prepare('uninstall', $package, $package);
		}

		// Copy stateful files and vendor for restoration after removal
		$temporaryPath = $this->getTemporaryPath($package);
		$this->copyStateAndReplace($installPath, $temporaryPath, $package);

		// TODO: Potential performance improvement
		//       If we know there's no state, no modules and no vendor, we can delete asynchronously
		//       return $this->downloadManager->remove($package, $installPath);

		return null;
	}

	/**
	 * Copy the stateful paths from a source installation path to a target installation path, then replace the source
	 * path with the target path.
	 *
	 * If the target path doesn't exist, then the source path is left empty or removed.
	 *
	 * Crucial part of the stateful update algorithm that works as efficiently as it can to update a package while
	 * retaining any known stateful file paths.
	 *
	 * Used by the update and uninstallation processes.
	 *
	 * @param string                            $sourcePath          The source path to copy state from and replace.
	 * @param string                            $targetPath          The target path to copy state to and replace with.
	 * @param PackageInterface                  $package             The new package to replace with.
	 * @param PackageInterface|null             $prevPackage         The package being replaced.
	 * @param InstalledRepositoryInterface|null $installedRepository Optional installed package repository.
	 * @return void
	 */
	public function copyStateAndReplace(
		string                       $sourcePath,
		string                       $targetPath,
		PackageInterface             $package,
		PackageInterface             $prevPackage = null,
		InstalledRepositoryInterface $installedRepository = null
	): void {
		$this->copyStatefulPackageFiles($package, $sourcePath, $targetPath, $installedRepository);
		$this->copyVendor($prevPackage ?: $package, $targetPath, $sourcePath);

		// Remove and replace
		$this->logVerbose("Removing $sourcePath");
		$this->filesystem->removeDirectory($sourcePath);

		if (is_dir($targetPath)) {
			$this->logVerbose("Moving $targetPath to $sourcePath");
			$this->filesystem->rename($targetPath, $sourcePath);
		}
	}

	/**
	 * Get potentially stateful file paths, relative to the package installation path.
	 *
	 * These file paths do not need to exist. This only needs to be a list of potentially stateful file paths to retain
	 * during installations, updates and uninstallations.
	 *
	 * This returns an empty array by default, as packages should ideally be stateless.
	 *
	 * @see findStatefulPaths() to find existing stateful paths
	 * @param PackageInterface $package
	 * @return string[]
	 */
	public function getStatefulPaths(PackageInterface $package): array
	{
		return [];
	}

	/**
	 * Find existing stateful file paths, relative to the package installation path.
	 *
	 * @see getStatefulPaths() to list potential stateful paths
	 * @param PackageInterface $package Package to find stateful paths for
	 * @return string[] Stateful paths that exist within the package installation path
	 */
	public function findStatefulPaths(PackageInterface $package): array
	{
		$installPath = $this->getInstallPath($package);
		$potentialStatefulPaths = $this->getStatefulPaths($package);
		$statefulPaths = [];

		foreach ($potentialStatefulPaths as $statefulPath) {
			$fullStatefulPath = $this->util->joinPath($installPath, $statefulPath);

			if (file_exists($fullStatefulPath)) {
				$statefulPaths[] = $statefulPath;
			}
		}

		return $statefulPaths;
	}

	/**
	 * Check whether a package has any existing stateful paths within its installation path.
	 *
	 * @param PackageInterface $package Package to check for state
	 * @return bool
	 */
	public function containsStatefulPaths(PackageInterface $package): bool
	{
		return !empty($this->findStatefulPaths($package));
	}

	/**
	 * Check whether a package _may_ contain the vendor directory.
	 *
	 * @param PackageInterface $package
	 * @return bool
	 */
	public function containsVendor(PackageInterface $package): bool
	{
		$installPath = $this->getInstallPath($package);
		$vendorPath = $this->getVendorPath();

		return $this->util->relativePathStartsWith($installPath, $vendorPath);
	}

	/**
	 * Copy package files that should be retained during an update.
	 *
	 * This serves a few different purposes when updating Core and Modules:
	 * - Retaining installed modules
	 * - Retaining legacy configuration files
	 * - Retaining user data
	 *
	 * This is a no-op by default, as packages should ideally be stateless.
	 *
	 * @param PackageInterface                  $package             The package being updated.
	 * @param string                            $sourcePath          The source installation path to copy stateful files from.
	 * @param string                            $targetPath          The target path to copy stateful files to.
	 * @param InstalledRepositoryInterface|null $installedRepository The installed repository (vendor/composer/installed.json).
	 * @return void
	 */
	public function copyStatefulPackageFiles(
		PackageInterface             $package,
		string                       $sourcePath,
		string                       $targetPath,
		InstalledRepositoryInterface $installedRepository = null
	): void {
		// No-op
	}

	/**
	 * Copy the vendor directory from one package installation path to another if it's configured to be contained
	 * within the package installation directory.
	 *
	 * This useful for retaining installation state during the installation of a package that contains the root
	 * project's vendor directory, i.e. Claromentis Core.
	 *
	 * The method is generalised enough to work with any package.
	 *
	 * @param PackageInterface $package     The package to copy the vendor directory from
	 * @param string           $targetPath  The target installation path to copy the vendor directory to, relative to the root package.
	 * @param string|null      $sourcePath  Optional package installation source path to copy the vendor directory from, relative from the root package.
	 *                                      Defaults to the package's installation path.
	 * @return void
	 */
	public function copyVendor(PackageInterface $package, string $targetPath, string $sourcePath = null): void
	{
		if (!$this->containsVendor($package)) {
			$this->logDebug("Vendor directory is not configured within {$package->getPrettyString()} install path, not copying");

			return;
		}

		$this->waitUntilSynchronous();

		$installPath = $this->getInstallPath($package);
		$sourcePath  = $sourcePath ?: $installPath;

		if (!is_dir($sourcePath)) {
			$this->logVerbose("Source path $sourcePath does not exist, not copying vendor to $targetPath");
			return;
		}

		// Determine the package-relative vendor directory paths for source and target paths
		// This is a simple path prefix replacement because we've asserted above that the
		// package installation path contains the configured vendor path
		$vendorPath       = $this->getVendorPath();
		$sourceVendorPath = $this->util->replaceRelativePathPrefix($installPath, $sourcePath, $vendorPath);
		$targetVendorPath = $this->util->replaceRelativePathPrefix($installPath, $targetPath, $vendorPath);

		$this->logDebug([
			"copyVendor() debug",
			"  Package:            {$package->getPrettyString()}",
			"  Install path:       $installPath",
			"  Source path:        $sourcePath",
			"  Target path:        $targetPath",
			"  Vendor path:        $vendorPath",
			"  Source vendor path: $sourceVendorPath",
			"  Target vendor path: $targetVendorPath"
		]);

		if (is_dir($sourceVendorPath)) {
			$this->logVerbose("Copying $sourceVendorPath to $targetVendorPath");

			$this->filesystem->copy($sourceVendorPath, $targetVendorPath);
		} else {
			$this->logVerbose("Vendor directory $sourceVendorPath not found, not copying");
		}
	}

	/**
	 * Get the backup path for a package.
	 *
	 * Checks package 'extra' data for an existing backup path. Generates and sets one otherwise.
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	public function getBackupPath(PackageInterface $package): string
	{
		$extra = $package->getExtra();

		if (isset($extra['claromentis']['installer']['backup-path'])) {
			return $extra['claromentis']['installer']['backup-path'];
		}

		$timestamp   = date('Y-m-d_H.i.s'); // TODO: Time as a service, for unit tests
		$backupsPath = $this->getBackupsPath();
		$packageName = $this->util->getSanitizedPackageName($package);
		$backupPath  = $this->util->joinPath($backupsPath, "{$timestamp}_$packageName");

		// Persist the backup path in the package's extra data for installation to reuse
		// This never gets saved to a file, it's only used in-memory
		$extra['claromentis']['installer']['backup-path'] = $backupPath;

		if ($package instanceof Package) {
			$package->setExtra($extra);
		}

		return $backupPath;
	}

	/**
	 * Remove the backup path stored with a package if possible.
	 *
	 * @param PackageInterface $package
	 * @return void
	 */
	public function unsetBackupPath(PackageInterface $package)
	{
		if ($package instanceof Package) {
			$extra = $package->getExtra();
			unset($extra['claromentis']['installer']['backup-path']);
			$package->setExtra($extra);
		}
	}

	/**
	 * Get the backups path for Claromentis Composer packages.
	 *
	 * This can be configured in the root project's `composer.json` file, and defaults to `'backups'`.
	 *
	 * This must **always** be a path relative to the root project directory, with slashes and whitespace trimmed.
	 *
	 * @return string
	 */
	public function getBackupsPath(): string
	{
		$defaultPath = $this->isLegacyInstallation() ? '../../backups' : 'backups';

		$rootConfig = $this->composer->getPackage()->getConfig();

		return $rootConfig['claromentis']['installer']['paths']['backups'] ?? $defaultPath;
	}

	/**
	 * Get the download path for a Claromentis package.
	 *
	 * This is used as a temporary installation directory for stateful package updates.
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	public function getDownloadPath(PackageInterface $package): string
	{
		$backupsPath = $this->getBackupsPath();
		$packageName = $this->util->getSanitizedPackageName($package);

		return $this->util->joinPath($backupsPath, "{$packageName}_download");
	}

	/**
	 * Get a temporary path for storing files from a Claromentis package.
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	public function getTemporaryPath(PackageInterface $package): string
	{
		$backupsPath = $this->getBackupsPath();
		$packageName = $this->util->getSanitizedPackageName($package);

		return $this->util->joinPath($backupsPath, "{$packageName}_temp");
	}

	/**
	 * Get the installation path for Claromentis Core.
	 *
	 * This can be configured in the root project's `composer.json` file, and defaults to `'application'`, or `'../'`
	 * for legacy installations.
	 *
	 * This must **always** be a path relative to the root project directory, with slashes and whitespace trimmed.
	 *
	 * @return string
	 */
	public function getCoreInstallPath(): string
	{
		if ($this->isRootPackageModule()) {
			$defaultPath = $this->getVendorPath() . '/claromentis/framework';
		} else {
			$defaultPath = $this->isLegacyInstallation() ? '../' : 'application';
		}

		$rootPackage = $this->composer->getPackage();
		$rootConfig = $rootPackage->getConfig();

		return $rootConfig['claromentis']['installer']['paths']['core'] ?? $defaultPath;
	}

	/**
	 * Get the installation path for Claromentis Modules.
	 *
	 * This can be configured in the root project's composer.json, and defaults to `'application/web/intranet'`, or
	 * `'../web/intranet'` for legacy installations.
	 *
	 * This must **always** be a path relative to the root project directory, with slashes and whitespace trimmed.
	 *
	 * @return string
	 */
	public function getModuleInstallPath(): string
	{
		$rootConfig = $this->composer->getPackage()->getConfig();

		return $rootConfig['claromentis']['installer']['paths']['module']
			?? $this->util->joinPath($this->getCoreInstallPath(), 'web', 'intranet');
	}

	/**
	 * Get the installation path for Claromentis Custom Modules.
	 *
	 * This can be configured in the root project's composer.json, and defaults to `'application/web/custom'`, or
	 * `'../web/custom'` for legacy installations.
	 *
	 * This must **always** be a path relative to the root project directory, with slashes and whitespace trimmed.
	 *
	 * @return string
	 */
	public function getCustomModuleInstallPath(): string
	{
		$rootConfig = $this->composer->getPackage()->getConfig();

		return $rootConfig['claromentis']['installer']['paths']['custom']
			?? $this->util->joinPath($this->getCoreInstallPath(), 'web', 'custom');
	}

	/**
	 * Get the vendor directory path.
	 *
	 * This will either be as configured by the root project's `composer.json`, or fall back to a default of `'vendor'`.
	 *
	 * @see getAbsoluteVendoirPath() to get the absolute vendor directory path
	 * @return string
	 */
	public function getVendorPath(): string
	{
		// We don't use $this->composer->getConfig()->get('vendor-dir') because it provides an absolute path
		// We need a relative path, and paths will always be relative when defined in composer.json,
		// which is what we're reading directly from below
		$rootPackage = $this->composer->getPackage();
		$rootConfig = $rootPackage->getConfig();

		return $rootConfig['vendor-dir'] ?? 'vendor';
	}

	/**
	 * Get the absolute vendor directory path.
	 *
	 * @return string
	 */
	public function getAbsoluteVendorPath(): string
	{
		return $this->composer->getConfig()->get('vendor-dir');
	}

	/**
	 * Best-effort detection of legacy Claromentis installations that use `installer/composer.json`.
	 *
	 * Used to adapt default installation paths.
	 *
	 * It can only be a best-effort attempt due to the lack of truly uniquely identifying features of this legacy
	 * `composer.json` file. The following serve as the most identifying features:
	 *
	 * - Required dependency of Claromentis' fork of Wikimedia's Composer Merge Plugin
	 * - Merge plugin configuration to include `'../modules.json'`
	 * - `'../vendor'` vendor path
	 *
	 * @see getBackupsPath()
	 * @see getCoreInstallPath()
	 * @see getModuleInstallPath()
	 * @see getVendorPath()
	 * @return bool
	 */
	protected function isLegacyInstallation(): bool
	{
		$rootPackage  = $this->composer->getPackage();
		$rootRequires = $rootPackage->getRequires();
		$rootConfig   = $rootPackage->getConfig();
		$rootExtra    = $rootPackage->getExtra();

		$hasMergePluginRequire = false;

		foreach ($rootRequires as $rootRequire) {
			if ($rootRequire->getTarget() === 'claromentis/composer-merge-plugin') {
				$hasMergePluginRequire = true;
				break;
			}
		}

		$hasModulesJsonInclude = in_array('../modules.json', $rootExtra['merge-plugin']['include'] ?? []);
		$vendorPath            = $rootConfig['vendor-dir'] ?? null;

		return $hasMergePluginRequire && $hasModulesJsonInclude && $vendorPath === '../vendor';
	}

	/**
	 * Check whether the root package is a Claromentis Module; whether it's a package supported by `ModuleInstaller`.
	 *
	 * @return bool
	 */
	protected function isRootPackageModule(): bool
	{
		$rootPackage = $this->composer->getPackage();
		$moduleInstaller = $this->composer->getInstallationManager()->getInstaller('claromentis-module');

		return $moduleInstaller->supports($rootPackage->getType());
	}

	/**
	 * Verifies that the given package is installed.
	 *
	 * Throws an invalid argument exception if the package is not installed.
	 *
	 * Composer's installers duplicate this snippet across several methods. Here, we implement it as a method for reuse.
	 *
	 * @param InstalledRepositoryInterface $repo
	 * @param PackageInterface             $package
	 * @throws InvalidArgumentException If the package is not installed.
	 */
	protected function verifyInstalledPackage(InstalledRepositoryInterface $repo, PackageInterface $package)
	{
		if (!$repo->hasPackage($package)) {
			throw new InvalidArgumentException('Package is not installed: ' . $package);
		}
	}

	/**
	 * Register the scripts of a package with the event dispatcher.
	 *
	 * Mimics some behaviour from `EventDispatcher::getScriptListeners()` to register listeners for a non-root package.
	 *
	 * @see EventDispatcher::getScriptListeners()
	 * @param CompletePackageInterface $package The package to register scripts for.
	 * @param string[]                 $scripts The scripts to register. Defaults to `null`, which means all scripts.
	 * @return void
	 */
	protected function registerPackageScripts(CompletePackageInterface $package, ?array $scripts = null)
	{
		$validScriptEvents = $this->getValidScriptEvents();

		if (is_array($scripts)) {
			$validScriptEvents = array_intersect($validScriptEvents, $scripts);
		}

		// Retrieve the package's scripts
		$packageScripts = $package->getScripts();

		if (empty($packageScripts)) {
			return;
		}

		$this->registerAutoloaderForPackage($package);

		$this->logVerbose("<info>Registering scripts for package {$package->getPrettyName()}</info>");

		$dispatcher = $this->composer->getEventDispatcher();

		foreach ($packageScripts as $scriptName => $scriptCommands) {
			if (!in_array($scriptName, $validScriptEvents)) {
				$this->logDebug("Skipping non-event script '$scriptName'");
				continue;
			}

			if (empty($scriptCommands)) {
				$this->logDebug("Skipping empty script for event '$scriptName'");
				continue;
			}

			foreach ($scriptCommands as $command) {
				$this->logDebug("Registering script for event '$scriptName', command '$command'");
				$dispatcher->addListener($scriptName, $command);
			}
		}
	}

	/**
	 * Gets a list of all valid script event names defined in Composer.
	 *
	 * @return string[]
	 */
	protected function getValidScriptEvents(): array
	{
		$scriptEventsClass = new ReflectionClass(ScriptEvents::class);
		return array_values($scriptEventsClass->getConstants());
	}

	/**
	 * Register a Composer autoloader for a package.
	 *
	 * Adapted from Composer's `PluginManager::registerPackage()` method.
	 *
	 * @see PluginManager::registerPackage() for the inspiration
	 * @param PackageInterface $package
	 * @return void
	 */
	protected function registerAutoloaderForPackage(PackageInterface $package)
	{
		$this->logVerbose("<info>Generating and registering autoloader for package {$package->getPrettyName()}</info>");

		$installPath = $this->getInstallPath($package);

		$autoloads[] = [$package, $installPath];
		$rootPackage = $this->composer->getPackage();

		$generator = $this->composer->getAutoloadGenerator();
		$map = $generator->parseAutoloads($autoloads, $rootPackage);
		$classLoader = $generator->createLoader($map, $this->composer->getConfig()->get('vendor-dir'));
		$classLoader->register(false);
	}

	/**
	 * Check whether backups are enabled.
	 *
	 * Checks the root package's configuration for whether replaced application directories should be retained.
	 *
	 * Defaults to `false` when configuration isn't set.
	 *
	 * @return bool
	 */
	protected function backupsEnabled(): bool
	{
		$config = $this->composer->getPackage()->getConfig();

		return (bool) ($config['claromentis']['installer']['backups'] ?? false);
	}

	/**
	 * Check whether a package should be backed up.
	 *
	 * Checks whether a package's installation directory contains stateful paths, or whether it _may_ contain the
	 * vendor directory.
	 *
	 * TODO: Optionally accepts a package installation path to check.
	 *
	 * @param PackageInterface $package
	 * @return bool
	 */
	protected function shouldBackup(PackageInterface $package): bool
	{
		return $this->containsStatefulPaths($package) || $this->containsVendor($package);
	}

	/**
	 * Resolve the operation type for a set of initial and target packages.
	 *
	 * Returns `'install'` if both packages are the same, and `'update'` if they are not.
	 *
	 * @param PackageInterface $initial
	 * @param PackageInterface $target
	 * @return string 'install' or 'update'
	 */
	protected function resolveOperationType(PackageInterface $initial, PackageInterface $target): string
	{
		return  $initial === $target ? 'install' : 'update';
	}

	/**
	 * Wait until all asynchronous Composer processes have finished.
	 *
	 * @see \Composer\Util\Loop::wait() for what this relies on
	 */
	protected function waitUntilSynchronous()
	{
		// Return immediately if we're using Composer 1; there are no asynchronous processes
		if ($this->util->composer1Api()) {
			return;
		}

		// Otherwise, wait until Composer's asynchronous loop has finished
		$loop = $this->composer->getLoop();

		$loop->wait([]);
	}

	/**
	 * @param string|string[] $messages
	 * @param int $verbosity Optional `IOInterface` verbosity. Defaults to `IOInterface::NORMAL`.
	 */
	protected function log($messages, int $verbosity = IOInterface::NORMAL)
	{
		$this->io->writeError($messages, true, $verbosity);
	}

	/**
	 * @param string|string[] $messages
	 */
	protected function logVerbose($messages)
	{
		$this->log($messages, IOInterface::VERBOSE);
	}

	/**
	 * @param string|string[] $messages
	 */
	protected function logDebug($messages)
	{
		$this->log($messages, IOInterface::DEBUG);
	}
}
