<?php

namespace Claromentis\Composer;

use Composer\IO\IOInterface;
use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Repository\InstalledRepositoryInterface;
use Composer\Semver\Comparator;
use Composer\Semver\Constraint\Constraint;
use RuntimeException;

/**
 * Claromentis Core installer.
 *
 * @author Chris Andrew <chris.andrew@claromentis.com>
 */
class CoreInstaller extends BaseInstaller
{
	/**
	 * Composer package names for Claromentis Core.
	 *
	 * - `claromentis/framework` is the current name
	 * - `claromentis/core` is the intended name in future
	 *
	 * @var string[]
	 */
	public const PACKAGE_NAMES = [
		'claromentis/core', // Future-proofing for project name change
		'claromentis/framework'
	];

	/**
	 * Composer package types for Claromentis Core.
	 *
	 * - `claromentis-core` is canonical starting with Claromentis 9.
	 * - `claromentis-framework[-v8]` are legacy package types supported for backwards-compatibility.
	 *
	 * @var string[]
	 */
	public const PACKAGE_TYPES = [
		'claromentis-core',
		'claromentis-framework',
		'claromentis-framework-v8'
	];

	/**
	 * Stateful paths in Core to copy from an old installation to a new installation.
	 *
	 * This excludes modules, and should only include files present in the Core packages and installations themselves.
	 *
	 * This is primarily for backwards-compatibility with older versions of Core, to avoid data loss. Moving forward,
	 * the Claromentis application directory should be completely stateless.
	 *
	 * @var string[]
	 */
	public const STATEFUL_PATHS = [
		'.env',
		'data',
		'local_data',
		'modules.json',
		'web/appdata',
		'web/custom',
		'web/intranet/common/config.php'
	];

	public function supports($packageType): bool
	{
		return in_array($packageType, static::PACKAGE_TYPES);
	}

	public function backup(PackageInterface $package): void
	{
		$this->waitUntilSynchronous();

		// TODO: #15 Check whether the installation contains modules
		//       This could be performed in CoreInstaller only, perhaps in a `shouldBackup()` override

		parent::backup($package);
	}

	public function uninstall(InstalledRepositoryInterface $repo, PackageInterface $package)
	{
		$this->io->write('<comment>Uninstalling Claromentis Core</comment>');

		$this->waitUntilSynchronous();

		return parent::uninstall($repo, $package);
	}

	public function getInstallPath(PackageInterface $package): string
	{
		// TODO: Unit tests
		// Legacy Core 7.x support for non-branch versions
		$version = $package->getVersion();
		/** @see \Composer\Semver\Constraint\Constraint::versionCompare() */
		$versionIsBranch = strpos($version, 'dev-') === 0;

		if (!$versionIsBranch && Comparator::lessThan($version, '8.0.0-dev')) {
			return $this->util->joinPath($this->getCoreInstallPath(), 'web');
		}

		// Core 8.x+
		return $this->getCoreInstallPath();
	}

	public function copyStateAndReplace(
		string                       $sourcePath,
		string                       $targetPath,
		PackageInterface             $package,
		PackageInterface             $prevPackage = null,
		InstalledRepositoryInterface $installedRepository = null
	): void {
		if ($this->isLegacyInstallation()) {
			// Handle legacy installer/composer.json installations that install on top of themselves
			$this->copyStatefulPackageFiles($package, $sourcePath, $targetPath, $installedRepository);
			$this->copyVendor($package, $targetPath, $sourcePath);

			$originalWorkingDirectory = getcwd();
			$realSourcePath           = realpath($sourcePath);
			$this->logVerbose("Changing working directory to $realSourcePath");
			chdir($realSourcePath);
			$this->logVerbose("Emptying $realSourcePath");
			$this->filesystem->emptyDirectory($realSourcePath);
			$this->logVerbose("Changing working directory to $originalWorkingDirectory");
			$this->filesystem->ensureDirectoryExists($originalWorkingDirectory);
			chdir($originalWorkingDirectory);

			if (is_dir($targetPath)) {
				$this->logVerbose("Copying $targetPath to $sourcePath");
				$this->filesystem->copy($targetPath, $sourcePath);
				$this->logVerbose("Removing $targetPath");
				$this->filesystem->removeDirectory($targetPath);
			}
		} else {
			parent::copyStateAndReplace($sourcePath, $targetPath, $package, $prevPackage, $installedRepository);
		}
	}

	public function getStatefulPaths(PackageInterface $package): array
	{
		return self::STATEFUL_PATHS;
	}

	/**
	 * Copy Claromentis modules and user data from one Core installation to another.
	 *
	 * @inheritDoc
	 */
	public function copyStatefulPackageFiles(
		PackageInterface             $package,
		string                       $sourcePath,
		string                       $targetPath,
		InstalledRepositoryInterface $installedRepository = null
	): void {
		$installationManager = $this->composer->getInstallationManager();
		$installedRepository = $installedRepository ?: $this->composer->getRepositoryManager()->getLocalRepository();
		$installedPackages   = $installedRepository->getCanonicalPackages();
		$moduleInstaller     = $installationManager->getInstaller('claromentis-module');
		$coreInstallPath     = $this->getCoreInstallPath();
		$moduleInstallPath   = $this->getModuleInstallPath();
		$coreContainsModules = $this->util->relativePathStartsWith($coreInstallPath, $moduleInstallPath);

		$this->logVerbose("Copying stateful file paths from $sourcePath to $targetPath");

		$this->filesystem->ensureDirectoryExists($targetPath);

		$this->logDebug("Copying Composer-installed Claromentis modules");

		if ($coreContainsModules) {
			foreach ($installedPackages as $installedPackage) {
				$this->logDebug("Checking {$installedPackage->getPrettyString()}");

				if (!$moduleInstaller->supports($installedPackage->getType())) {
					continue;
				}

				// Use path replacement to build the source and target paths for this module
				$installedPackagePath       = $moduleInstaller->getInstallPath($installedPackage);
				$sourceInstalledPackagePath = $this->util->replaceRelativePathPrefix($coreInstallPath, $sourcePath, $installedPackagePath);
				$targetInstalledPackagePath = $this->util->replaceRelativePathPrefix($coreInstallPath, $targetPath, $installedPackagePath);

				$this->logVerbose("Copying {$installedPackage->getPrettyString()} from $sourceInstalledPackagePath to $targetInstalledPackagePath");

				// The target directory could exist if Core introduces directories with the same name as a module
				// We can only throw an exception in this situation
				// TODO: Another approach would be force-emptying the directory with $this->filesystem->removeDirectory($newInstalledPackagePath)
				if (is_dir($targetInstalledPackagePath)) {
					throw new RuntimeException("Module {$installedPackage->getPrettyString()} install path $targetInstalledPackagePath already exists");
				}

				$this->filesystem->ensureDirectoryExists(dirname($targetInstalledPackagePath));
				$this->filesystem->copy($sourceInstalledPackagePath, $targetInstalledPackagePath);
			}
		}

		// Copy across stateful paths from the current installation
		$statefulPaths = $this->getStatefulPaths($package);

		foreach ($statefulPaths as $statefulPath) {
			$sourceStatefulPath = $this->util->joinPath($sourcePath, $statefulPath);
			$targetStatefulPath = $this->util->joinPath($targetPath, $statefulPath);

			if (!file_exists($sourceStatefulPath)) {
				$this->logDebug("Stateful path $sourceStatefulPath does not exist, not copying");
				continue;
			}

			$this->logVerbose("Copying $sourceStatefulPath to $targetStatefulPath");

			$this->filesystem->ensureDirectoryExists(dirname($targetStatefulPath));
			$this->filesystem->copy($sourceStatefulPath, $targetStatefulPath);
		}
	}

	/**
	 * Normalize a Claromentis Core Composer package type to a canonical value.
	 *
	 * No-op if the package type is not a Claromentis Core package type.
	 *
	 * @param Package $package
	 * @return void
	 */
	public function normalizePackageType(Package $package): void
	{
		if (!$this->supports($package->getType())) {
			return;
		}

		$package->setType(self::PACKAGE_TYPES[0]);
	}
}
