<?php

namespace Claromentis\Composer;

use Composer\EventDispatcher\Event;
use Composer\IO\IOInterface;
use Composer\Package\AliasPackage;
use Composer\Package\Link;
use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Package\RootPackageInterface;
use Composer\Plugin\PrePoolCreateEvent;
use Composer\Repository\RepositoryManager;
use Composer\Semver\Comparator;
use Composer\Semver\VersionParser;

/**
 * Package dependency processing service.
 *
 * Processes dependencies for legacy Composer usage, compatibility or convenience.
 *
 * - Restores `require`s for Core 7.x & 8.x dependencies installed from source
 * - Strips Core from root-project Claromentis Module `require`s so they can `composer install`
 *   comfortably and run automated tests in isolation
 * - Normalizes Claromentis Composer package types to improve Composer's update behaviour
 *
 * TODO: Unit tests
 *
 * @author Chris Andrew <chris.andrew@claromentis.com>
 */
class DependencyProcessor
{
	/**
	 * @var IOInterface
	 */
	private $io;

	/**
	 * @var RepositoryManager
	 */
	private $repositoryManager;

	/**
	 * @var CoreInstaller
	 */
	private $coreInstaller;

	/**
	 * @var ModuleInstaller
	 */
	private $moduleInstaller;

	public function __construct(IOInterface $io, RepositoryManager $repositoryManager, CoreInstaller $coreInstaller, ModuleInstaller $moduleInstaller)
	{
		$this->io                = $io;
		$this->repositoryManager = $repositoryManager;
		$this->coreInstaller     = $coreInstaller;
		$this->moduleInstaller   = $moduleInstaller;
	}

	/**
	 * Normalizes Claromentis Composer package types to their canonical values.
	 *
	 * Processes Claromentis packages from the installed package repository and installation pool to ensure that their
	 * package types match canonically to avoid a bug in Composer. See the linked GitLab issue for more information.
	 *
	 * @link https://gitlab.com/claromentis/product/composer-installer/-/issues/14
	 * @param PackageInterface[] $packages The packages to normalize
	 * @return PackageInterface[] The normalized packages
	 */
	public function normalizePackageTypes(array $packages): array
	{
		$this->io->write(__METHOD__, true, IOInterface::DEBUG);

		$packagesCount = count($packages);

		if (!$packagesCount) {
			return $packages;
		}

		$this->io->write(sprintf("<info>Normalizing package types for %s packages</info>", $packagesCount));

		$normalizedPackagesCount = 0;

		foreach ($packages as $package) {
			// Skip unsupported packages nice and early
			if (!($this->coreInstaller->supports($package->getType()) || $this->moduleInstaller->supports($package->getType()))) {
				continue;
			}

			$this->io->write("Normalizing {$package->getPrettyString()}", true, IOInterface::DEBUG);

			// Resolve aliased packages
			if ($package instanceof AliasPackage) {
				$package = $package->getAliasOf();
				$this->io->write("  Resolved alias to {$package->getPrettyString()}", true, IOInterface::DEBUG);
			}

			// Let the Core & Module installers to normalize the package
			if ($package instanceof Package) {
				$originalType = $package->getType();
				$this->coreInstaller->normalizePackageType($package);
				$this->moduleInstaller->normalizePackageType($package);
				$newType = $package->getType();

				$typeChanged = $originalType !== $newType;

				if ($typeChanged) {
					$this->io->write("  <info>Normalized {$package->getPrettyString()} type from $originalType to $newType</info>", true, IOInterface::DEBUG);
					$normalizedPackagesCount++;
				}
			} else {
				$packageClass = get_class($package);
				$this->io->write("  <info>Skipped normalizing {$package->getPrettyString()} of class $packageClass</info>", true, IOInterface::DEBUG);
			}
		}

		if ($normalizedPackagesCount > 0) {
			$this->io->write("<info>Normalized package types of $normalizedPackagesCount packages</info>");
		} else {
			$this->io->write("<info>No package types needed normalizing</info>");
		}

		return $packages;
	}

	/**
	 * Restore the dependencies of a Claromentis Core dependency of the given root package.
	 *
	 * Claromentis Core 7.x and 8.x distributions bundle their `composer.lock`ed `require` dependencies and mark them as
	 * `replace`s in their `composer.json` and in Composer repositories.
	 *
	 * This method undoes this for source installations by moving all non-Core `replace`s back to `require`s in-memory
	 * at runtime, for all matching Core packages found.
	 *
	 * The affected packages include those found in the local (`vendor/composer/installed.json`) repository and any
	 * remote repositories, matching the version constraints of the `require`s in the given root project.
	 *
	 * **Note:** This method pre-filters the packages by Core package name to save on logging and processing.
	 *           The best thing to do is pass only Core packages to this method if you can.
	 *
	 * @param PackageInterface[] $packages Packages to process
	 * @return array The processed packages
	 */
	public function restoreCoreDependencyRequires(array $packages): array
	{
		$io = $this->io;

		$io->write(__METHOD__, true, IOInterface::DEBUG);

		// Pre-filter the packages to save on logging and processing
		$corePackages = [];

		foreach ($packages as $package) {
			if (in_array($package->getName(), CoreInstaller::PACKAGE_NAMES) || $this->coreInstaller->supports($package->getType())) {
				$corePackages[] = $package;
			}
		}

		if (empty($corePackages)) {
			$io->write("<info>No Core packages found, skipping processing</info>", true, IOInterface::DEBUG);

			return $packages;
		}

		// Let's go!
		$io->write('<info>Restoring Core package dependencies</info>');

		$corePackagesProcessed = 0;

		foreach ($corePackages as $corePackage) {
			$corePrettyString       = $corePackage->getPrettyString();
			$coreInstallationSource = $corePackage->getInstallationSource();

			if (!empty($coreInstallationSource)) {
				$corePrettyString .= " from $coreInstallationSource";
			}

			$repositoryName = $corePackage->getRepository()->getRepoName();

			$io->write(sprintf('<info>Checking %s from %s</info>', $corePrettyString, $repositoryName), true, IOInterface::VERBOSE);

			if (!$corePackage instanceof Package) {
				$io->write(sprintf("<info>%s is not an instance of %s, skipping</info>", $corePrettyString, Package::class), true, IOInterface::VERBOSE);
				continue;
			}

			if (empty($corePackage->getReplaces())) {
				$io->write(sprintf("<info>No replaces found in %s, skipping</info>", $corePrettyString), true, IOInterface::VERBOSE);
				continue;
			}

			// Process any package earlier than Core 9, or any development version because we can't know what branch
			// it they could be based on
			$earlierThanNine = Comparator::lessThan($corePackage->getVersion(), '9.0.0-dev');
			$developmentVersion = VersionParser::parseStability($corePackage->getVersion()) === 'dev';

			if ($earlierThanNine || $developmentVersion) {
				$io->write(sprintf('<info>Merging replaces into requires for %s</info>', $corePrettyString));

				// Extract its replaces, merge them into the requires, and empty the replaces
				$requires = $corePackage->getRequires();
				$replaces = $corePackage->getReplaces();
				$io->write(sprintf('<info>Found:      %s requires, %s replaces</info>', count($requires), count($replaces)), true, IOInterface::VERBOSE);

				$requires = array_merge($requires, $replaces);
				$replaces = [];

				// Retain Core package replaces
				foreach (CoreInstaller::PACKAGE_NAMES as $packageName) {
					if (isset($requires[$packageName])) {
						$replaces[$packageName] = $requires[$packageName];
						unset($requires[$packageName]);
					}
				}

				$corePackage->setRequires($requires);
				$corePackage->setReplaces($replaces);

				$io->write(sprintf('<info>Updated to: %s requires, %s replaces</info>', count($requires), count($replaces)), true, IOInterface::VERBOSE);

				$corePackagesProcessed++;
			} else {
				$io->write(sprintf("<info>Skipping replace merging for %s; not needed</info>", $corePrettyString), true, IOInterface::VERBOSE);
			}
		}

		$io->write(sprintf("<info>Done: Dependencies restored for %s Core packages</info>", $corePackagesProcessed));

		return $packages;
	}
}
