<?php

namespace Claromentis\Composer;

use Composer\Composer;
use Composer\EventDispatcher\Event;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginEvents;
use Composer\Plugin\PluginInterface;
use Composer\Repository\ComposerRepository;
use Composer\Repository\FilterRepository;
use Composer\Repository\RepositoryFactory;
use Composer\Repository\RepositoryInterface;
use Composer\Repository\RepositoryManager;
use Composer\Util\Filesystem;

/**
 * Claromentis Composer Installer Plugin.
 *
 * Registers Composer installers for Claromentis Core & Modules.
 *
 * @author Chris Andrew <chris@claromentis.com>
 */
class InstallerPlugin implements PluginInterface, EventSubscriberInterface
{
	/**
	 * Package name.
	 */
	public const NAME = 'claromentis/composer-installer-plugin';

	/**
	 * Package version.
	 */
	public const VERSION = '2.1.0';

	/**
	 * Package aliases.
	 */
	public const ALIASES = [
		'claromentis/installer-composer-plugin',
		'claromentis/installer_composer_plugin'
	];

	/**
	 * Canonical Claromentis Composer repository URL.
	 */
	public const REPOSITORY_URL = 'https://packages.claromentis.net';

	/**
	 * @var Composer
	 */
	protected $composer;

	/**
	 * @var IOInterface
	 */
	protected $io;

	/**
	 * @var CoreInstaller
	 */
	protected $coreInstaller;

	/**
	 * @var ModuleInstaller
	 */
	protected $moduleInstaller;

	/**
	 * @var DependencyProcessor
	 */
	protected $dependencyProcessor;

	/**
	 * @see onInit()
	 * @see onDependencySolve()
	 * @return array[]
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			PluginEvents::INIT => ['onInit', -1]
		];
	}

	/**
	 * Activate the plugin by extending Composer.
	 *
	 * - Adds Core & Module installers
	 * - Initializes the Core dependency processor for Core 8.x support
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function activate(Composer $composer, IOInterface $io)
	{
		$this->composer = $composer;
		$this->io       = $io;

		$filesystem        = new Filesystem();
		$repositoryManager = $composer->getRepositoryManager();

		$this->coreInstaller       = new CoreInstaller($io, $composer, $filesystem);
		$this->moduleInstaller     = new ModuleInstaller($io, $composer, $filesystem);

		// TODO: This should ideally no longer require RepositoryManager, and instead operate on a given list of packages
		//       If RepositoryManager is still needed, e.g. for finding packages for type normalization, then make it the decorated one
		//       This could be achieved by passing in Composer itself and always retrieving the RepositoryManager from
		//       Composer, instead of holding on to a single instance
		$this->dependencyProcessor = new DependencyProcessor($io, $repositoryManager, $this->coreInstaller, $this->moduleInstaller);

		$installationManager = $composer->getInstallationManager();
		$installationManager->addInstaller($this->coreInstaller);
		$installationManager->addInstaller($this->moduleInstaller);

		$io->write(sprintf('Composer %s', $composer::getVersion()), true, IOInterface::DEBUG);
		$io->write(sprintf('Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);
	}

	/**
	 * Deactivate the plugin.
	 *
	 * Removes Core & Module installers.
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function deactivate(Composer $composer, IOInterface $io)
	{
		$io->write(sprintf('Deactivating Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);

		$installationManager = $composer->getInstallationManager();
		$installationManager->removeInstaller($this->coreInstaller);
		$installationManager->removeInstaller($this->moduleInstaller);

		// TODO: Restore original RepositoryManager?
	}

	/**
	 * Prepare the plugin for uninstallation.
	 *
	 * @param Composer    $composer
	 * @param IOInterface $io
	 */
	public function uninstall(Composer $composer, IOInterface $io)
	{
		$io->write(sprintf('Uninstalling Claromentis Composer Installer %s', static::VERSION), true, IOInterface::VERBOSE);
	}

	/**
	 * Initialise Composer once it's ready.
	 *
	 * @see prepareComposer()
	 * @param Event $event
	 */
	public function onInit(Event $event)
	{
		$this->io->write(__METHOD__, true, IOInterface::VERBOSE);

		$this->prepareComposer($event);
	}

	/**
	 * Prepares Composer for Claromentis installations.
	 *
	 * - Prepends the Claromentis Composer repository (https://packages.claromentis.net)
	 * - Decorates Claromentis Composer repositories for legacy support (Claromentis 8.13 and earlier)
	 *   - Normalizes Claromentis Composer package types to canonical values
	 *   - Restores legacy Core dependencies from `"replace"` to `"require"`
	 *
	 * @param Event|null $event Optional event that instigated the preparation
	 */
	protected function prepareComposer(Event $event = null)
	{
		// Prepend the Claromentis repository if it's not there already
		// This will be very useful if we ever publish Composer Installer Plugin to Packagist (packagist.org) because
		// we won't need to add the repository manually in composer.json files
		$repositoryManager = $this->composer->getRepositoryManager();
		$this->prependClaromentisRepository($repositoryManager);

		// Replace the repository manager with the new one that consists only of decorated repositories
		$newRepositoryManager = $this->decorateRepositoryManager($repositoryManager);
		$this->composer->setRepositoryManager($newRepositoryManager);
	}

	/**
	 * Add the Claromentis Composer repository to the given repository manager.
	 *
	 * @param RepositoryManager $repositoryManager The repository manager to add the Claromentis Composer repository to
	 */
	protected function prependClaromentisRepository(RepositoryManager $repositoryManager)
	{
		// Bail if the repository is already present
		foreach ($repositoryManager->getRepositories() as $repository) {
			// Unravel filtered repositories (Composer 2)
			if ($repository instanceof FilterRepository) {
				$repository = $repository->getRepository();
			}

			if (
				$repository instanceof ComposerRepository
				&& isset($repository->getRepoConfig()['url'])
				&& $repository->getRepoConfig()['url'] === self::REPOSITORY_URL
			) {
				$this->io->write('<info>Claromentis Packages repository already present, no need to prepend</info>', true, IOInterface::VERBOSE);
				return;
			}
		}

		// Prepend the repository so that we get in front of Packagist
		$this->io->write('<info>Prepending Claromentis Packages repository</info>');

		$repositoryManager->prependRepository(
			$repositoryManager->createRepository('composer', ['url' => self::REPOSITORY_URL], 'Packages')
		);
	}

	/**
	 * Create a new repository manager with its repositories decorated for legacy compatibility.
	 *
	 * @param RepositoryManager $repositoryManager
	 * @return RepositoryManager
	 */
	protected function decorateRepositoryManager(RepositoryManager $repositoryManager): RepositoryManager
	{
		// Create a new repository manager
		$newRepositoryManager = RepositoryFactory::manager(
			$this->io,
			$this->composer->getConfig(),
			$this->composer->getLoop()->getHttpDownloader(),
			$this->composer->getEventDispatcher()
		);

		// Decorate all non-local repositories for legacy package support and add them to the new repository manager
		$decoratedRepositories = $this->decorateRepositories($repositoryManager->getRepositories());

		foreach ($decoratedRepositories as $repository) {
			$newRepositoryManager->addRepository($repository);
		}

		// Ensure that the local repository remains the same
		$newRepositoryManager->setLocalRepository($repositoryManager->getLocalRepository());

		return $newRepositoryManager;
	}

	/**
	 * Decorate Claromentis Composer repositories for Claromentis package type normalization, and for legacy
	 * compatibility with Claromentis Core 8.13 and earlier.
	 *
	 * @param RepositoryInterface[] $repositories
	 * @return RepositoryInterface[]
	 */
	protected function decorateRepositories(array $repositories): array
	{
		$this->io->write("<info>Decorating Claromentis Composer repositories for legacy compatibility</info>", true, IOInterface::VERBOSE);

		$decoratedRepositories = [];

		foreach ($repositories as $repository) {
			if (
				$repository instanceof ComposerRepository
				&& isset($repository->getRepoConfig()['url'])
				&& $repository->getRepoConfig()['url'] === self::REPOSITORY_URL
			) {
				$decoratedRepository = new LegacyRepositoryDecorator(
					$repository,
					$this->dependencyProcessor,
					$this->io,
					$this->composer->getConfig(),
					$this->composer->getLoop()->getHttpDownloader(),
					$this->composer->getEventDispatcher()
				);

				$this->io->write(sprintf("<info>Decorated %s</info>", $decoratedRepository->getRepoName()), true, IOInterface::VERBOSE);

				$decoratedRepositories[] = $decoratedRepository;
			} else {
				$decoratedRepositories[] = $repository;
			}
		}

		$this->logRepositories($decoratedRepositories);

		return $decoratedRepositories;
	}

	protected function logRepositories(array $repositories)
	{
		$this->io->write('<info>Repositories:</info>', true, IOInterface::VERBOSE);

		foreach ($repositories as $repository) {
			$repositoryClass = get_class($repository);
			$repositoryName = method_exists($repository, 'getRepoName') ? $repository->getRepoName() : '';

			$this->io->write("  $repositoryClass $repositoryName", true, IOInterface::VERBOSE);
		}
	}
}
