<?php

namespace Claromentis\Composer;

use Composer\Config;
use Composer\EventDispatcher\EventDispatcher;
use Composer\IO\IOInterface;
use Composer\Repository\ComposerRepository;
use Composer\Repository\RepositoryInterface;
use Composer\Util\HttpDownloader;

/**
 * Composer repository decorator that maintains support for legacy Claromentis packages by normalizing package types,
 * and by moving "replace" dependencies to "require" dependencies for Core 8.13 and earlier.
 */
class LegacyRepositoryDecorator extends ComposerRepository
{
	/**
	 * The decorated repository.
	 *
	 * @var RepositoryInterface
	 */
	private $repository;

	/**
	 * @var DependencyProcessor
	 */
	private $dependencyProcessor;

	/**
	 * @var IOInterface
	 */
	private $io;

	public function __construct(ComposerRepository $repository, DependencyProcessor $dependencyProcessor, IOInterface $io, Config $config, HttpDownloader $httpDownloader, ?EventDispatcher $eventDispatcher = null)
	{
		parent::__construct($repository->getRepoConfig(), $io, $config, $httpDownloader, $eventDispatcher);

		$this->repository          = $repository;
		$this->dependencyProcessor = $dependencyProcessor;
		$this->io                  = $io;
	}

	/**
	 * This override processes legacy Claromentis Composer packages to allows source installations of Claromentis 8 and
	 * earlier with their dependencies.
	 *
	 * This method _MUST_ be idempotent for the Claromentis Composer Installer, and will exhibit undefined behaviour
	 * if it's not. It will be called multiple times throughout the Composer `install`/`update` event lifecycle.
	 *
	 * @inheritDoc
	 */
	public function loadPackages(array $packageNameMap, array $acceptableStabilities, array $stabilityFlags, array $alreadyLoaded = []): array
	{
		// Note that this method is called multiple times due to Composer batching requests for packages, likely for
		// performance reasons
		$this->io->write(__METHOD__, true, IOInterface::DEBUG);

		$result = $this->repository->loadPackages($packageNameMap, $acceptableStabilities, $stabilityFlags, $alreadyLoaded);

		$packages = $result['packages'] ?? [];

		// TODO: Decorated packages should be cloned to avoid messing with the decorated repository's cache
		//       Either we inject this repository as an argument to these methods, or we extract the
		//       DependencyProcessor methods to this class, and remove DependencyProcessor altogether
		//       We also don't want to clone every package, because that would be wasteful; we only want to clone
		//       the packages that we actually modify
		//       Also, packages should only be cloned once, because cloning per-modification would also be wasteful
		$this->dependencyProcessor->normalizePackageTypes($packages);
		$this->dependencyProcessor->restoreCoreDependencyRequires($packages);

		return $result;
	}

	public function getRepoName(): string
	{
		return ucfirst(parent::getRepoName()) . ' (decorated)';
	}
}
