<?php

namespace Claromentis\Composer;

use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Repository\InstalledRepositoryInterface;

/**
 * Claromentis module installer.
 *
 * @author Chris Andrew <chris.andrew@claromentis.com>
 */
class ModuleInstaller extends BaseInstaller
{
	public function supports($packageType): bool
	{
		return in_array($packageType, [
			'claromentis-module',
			'claromentis-module-v7',
			'claromentis-custom-module'
		]);
	}

	public function prepare($type, PackageInterface $package, PackageInterface $prevPackage = null)
	{
		// TODO: Ideally we can remove this override entirely and rely on the base implementation for all packages
		$corePath    = $this->getCoreInstallPath();
		$installPath = $this->getInstallPath($package);

		$coreContainsInstallation = $installPath !== $corePath
			&& $this->util->relativePathStartsWith($corePath, $installPath);

		if ($this->backupsEnabled() && $this->shouldBackup($package)) {
			$this->backup($package);
		}

		// Skip preparation (emptying or git cleaning the installation) if Core contains the module
		if ($coreContainsInstallation) {
			return $this->util->resolve();
		}

		$this->logDebug("Running downloader preparation for module {$package->getPrettyString()} (emptying \"$installPath\" or cleaning changes)");

		return $this->downloadManager->prepare($type, $package, $installPath, $prevPackage);
	}

	public function getInstallPath(PackageInterface $package): string
	{
		$isCustomModule  = $package->getType() === 'claromentis-custom-module';
		$installPath     = $isCustomModule ? $this->getCustomModuleInstallPath() : $this->getModuleInstallPath();
		$moduleDirectory = $this->getApplicationCode($package);

		return $this->util->joinPath($installPath, $moduleDirectory);
	}

	public function getStatefulPaths(PackageInterface $package): array
	{
		return [
			$this->getConfigFileName($package)
		];
	}

	public function copyStatefulPackageFiles(
		PackageInterface             $package,
		string                       $sourcePath,
		string                       $targetPath,
		InstalledRepositoryInterface $installedRepository = null
	): void {
		$configFileName = $this->getConfigFileName($package);

		$source = $this->util->joinPath($sourcePath, $configFileName);

		if (is_file($source)) {
			$target = $this->util->joinPath($targetPath, $configFileName);

			$this->logDebug("Copying module config from $source to $target");

			$this->filesystem->ensureDirectoryExists(dirname($target));
			$this->filesystem->copy($source, $target);
		}
	}

	public function copyVendor(PackageInterface $package, string $targetPath, string $sourcePath = null): void
	{
		// No-op: modules will not contain vendor
	}

	public function shouldBackup(PackageInterface $package): bool
	{
		// Modules will not contain vendor, so we can skip the extra check
		return $this->containsStatefulPaths($package);
	}

	/**
	 * Normalize a Claromentis module's Composer package type to a canonical value.
	 *
	 * No-op if the package type is not a Claromentis Module package type.
	 *
	 * @param Package $package
	 * @return void
	 */
	public function normalizePackageType(Package $package): void
	{
		if (!$this->supports($package->getType())) {
			return;
		}

		if ($package->getType() !== 'claromentis-custom-module') {
			$package->setType('claromentis-module');
		}
	}

	/**
	 * Get the PHP config file name for the given Claromentis module package.
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	protected function getConfigFileName(PackageInterface $package): string
	{
		$applicationCode = $this->getApplicationCode($package);

		return "config_$applicationCode.php";
	}

	/**
	 * Get the application code of the given package.
	 *
	 * This is the package name without its vendor prefix, normalized. It used as the name of the package's install
	 * directory.
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	protected function getApplicationCode(PackageInterface $package): string
	{
		// Discern the project name
		$packageName = $package->getName();
		[, $projectName] = explode('/', $packageName, 2);

		// Remove legacy suffixes
		$projectName = preg_replace("/-(?:src|obf|php5?|php7)$/", '', $projectName);

		// Remove custom prefix
		$projectName = preg_replace("/^custom[\/-]/", '', $projectName);

		return trim($projectName);
	}
}
