<?php

namespace Claromentis\Composer\Tests\System;

use Claromentis\Composer\InstallerPlugin;
use RuntimeException;

/**
 * Abstract test case for using Composer with the Claromentis Composer Installer plugin.
 *
 * Useful for testing Claromentis installations using Composer.
 */
abstract class AbstractComposerInstallerTest extends AbstractComposerTest
{
	/**
	 * Configure a composer.json file to use the current codebase as the installer plugin.
	 *
	 * Configures:
	 * - An absolute path, canonical repository for the Composer Installer Plugin's project root; we're testing the current codebase
	 * - Adds `claromentis/composer-installer-plugin` "require" dependency, using git to read the branch or tag under test
	 * - Merges any given "require" dependencies given
	 *
	 * How very... meta!
	 *
	 * TODO: ComposerJson helper class; perhaps we can use Composer's?
	 *
	 * @param string      $composerJsonPath Path to the composer.json to configure
	 * @param string|null $versionAlias     Composer Installer version alias
	 */
	protected function prepareClaromentisComposerJson(string $composerJsonPath, ?string $versionAlias = null): void
	{
		// Read the composer.json fixture and add the Composer Installer as a path repository
		$composerJson = $this->addComposerInstallerRepository($composerJsonPath, $versionAlias);

		// Share the cache directory across all tests, disable process timeout
		$config = $composerJson['config'] ?? [];
		$config['cache-dir'] = $this->getCacheDirectory();
		$config['process-timeout'] = 0;

		// Set preferred install config, defaulting to source for claromentis/* packages
		// This allows the --prefer-dist flag to set everything to dist, and avoids us
		// installing all vendor dependencies from source, which is slow, and not an
		// approach we ever take in practice
		$config['preferred-install'] = [
			'claromentis/*' => 'source',
			'*' => 'dist'
		];

		// Write the new composer.json fixture
		$composerJson['config'] = $config;

		$this->writeComposerJson($composerJsonPath, $composerJson);
	}

	/**
	 * Add the Composer Installer (this source code) under test to a composer.json as a Composer path repository.
	 *
	 * @param string      $composerJsonPath Path to the composer.json to configure
	 * @param string|null $versionAlias     Composer Installer version alias
	 * @return array The modified composer.json array
	 */
	protected function addComposerInstallerRepository(string $composerJsonPath, ?string $versionAlias = null): array
	{
		$composerJson = $this->readComposerJson($composerJsonPath);

		// TODO: getPluginDirectory(); we could be in the vendor directory of different root project
		$sourceDirectory = $this->getSourceDirectory();
		$pathToPluginSource = $this->filesystem->findShortestPath(dirname($composerJsonPath), $sourceDirectory, true);

		// Add the plugin source path as a Composer repository
		$repositories = $composerJson['repositories'] ?? [];
		array_unshift($repositories, [
			'type' => 'path',
			'url' => $pathToPluginSource,
			'options' => [
				'symlink' => false
			]
		]);

		// Replace the plugin's require to suit the current version under test
		$requires = $composerJson['require'] ?? [];

		// First remove any existing references to the plugin in the fixtures
		$packageNames = $this->getPluginPackageNames();

		assert(in_array(InstallerPlugin::NAME, $packageNames), "I've got a bad feeling about this");

		$requires = array_filter($requires, function ($key) use ($packageNames) {
			return !in_array($key, $packageNames);
		}, ARRAY_FILTER_USE_KEY);

		// Use git to find the current plugin branch/tag version and add it as a dependency
		$pluginVersion = $this->getPluginVersion();

		if ($versionAlias) {
			$pluginVersion = "$pluginVersion as $versionAlias";
		}

		$requires = array_merge([
			InstallerPlugin::NAME => "$pluginVersion",
		], $requires);

		$composerJson['repositories'] = $repositories;
		$composerJson['require'] = $requires;

		$this->writeComposerJson($composerJsonPath, $composerJson);

		return $composerJson;
	}

	/**
	 * Get the list of possible names for the Composer Installer Plugin.
	 *
	 * @return string[]
	 */
	protected function getPluginPackageNames(): array
	{
		return array_merge(
			[InstallerPlugin::NAME],
			InstallerPlugin::ALIASES
		);
	}

	/**
	 * Read the current Composer-normalized version of the plugin version using git.
	 *
	 * If a tag is checked out, it will return that tag. Otherwise, it tries to determine the current branch.
	 *
	 * @return string
	 * @throws RuntimeException If the current branch is HEAD; this likely indicates a detached head state, and we
	 *                          cannot continue with tests in this case.
	 */
	protected function getPluginVersion(): string
	{
		if ($this->processExecutor->execute('git describe --tags --exact-match', $currentTag) === 0) {
			// Normalize the tag without using the result, just to confirm that Composer can use it
			$this->versionParser->normalize($currentTag);

			// Return the tag ("pretty version")
			return trim($currentTag);
		}

		$this->processExecutor->execute('git rev-parse --abbrev-ref HEAD', $currentBranch);

		if ($currentBranch === 'HEAD') {
			throw new RuntimeException('Composer Installer Plugin may be in a detached head state. Please check out a branch or tag in order to run system tests.');
		}

		return $this->versionParser->normalizeBranch(trim($currentBranch));
	}
}