<?php

namespace Claromentis\Composer\Tests\System;

use Claromentis\Composer\InstallerPlugin;
use Claromentis\Composer\Util;
use Composer\Semver\VersionParser;
use Composer\Util\Filesystem;
use Composer\Util\ProcessExecutor;
use InvalidArgumentException;
use PHPUnit\Framework\TestCase;
use RuntimeException;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

/**
 * Abstract test case that provides helpers for running Composer and asserting results.
 */
abstract class AbstractComposerTest extends TestCase
{
	/**
	 * Composer preference for distribution/distributive downloads.
	 *
	 * @var string
	 */
	const PREFER_DIST = 'dist';

	/**
	 * Composer preference for source code downloads.
	 *
	 * @var string
	 */
	const PREFER_SOURCE = 'source';

	/**
	 * Composer flags.
	 *
	 * @var string[]
	 */
	const COMPOSER_FLAGS = [
		self::PREFER_DIST   => '--prefer-dist',
		self::PREFER_SOURCE => '--prefer-source'
	];

	/**
	 * The plugin's package name and aliases in a single array.
	 *
	 * Used to replace plugin entries in the `require`s of `composer.json` fixtures.
	 */
	const PACKAGE_NAMES = [InstallerPlugin::NAME] + InstallerPlugin::ALIASES;

	/**
	 * @var Filesystem
	 */
	protected $filesystem;

	/**
	 * @var ProcessExecutor
	 */
	protected $processExecutor;

	/**
	 * @var Util
	 */
	protected $util;

	/**
	 * @var VersionParser
	 */
	protected $versionParser;

	/**
	 * Directory to run Composer tests within.
	 *
	 * Generated by `getTemporaryTestbedDirectory()`.
	 *
	 * This path needs to be outside the project directory if the tests include installing project itself; otherwise
	 * Composer 1 will complain about installing the project inside itself.
	 *
	 * @see getTestbedDirectory()
	 * @var string
	 */
	protected $tempTestbedDirectory;

	public function __construct(?string $name = null, array $data = [], $dataName = '')
	{
		parent::__construct($name, $data, $dataName);

		$this->filesystem      = new Filesystem();
		$this->processExecutor = new ProcessExecutor();
		$this->util            = new Util($this->filesystem);
		$this->versionParser   = new VersionParser();
	}

	protected function tearDown(): void
	{
		$this->copyAndEmptyTemporaryTestbed();
	}

	/**
	 * Get the source directory of the project.
	 *
	 * Uses `getcwd()` by default, assuming that PHPUnit is run from the project root.
	 *
	 * @return string
	 */
	protected function getSourceDirectory(): string
	{
		return getcwd();
	}

	/**
	 * Get the Composer fixtures directory.
	 *
	 * Contains `composer.json` fixtures organized in sub directories. Defaults to `'tests/fixtures'`.
	 *
	 * @return string
	 */
	protected function getFixturesDirectory(): string
	{
		return $this->getSourceDirectory() . '/tests/fixtures';
	}

	/**
	 * Get the Composer testbed directory.
	 *
	 * This is the directory where Composer tests will be executed. Fixtures are copied from the fixtures directory
	 * to the testbed directory, and then test cases can run Composer commands.
	 *
	 * @return string
	 */
	protected function getTestbedDirectory(): string
	{
		return $this->getSourceDirectory() . '/tests/testbed';
	}

	/**
	 * Get the temporary Composer testbed directory.
	 *
	 * This directory is usually `"/tmp/composer-installer-plugin/tests"`.
	 *
	 * This can be used when running Composer 1 to install a project inside itself with Composer "path" repositories.
	 * Composer 1 otherwise rejects these kinds of installations.
	 *
	 * @return string
	 */
	protected function getTemporaryTestbedDirectory(): string
	{
		if (!isset($this->tempTestbedDirectory)) {
			$this->tempTestbedDirectory = sys_get_temp_dir() . '/composer-installer-plugin/tests';
//			$this->filesystem->ensureDirectoryExists($this->tempTestbedDirectory);
		}

		return $this->tempTestbedDirectory;
	}

	/**
	 * Composer cache directory.
	 *
	 * Used to configure tests to use the same Composer cache directory.
	 *
	 * @return string
	 */
	protected function getCacheDirectory(): string
	{
		$process = $this->runComposer(
			'config cache-dir', $this->getSourceDirectory(), 2, 10, false
		);

		return trim($process->getOutput());
	}

	/**
	 * Prepare a testbed directory for using Composer.
	 *
	 * Copies file paths from tests/fixtures to tests/testbed for testing Composer commands.
	 *
	 * @param string      $fixturesPath     The relative file path to copy from the test fixtures directory.
	 * @param string|null $testbedPath      The relative file path to copy to within the testbed directory. Defaults to the same as `$fixturesPath`.
	 * @param string      $composerJsonPath [optional] The path to the Composer JSON, relative to the fixtures path. Defaults to `'composer.json'`.
	 *                                      Used to update the Composer Installer Plugin's version constraint to match the version under test.
	 * @param string|null $versionAlias     [optional] Composer Installer version alias to use to get Core & Modules to play ball.
	 * @param bool        $useTemporary     [optional] Whether to use a temporary directory for the testbed. Defaults to `false`.
	 *                                      This is useful when running Composer 1 installations that would install a project inside its own source directory.
	 *                                      Composer would otherwise complain about installing the Composer plugin deeper into its own root directory.
	 * @return string The working testbed path. This is where the test fixtures are copied to, and where Composer should be run for testing.
	 */
	protected function prepareTestbed(string $fixturesPath, ?string $testbedPath = null, string $composerJsonPath = 'composer.json', ?string $versionAlias = null, bool $useTemporary = false): string
	{
		$fixturesDirectory = $this->getFixturesDirectory();
		$workingTestbedDirectory = $useTemporary
			? $this->getTemporaryTestbedDirectory()
			: $this->getTestbedDirectory();

		$this->filesystem->ensureDirectoryExists($workingTestbedDirectory);

		if (!is_readable($workingTestbedDirectory) || !is_writable($workingTestbedDirectory)) {
			throw new RuntimeException("Working testbed directory '$workingTestbedDirectory' is not readable/writable");
		}

		// Sanitize the given paths
		$fixturesPath = $this->util->sanitizePath($fixturesPath);
		$testbedPath = isset($testbedPath) ? $this->util->sanitizePath($testbedPath) : $fixturesPath;

		// Join the base directories
		$absoluteFixturesPath = $this->util->joinPath($fixturesDirectory, $fixturesPath);
		$absoluteWorkingTestbedPath = $this->util->joinPath($workingTestbedDirectory, $testbedPath);

		// Clear the working testbed, copy from fixtures to working testbed
		$this->filesystem->emptyDirectory($absoluteWorkingTestbedPath);
		$this->filesystem->copy($absoluteFixturesPath, $absoluteWorkingTestbedPath);

		if ($useTemporary) {
			// Also clear the in-project testbed if the working testbed is a temporary directory;
			// we copy the resulting directories here after tests have finished, and we need a
			// clean target to copy to
			$this->filesystem->emptyDirectory(
				$this->util->joinPath($this->getTestbedDirectory(), $testbedPath)
			);
		}

		// Configure the testbed's composer.json to use the current codebase as the installer
		$workingTestbedComposerJsonPath = $this->util->joinPath($absoluteWorkingTestbedPath, $composerJsonPath);

		$this->prepareClaromentisComposerJson($workingTestbedComposerJsonPath, $versionAlias);

		return $absoluteWorkingTestbedPath;
	}

	/**
	 * Prepare the given composer.json before installation tests.
	 *
	 * Called by prepareTestbed() after preparing the testbed directory.
	 *
	 * Override this to configure Composer projects for each test case, or configure them separately instead.
	 *
	 * @see prepareTestbed()
	 * @param string $composerJsonPath Path to the composer.json to configure
	 * @return void
	 */
	protected function prepareClaromentisComposerJson(string $composerJsonPath): void
	{
		// No-op
	}

	/**
	 * Read a composer.json file.
	 *
	 * TODO: JSON Schema?
	 *
	 * @param string $composerJsonPath
	 * @return mixed
	 */
	protected function readComposerJson(string $composerJsonPath): array
	{
		if (!is_readable($composerJsonPath)) {
			throw new RuntimeException("Unreadable composer.json file '$composerJsonPath'");
		}

		$composerJson = json_decode(file_get_contents($composerJsonPath), true);

		if (!is_array($composerJson)) {
			throw new RuntimeException("Invalid composer.json data from '$composerJsonPath'; expected array");
		}

		return $composerJson;
	}

	/**
	 * Write a composer.json file.
	 *
	 * TODO: JSON Schema?
	 *
	 * @param string $composerJsonPath
	 * @param array  $composerJson
	 */
	protected function writeComposerJson(string $composerJsonPath, array $composerJson): void
	{
		file_put_contents(
			$composerJsonPath,
			json_encode(
				$composerJson,
				JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
			)
		);
	}

	/**
	 * Merge a set of "require" dependencies into a composer.json file.
	 *
	 * @param string $composerJsonPath The composer.json file to update
	 * @param array  $requires The "require" dependencies to merge into the file
	 */
	protected function mergeRequires(string $composerJsonPath, array $requires = []): void
	{
		$composerJson = $this->readComposerJson($composerJsonPath);

		$composerJson['require'] = array_merge(
			$composerJson['require'] ?? [],
			$requires
		);

		$this->writeComposerJson($composerJsonPath, $composerJson);
	}

	/**
	 * Remove "require" dependency matching the given regex pattern.
	 *
	 * @param string $composerJsonPath The composer.json file to update
	 * @param string $pattern          Regex pattern of package names to remove from "require" dependencies.
	 */
	protected function removeRequires(string $composerJsonPath, string $pattern): void
	{
		$composerJson = $this->readComposerJson($composerJsonPath);

		$requires = $composerJson['require'] ?? [];
		$newRequires = [];

		foreach ($requires as $packageName => $versionConstraint) {
			if (!preg_match($pattern, $packageName)) {
				$newRequires[$packageName] = $versionConstraint;
			}
		}

		$composerJson['require'] = $newRequires;

		$this->writeComposerJson($composerJsonPath, $composerJson);
	}

	/**
	 * Run a Composer command.
	 *
	 * Throws an exception if the command fails with a non-zero return code, or if the timeout is reached.
	 *
	 * @param string|string[] $command         Composer command and options. e.g. `"install --ignore-platform-reqs"`
	 * @param string          $directory       The directory within which to run Composer.
	 * @param int             $composerVersion The major version of Composer to run. 1 or 2, defaults to 2.
	 * @param int|null        $timeout         Timeout in seconds. `null` to disable, defaults to `null`.
	 *                                         Execution is considered a failure if the timeout is reached.
	 * @param bool            $log             Whether to log output in a `composer.log` in the working directory. Defaults to `true`.
	 * @return Process The Symfony Process that ran the command. Use this to access exit code and output.
	 * @throws ProcessFailedException If the process failed.
	 */
	protected function runComposer($command, string $directory, int $composerVersion = 2, ?int $timeout = null, bool $log = true): Process
	{
		if (!is_array($command) && !is_string($command)) {
			throw new InvalidArgumentException('$command must be a string or an array of strings');
		}

		if (is_string($command)) {
			$command = array_filter(array_map('trim', explode(' ', $command)));
		}

		if (!in_array($composerVersion, [1, 2])) {
			throw new InvalidArgumentException("Composer $composerVersion not supported; Composer 1 and 2 are the only supported versions");
		}

		$composerBinary = $composerVersion === 2 ? 'composer' : 'composer1';
		$verbosityFlag = $this->isDebug() ? '-vvv' : '-v';
		$command = array_merge([$composerBinary, $verbosityFlag], $command);
		$env     = array_merge(['PATH' => getenv()['PATH'] ?? null], ['COMPOSER_MEMORY_LIMIT' => '-1']);

		$process = new Process($command, $directory, $env);
		$process->setTimeout($timeout);
		$process->start();

		$output = '';
		$exitCode = $process->wait(function ($type, $data) use (&$output) {
			$output .= $data;
//			echo $data; // Debug
		});

		$commandString = implode(' ', $command);

		if ($log) {
			file_put_contents("$directory/composer.log", "\n\$ $commandString:\n$output", FILE_APPEND);
		}

		if ($exitCode !== 0) {
			throw new ProcessFailedException($process);
		}

		return $process;
	}

	/**
	 * Copy any temporary testbed directories to the source testbed directory.
	 */
	protected function copyAndEmptyTemporaryTestbed()
	{
		$temporaryTestbedDirectory = $this->getTemporaryTestbedDirectory();
		$testbedDirectory = $this->getTestbedDirectory();

		if (is_dir($temporaryTestbedDirectory) && !$this->filesystem->isDirEmpty($temporaryTestbedDirectory)) {
			$this->filesystem->copy($temporaryTestbedDirectory, $testbedDirectory);
			$this->filesystem->emptyDirectory($temporaryTestbedDirectory);
		}
	}

	/**
	 * Get the major version from a given version string.
	 *
	 * @param string $version Version string
	 * @return string Major version
	 */
	protected function getMajorVersion(string $version): string
	{
		if (!strlen($version)) {
			throw new InvalidArgumentException('Version string cannot be empty');
		}

		return explode('.', $version)[0];
	}

	/**
	 * Get the major and minor version from a given version string.
	 *
	 * @param string $version Version string
	 * @return string <major>.<minor>
	 */
	protected function getMinorVersion(string $version): string
	{
		if (!strlen($version)) {
			throw new InvalidArgumentException('Version string cannot be empty');
		}

		$versionComponents = explode('.', $version);
		$versionComponents = array_slice($versionComponents, 0, 2);

		return implode('.', array_filter($versionComponents, 'strlen'));
	}

	/**
	 * Use Composer to find the available versions of a given package.
	 *
	 * This can be used to decide which versions of Claromentis to test with installs and upgrades, for example.
	 *
	 * @param string $packageName Composer package name
	 * @return array Available version numbers and/or branch names for a given package
	 */
	protected function getAvailableVersions(string $packageName): array
	{
		// TODO: Use Composer directly here, via the command-line it doesn't seem possible to list all available versions
		//       We want to avoid having to hard-code Claromentis versions to test in this repo wherever possible
		//       Perhaps the only exception would be testing 7.4 to 8.0 upgrades, if we want to go that far
		throw new \BadMethodCallException('Not yet implemented');
	}

	/**
	 * Check whether PHPUnit is running with the --verbose or --debug flags.
	 *
	 * @return bool
	 */
	protected function isDebug(): bool
	{
		return !empty(array_intersect(['-v', '--verbose', '--debug'], $_SERVER['argv']));
	}
}