<?php

namespace Claromentis\Composer\Tests\System;

use Claromentis\Composer\InstallerPlugin;
use Composer\Util\Filesystem;
use Symfony\Component\Process\Process;

/**
 * Tests Claromentis installations and upgrades.
 *
 * ## TODOs:
 *
 * ### Dynamic version selection
 *
 * Ideally, we can use _dynamic_ data providers to:
 *
 * 1. Discover available Claromentis package versions using Composer
 * 2. Select sets of versions to install and update to for testing
 *
 * ### \#3 Test installing Claromentis using Claromentis Core's installer directory.
 *
 * 1. Set up `/installer/composer.json` and `/modules.json`
 * 2. Run `composer install -d installer`
 */
class ClaromentisTest extends AbstractComposerInstallerTest
{
	protected static $initialized = false;

	public function setUp(): void
	{
		parent::setUp();

		if (!static::$initialized) {
			// Empty the testbed directory once before all tests
			// A clean slate is useful for diagnosing failing tests
			$this->filesystem->emptyDirectory($this->getTestbedDirectory());

			static::$initialized = true;
		}
	}

	/**
	 * Test installing, updating and uninstalling the Claromentis codebase using Composer.
	 *
	 * These tests use composer.json fixtures from tests/fixtures to run Composer installations against live systems.
	 * Fixtures are separated by major Claromentis version.
	 *
	 * Composer 1 & 2 are tested with source (Git) and dist (zip archive) installations and updates.
	 *
	 * Claromentis versions to test with are decided by the `getClaromentisVersions()` data provider.
	 *
	 * Composer's `--prefer-source` flag is not used for source installations because:
	 *   - Test fixtures specify source preference for claromentis/* packages by default
	 *   - Composer would install all third-party dependencies from source, which is slow
	 *
	 * @dataProvider getClaromentisVersions()
	 * @param string      $from         Claromentis version to install
	 * @param string|null $to           Claromentis version to update to, if any.
	 * @param bool|null   $preferStable Whether to prefer stable versions for updates.
	 */
	public function testClaromentisInstallations(string $from, ?string $to = null, bool $preferStable = false)
	{
		$fromMajorVersion = $this->getMajorVersion($from);
		$toMajorVersion   = $this->getMajorVersion($to);
		$fromVersionAlias = $fromMajorVersion < 9 ? '1.2.x-dev' : '2.0.x-dev';
		$toVersionAlias   = $toMajorVersion < 9 ? '1.2.x-dev' : '2.0.x-dev';
		$preferStableFlag = $preferStable ? '--prefer-stable' : '';

		foreach ([1, 2] as $composerVersion) {
			foreach ([self::PREFER_DIST, self::PREFER_SOURCE] as $preference) {
				// Prepare testbed directory and file paths
				$preferenceFlag   = $preference === self::PREFER_DIST ? self::COMPOSER_FLAGS[self::PREFER_DIST] : '';
				$sourceFixture    = "claromentis-$fromMajorVersion";
				$targetTestbed    = "composer-$composerVersion-$preference-claromentis-$from-to-$to";
				$testbedPath      = $this->prepareTestbed($sourceFixture, $targetTestbed, 'composer.json', $fromVersionAlias, $composerVersion === 1);
				$composerJsonPath = "$testbedPath/composer.json";
				$composerLockPath = "$testbedPath/composer.lock";
				$vendorPath       = "$testbedPath/application/vendor_core";

//				$statefulFiles = $this->mockStatefulFiles($testbedPath); // TODO: #15 Installation discovery is required to support this

				// Install the $from version
				$this->mergeRequires($composerJsonPath, ['claromentis/framework' => $from]);

				if (is_file($composerLockPath)) {
					// Run `composer install` if the fixture has a lock file
					$this->runComposer("install --ignore-platform-reqs $preferenceFlag --no-dev --no-progress", $testbedPath, $composerVersion);
				} else {
					// Initialize with a `composer update --prefer-stable` when we have no lock file
					// `composer install` does not have a `--prefer-stable` flag that we can use to prefer stable versions
					$this->runComposer("update --ignore-platform-reqs $preferenceFlag $preferStableFlag --no-dev --no-progress", $testbedPath, $composerVersion);
				}

				// Run a second installation for Claromentis versions less than 9, for Core dependency reprocessing
				if ($fromMajorVersion < 9) {
					// We use 'update --lock' to force Composer to reconsider the currently locked and installed
					// packages, now that the the plugin's dependency processing has a chance to run early enough
					$this->runComposer("update --lock --ignore-platform-reqs $preferenceFlag $preferStableFlag --no-dev --no-progress", $testbedPath, $composerVersion);
				}

				$this->assertCoreVersion($from, $testbedPath);
				//$this->assertStatefulFiles($testbedPath, $statefulFiles); // TODO: #15 Installation discovery is required to support this
				$this->assertVendorDependencies($vendorPath);
				$this->assertInstallationPreference($testbedPath, $preference);

				$statefulFiles = $this->mockStatefulFiles($testbedPath);

				// Update from $from version to $to version
				if (isset($to)) {
					$this->mergeRequires($composerJsonPath, [
						'claromentis/framework' => $to,
						InstallerPlugin::NAME   => "{$this->getPluginVersion()} as $toVersionAlias"
					]);

					$this->runComposer("update --ignore-platform-reqs $preferenceFlag $preferStableFlag --no-dev --no-progress", $testbedPath, $composerVersion);

					$this->assertCoreVersion($to, $testbedPath);
					$this->assertStatefulFiles($testbedPath, $statefulFiles);
					$this->assertVendorDependencies($vendorPath);
					$this->assertInstallationPreference($testbedPath, $preference);
				}

				// Uninstall Core & Module dependencies and assert state retention
				$this->removeRequires($composerJsonPath, '/^claromentis\\//');
				$this->mergeRequires($composerJsonPath, [
					InstallerPlugin::NAME => "{$this->getPluginVersion()} as $toVersionAlias"
				]);

				$this->runComposer("update --ignore-platform-reqs $preferenceFlag $preferStableFlag --no-dev --no-progress", $testbedPath, $composerVersion);

				$this->assertStatefulFiles($testbedPath, $statefulFiles);
			}
		}
	}

	/**
	 * Tests legacy Claromentis codebase installations using Composer 1 & 2.
	 *
	 * Legacy installations rely on:
	 * - `installer/composer.json` for an initial installer dependency installation
	 * - `modules.json` in the project root for listing Core & Module requirements
	 *
	 * Claromentis 7.4.x and 8.x only, because `installer` was removed from Core in 9.x.
	 *
	 * Also tests upgrading from this legacy installation to a parent-project based one.
	 */
	public function testLegacyInstallations()
	{
		$composerVersion = 1;
		$installerVersionAlias = '1.2.x-dev';

		foreach ([self::PREFER_DIST, self::PREFER_SOURCE] as $preference) {
			// Prepare testbed directory and file paths
			$preferenceFlag   = $preference === self::PREFER_DIST ? self::COMPOSER_FLAGS[self::PREFER_DIST] : '';
			$sourceFixture    = "core";
			$targetTestbed    = "composer-$composerVersion-$preference-core";
			$testbedPath      = $this->prepareTestbed($sourceFixture, $targetTestbed, 'application/installer/composer.json', $installerVersionAlias, $composerVersion === 1);
			$composerJsonPath = "$testbedPath/application/installer/composer.json";
			$modulesJsonPath  = "$testbedPath/application/modules.json";
			$vendorPath       = $preference === self::PREFER_DIST
				? "$testbedPath/application/vendor_core"
				: "$testbedPath/application/vendor";

			$statefulFiles = $this->mockStatefulFiles($testbedPath);

			$this->prepareClaromentisComposerJson($composerJsonPath, $installerVersionAlias);

			// Composer update twice, once for Core installer dependencies, second time for modules.json includes
			for ($i = 0; $i < 2; $i++) {
				$this->runComposer("update -d application/installer --ignore-platform-reqs $preferenceFlag --no-dev --no-progress", $testbedPath, $composerVersion);
			}

			// Third time for source, to give the Installer Plugin a chance to process Core dependencies
			if ($preference === self::PREFER_SOURCE) {
				$this->prepareClaromentisComposerJson($composerJsonPath, $installerVersionAlias);
				$this->runComposer("update -d application/installer --ignore-platform-reqs $preferenceFlag --no-dev --no-progress", $testbedPath, $composerVersion);
			}

			$this->assertCoreVersion('8.13', $testbedPath);
			$this->assertStatefulFiles($testbedPath, $statefulFiles);
			$this->assertVendorDependencies($vendorPath);
			$this->assertInstallationPreference($testbedPath, $preference);

			// TODO: #16 Instate parent project fixtures (composer.json/composer.lock)
			//       This could actually be performed by the plugin instead of these tests implying a manual step: AUTOMATE!

			// TODO: Update using parent project approach
			//       Assert retention of state

			// Uninstall Core & Module dependencies and assert state retention
			file_put_contents($modulesJsonPath, '{}');
			$this->prepareClaromentisComposerJson($composerJsonPath, $installerVersionAlias);
			$this->runComposer("update -d application/installer --ignore-platform-reqs $preferenceFlag --no-dev --no-progress", $testbedPath, $composerVersion);

			$this->assertStatefulFiles($testbedPath, $statefulFiles);
		}
	}

	/**
	 * Data provider for Claromentis versions to test Composer installations and updates with.
	 *
	 * When adding versions to this, only use `true` for `$preferStable` if all dependencies have properly matching
	 * "stable" (alpha or higher) versions tagged. Otherwise, the wrong versions will be installed.
	 *
	 * @return string[][]
	 */
	public function getClaromentisVersions(): array
	{
		// TODO: Try to acquire these dynamically using Composer
		//       $this->getAvailableVersions("claromentis/claromentis');
		//       This doesn't preclude having specific versions that are always tested, with specific requirements
		return [
			// [string $from, string|null $to, bool|null $preferStable]
//			['8.11.x', '8.12.x', true],
//			['8.12.x', '8.13.x', true],
			['8.13.x', '9.0.x@dev'], // TODO: #16 Could an upgrade like this put parent project composer.json/composer.lock in place above /application?
		];
	}

	/**
	 * Create mock stateful files for a Claromentis installation.
	 *
	 * Used to test stateful file retention during Claromentis package installations and updates.
	 *
	 * @see assertStatefulFiles() to assert that these files exist
	 * @param string $installationPath Claromentis installation path.
	 * @return string[] File paths as keys, and file contents as values, of stateful files added to the installation. Useful for asserting retention later.
	 */
	protected function mockStatefulFiles(string $installationPath): array
	{
		$mockFiles = [
			'application/.env'                                => "CLARO_TEST=\"Mock .env file\"",
			'application/data/mock.txt'                       => 'Mock data text file',
			'application/web/appdata/mock.txt'                => 'Mock appdata text file',
			'application/web/intranet/common/config.php'      => "<?php\n// Mock config.php file",
//			'application/web/intranet/pages/config_pages.php' => "<?php\n// Mock pages config file" // TODO: #15 Installation discovery is required for this to pass on initial installation
		];

		foreach ($mockFiles as $filePath => $fileContents) {
			$absoluteFilePath = "$installationPath/$filePath";
			$this->filesystem->ensureDirectoryExists(dirname($absoluteFilePath));
			file_put_contents($absoluteFilePath, $fileContents);
		}

		return $mockFiles;
	}

	/**
	 * Assert the existence and contents of stateful files for a Claromentis installation.
	 *
	 * Useful in tandem with the return value of `mockStatefulFiles()`.
	 *
	 * @see mockStatefulFiles() to setup mock files to use with this method
	 * @param string $installationPath Claromentis installation path.
	 * @param array  $files            File paths as keys, and file contents as values, of files to assert existing in the installation.
	 * @return void
	 */
	protected function assertStatefulFiles(string $installationPath, array $files)
	{
		foreach ($files as $filePath => $fileContents) {
			$this->assertFileExists("$installationPath/$filePath", "Stateful file path should exist");
			$this->assertEquals($fileContents, file_get_contents("$installationPath/$filePath"), "Stateful file contents should stay the same");
		}
	}

	/**
	 * Assert that a Claromentis version is installed at a particular directory path.
	 *
	 * Versions constraints can vary in specificity; they can be Composer version constraints.
	 *
	 * - Reads the Core version file to determine the version of the code
	 * - TODO: Uses Git, if a repository is available, to check the current branch/tag
	 *
	 * TODO: Use Composer's version comparator to check the given constraint as a constraint, rather
	 *       than a cheeky "assertStringStartsWith()"
	 *
	 * @param string $versionConstraint Composer version constraint to check
	 * @param string $installationPath  Claromentis installation path to check
	 * @param string $message           Assertion failure message
	 */
	protected function assertCoreVersion(
		string $versionConstraint,
		string $installationPath,
		string $message = 'Core version mismatch'
	) {
		$minorVersion = $this->getMinorVersion($versionConstraint);
		$versionTxtPath = "$installationPath/application/web/intranet/setup/_init/version.txt";

		$this->assertFileExists($versionTxtPath, "Core's version.txt file should exist at $versionTxtPath");

		$versionTxt = file($versionTxtPath);

		$this->assertStringStartsWith($minorVersion, $versionTxt[1] ?? '', $message);
	}

	/**
	 * Assert that Claromentis was installed with the expected installation preference.
	 *
	 * - Git repositories should exist for a `'source'` preference (`--prefer-source`)
	 * - Git repositories should NOT exist for a `'dist'` preference (`--prefer-dist`)
	 *
	 * Does nothing if a `$preference` is not given.
	 *
	 * @param string      $installationPath Claromentis installation path.
	 * @param string|null $preference       Composer installation preference.
	 * @return void
	 */
	protected function assertInstallationPreference(string $installationPath, ?string $preference = null)
	{
		$gitDirectoryPath = "$installationPath/application/.git";
		$gitDirectoryExists = is_dir($gitDirectoryPath);

		if ($preference === self::PREFER_SOURCE) {
			$this->assertTrue($gitDirectoryExists, "Git repository should be cloned for source installation at $gitDirectoryPath");
		}

		if ($preference === self::PREFER_DIST) {
			$this->assertFalse($gitDirectoryExists, "Git repository should NOT be cloned for dist installation at $gitDirectoryPath");
		}
	}

	/**
	 * Check that vendor dependencies have been installed properly.
	 *
	 * Checks `vendor_core` for `psr/log` by default, but `$packageNames` can be used to check for more packages.
	 *
	 * @param string   $vendorPath   Vendor path
	 * @param string[] $packageNames Vendor packages to check for
	 */
	protected function assertVendorDependencies(string $vendorPath, array $packageNames = ['psr/log'])
	{
		foreach ($packageNames as $packageName) {
			$this->assertDirectoryExists(
				"$vendorPath/$packageName",
				"Vendor dependency $packageName should be installed"
			);
		}
	}
}