<?php

namespace Claromentis\Composer;

use Composer\Config;
use Composer\Installer\BinaryInstaller;
use Composer\IO\IOInterface;
use Composer\Package\PackageInterface;
use Composer\Plugin\PluginInterface;
use Composer\Util\Filesystem;
use function React\Promise\resolve as reactResolve;

/**
 * Utility service for Claromentis Composer installers.
 *
 * TODO: UNIT TESTS
 *
 * @author Chris Andrew <chris@claromentis.com>
 */
class Util
{
	/**
	 * @var Filesystem
	 */
	private $filesystem;

	public function __construct(Filesystem $filesystem)
	{
		$this->filesystem = $filesystem;
	}

	/**
	 * Normalize a relative file path.
	 *
	 * Forwards to Composer's path normalization method.
	 *
	 * @see Filesystem::normalizePath()
	 * @param string $path The file path to normalize.
	 * @return string
	 */
	public function normalizePath(string $path): string
	{
		return $this->filesystem->normalizePath($path);
	}

	/**
	 * Join the given path segments and normalize the resulting path.
	 *
	 * @param string ...$segments File path segments to join and normalize.
	 * @return string The joined and normalized file path.
	 */
	public function joinPath(string ...$segments): string
	{
		// Trim all segments
		$segments = array_map(function ($segment, $index) {
			return $index === 0
				? $this->trimAbsolutePath($segment)
				: $this->trimRelativePath($segment);
		}, $segments, array_keys($segments));

		// Join and normalize
		return $this->normalizePath(join('/', $segments));
	}

	/**
	 * Trim whitespace and slashes from the given absolute file path.
	 *
	 * Preserves leading slashes, unlike trimRelativePath().
	 *
	 * @param string $path Absolute path to trim.
	 * @return string The trimmed file path.
	 */
	public function trimAbsolutePath(string $path): string
	{
		return ltrim(rtrim($path, ' /\\'), ' ');
	}

	/**
	 * Trim whitespace and slashes from the given relative file path.
	 *
	 * @param string $path Relative file path to trim.
	 * @return string The trimmed file path.
	 */
	public function trimRelativePath(string $path): string
	{
		return trim($path, ' /\\');
	}

	/**
	 * Replace the prefix of a relative file path.
	 *
	 * @param string $prefix      The prefix to be replaced.
	 * @param string $replacement The string to replace the prefix with.
	 * @param string $subject     The string whose prefix will be replaced.
	 * @return string The $subject string with its prefix replaced.
	 */
	public function replaceRelativePathPrefix(string $prefix, string $replacement, string $subject): string
	{
		$prefix      = $this->trimRelativePath($prefix);
		$replacement = $this->trimRelativePath($replacement);
		$subject     = $this->trimRelativePath($subject);

		$escapedPrefix = preg_quote($prefix, '/');

		return preg_replace("/^$escapedPrefix/", $replacement, $subject, 1);
	}

	/**
	 * Determine whether a file path intersects - is contained within - another path.
	 *
	 * Useful for short-circuiting some installer operations.
	 *
	 * @param string $prefix      The path prefix that `$subjectPath` may begin with.
	 * @param string $subjectPath The path to check.
	 * @return bool
	 */
	public function relativePathStartsWith(string $prefix, string $subjectPath): bool
	{
		$subjectPath = $this->trimRelativePath($subjectPath);
		$prefix      = $this->trimRelativePath($prefix);

		return strpos($subjectPath, $prefix) === 0;
	}

	/**
	 * Sanitize a file path so that it only contains acceptable characters for Linux or Windows filesystems.
	 *
	 * @param string $pathSegment The path segment to sanitize.
	 * @return string The sanitized path segment.
	 */
	public function sanitizePath(string $pathSegment): string
	{
		return preg_replace('/[^A-Za-z0-9\\/\\\._]+/', '-', $pathSegment);
	}

	/**
	 * Sanitize a package name so that it won't create nested directories when used as a file path.
	 *
	 * @param string $packageName The package name to sanitize.
	 * @return string The sanitized package name.
	 */
	public function sanitizePackageName(string $packageName): string
	{
		return preg_replace('/[^A-Za-z0-9._]+/', '-', $packageName);
	}

	/**
	 * Get a package name sanitized for use as a file path.
	 *
	 * - Strips out slashes to avoid creating nested directories
	 * - Includes source reference for uniqueness
	 *
	 * @param PackageInterface $package
	 * @return string
	 */
	public function getSanitizedPackageName(PackageInterface $package): string
	{
		$packageName = $package->getPrettyName();
		$packageVersion = $package->getFullPrettyVersion(true);

		return $this->sanitizePackageName("$packageName-$packageVersion");
	}

	/*
	 * Backwards-compatible equivalent of React's promise `resolve()` function.
	 *
	 * Abstracts direct usage of the React library.
	 *
	 * Composer 1's service methods return `null` in many cases, whereas Composer 2 _may_ return a React promise.
	 *
	 * To unify the approach for both, this resolve() method will use React if it's available, or return a "fake"
	 * Promise if not.
	 *
	 * @see React\Promise\resolve() for the function being mimicked
	 * @param mixed ...$arguments Optional promises to resolve
	 * @return \React\Promise\PromiseInterface|\Claromentis\Composer\Promise A promise or promise-like object
	 */
	public function resolve(...$arguments)
	{
		if (function_exists('React\Promise\resolve')) {
			if (count($arguments) === 0) {
				$arguments = [null];
			}

			return reactResolve(...$arguments);
		}

		return new \Claromentis\Composer\Promise(); // FQCN for explicitness
	}

	/**
	 * Check whether the current Composer API is less than version 2.
	 *
	 * @return bool
	 */
	public function composer1Api(): bool
	{
		return version_compare(PluginInterface::PLUGIN_API_VERSION, '2.0.0', '<');
	}

	/**
	 * Create a default Composer binary installer.
	 *
	 * @param IOInterface $io
	 * @param Config      $config
	 * @return BinaryInstaller
	 */
	public function createBinaryInstaller(IOInterface $io, Config $config): BinaryInstaller
	{
		return new BinaryInstaller(
			$io,
			rtrim($config->get('bin-dir'), '/'),
			$config->get('bin-compat'),
			$this->filesystem
		);
	}
}
