<?php

namespace Claromentis\Composer\Tests\Unit;

use Claromentis\Composer\Promise;
use Claromentis\Composer\Util;
use Composer\Util\Filesystem;
use PHPUnit\Framework\TestCase;
use React\Promise\PromiseInterface;

class UtilTest extends TestCase
{
	protected $util;

	public function setUp(): void
	{
		$filesystem = $this->getMockBuilder(Filesystem::class)
			->setMethodsExcept(['normalizePath'])
			->disableOriginalConstructor()
			->getMock();
		$this->util = new Util($filesystem);
	}

	/**
	 * Given a set of path segments
	 * when I attempt to join them together
	 * they should join in a normalized way, whether absolute or relative
	 *
	 * @dataProvider joinPathProvider
	 * @param array  $segments Path segments to join
	 * @param string $expected Expected resulting path
	 */
	public function testJoinPath(array $segments, string $expected)
	{
		$this->assertSame($expected, $this->util->joinPath(...$segments));
	}

	/**
	 * Data provider for different path join test cases.
	 *
	 * @return array[]
	 */
	public function joinPathProvider()
	{
		return [
			'Clean path segments'        => [
				['workspace', 'composer-installer', 'tests', 'testbed'],
				'workspace/composer-installer/tests/testbed'
			],
			'Dirty path segments'        => [
				['workspace/', '/composer-installer', 'tests//', '//testbed'],
				'workspace/composer-installer/tests/testbed'
			],
			'Very dirty path segments'   => [
				['  workspace/', '/composer-installer ', ' //tests// ', '//testbed////'],
				'workspace/composer-installer/tests/testbed'
			],
			'Unix relative directory' => [
				['workspace', 'core', 'installer/', '/../', 'vendor'],
				'workspace/core/vendor'
			],
			'Unix upwards relative directory' => [
				['../', 'web', 'intranet'],
				'../web/intranet'
			],
			'Unix absolute directory'    => [
				['/home', 'claromentis', 'composer-installer'],
				'/home/claromentis/composer-installer'
			],
			'Windows relative directory' => [
				['C:\\', 'Claromentis', 'application\\', '\\installer', '..\\', 'vendor'],
				'C:/Claromentis/application/vendor'
			],
			'Windows upwards relative directory' => [
				['..\\', 'web', 'intranet'],
				'../web/intranet'
			],
			'Windows absolute directory' => [
				['C:\\', 'Claromentis', ' \\application'],
				'C:/Claromentis/application'
			]
		];
	}

	/**
	 * Test resolving a promise with Util::resolve().
	 *
	 * This test is split in two depending on the availability of `react/promise` (Composer 1 vs Composer 2).
	 *
	 *
	 * @return void
	 */
	public function testResolve()
	{
		if (function_exists('React\Promise\resolve')) {
			// Tests that we can only assert if React\Promise\resolve() is available
			$this->assertInstanceOf(PromiseInterface::class, $this->util->resolve());
			$this->assertInstanceOf(PromiseInterface::class, $this->util->resolve(null));

			$this->util->resolve('test')->then(function ($value) {
				$this->assertSame('test', $value);
			});
		} else {
			// Tests that we can only assert if React\Promise\resolve() is NOT available
			$this->assertInstanceOf(Promise::class, $this->util->resolve());
			$this->assertInstanceOf(Promise::class, $this->util->resolve(null));

			$this->util->resolve()->then(function() {
				$this->assertTrue(true);
			});
		}
	}
}