var moduleName = 'commentsModule';
var module;

try {
    module = angular.module(moduleName);
} catch(err) {
    // named module does not exist, so create one
    module = angular.module(moduleName, ['ngResource', 'fileupload', 'doclink', 'link']);
}

angular.module(moduleName)
    .factory('CommentsCollection', ['$resource', 'Comment', function($resource, Comment) {
        // set up a $resource for the all endpoint
        var resource = $resource('/api/comments/discussions', {}, {
            fetch: {
                method: 'GET',
                isArray: false,
                params: {
                    objects: '@objects',
                    limit: '@limit',
                    offset: '@offset',
                    sortLatestTop: '@sortLatestTop',
                },
                paramSerializer: function(params) {
                    var getRequestParameters = [];

                    angular.forEach(params, function(value, key) {
                        // ex. objects[discuss]=1,2,3&
                        if (key === 'objects' && angular.isObject(value)) {
                            angular.forEach(value, function(objVal, objKey) {
                                if (angular.isArray(objVal)) {
                                    getRequestParameters.push(encodeURIComponent('objects[' + objKey + ']') + '=' + objVal.join(','));
                                } else {
                                    getRequestParameters.push(encodeURIComponent('objects[' + objKey + ']') + '=' + encodeURIComponent(objVal));
                                }
                            });
                        } else if (angular.isArray(value)) {
                            angular.forEach(value, function(v) {
                                getRequestParameters.push(encodeURIComponent(key) + '=' + encodeURIComponent(v));
                            });
                        } else if (value !== undefined && value !== null) {
                            getRequestParameters.push(encodeURIComponent(key) + '=' + encodeURIComponent(value));
                        }
                    });

                    return getRequestParameters.join('&');
                }
            }
        });

        /**
         * Get collection of the comments filtered by columns aggregation_id and object_id
         *
         * @param objectIds
         * @param limit
         * @param offset
         * @param sortLatestTop
         * @constructor
         */
        function CommentsCollection(objectIds, limit, offset, sortLatestTop) {
            if (objectIds !== undefined) {
                this.objects = {
                    discuss: objectIds
                };
            }

            this.offset = offset;
            this.limit = limit;
            this.sortLatestTop = true;

            if (sortLatestTop !== undefined)
                this.sortLatestTop = sortLatestTop;

            this.comments = []; // Comment instances
            this.paging = {};   // { offset, limit, total, prev, next }
        }

        CommentsCollection.prototype = {
            /**
             * Fetch page of comments from the server, wrap each in a Comment,
             * and then call back with (err, this).
             */
            loadPage: function(callback) {
                var self = this;
                resource.fetch(
                    {
                        objects: this.objects,
                        limit: this.limit,
                        offset: this.offset,
                        sortLatestTop: this.sortLatestTop,
                    },
                    function (response) {
                        // clear any existing
                        self.comments.length = 0;

                        // wrap raw data items in Comment model
                        angular.forEach(response.data, function(raw) {
                            var comment = new Comment();
                            comment.load(raw);
                            self.comments.push(comment);
                        });

                        // copy over paging
                        self.paging = response.paging || {};
                        callback && callback(null, self);
                    },
                    function (err) {
                        callback && callback(err);
                    }
                );
            },

            /**
             * Get the next page
             */
            loadNextPage: function(callback) {
                if (!this.paging.next) {
                    return callback && callback(new Error('No next page'));
                }

                // extract offset, limit from next URL (simple regex)
                var regex = /[?&]offset=(\d+)/.exec(this.paging.next);

                if (regex)
                    this.offset = parseInt(regex[1], 10);

                this.loadPage(callback);
            },

            setOffset: function (offset) {
                this.offset = offset;
            }
        };

        return CommentsCollection;
    }]);
